/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.internal.BufferedBlock;
import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.hapi.node.base.Timestamp;
import com.hedera.node.app.blocks.impl.streaming.BlockBufferIO;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnectionManager;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockBufferConfig;
import com.hedera.node.config.data.BlockStreamConfig;
import com.hedera.node.config.types.BlockStreamWriterMode;
import com.hedera.node.config.types.StreamMode;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.Instant;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Singleton
public class BlockBufferService {
    private static final Logger logger = LogManager.getLogger(BlockBufferService.class);
    private static final int DEFAULT_BUFFER_SIZE = 150;
    private static final Duration DEFAULT_WORKER_INTERVAL = Duration.ofSeconds(1L);
    private final ConcurrentMap<Long, BlockState> blockBuffer = new ConcurrentHashMap<Long, BlockState>();
    private final AtomicLong earliestBlockNumber = new AtomicLong(Long.MIN_VALUE);
    private final AtomicLong highestAckedBlockNumber = new AtomicLong(Long.MIN_VALUE);
    private final ScheduledExecutorService execSvc = Executors.newSingleThreadScheduledExecutor();
    private final AtomicReference<CompletableFuture<Boolean>> backpressureCompletableFutureRef = new AtomicReference();
    private final AtomicLong lastProducedBlockNumber = new AtomicLong(-1L);
    private final ConfigProvider configProvider;
    private BlockNodeConnectionManager blockNodeConnectionManager;
    private final BlockStreamMetrics blockStreamMetrics;
    private final boolean grpcStreamingEnabled;
    private final boolean backpressureEnabled;
    private Instant lastRecoveryActionTimestamp = Instant.MIN;
    private PruneResult lastPruningResult = PruneResult.NIL;
    private boolean awaitingRecovery = false;
    private final BlockBufferIO bufferIO;
    private final AtomicBoolean isStarted = new AtomicBoolean(false);

    @Inject
    public BlockBufferService(@NonNull ConfigProvider configProvider, @NonNull BlockStreamMetrics blockStreamMetrics) {
        this.configProvider = configProvider;
        this.blockStreamMetrics = blockStreamMetrics;
        this.bufferIO = new BlockBufferIO(this.bufferDirectory());
        BlockStreamConfig blockStreamConfig = (BlockStreamConfig)configProvider.getConfiguration().getConfigData(BlockStreamConfig.class);
        this.grpcStreamingEnabled = blockStreamConfig.writerMode() != BlockStreamWriterMode.FILE;
        this.backpressureEnabled = blockStreamConfig.streamMode() == StreamMode.BLOCKS && this.grpcStreamingEnabled;
    }

    public void start() {
        if (this.grpcStreamingEnabled && this.isStarted.compareAndSet(false, true)) {
            this.loadBufferFromDisk();
            this.scheduleNextWorkerTask();
        }
    }

    public void shutdown() {
        this.execSvc.shutdownNow();
        this.blockNodeConnectionManager.shutdown();
    }

    private Duration workerTaskInterval() {
        Duration interval = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).workerInterval();
        if (interval.isNegative() || interval.isZero()) {
            return DEFAULT_WORKER_INTERVAL;
        }
        return interval;
    }

    private Duration blockBufferTtl() {
        return ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).blockTtl();
    }

    private Duration blockPeriod() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).blockPeriod();
    }

    private double actionStageThreshold() {
        double threshold = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).actionStageThreshold();
        return Math.max(0.0, threshold);
    }

    private Duration actionGracePeriod() {
        Duration gracePeriod = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).actionGracePeriod();
        return gracePeriod == null || gracePeriod.isNegative() ? Duration.ZERO : gracePeriod;
    }

    private double recoveryThreshold() {
        double threshold = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).recoveryThreshold();
        return Math.max(0.0, threshold);
    }

    private boolean isBufferPersistenceEnabled() {
        return ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).isBufferPersistenceEnabled();
    }

    private String bufferDirectory() {
        return ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).bufferDirectory();
    }

    private int blockItemBatchSize() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).blockItemBatchSize();
    }

    public void setBlockNodeConnectionManager(@NonNull BlockNodeConnectionManager blockNodeConnectionManager) {
        this.blockNodeConnectionManager = Objects.requireNonNull(blockNodeConnectionManager, "blockNodeConnectionManager must not be null");
    }

    public void openBlock(long blockNumber) {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        if (blockNumber < 0L) {
            throw new IllegalArgumentException("Block number must be non-negative");
        }
        BlockState existingBlock = (BlockState)this.blockBuffer.get(blockNumber);
        if (existingBlock != null && existingBlock.isBlockProofSent()) {
            logger.error("Attempted to open block {}, but this block already has the block proof sent", (Object)blockNumber);
            throw new IllegalStateException("Attempted to open block " + blockNumber + ", but this block already has the block proof sent");
        }
        BlockState blockState = new BlockState(blockNumber);
        this.blockBuffer.put(blockNumber, blockState);
        this.earliestBlockNumber.updateAndGet(current -> current == Long.MIN_VALUE ? blockNumber : Math.min(current, blockNumber));
        this.lastProducedBlockNumber.updateAndGet(old -> Math.max(old, blockNumber));
        this.blockStreamMetrics.recordLatestBlockOpened(blockNumber);
        this.blockStreamMetrics.recordBlockOpened();
        this.blockNodeConnectionManager.openBlock(blockNumber);
    }

    public void addItem(long blockNumber, @NonNull BlockItem blockItem) {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        Objects.requireNonNull(blockItem, "blockItem must not be null");
        BlockState blockState = this.getBlockState(blockNumber);
        if (blockState == null) {
            throw new IllegalStateException("Block state not found for block " + blockNumber);
        }
        blockState.addItem(blockItem);
    }

    public void closeBlock(long blockNumber) {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        BlockState blockState = this.getBlockState(blockNumber);
        if (blockState == null) {
            throw new IllegalStateException("Block state not found for block " + blockNumber);
        }
        this.blockStreamMetrics.recordBlockClosed();
        blockState.closeBlock();
    }

    @Nullable
    public BlockState getBlockState(long blockNumber) {
        BlockState block = (BlockState)this.blockBuffer.get(blockNumber);
        if (block == null) {
            this.blockStreamMetrics.recordBlockMissing();
        }
        return block;
    }

    public boolean isAcked(long blockNumber) {
        return this.highestAckedBlockNumber.get() >= blockNumber;
    }

    public void setLatestAcknowledgedBlock(long blockNumber) {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        long highestBlock = this.highestAckedBlockNumber.updateAndGet(current -> Math.max(current, blockNumber));
        this.blockStreamMetrics.recordLatestBlockAcked(highestBlock);
    }

    public long getLastBlockNumberProduced() {
        return this.lastProducedBlockNumber.get();
    }

    public long getLowestUnackedBlockNumber() {
        return this.highestAckedBlockNumber.get() == Long.MIN_VALUE ? -1L : this.highestAckedBlockNumber.get() + 1L;
    }

    public long getHighestAckedBlockNumber() {
        return this.highestAckedBlockNumber.get() == Long.MIN_VALUE ? -1L : this.highestAckedBlockNumber.get();
    }

    public long getEarliestAvailableBlockNumber() {
        return this.earliestBlockNumber.get() == Long.MIN_VALUE ? -1L : this.earliestBlockNumber.get();
    }

    public void ensureNewBlocksPermitted() {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        CompletableFuture<Boolean> cf = this.backpressureCompletableFutureRef.get();
        if (cf != null && !cf.isDone()) {
            try {
                logger.error("!!! Block buffer is saturated; blocking thread until buffer is no longer saturated");
                long startMs = System.currentTimeMillis();
                boolean bufferAvailable = cf.get();
                long durationMs = System.currentTimeMillis() - startMs;
                logger.warn("Thread was blocked for {}ms waiting for block buffer to free space", (Object)durationMs);
                if (!bufferAvailable) {
                    logger.warn("Block buffer still not available to accept new blocks; reentering wait...");
                    this.ensureNewBlocksPermitted();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                logger.warn("Failed to wait for block buffer to be available", (Throwable)e);
            }
        }
    }

    private void loadBufferFromDisk() {
        List<BufferedBlock> blocks;
        if (!this.isBufferPersistenceEnabled()) {
            return;
        }
        try {
            blocks = this.bufferIO.read();
        }
        catch (IOException e) {
            logger.error("Failed to read block buffer from disk!", (Throwable)e);
            return;
        }
        if (blocks.isEmpty()) {
            logger.info("Block buffer will not be repopulated (reason: no blocks found on disk)");
            return;
        }
        int batchSize = this.blockItemBatchSize();
        logger.info("Block buffer is being restored from disk (blocksRead: {})", (Object)blocks.size());
        for (BufferedBlock bufferedBlock : blocks) {
            BlockState block = new BlockState(bufferedBlock.blockNumber());
            bufferedBlock.block().items().forEach(block::addItem);
            block.processPendingItems(batchSize);
            if (bufferedBlock.isProofSent()) {
                for (int i = 0; i < block.numRequestsCreated(); ++i) {
                    block.markRequestSent(i);
                }
            }
            Timestamp closedTimestamp = bufferedBlock.closedTimestamp();
            Instant closedInstant = Instant.ofEpochSecond(closedTimestamp.seconds(), closedTimestamp.nanos());
            block.closeBlock(closedInstant);
            if (bufferedBlock.isAcknowledged()) {
                this.setLatestAcknowledgedBlock(bufferedBlock.blockNumber());
            }
            if (this.blockBuffer.putIfAbsent(bufferedBlock.blockNumber(), block) == null) continue;
            logger.debug("Block {} was read from disk but it was already in the buffer; ignoring block from disk", (Object)bufferedBlock.blockNumber());
        }
    }

    public void persistBuffer() {
        if (!this.grpcStreamingEnabled || !this.isBufferPersistenceEnabled()) {
            return;
        }
        List<BlockState> blocksToPersist = this.blockBuffer.values().stream().filter(BlockState::isClosed).toList();
        int batchSize = this.blockItemBatchSize();
        blocksToPersist.forEach(block -> block.processPendingItems(batchSize));
        try {
            this.bufferIO.write(blocksToPersist, this.highestAckedBlockNumber.get());
            logger.info("Block buffer persisted to disk (blocksWritten: {})", (Object)blocksToPersist.size());
        }
        catch (IOException | RuntimeException e) {
            logger.error("Failed to write block buffer to disk!", (Throwable)e);
        }
    }

    @NonNull
    private PruneResult pruneBuffer() {
        Duration ttl = this.blockBufferTtl();
        Instant cutoffInstant = Instant.now().minus(ttl);
        Iterator it = this.blockBuffer.entrySet().iterator();
        long highestBlockAcked = this.highestAckedBlockNumber.get();
        Duration blockPeriod = this.blockPeriod();
        long idealMaxBufferSize = blockPeriod.isZero() || blockPeriod.isNegative() ? 150L : ttl.dividedBy(blockPeriod);
        int numPruned = 0;
        int numChecked = 0;
        int numPendingAck = 0;
        long newEarliestBlock = Long.MAX_VALUE;
        long newLatestBlock = Long.MIN_VALUE;
        while (it.hasNext()) {
            Map.Entry blockEntry = it.next();
            BlockState block = (BlockState)blockEntry.getValue();
            long blockNum = (Long)blockEntry.getKey();
            ++numChecked;
            Instant closedTimestamp = block.closedTimestamp();
            if (closedTimestamp == null) continue;
            if (!this.backpressureEnabled) {
                if (closedTimestamp.isBefore(cutoffInstant)) {
                    it.remove();
                    ++numPruned;
                    continue;
                }
                if (block.blockNumber() > highestBlockAcked) {
                    ++numPendingAck;
                }
                newEarliestBlock = Math.min(newEarliestBlock, blockNum);
                newLatestBlock = Math.max(newLatestBlock, blockNum);
                continue;
            }
            if (block.blockNumber() <= highestBlockAcked) {
                if (closedTimestamp.isBefore(cutoffInstant)) {
                    it.remove();
                    ++numPruned;
                    continue;
                }
                newEarliestBlock = Math.min(newEarliestBlock, blockNum);
                newLatestBlock = Math.max(newLatestBlock, blockNum);
                continue;
            }
            ++numPendingAck;
            newEarliestBlock = Math.min(newEarliestBlock, blockNum);
            newLatestBlock = Math.max(newLatestBlock, blockNum);
        }
        newEarliestBlock = newEarliestBlock == Long.MAX_VALUE ? -1L : newEarliestBlock;
        newLatestBlock = newLatestBlock == Long.MIN_VALUE ? -1L : newLatestBlock;
        this.earliestBlockNumber.set(newEarliestBlock);
        this.blockStreamMetrics.recordNumberOfBlocksPruned(numPruned);
        this.blockStreamMetrics.recordBufferOldestBlock(newEarliestBlock);
        this.blockStreamMetrics.recordBufferNewestBlock(newLatestBlock);
        return new PruneResult(idealMaxBufferSize, numChecked, numPendingAck, numPruned, newEarliestBlock, newLatestBlock);
    }

    private void checkBuffer() {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        PruneResult pruningResult = this.pruneBuffer();
        PruneResult previousPruneResult = this.lastPruningResult;
        this.lastPruningResult = pruningResult;
        logger.debug("Block buffer status: idealMaxBufferSize={}, blocksChecked={}, blocksPruned={}, blocksPendingAck={}, blockRange=[{}, {}], saturation={}%", (Object)pruningResult.idealMaxBufferSize, (Object)pruningResult.numBlocksChecked, (Object)pruningResult.numBlocksPruned, (Object)pruningResult.numBlocksPendingAck, pruningResult.oldestBlockNumber == -1L ? "-" : Long.valueOf(pruningResult.oldestBlockNumber), pruningResult.newestBlockNumber == -1L ? "-" : Long.valueOf(pruningResult.newestBlockNumber), (Object)pruningResult.saturationPercent);
        this.blockStreamMetrics.recordBufferSaturation(pruningResult.saturationPercent);
        double actionStageThreshold = this.actionStageThreshold();
        if (previousPruneResult.saturationPercent < actionStageThreshold) {
            if (pruningResult.isSaturated) {
                this.blockStreamMetrics.recordBackPressureActive();
                this.enableBackPressure(pruningResult);
                this.switchBlockNodeIfPermitted(pruningResult);
            } else if (pruningResult.saturationPercent >= actionStageThreshold) {
                this.blockStreamMetrics.recordBackPressureActionStage();
                this.switchBlockNodeIfPermitted(pruningResult);
            } else {
                this.blockStreamMetrics.recordBackPressureDisabled();
            }
        } else if (!previousPruneResult.isSaturated && previousPruneResult.saturationPercent >= actionStageThreshold) {
            if (pruningResult.isSaturated) {
                this.blockStreamMetrics.recordBackPressureActive();
                this.enableBackPressure(pruningResult);
                this.switchBlockNodeIfPermitted(pruningResult);
            } else if (pruningResult.saturationPercent >= actionStageThreshold) {
                this.blockStreamMetrics.recordBackPressureActionStage();
                this.switchBlockNodeIfPermitted(pruningResult);
            } else {
                this.blockStreamMetrics.recordBackPressureDisabled();
            }
        } else if (previousPruneResult.isSaturated) {
            if (pruningResult.isSaturated) {
                this.blockStreamMetrics.recordBackPressureActive();
                this.switchBlockNodeIfPermitted(pruningResult);
                this.enableBackPressure(pruningResult);
            } else if (pruningResult.saturationPercent >= actionStageThreshold) {
                this.disableBackPressureIfRecovered(pruningResult);
                if (this.awaitingRecovery) {
                    this.blockStreamMetrics.recordBackPressureRecovering();
                } else {
                    this.blockStreamMetrics.recordBackPressureActionStage();
                }
            } else {
                this.disableBackPressureIfRecovered(pruningResult);
                if (this.awaitingRecovery) {
                    this.blockStreamMetrics.recordBackPressureRecovering();
                } else {
                    this.blockStreamMetrics.recordBackPressureDisabled();
                }
            }
        }
        if (this.awaitingRecovery && !pruningResult.isSaturated) {
            this.disableBackPressureIfRecovered(pruningResult);
        }
    }

    private void switchBlockNodeIfPermitted(PruneResult pruneResult) {
        Duration actionGracePeriod = this.actionGracePeriod();
        Instant now = Instant.now();
        Duration periodSinceLastAction = Duration.between(this.lastRecoveryActionTimestamp, now);
        if (periodSinceLastAction.compareTo(actionGracePeriod) <= 0) {
            return;
        }
        logger.info("Attempting to forcefully switch block node connections due to increasing block buffer saturation (saturation={}%)", (Object)pruneResult.saturationPercent);
        this.lastRecoveryActionTimestamp = now;
        this.blockNodeConnectionManager.selectNewBlockNodeForStreaming(true);
    }

    private void disableBackPressureIfRecovered(PruneResult latestPruneResult) {
        if (!this.backpressureEnabled) {
            return;
        }
        double recoveryThreshold = this.recoveryThreshold();
        if (latestPruneResult.saturationPercent > recoveryThreshold) {
            this.awaitingRecovery = true;
            logger.debug("Attempted to disable back pressure, but buffer saturation is not less than or equal to recovery threshold (saturation={}%, recoveryThreshold={}%)", (Object)latestPruneResult.saturationPercent, (Object)recoveryThreshold);
            return;
        }
        this.awaitingRecovery = false;
        logger.debug("Buffer saturation is below or equal to the recovery threshold; back pressure will be disabled. (saturation={}%, recoveryThreshold={}%)", (Object)latestPruneResult.saturationPercent, (Object)recoveryThreshold);
        CompletableFuture<Boolean> cf = this.backpressureCompletableFutureRef.get();
        if (cf != null && !cf.isDone()) {
            cf.complete(true);
        }
    }

    private void enableBackPressure(PruneResult latestPruneResult) {
        CompletableFuture<Object> newCf;
        CompletableFuture<Boolean> oldCf;
        if (!this.backpressureEnabled) {
            return;
        }
        do {
            if ((oldCf = this.backpressureCompletableFutureRef.get()) == null || oldCf.isDone()) {
                newCf = new CompletableFuture();
                logger.warn("Block buffer is saturated; backpressure is being enabled (idealMaxBufferSize={}, blocksChecked={}, blocksPruned={}, blocksPendingAck={}, saturation={}%)", (Object)latestPruneResult.idealMaxBufferSize, (Object)latestPruneResult.numBlocksChecked, (Object)latestPruneResult.numBlocksPruned, (Object)latestPruneResult.numBlocksPendingAck, (Object)latestPruneResult.saturationPercent);
                continue;
            }
            newCf = oldCf;
        } while (!this.backpressureCompletableFutureRef.compareAndSet(oldCf, newCf));
    }

    private void scheduleNextWorkerTask() {
        if (!this.grpcStreamingEnabled) {
            return;
        }
        Duration interval = this.workerTaskInterval();
        this.execSvc.schedule(new BufferWorkerTask(), interval.toMillis(), TimeUnit.MILLISECONDS);
    }

    static class PruneResult {
        static final PruneResult NIL = new PruneResult(0L, 0, 0, 0, 0L, 0L);
        final long idealMaxBufferSize;
        final int numBlocksChecked;
        final int numBlocksPendingAck;
        final int numBlocksPruned;
        final long oldestBlockNumber;
        final long newestBlockNumber;
        final double saturationPercent;
        final boolean isSaturated;

        PruneResult(long idealMaxBufferSize, int numBlocksChecked, int numBlocksPendingAck, int numBlocksPruned, long oldestBlockNumber, long newestBlockNumber) {
            this.idealMaxBufferSize = idealMaxBufferSize;
            this.numBlocksChecked = numBlocksChecked;
            this.numBlocksPendingAck = numBlocksPendingAck;
            this.numBlocksPruned = numBlocksPruned;
            this.oldestBlockNumber = oldestBlockNumber;
            this.newestBlockNumber = newestBlockNumber;
            boolean bl = this.isSaturated = idealMaxBufferSize != 0L && (long)numBlocksPendingAck >= idealMaxBufferSize;
            if (idealMaxBufferSize == 0L) {
                this.saturationPercent = 0.0;
            } else {
                BigDecimal size = BigDecimal.valueOf(idealMaxBufferSize);
                BigDecimal pending = BigDecimal.valueOf(numBlocksPendingAck);
                this.saturationPercent = pending.divide(size, 6, RoundingMode.HALF_EVEN).multiply(BigDecimal.valueOf(100L)).doubleValue();
            }
        }

        public String toString() {
            return "PruneResult{idealMaxBufferSize=" + this.idealMaxBufferSize + ", numBlocksChecked=" + this.numBlocksChecked + ", numBlocksPendingAck=" + this.numBlocksPendingAck + ", numBlocksPruned=" + this.numBlocksPruned + ", saturationPercent=" + this.saturationPercent + ", isSaturated=" + this.isSaturated + "}";
        }
    }

    private class BufferWorkerTask
    implements Runnable {
        private BufferWorkerTask() {
        }

        @Override
        public void run() {
            try {
                BlockBufferService.this.checkBuffer();
            }
            catch (RuntimeException e) {
                logger.warn("Periodic buffer worker task failed", (Throwable)e);
            }
            finally {
                BlockBufferService.this.scheduleNextWorkerTask();
            }
        }
    }
}

