/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.besu.ethereum.rlp;

import com.google.common.base.Preconditions;
import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.function.Function;
import org.apache.tuweni.bytes.Bytes;
import org.apache.tuweni.bytes.Bytes32;
import org.apache.tuweni.bytes.Bytes48;
import org.apache.tuweni.bytes.MutableBytes;
import org.apache.tuweni.bytes.MutableBytes32;
import org.apache.tuweni.units.bigints.UInt256;
import org.apache.tuweni.units.bigints.UInt64;
import org.hyperledger.besu.ethereum.rlp.CorruptedRLPInputException;
import org.hyperledger.besu.ethereum.rlp.MalformedRLPInputException;
import org.hyperledger.besu.ethereum.rlp.RLP;
import org.hyperledger.besu.ethereum.rlp.RLPDecodingHelpers;
import org.hyperledger.besu.ethereum.rlp.RLPEncodingHelpers;
import org.hyperledger.besu.ethereum.rlp.RLPException;
import org.hyperledger.besu.ethereum.rlp.RLPInput;

abstract class AbstractRLPInput
implements RLPInput {
    private static final String errorMessageSuffix = " (at bytes %d-%d: %s%s[%s]%s%s)";
    private final boolean lenient;
    protected long size;
    protected long currentItem;
    private RLPDecodingHelpers.Kind currentKind;
    private long currentPayloadOffset;
    private int currentPayloadSize;
    private int depth;
    private long[] endOfListOffset = new long[4];

    AbstractRLPInput(boolean lenient) {
        this.lenient = lenient;
    }

    protected void init(long inputSize, boolean shouldFitInputSizeExactly) {
        if (inputSize == 0L) {
            return;
        }
        this.currentItem = 0L;
        this.size = inputSize;
        this.prepareCurrentItem();
        if (this.currentKind.isList()) {
            this.size = this.nextItem();
        }
        if (this.size > inputSize) {
            long itemEnd = this.size;
            this.size = inputSize;
            throw this.corrupted("Input doesn't have enough data for RLP encoding: encoding advertise a payload ending at byte %d but input has size %d", itemEnd, inputSize);
        }
        if (shouldFitInputSizeExactly && inputSize > this.size) {
            this.throwMalformed("Input has extra data after RLP encoding: encoding ends at byte %d but input has size %d", this.size, inputSize);
        }
        this.validateCurrentItem();
    }

    protected abstract byte inputByte(long var1);

    protected abstract Bytes inputSlice(long var1, int var3);

    protected abstract Bytes32 inputSlice32(long var1);

    protected abstract Bytes48 inputSlice48(long var1);

    protected abstract String inputHex(long var1, int var3);

    protected abstract BigInteger getUnsignedBigInteger(long var1, int var3);

    protected abstract int getInt(long var1);

    protected abstract long getLong(long var1);

    protected void setTo(long item) {
        this.currentItem = item;
        if (this.currentItem >= this.size) {
            this.currentKind = null;
            this.currentPayloadOffset = item;
            this.currentPayloadSize = 0;
            return;
        }
        this.prepareCurrentItem();
        this.validateCurrentItem();
    }

    private void prepareCurrentItem() {
        try {
            RLPDecodingHelpers.RLPElementMetadata elementMetadata = RLPDecodingHelpers.rlpElementMetadata(this::inputByte, this.size, this.currentItem);
            this.currentKind = elementMetadata.kind;
            this.currentPayloadOffset = elementMetadata.payloadStart;
            this.currentPayloadSize = elementMetadata.payloadSize;
        }
        catch (RLPException exception) {
            String message = String.format(exception.getMessage() + errorMessageSuffix, this.getErrorMessageSuffixParams());
            throw new RLPException(message, exception);
        }
    }

    private void validateCurrentItem() {
        if (this.currentKind == RLPDecodingHelpers.Kind.SHORT_ELEMENT && this.currentPayloadSize == 1 && this.currentPayloadOffset < this.size && (this.payloadByte(0) & 0xFF) <= 127) {
            this.throwMalformed("Malformed RLP item: single byte value 0x%s should have been written without a prefix", this.hex(this.currentPayloadOffset, this.currentPayloadOffset + 1L));
        }
        if (this.currentPayloadSize > 0 && this.currentPayloadOffset >= this.size) {
            throw this.corrupted("Invalid RLP item: payload should start at offset %d but input has only %d bytes", this.currentPayloadOffset, this.size);
        }
        if (this.size - this.currentPayloadOffset < (long)this.currentPayloadSize) {
            throw this.corrupted("Invalid RLP item: payload starting at byte %d should be %d bytes long, but input has only %d bytes from that offset", this.currentPayloadOffset, this.currentPayloadSize, this.size - this.currentPayloadOffset);
        }
    }

    private long nextItem() {
        return this.currentPayloadOffset + (long)this.currentPayloadSize;
    }

    @Override
    public boolean isDone() {
        return this.currentItem >= this.size && this.depth == 0;
    }

    private String hex(long start, long taintedEnd) {
        long end = Math.min(taintedEnd, this.size);
        long length = end - start;
        if (length < 10L) {
            return this.inputHex(start, Math.toIntExact(length));
        }
        return String.format("%s...%s", this.inputHex(start, 4), this.inputHex(end - 4L, 4));
    }

    private void throwMalformed(String msg, Object ... params) {
        if (!this.lenient) {
            throw new MalformedRLPInputException(this.errorMsg(msg, params));
        }
    }

    private CorruptedRLPInputException corrupted(String msg, Object ... params) {
        throw new CorruptedRLPInputException(this.errorMsg(msg, params));
    }

    private RLPException error(String msg, Object ... params) {
        throw new RLPException(this.errorMsg(msg, params));
    }

    private RLPException error(Throwable cause, String msg, Object ... params) {
        throw new RLPException(this.errorMsg(msg, params), cause);
    }

    private String errorMsg(String message, Object ... params) {
        return String.format(message + errorMessageSuffix, AbstractRLPInput.concatParams(params, this.getErrorMessageSuffixParams()));
    }

    private Object[] getErrorMessageSuffixParams() {
        long start = this.currentItem;
        long end = Math.min(this.size, this.nextItem());
        long realStart = Math.max(0L, start - 4L);
        long realEnd = Math.min(this.size, end + 4L);
        return new Object[]{start, end, realStart == 0L ? "" : "...", this.hex(realStart, start), this.hex(start, end), this.hex(end, realEnd), realEnd == this.size ? "" : "..."};
    }

    private static Object[] concatParams(Object[] initial, Object ... others) {
        Object[] params = Arrays.copyOf(initial, initial.length + others.length);
        System.arraycopy(others, 0, params, initial.length, others.length);
        return params;
    }

    private void checkElt(String what) {
        if (this.currentItem >= this.size) {
            throw this.error("Cannot read a %s, input is fully consumed", what);
        }
        if (this.depth > 0 && this.currentPayloadOffset + (long)this.currentPayloadSize > this.endOfListOffset[this.depth - 1]) {
            throw this.error("Cannot read a %s, too large for enclosing list", what);
        }
        if (this.isEndOfCurrentList()) {
            throw this.error("Cannot read a %s, reached end of current list", what);
        }
        if (this.currentKind.isList()) {
            throw this.error("Cannot read a %s, current item is a list", what);
        }
    }

    private void checkElt(String what, int expectedSize) {
        this.checkElt(what);
        if (this.currentPayloadSize != expectedSize) {
            throw this.error("Cannot read a %s, expecting %d bytes but current element is %d bytes long", what, expectedSize, this.currentPayloadSize);
        }
    }

    private void checkScalar(String what) {
        this.checkElt(what);
        if (this.currentPayloadSize > 0 && this.payloadByte(0) == 0) {
            this.throwMalformed("Invalid scalar, has leading zeros bytes", new Object[0]);
        }
    }

    private void checkScalar(String what, int maxExpectedSize) {
        this.checkScalar(what);
        if (this.currentPayloadSize > maxExpectedSize) {
            throw this.error("Cannot read a %s, expecting a maximum of %d bytes but current element is %d bytes long", what, maxExpectedSize, this.currentPayloadSize);
        }
    }

    private byte payloadByte(int offsetInPayload) {
        return this.inputByte(this.currentPayloadOffset + (long)offsetInPayload);
    }

    private Bytes payloadSlice() {
        return this.inputSlice(this.currentPayloadOffset, this.currentPayloadSize);
    }

    @Override
    public void skipNext() {
        this.setTo(this.nextItem());
    }

    @Override
    public long readLongScalar() {
        this.checkScalar("long scalar", 8);
        long res = this.readGenericLongScalar();
        this.setTo(this.nextItem());
        return res;
    }

    private long readGenericLongScalar() {
        long res = 0L;
        int shift = 0;
        for (int i = 0; i < this.currentPayloadSize; ++i) {
            res |= ((long)this.payloadByte(this.currentPayloadSize - i - 1) & 0xFFL) << shift;
            shift += 8;
        }
        return res;
    }

    @Override
    public int readIntScalar() {
        this.checkScalar("int scalar", 4);
        int res = 0;
        int shift = 0;
        for (int i = 0; i < this.currentPayloadSize; ++i) {
            res |= (this.payloadByte(this.currentPayloadSize - i - 1) & 0xFF) << shift;
            shift += 8;
        }
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public long readUnsignedIntScalar() {
        this.checkScalar("unsigned int scalar", 4);
        return this.readLongScalar();
    }

    @Override
    public int readUnsignedByteScalar() {
        this.checkScalar("unsigned byte scalar", 1);
        int result = this.currentPayloadSize == 0 ? 0 : this.payloadByte(0) & 0xFF;
        this.setTo(this.nextItem());
        return result;
    }

    @Override
    public BigInteger readBigIntegerScalar() {
        this.checkScalar("arbitrary precision scalar");
        BigInteger res = this.getUnsignedBigInteger(this.currentPayloadOffset, this.currentPayloadSize);
        this.setTo(this.nextItem());
        return res;
    }

    private Bytes readBytes8Scalar() {
        this.checkScalar("8-bytes scalar", 8);
        MutableBytes res = MutableBytes.create((int)8);
        this.payloadSlice().copyTo(res, res.size() - this.currentPayloadSize);
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public UInt64 readUInt64Scalar() {
        return UInt64.fromBytes((Bytes)this.readBytes8Scalar());
    }

    private Bytes32 readBytes32Scalar() {
        this.checkScalar("32-bytes scalar", 32);
        MutableBytes32 res = MutableBytes32.create();
        this.payloadSlice().copyTo((MutableBytes)res, res.size() - this.currentPayloadSize);
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public UInt256 readUInt256Scalar() {
        return UInt256.fromBytes((Bytes)this.readBytes32Scalar());
    }

    @Override
    public byte readByte() {
        this.checkElt("byte", 1);
        byte b = this.payloadByte(0);
        this.setTo(this.nextItem());
        return b;
    }

    @Override
    public short readShort() {
        this.checkElt("2-byte short", 2);
        short s = (short)(this.payloadByte(0) << 8 | this.payloadByte(1) & 0xFF);
        this.setTo(this.nextItem());
        return s;
    }

    @Override
    public int readInt() {
        this.checkElt("4-byte int", 4);
        int res = this.getInt(this.currentPayloadOffset);
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public long readLong() {
        this.checkElt("8-byte long", 8);
        long res = this.getLong(this.currentPayloadOffset);
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public InetAddress readInetAddress() {
        this.checkElt("inet address");
        if (this.currentPayloadSize != 4 && this.currentPayloadSize != 16) {
            throw this.error("Cannot read an inet address, current element is %d bytes long", this.currentPayloadSize);
        }
        byte[] address = new byte[this.currentPayloadSize];
        for (int i = 0; i < this.currentPayloadSize; ++i) {
            address[i] = this.payloadByte(i);
        }
        this.setTo(this.nextItem());
        try {
            return InetAddress.getByAddress(address);
        }
        catch (UnknownHostException e) {
            throw new AssertionError((Object)e);
        }
    }

    @Override
    public Bytes readBytes() {
        this.checkElt("arbitrary bytes value");
        Bytes res = this.payloadSlice();
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public Bytes32 readBytes32() {
        this.checkElt("32 bytes value", 32);
        Bytes32 res = this.inputSlice32(this.currentPayloadOffset);
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public Bytes48 readBytes48() {
        this.checkElt("48 bytes value", 48);
        Bytes48 res = this.inputSlice48(this.currentPayloadOffset);
        this.setTo(this.nextItem());
        return res;
    }

    @Override
    public <T> T readBytes(Function<Bytes, T> mapper) {
        Bytes res = this.readBytes();
        try {
            return mapper.apply(res);
        }
        catch (Exception e) {
            throw this.error(e, "Problem decoding bytes value", new Object[0]);
        }
    }

    @Override
    public RLPInput readAsRlp() {
        if (this.currentItem >= this.size) {
            throw this.error("Cannot read current element as RLP, input is fully consumed", new Object[0]);
        }
        long next = this.nextItem();
        RLPInput res = RLP.input(this.inputSlice(this.currentItem, Math.toIntExact(next - this.currentItem)));
        this.setTo(next);
        return res;
    }

    @Override
    public int enterList() {
        return this.enterList(false);
    }

    public int enterList(boolean skipCount) {
        if (this.currentItem >= this.size) {
            throw this.error("Cannot enter a lists, input is fully consumed", new Object[0]);
        }
        if (!this.currentKind.isList()) {
            throw this.error("Expected current item to be a list, but it is: " + this.currentKind, new Object[0]);
        }
        ++this.depth;
        if (this.depth > this.endOfListOffset.length) {
            this.endOfListOffset = Arrays.copyOf(this.endOfListOffset, this.endOfListOffset.length * 3 / 2);
        }
        long listStart = this.currentPayloadOffset;
        long listEnd = this.nextItem();
        if (listEnd > this.size) {
            throw this.corrupted("Invalid RLP item: list payload should end at offset %d but input has only %d bytes", listEnd, this.size);
        }
        if (this.depth > 1 && listEnd > this.endOfListOffset[this.depth - 2]) {
            throw this.corrupted("Invalid RLP item: list ends outside of enclosing list (inner: %d, outer: %d)", listEnd, this.endOfListOffset[this.depth - 2]);
        }
        this.endOfListOffset[this.depth - 1] = listEnd;
        int count = -1;
        if (!skipCount) {
            count = 0;
            this.setTo(listStart);
            while (this.currentItem < listEnd) {
                ++count;
                this.setTo(this.nextItem());
            }
        }
        this.setTo(listStart);
        return count;
    }

    @Override
    public void leaveList() {
        this.leaveList(false);
    }

    @Override
    public void leaveListLenient() {
        this.leaveList(true);
    }

    private void leaveList(boolean ignoreRest) {
        long listEndOffset;
        Preconditions.checkState((this.depth > 0 ? 1 : 0) != 0, (Object)"Not within an RLP list");
        if (!ignoreRest && this.currentItem < (listEndOffset = this.endOfListOffset[this.depth - 1])) {
            throw this.error("Not at the end of the current list", new Object[0]);
        }
        --this.depth;
    }

    @Override
    public boolean nextIsList() {
        return this.currentKind != null && this.currentKind.isList();
    }

    @Override
    public boolean nextIsNull() {
        return this.currentKind == RLPDecodingHelpers.Kind.SHORT_ELEMENT && this.currentPayloadSize == 0;
    }

    @Override
    public int nextSize() {
        return this.currentPayloadSize;
    }

    @Override
    public int nextOffset() {
        return Math.toIntExact(this.currentPayloadOffset);
    }

    @Override
    public boolean isEndOfCurrentList() {
        return this.depth > 0 && this.currentItem >= this.endOfListOffset[this.depth - 1];
    }

    @Override
    public boolean isZeroLengthString() {
        return this.currentKind == RLPDecodingHelpers.Kind.SHORT_ELEMENT && this.currentPayloadSize == 0;
    }

    @Override
    public void reset() {
        this.setTo(0L);
    }

    @Override
    public Bytes currentListAsBytes() {
        if (this.currentItem >= this.size) {
            throw this.error("Cannot read list, input is fully consumed", new Object[0]);
        }
        if (!this.currentKind.isList()) {
            throw this.error("Cannot read list, current item is not a list list", new Object[0]);
        }
        MutableBytes scratch = MutableBytes.create((int)(this.currentPayloadSize + 10));
        int headerSize = RLPEncodingHelpers.writeListHeader(this.currentPayloadSize, scratch, 0);
        this.payloadSlice().copyTo(scratch, headerSize);
        Bytes res = scratch.slice(0, this.currentPayloadSize + headerSize);
        this.setTo(this.nextItem());
        return res;
    }
}

