/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.http.http2;

import io.helidon.http.http2.ConnectionFlowControl;
import io.helidon.http.http2.FlowControl;
import io.helidon.http.http2.Http2ErrorCode;
import io.helidon.http.http2.Http2Exception;
import io.helidon.http.http2.Http2WindowUpdate;
import io.helidon.http.http2.WindowSize;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;

abstract class WindowSizeImpl
implements WindowSize {
    private static final System.Logger LOGGER_INBOUND = System.getLogger(FlowControl.class.getName() + ".ifc");
    private static final System.Logger LOGGER_OUTBOUND = System.getLogger(FlowControl.class.getName() + ".ofc");
    private final ConnectionFlowControl.Type type;
    private final int streamId;
    private final AtomicInteger remainingWindowSize;
    private int windowSize;

    private WindowSizeImpl(ConnectionFlowControl.Type type, int streamId, int initialWindowSize) {
        this.type = type;
        this.streamId = streamId;
        this.windowSize = initialWindowSize;
        this.remainingWindowSize = new AtomicInteger(initialWindowSize);
    }

    @Override
    public void resetWindowSize(int size) {
        this.remainingWindowSize.updateAndGet(o -> o + size - this.windowSize);
        this.windowSize = size;
        if (LOGGER_OUTBOUND.isLoggable(System.Logger.Level.DEBUG)) {
            LOGGER_OUTBOUND.log(System.Logger.Level.DEBUG, String.format("%s OFC STR %d: Recv INITIAL_WINDOW_SIZE %d(%d)", new Object[]{this.type, this.streamId, this.windowSize, this.remainingWindowSize.get()}));
        }
    }

    @Override
    public long incrementWindowSize(int increment) {
        int remaining = this.remainingWindowSize.getAndUpdate(r -> r < 0 || Integer.MAX_VALUE - r > increment ? increment + r : Integer.MAX_VALUE);
        return remaining + increment;
    }

    @Override
    public int decrementWindowSize(int decrement) {
        return this.remainingWindowSize.updateAndGet(operand -> operand - decrement);
    }

    @Override
    public int getRemainingWindowSize() {
        return this.remainingWindowSize.get();
    }

    public String toString() {
        return String.valueOf(this.remainingWindowSize.get());
    }

    static abstract class Strategy {
        private static final StrategyConstructor[] CREATORS = new StrategyConstructor[]{Simple::new, Bisection::new};
        private final Context context;
        private final int streamId;
        private final BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter;

        private Strategy(Context context, int streamId, BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter) {
            this.context = context;
            this.streamId = streamId;
            this.windowUpdateWriter = windowUpdateWriter;
        }

        private static Strategy create(Context context, int streamId, BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter) {
            return CREATORS[Type.select(context).ordinal()].create(context, streamId, windowUpdateWriter);
        }

        abstract void windowUpdate(ConnectionFlowControl.Type var1, int var2, int var3);

        Context context() {
            return this.context;
        }

        int streamId() {
            return this.streamId;
        }

        BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter() {
            return this.windowUpdateWriter;
        }

        private record Context(int maxFrameSize, int initialWindowSize) {
        }

        private static interface StrategyConstructor {
            public Strategy create(Context var1, int var2, BiConsumer<Integer, Http2WindowUpdate> var3);
        }

        private static enum Type {
            SIMPLE,
            BISECTION;


            private static Type select(Context context) {
                return context.maxFrameSize * 4 <= context.initialWindowSize ? BISECTION : SIMPLE;
            }
        }

        private static final class Bisection
        extends Strategy {
            private final int watermark = this.context().initialWindowSize() / 2;
            private int delayedIncrement = 0;

            private Bisection(Context context, int streamId, BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter) {
                super(context, streamId, windowUpdateWriter);
            }

            @Override
            void windowUpdate(ConnectionFlowControl.Type type, int streamId, int increment) {
                if (LOGGER_INBOUND.isLoggable(System.Logger.Level.DEBUG)) {
                    LOGGER_INBOUND.log(System.Logger.Level.DEBUG, String.format("%s IFC STR %d: Deferred WINDOW_UPDATE %d, total %d, watermark %d", new Object[]{type, streamId, increment, this.delayedIncrement, this.watermark}));
                }
                this.delayedIncrement += increment;
                if (this.delayedIncrement > this.watermark) {
                    if (LOGGER_INBOUND.isLoggable(System.Logger.Level.DEBUG)) {
                        LOGGER_INBOUND.log(System.Logger.Level.DEBUG, String.format("%s IFC STR %d: Send WINDOW_UPDATE %d, watermark %d", new Object[]{type, streamId, this.delayedIncrement, this.watermark}));
                    }
                    this.windowUpdateWriter().accept(this.streamId(), new Http2WindowUpdate(this.delayedIncrement));
                    this.delayedIncrement = 0;
                }
            }
        }

        private static final class Simple
        extends Strategy {
            private Simple(Context context, int streamId, BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter) {
                super(context, streamId, windowUpdateWriter);
            }

            @Override
            void windowUpdate(ConnectionFlowControl.Type type, int streamId, int increment) {
                if (LOGGER_INBOUND.isLoggable(System.Logger.Level.DEBUG)) {
                    LOGGER_INBOUND.log(System.Logger.Level.DEBUG, String.format("%s IFC STR %d: Send WINDOW_UPDATE %s", new Object[]{type, streamId, increment}));
                }
                this.windowUpdateWriter().accept(this.streamId(), new Http2WindowUpdate(increment));
            }
        }
    }

    public static final class InboundNoop
    implements WindowSize.Inbound {
        private static final int WIN_SIZE_WATERMARK = 0x3FFFFFFF;
        private final int streamId;
        private final BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter;
        private int delayedIncrement;

        InboundNoop(int streamId, BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter) {
            this.streamId = streamId;
            this.windowUpdateWriter = windowUpdateWriter;
            this.delayedIncrement = 0;
        }

        @Override
        public long incrementWindowSize(int increment) {
            this.delayedIncrement += increment;
            if (this.delayedIncrement > 0x3FFFFFFF) {
                this.windowUpdateWriter.accept(this.streamId, new Http2WindowUpdate(this.delayedIncrement));
                this.delayedIncrement = 0;
            }
            return this.getRemainingWindowSize();
        }

        @Override
        public void resetWindowSize(int size) {
        }

        @Override
        public int decrementWindowSize(int decrement) {
            return Integer.MAX_VALUE;
        }

        @Override
        public int getRemainingWindowSize() {
            return Integer.MAX_VALUE;
        }

        public String toString() {
            return String.valueOf(Integer.MAX_VALUE);
        }
    }

    static final class Outbound
    extends WindowSizeImpl
    implements WindowSize.Outbound {
        private static final int BACKOFF_MIN = 50;
        private static final int BACKOFF_MAX = 5000;
        private final Semaphore updatedSemaphore = new Semaphore(1);
        private final ConnectionFlowControl.Type type;
        private final int streamId;
        private final long timeoutMillis;

        Outbound(ConnectionFlowControl.Type type, int streamId, ConnectionFlowControl connectionFlowControl) {
            super(type, streamId, connectionFlowControl.initialWindowSize());
            this.type = type;
            this.streamId = streamId;
            this.timeoutMillis = connectionFlowControl.timeout().toMillis();
        }

        @Override
        public long incrementWindowSize(int increment) {
            long remaining = super.incrementWindowSize(increment);
            if (LOGGER_OUTBOUND.isLoggable(System.Logger.Level.DEBUG)) {
                LOGGER_OUTBOUND.log(System.Logger.Level.DEBUG, String.format("%s OFC STR %d: +%d(%d)", new Object[]{this.type, this.streamId, increment, remaining}));
            }
            this.triggerUpdate();
            return remaining;
        }

        @Override
        public void resetWindowSize(int size) {
            super.resetWindowSize(size);
            this.triggerUpdate();
        }

        @Override
        public int decrementWindowSize(int decrement) {
            int n = super.decrementWindowSize(decrement);
            this.triggerUpdate();
            return n;
        }

        @Override
        public void triggerUpdate() {
            this.updatedSemaphore.release();
        }

        @Override
        public void blockTillUpdate() {
            long startTime = System.currentTimeMillis();
            int backoff = 50;
            while (this.getRemainingWindowSize() < 1) {
                try {
                    this.updatedSemaphore.drainPermits();
                    boolean ignored = this.updatedSemaphore.tryAcquire(backoff, TimeUnit.MILLISECONDS);
                    backoff = Math.min(backoff * 2, 5000);
                }
                catch (InterruptedException e) {
                    this.debugLog("%s OFC STR %d: Window depleted, waiting for update interrupted.", e);
                    throw new Http2Exception(Http2ErrorCode.FLOW_CONTROL, "Flow control update wait interrupted.");
                }
                if (System.currentTimeMillis() - startTime > this.timeoutMillis) {
                    this.debugLog("%s OFC STR %d: Window depleted, waiting for update time-out.", null);
                    throw new Http2Exception(Http2ErrorCode.FLOW_CONTROL, "Flow control update wait time-out.");
                }
                this.debugLog("%s OFC STR %d: Window depleted, waiting for update.", null);
            }
        }

        private void debugLog(String message, Exception e) {
            if (LOGGER_OUTBOUND.isLoggable(System.Logger.Level.DEBUG)) {
                if (e != null) {
                    LOGGER_OUTBOUND.log(System.Logger.Level.DEBUG, String.format(message, new Object[]{this.type, this.streamId}), (Throwable)e);
                } else {
                    LOGGER_OUTBOUND.log(System.Logger.Level.DEBUG, String.format(message, new Object[]{this.type, this.streamId}));
                }
            }
        }
    }

    static final class Inbound
    extends WindowSizeImpl
    implements WindowSize.Inbound {
        private final Strategy strategy;
        private final ConnectionFlowControl.Type type;
        private final int streamId;

        Inbound(ConnectionFlowControl.Type type, int streamId, int initialWindowSize, int maxFrameSize, BiConsumer<Integer, Http2WindowUpdate> windowUpdateWriter) {
            super(type, streamId, initialWindowSize);
            this.type = type;
            this.streamId = streamId;
            this.strategy = Strategy.create(new Strategy.Context(maxFrameSize, initialWindowSize), streamId, windowUpdateWriter);
        }

        @Override
        public long incrementWindowSize(int increment) {
            if (increment > 0) {
                long result = super.incrementWindowSize(increment);
                this.strategy.windowUpdate(this.type, this.streamId, increment);
                return result;
            }
            return super.getRemainingWindowSize();
        }
    }
}

