/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.besu.evm.processor;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import org.apache.tuweni.bytes.Bytes;
import org.hyperledger.besu.datatypes.Address;
import org.hyperledger.besu.evm.EVM;
import org.hyperledger.besu.evm.ModificationNotAllowedException;
import org.hyperledger.besu.evm.account.Account;
import org.hyperledger.besu.evm.account.MutableAccount;
import org.hyperledger.besu.evm.contractvalidation.ContractValidationRule;
import org.hyperledger.besu.evm.frame.ExceptionalHaltReason;
import org.hyperledger.besu.evm.frame.MessageFrame;
import org.hyperledger.besu.evm.gascalculator.GasCalculator;
import org.hyperledger.besu.evm.processor.AbstractMessageProcessor;
import org.hyperledger.besu.evm.tracing.OperationTracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContractCreationProcessor
extends AbstractMessageProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(ContractCreationProcessor.class);
    private final boolean requireCodeDepositToSucceed;
    private final GasCalculator gasCalculator;
    private final long initialContractNonce;
    private final List<ContractValidationRule> contractValidationRules;

    public ContractCreationProcessor(GasCalculator gasCalculator, EVM evm, boolean requireCodeDepositToSucceed, List<ContractValidationRule> contractValidationRules, long initialContractNonce, Collection<Address> forceCommitAddresses) {
        super(evm, forceCommitAddresses);
        this.gasCalculator = gasCalculator;
        this.requireCodeDepositToSucceed = requireCodeDepositToSucceed;
        this.contractValidationRules = contractValidationRules;
        this.initialContractNonce = initialContractNonce;
    }

    public ContractCreationProcessor(GasCalculator gasCalculator, EVM evm, boolean requireCodeDepositToSucceed, List<ContractValidationRule> contractValidationRules, long initialContractNonce) {
        this(gasCalculator, evm, requireCodeDepositToSucceed, contractValidationRules, initialContractNonce, Set.of());
    }

    private static boolean accountExists(Account account) {
        return account.getNonce() > 0L || !account.getCode().isEmpty();
    }

    @Override
    public void start(MessageFrame frame, OperationTracer operationTracer) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("Executing contract-creation");
        }
        try {
            MutableAccount sender = frame.getWorldUpdater().getSenderAccount(frame);
            sender.decrementBalance(frame.getValue());
            Address contractAddress = frame.getContractAddress();
            MutableAccount contract = frame.getWorldUpdater().getOrCreate(contractAddress);
            if (ContractCreationProcessor.accountExists(contract)) {
                LOG.trace("Contract creation error: account has already been created for address {}", (Object)contractAddress);
                frame.setExceptionalHaltReason(Optional.of(ExceptionalHaltReason.ILLEGAL_STATE_CHANGE));
                frame.setState(MessageFrame.State.EXCEPTIONAL_HALT);
                operationTracer.traceAccountCreationResult(frame, Optional.of(ExceptionalHaltReason.ILLEGAL_STATE_CHANGE));
            } else {
                frame.addCreate(contractAddress);
                contract.incrementBalance(frame.getValue());
                contract.setNonce(this.initialContractNonce);
                contract.clearStorage();
                frame.setState(MessageFrame.State.CODE_EXECUTING);
            }
        }
        catch (ModificationNotAllowedException ex) {
            LOG.trace("Contract creation error: attempt to mutate an immutable account");
            frame.setExceptionalHaltReason(Optional.of(ExceptionalHaltReason.ILLEGAL_STATE_CHANGE));
            frame.setState(MessageFrame.State.EXCEPTIONAL_HALT);
        }
    }

    @Override
    public void codeSuccess(MessageFrame frame, OperationTracer operationTracer) {
        Bytes contractCode = frame.getOutputData();
        long depositFee = this.gasCalculator.codeDepositGasCost(contractCode.size());
        if (frame.getRemainingGas() < depositFee) {
            LOG.trace("Not enough gas to pay the code deposit fee for {}: remaining gas = {} < {} = deposit fee", new Object[]{frame.getContractAddress(), frame.getRemainingGas(), depositFee});
            if (this.requireCodeDepositToSucceed) {
                LOG.trace("Contract creation error: insufficient funds for code deposit");
                frame.setExceptionalHaltReason(Optional.of(ExceptionalHaltReason.INSUFFICIENT_GAS));
                frame.setState(MessageFrame.State.EXCEPTIONAL_HALT);
                operationTracer.traceAccountCreationResult(frame, Optional.of(ExceptionalHaltReason.INSUFFICIENT_GAS));
            } else {
                frame.setState(MessageFrame.State.COMPLETED_SUCCESS);
            }
        } else {
            Optional<Optional> invalidReason = this.contractValidationRules.stream().map(rule -> rule.validate(contractCode, frame)).filter(Optional::isPresent).findFirst();
            if (invalidReason.isEmpty()) {
                frame.decrementRemainingGas(depositFee);
                MutableAccount contract = frame.getWorldUpdater().getOrCreate(frame.getContractAddress());
                contract.setCode(contractCode);
                LOG.trace("Successful creation of contract {} with code of size {} (Gas remaining: {})", new Object[]{frame.getContractAddress(), contractCode.size(), frame.getRemainingGas()});
                frame.setState(MessageFrame.State.COMPLETED_SUCCESS);
                if (operationTracer.isExtendedTracing()) {
                    operationTracer.traceAccountCreationResult(frame, Optional.empty());
                }
            } else {
                Optional exceptionalHaltReason = invalidReason.get();
                frame.setExceptionalHaltReason(exceptionalHaltReason);
                frame.setState(MessageFrame.State.EXCEPTIONAL_HALT);
                operationTracer.traceAccountCreationResult(frame, exceptionalHaltReason);
            }
        }
    }
}

