/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.internal.BufferedBlock;
import com.hedera.hapi.block.stream.Block;
import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.hapi.node.base.Timestamp;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.pbj.runtime.ParseException;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class BlockBufferIO {
    private static final Logger logger = LogManager.getLogger(BlockBufferIO.class);
    private final File rootDirectory;

    public BlockBufferIO(String rootDirectory) {
        this.rootDirectory = new File(Objects.requireNonNull(rootDirectory));
    }

    public void write(List<BlockState> blocks, long latestAcknowledgedBlockNumber) throws IOException {
        new Writer(blocks, latestAcknowledgedBlockNumber).write();
    }

    public List<BufferedBlock> read() throws IOException {
        return new Reader().read();
    }

    private class Writer {
        private final List<BlockState> blocks;
        private final long latestAcknowledgedBlockNumber;

        Writer(List<BlockState> blocks, long latestAcknowledgedBlockNumber) {
            this.blocks = new ArrayList<BlockState>((Collection)Objects.requireNonNull(blocks));
            this.latestAcknowledgedBlockNumber = latestAcknowledgedBlockNumber;
        }

        private void write() throws IOException {
            Instant now = Instant.now();
            File directory = new File(BlockBufferIO.this.rootDirectory, Long.toString(now.toEpochMilli()));
            Path directoryPath = directory.toPath();
            Files.createDirectories(directoryPath, new FileAttribute[0]);
            logger.debug("Created new block buffer directory: {}", (Object)directoryPath.toFile().getAbsolutePath());
            for (BlockState block : this.blocks) {
                String fileName = "block-" + block.blockNumber() + ".bin";
                Path path = new File(directory, fileName).toPath();
                this.writeBlock(path, block);
            }
            this.cleanupOldFiles(directoryPath);
        }

        private void writeBlock(Path path, BlockState block) throws IOException {
            ArrayList<BlockItem> items = new ArrayList<BlockItem>(block.itemCount());
            for (int i = 0; i < block.itemCount(); ++i) {
                BlockItem item = block.blockItem(i);
                if (item == null) continue;
                items.add(item);
            }
            Block blk = new Block(items);
            Instant closedInstant = block.closedTimestamp();
            Timestamp closedTimestamp = Timestamp.newBuilder().seconds(closedInstant.getEpochSecond()).nanos(closedInstant.getNano()).build();
            BufferedBlock bufferedBlock = BufferedBlock.newBuilder().blockNumber(block.blockNumber()).closedTimestamp(closedTimestamp).isAcknowledged(block.blockNumber() <= this.latestAcknowledgedBlockNumber).block(blk).build();
            Bytes payload = BufferedBlock.PROTOBUF.toBytes((Object)bufferedBlock);
            int length = (int)payload.length();
            byte[] lenArray = ByteBuffer.allocate(4).putInt(length).array();
            Bytes len = Bytes.wrap((byte[])lenArray);
            Bytes bytes = Bytes.merge((Bytes)len, (Bytes)payload);
            Files.write(path, bytes.toByteArray(), StandardOpenOption.WRITE, StandardOpenOption.CREATE_NEW, StandardOpenOption.TRUNCATE_EXISTING);
            if (logger.isDebugEnabled()) {
                logger.debug("Block {} (items: {}) written to file: {} (bytes: {})", (Object)block.blockNumber(), (Object)items.size(), (Object)path.toFile().getAbsolutePath(), (Object)bytes.length());
            }
        }

        private void cleanupOldFiles(final Path newestDirectory) throws IOException {
            Files.walkFileTree(BlockBufferIO.this.rootDirectory.toPath(), (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    if (dir.equals(newestDirectory)) {
                        return FileVisitResult.SKIP_SUBTREE;
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    logger.debug("Deleting old block buffer file: {}", (Object)file);
                    Files.delete(file);
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    if (!dir.equals(newestDirectory) && !dir.equals(BlockBufferIO.this.rootDirectory.toPath())) {
                        logger.debug("Deleting old block buffer directory: {}", (Object)dir);
                        Files.delete(dir);
                    }
                    return FileVisitResult.CONTINUE;
                }
            });
        }
    }

    private class Reader {
        private Reader() {
        }

        private List<BufferedBlock> read() throws IOException {
            File[] files = BlockBufferIO.this.rootDirectory.listFiles();
            if (files == null) {
                logger.info("Block buffer directory not found and/or no files present (directory: {})", (Object)BlockBufferIO.this.rootDirectory.getAbsolutePath());
                return List.of();
            }
            File dirToRead = null;
            long dirMillis = -1L;
            for (File file : files) {
                long millis;
                if (!file.isDirectory()) continue;
                String dirName = file.getName();
                try {
                    millis = Long.parseLong(dirName);
                }
                catch (NumberFormatException e) {
                    continue;
                }
                if (millis <= dirMillis) continue;
                dirToRead = file;
                dirMillis = millis;
            }
            if (dirToRead == null) {
                logger.warn("No valid block buffer directories found in: {}", (Object)BlockBufferIO.this.rootDirectory.getAbsolutePath());
                return List.of();
            }
            return this.read(dirToRead);
        }

        private List<BufferedBlock> read(File directory) throws IOException {
            List<File> files;
            logger.debug("Reading blocks from directory: {}", (Object)directory.getAbsolutePath());
            try (Stream<Path> stream = Files.list(directory.toPath());){
                files = stream.map(Path::toFile).toList();
            }
            ArrayList<BufferedBlock> blocks = new ArrayList<BufferedBlock>(files.size());
            for (File file : files) {
                try {
                    BufferedBlock bufferedBlock = this.readBlockFile(file);
                    logger.debug("Block {} (items: {}) read from file: {}", (Object)bufferedBlock.blockNumber(), (Object)bufferedBlock.block().items().size(), (Object)file.getAbsolutePath());
                    blocks.add(bufferedBlock);
                }
                catch (Exception e) {
                    logger.warn("Failed to read block file; ignoring block (file={})", (Object)file.getAbsolutePath(), (Object)e);
                }
            }
            return blocks;
        }

        private BufferedBlock readBlockFile(File file) throws IOException, ParseException {
            try (RandomAccessFile raf = new RandomAccessFile(file, "r");){
                FileChannel fileChannel = raf.getChannel();
                MappedByteBuffer byteBuffer = fileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, fileChannel.size());
                int length = byteBuffer.getInt();
                byte[] payload = new byte[length];
                byteBuffer.get(payload);
                Bytes bytes = Bytes.wrap((byte[])payload);
                BufferedBlock bufferedBlock = (BufferedBlock)BufferedBlock.PROTOBUF.parse(bytes);
                return bufferedBlock;
            }
        }
    }
}

