/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.internal.BufferedBlock;
import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.hapi.node.base.Timestamp;
import com.hedera.node.app.blocks.impl.streaming.BlockBufferIO;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnectionManager;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockBufferConfig;
import com.hedera.node.config.data.BlockStreamConfig;
import com.hedera.node.config.types.StreamMode;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Singleton
public class BlockBufferService {
    private static final Logger logger = LogManager.getLogger(BlockBufferService.class);
    private static final Duration DEFAULT_WORKER_INTERVAL = Duration.ofSeconds(1L);
    private static final int DEFAULT_BUFFER_SIZE = 150;
    private final ConcurrentMap<Long, BlockState> blockBuffer = new ConcurrentHashMap<Long, BlockState>();
    private final AtomicLong earliestBlockNumber = new AtomicLong(Long.MIN_VALUE);
    private final AtomicLong highestAckedBlockNumber = new AtomicLong(Long.MIN_VALUE);
    private ScheduledExecutorService execSvc;
    private final AtomicReference<CompletableFuture<Boolean>> backpressureCompletableFutureRef = new AtomicReference();
    private final AtomicLong lastProducedBlockNumber = new AtomicLong(-1L);
    private final ConfigProvider configProvider;
    private BlockNodeConnectionManager blockNodeConnectionManager;
    private final BlockStreamMetrics blockStreamMetrics;
    private Instant lastRecoveryActionTimestamp = Instant.MIN;
    private PruneResult lastPruningResult = PruneResult.NIL;
    private boolean awaitingRecovery = false;
    private final BlockBufferIO bufferIO;
    private final AtomicBoolean isStarted = new AtomicBoolean(false);

    @Inject
    public BlockBufferService(@NonNull ConfigProvider configProvider, @NonNull BlockStreamMetrics blockStreamMetrics) {
        this.configProvider = configProvider;
        this.blockStreamMetrics = blockStreamMetrics;
        this.bufferIO = new BlockBufferIO(this.bufferDirectory(), this.maxReadDepth());
    }

    private boolean isGrpcStreamingEnabled() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).streamToBlockNodes();
    }

    private boolean isBackpressureEnabled() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).streamMode() == StreamMode.BLOCKS && this.isGrpcStreamingEnabled();
    }

    public void start() {
        if (!this.isGrpcStreamingEnabled() || !this.isStarted.compareAndSet(false, true)) {
            return;
        }
        this.loadBufferFromDisk();
        this.execSvc = Executors.newSingleThreadScheduledExecutor();
        this.scheduleNextWorkerTask();
    }

    public void shutdown() {
        if (!this.isStarted.compareAndSet(true, false)) {
            return;
        }
        this.execSvc.shutdownNow();
        this.blockBuffer.clear();
        this.disableBackPressure();
        this.highestAckedBlockNumber.set(Long.MIN_VALUE);
        this.lastProducedBlockNumber.set(-1L);
        this.earliestBlockNumber.set(Long.MIN_VALUE);
        this.lastPruningResult = PruneResult.NIL;
        this.lastRecoveryActionTimestamp = Instant.MIN;
        this.awaitingRecovery = false;
    }

    private Duration workerTaskInterval() {
        Duration interval = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).workerInterval();
        if (interval.isNegative() || interval.isZero()) {
            return DEFAULT_WORKER_INTERVAL;
        }
        return interval;
    }

    private int maxBufferedBlocks() {
        int maxBufferedBlocks = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).maxBlocks();
        return maxBufferedBlocks <= 0 ? 150 : maxBufferedBlocks;
    }

    private double actionStageThreshold() {
        double threshold = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).actionStageThreshold();
        return Math.max(0.0, threshold);
    }

    private Duration actionGracePeriod() {
        Duration gracePeriod = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).actionGracePeriod();
        return gracePeriod == null || gracePeriod.isNegative() ? Duration.ZERO : gracePeriod;
    }

    private double recoveryThreshold() {
        double threshold = ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).recoveryThreshold();
        return Math.max(0.0, threshold);
    }

    private boolean isBufferPersistenceEnabled() {
        return ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).isBufferPersistenceEnabled();
    }

    private String bufferDirectory() {
        return ((BlockBufferConfig)this.configProvider.getConfiguration().getConfigData(BlockBufferConfig.class)).bufferDirectory();
    }

    private int maxReadDepth() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).maxReadDepth();
    }

    public void setBlockNodeConnectionManager(@NonNull BlockNodeConnectionManager blockNodeConnectionManager) {
        this.blockNodeConnectionManager = Objects.requireNonNull(blockNodeConnectionManager, "blockNodeConnectionManager must not be null");
    }

    public void openBlock(long blockNumber) {
        if (!this.isGrpcStreamingEnabled() || !this.isStarted.get()) {
            return;
        }
        logger.debug("Opening block {}.", (Object)blockNumber);
        if (blockNumber < 0L) {
            throw new IllegalArgumentException("Block number must be non-negative");
        }
        BlockState existingBlock = (BlockState)this.blockBuffer.get(blockNumber);
        if (existingBlock != null && existingBlock.isClosed()) {
            logger.debug("Block {} is already open and its closed; ignoring open request", (Object)blockNumber);
            return;
        }
        BlockState blockState = new BlockState(blockNumber);
        this.blockBuffer.put(blockNumber, blockState);
        this.earliestBlockNumber.updateAndGet(current -> current == Long.MIN_VALUE ? blockNumber : Math.min(current, blockNumber));
        this.lastProducedBlockNumber.updateAndGet(old -> Math.max(old, blockNumber));
        this.blockStreamMetrics.recordLatestBlockOpened(blockNumber);
        this.blockStreamMetrics.recordBlockOpened();
    }

    public void addItem(long blockNumber, @NonNull BlockItem blockItem) {
        if (!this.isGrpcStreamingEnabled() || !this.isStarted.get()) {
            return;
        }
        Objects.requireNonNull(blockItem, "blockItem must not be null");
        BlockState blockState = this.getBlockState(blockNumber);
        if (blockState == null || blockState.isClosed()) {
            return;
        }
        blockState.addItem(blockItem);
    }

    public void closeBlock(long blockNumber) {
        if (!this.isGrpcStreamingEnabled() || !this.isStarted.get()) {
            return;
        }
        BlockState blockState = this.getBlockState(blockNumber);
        if (blockState == null || blockState.isClosed()) {
            return;
        }
        this.blockStreamMetrics.recordBlockClosed();
        blockState.closeBlock();
    }

    @Nullable
    public BlockState getBlockState(long blockNumber) {
        BlockState block = (BlockState)this.blockBuffer.get(blockNumber);
        if (block == null && blockNumber <= this.lastProducedBlockNumber.get()) {
            this.blockStreamMetrics.recordBlockMissing();
        }
        return block;
    }

    public boolean isAcked(long blockNumber) {
        return this.highestAckedBlockNumber.get() >= blockNumber;
    }

    public void setLatestAcknowledgedBlock(long blockNumber) {
        if (!this.isGrpcStreamingEnabled() || !this.isStarted.get()) {
            return;
        }
        long highestBlock = this.highestAckedBlockNumber.updateAndGet(current -> Math.max(current, blockNumber));
        this.blockStreamMetrics.recordLatestBlockAcked(highestBlock);
    }

    public long getLastBlockNumberProduced() {
        return this.lastProducedBlockNumber.get();
    }

    public long getHighestAckedBlockNumber() {
        return this.highestAckedBlockNumber.get() == Long.MIN_VALUE ? -1L : this.highestAckedBlockNumber.get();
    }

    public long getEarliestAvailableBlockNumber() {
        return this.earliestBlockNumber.get() == Long.MIN_VALUE ? -1L : this.earliestBlockNumber.get();
    }

    public void ensureNewBlocksPermitted() {
        if (!this.isGrpcStreamingEnabled() || !this.isStarted.get()) {
            return;
        }
        CompletableFuture<Boolean> cf = this.backpressureCompletableFutureRef.get();
        if (cf != null && !cf.isDone()) {
            try {
                logger.error("!!! Block buffer is saturated; blocking thread until buffer is no longer saturated");
                long startMs = System.currentTimeMillis();
                boolean bufferAvailable = cf.get();
                long durationMs = System.currentTimeMillis() - startMs;
                logger.warn("Thread was blocked for {}ms waiting for block buffer to free space", (Object)durationMs);
                if (!bufferAvailable) {
                    logger.warn("Block buffer still not available to accept new blocks; reentering wait...");
                    this.ensureNewBlocksPermitted();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            catch (Exception e) {
                logger.warn("Failed to wait for block buffer to be available", (Throwable)e);
            }
        }
    }

    private void loadBufferFromDisk() {
        List<BufferedBlock> blocks;
        if (!this.isBufferPersistenceEnabled()) {
            return;
        }
        try {
            blocks = this.bufferIO.read();
        }
        catch (IOException e) {
            logger.error("Failed to read block buffer from disk!", (Throwable)e);
            return;
        }
        if (blocks.isEmpty()) {
            logger.info("Block buffer will not be repopulated (reason: no blocks found on disk)");
            return;
        }
        logger.info("Block buffer is being restored from disk (blocksRead: {})", (Object)blocks.size());
        for (BufferedBlock bufferedBlock : blocks) {
            BlockState block = new BlockState(bufferedBlock.blockNumber());
            bufferedBlock.block().items().forEach(block::addItem);
            Timestamp closedTimestamp = bufferedBlock.closedTimestamp();
            Instant closedInstant = Instant.ofEpochSecond(closedTimestamp.seconds(), closedTimestamp.nanos());
            logger.debug("Reconstructed block {} from disk and closed at {}", (Object)bufferedBlock.blockNumber(), (Object)closedInstant);
            block.closeBlock(closedInstant);
            if (bufferedBlock.isAcknowledged()) {
                this.setLatestAcknowledgedBlock(bufferedBlock.blockNumber());
            }
            if (this.blockBuffer.putIfAbsent(bufferedBlock.blockNumber(), block) == null) continue;
            logger.debug("Block {} was read from disk but it was already in the buffer; ignoring block from disk", (Object)bufferedBlock.blockNumber());
        }
    }

    public void persistBuffer() {
        if (!(this.isGrpcStreamingEnabled() && this.isStarted.get() && this.isBufferPersistenceEnabled())) {
            return;
        }
        List<BlockState> blocksToPersist = this.blockBuffer.values().stream().filter(BlockState::isClosed).filter(blockState -> blockState.blockNumber() > this.highestAckedBlockNumber.get()).toList();
        try {
            this.bufferIO.write(blocksToPersist, this.highestAckedBlockNumber.get());
            logger.info("Block buffer persisted to disk (blocksWritten: {})", (Object)blocksToPersist.size());
        }
        catch (IOException | RuntimeException e) {
            logger.error("Failed to write block buffer to disk!", (Throwable)e);
        }
    }

    @NonNull
    private PruneResult pruneBuffer() {
        long highestBlockAcked = this.highestAckedBlockNumber.get();
        int maxBufferSize = this.maxBufferedBlocks();
        int numPruned = 0;
        int numChecked = 0;
        int numPendingAck = 0;
        long newEarliestBlock = Long.MAX_VALUE;
        long newLatestBlock = Long.MIN_VALUE;
        ArrayList orderedBuffer = new ArrayList(this.blockBuffer.keySet());
        Collections.sort(orderedBuffer);
        int size = this.blockBuffer.size();
        Iterator iterator = orderedBuffer.iterator();
        while (iterator.hasNext()) {
            boolean shouldPrune;
            long blockNumber = (Long)iterator.next();
            BlockState block = (BlockState)this.blockBuffer.get(blockNumber);
            ++numChecked;
            if (block.closedTimestamp() == null) continue;
            if (!this.isBackpressureEnabled()) {
                shouldPrune = size > maxBufferSize;
            } else {
                boolean bl = shouldPrune = size > maxBufferSize && blockNumber <= highestBlockAcked;
            }
            if (shouldPrune) {
                this.blockBuffer.remove(blockNumber);
                ++numPruned;
                --size;
                continue;
            }
            if (blockNumber > highestBlockAcked) {
                ++numPendingAck;
            }
            newEarliestBlock = Math.min(newEarliestBlock, blockNumber);
            newLatestBlock = Math.max(newLatestBlock, blockNumber);
        }
        newEarliestBlock = newEarliestBlock == Long.MAX_VALUE ? Long.MIN_VALUE : newEarliestBlock;
        newLatestBlock = newLatestBlock == Long.MIN_VALUE ? -1L : newLatestBlock;
        this.earliestBlockNumber.set(newEarliestBlock);
        this.blockStreamMetrics.recordNumberOfBlocksPruned(numPruned);
        this.blockStreamMetrics.recordBufferOldestBlock(newEarliestBlock == Long.MIN_VALUE ? -1L : newEarliestBlock);
        this.blockStreamMetrics.recordBufferNewestBlock(newLatestBlock);
        return new PruneResult(maxBufferSize, numChecked, numPendingAck, numPruned, newEarliestBlock, newLatestBlock);
    }

    private void checkBuffer() {
        if (!this.isGrpcStreamingEnabled()) {
            return;
        }
        PruneResult pruningResult = this.pruneBuffer();
        PruneResult previousPruneResult = this.lastPruningResult;
        this.lastPruningResult = pruningResult;
        if (logger.isDebugEnabled()) {
            logger.debug("Block buffer status: idealMaxBufferSize={}, blocksChecked={}, blocksPruned={}, blocksPendingAck={}, blockRange={}, saturation={}%", (Object)pruningResult.idealMaxBufferSize, (Object)pruningResult.numBlocksChecked, (Object)pruningResult.numBlocksPruned, (Object)pruningResult.numBlocksPendingAck, (Object)BlockBufferService.getContiguousRangesAsString(new ArrayList<Long>(this.blockBuffer.keySet())), (Object)pruningResult.saturationPercent);
        }
        this.blockStreamMetrics.recordBufferSaturation(pruningResult.saturationPercent);
        double actionStageThreshold = this.actionStageThreshold();
        if (previousPruneResult.saturationPercent < actionStageThreshold) {
            if (pruningResult.isSaturated) {
                this.enableBackPressure(pruningResult);
                this.switchBlockNodeIfPermitted(pruningResult);
            } else if (pruningResult.saturationPercent >= actionStageThreshold) {
                this.blockStreamMetrics.recordBackPressureActionStage();
                this.switchBlockNodeIfPermitted(pruningResult);
            } else {
                this.blockStreamMetrics.recordBackPressureDisabled();
            }
        } else if (!previousPruneResult.isSaturated && previousPruneResult.saturationPercent >= actionStageThreshold) {
            if (pruningResult.isSaturated) {
                this.enableBackPressure(pruningResult);
                this.switchBlockNodeIfPermitted(pruningResult);
            } else if (pruningResult.saturationPercent >= actionStageThreshold) {
                this.blockStreamMetrics.recordBackPressureActionStage();
                this.switchBlockNodeIfPermitted(pruningResult);
            } else {
                this.blockStreamMetrics.recordBackPressureDisabled();
            }
        } else if (previousPruneResult.isSaturated) {
            if (pruningResult.isSaturated) {
                this.switchBlockNodeIfPermitted(pruningResult);
                this.enableBackPressure(pruningResult);
            } else if (pruningResult.saturationPercent >= actionStageThreshold) {
                this.disableBackPressureIfRecovered(pruningResult);
                if (this.awaitingRecovery) {
                    this.blockStreamMetrics.recordBackPressureRecovering();
                } else {
                    this.blockStreamMetrics.recordBackPressureActionStage();
                }
            } else {
                this.disableBackPressureIfRecovered(pruningResult);
                if (this.awaitingRecovery) {
                    this.blockStreamMetrics.recordBackPressureRecovering();
                } else {
                    this.blockStreamMetrics.recordBackPressureDisabled();
                }
            }
        }
        if (this.awaitingRecovery && !pruningResult.isSaturated) {
            this.disableBackPressureIfRecovered(pruningResult);
        }
    }

    private void switchBlockNodeIfPermitted(PruneResult pruneResult) {
        Duration actionGracePeriod = this.actionGracePeriod();
        Instant now = Instant.now();
        Duration periodSinceLastAction = Duration.between(this.lastRecoveryActionTimestamp, now);
        if (periodSinceLastAction.compareTo(actionGracePeriod) <= 0) {
            return;
        }
        logger.debug("Attempting to forcefully switch block node connections due to increasing block buffer saturation (saturation={}%)", (Object)pruneResult.saturationPercent);
        this.lastRecoveryActionTimestamp = now;
        this.blockNodeConnectionManager.selectNewBlockNodeForStreaming(true);
    }

    private void disableBackPressureIfRecovered(PruneResult latestPruneResult) {
        if (!this.isBackpressureEnabled()) {
            return;
        }
        double recoveryThreshold = this.recoveryThreshold();
        if (latestPruneResult.saturationPercent > recoveryThreshold) {
            this.awaitingRecovery = true;
            logger.debug("Attempted to disable back pressure, but buffer saturation is not less than or equal to recovery threshold (saturation={}%, recoveryThreshold={}%)", (Object)latestPruneResult.saturationPercent, (Object)recoveryThreshold);
            return;
        }
        this.awaitingRecovery = false;
        logger.debug("Buffer saturation is below or equal to the recovery threshold; back pressure will be disabled. (saturation={}%, recoveryThreshold={}%)", (Object)latestPruneResult.saturationPercent, (Object)recoveryThreshold);
        this.disableBackPressure();
    }

    private void disableBackPressure() {
        CompletableFuture<Boolean> cf = this.backpressureCompletableFutureRef.get();
        if (cf != null && !cf.isDone()) {
            cf.complete(true);
        }
    }

    private void enableBackPressure(PruneResult latestPruneResult) {
        CompletableFuture<Object> newCf;
        CompletableFuture<Boolean> oldCf;
        if (!this.isBackpressureEnabled()) {
            return;
        }
        do {
            if ((oldCf = this.backpressureCompletableFutureRef.get()) == null || oldCf.isDone()) {
                newCf = new CompletableFuture();
                this.blockStreamMetrics.recordBackPressureActive();
                logger.warn("Block buffer is saturated; backpressure is being enabled (idealMaxBufferSize={}, blocksChecked={}, blocksPruned={}, blocksPendingAck={}, saturation={}%)", (Object)latestPruneResult.idealMaxBufferSize, (Object)latestPruneResult.numBlocksChecked, (Object)latestPruneResult.numBlocksPruned, (Object)latestPruneResult.numBlocksPendingAck, (Object)latestPruneResult.saturationPercent);
                continue;
            }
            newCf = oldCf;
        } while (!this.backpressureCompletableFutureRef.compareAndSet(oldCf, newCf));
    }

    private void scheduleNextWorkerTask() {
        if (!this.isGrpcStreamingEnabled()) {
            return;
        }
        Duration interval = this.workerTaskInterval();
        this.execSvc.schedule(new BufferWorkerTask(), interval.toMillis(), TimeUnit.MILLISECONDS);
    }

    private static String getContiguousRangesAsString(List<Long> blockNumbers) {
        long start;
        Collections.sort(blockNumbers);
        if (blockNumbers.isEmpty()) {
            return "[]";
        }
        ArrayList<String> ranges = new ArrayList<String>();
        long prev = start = blockNumbers.getFirst().longValue();
        for (int i = 1; i < blockNumbers.size(); ++i) {
            long current = blockNumbers.get(i);
            if (current != prev + 1L) {
                ranges.add(BlockBufferService.formatRange(start, prev));
                start = current;
            }
            prev = current;
        }
        ranges.add(BlockBufferService.formatRange(start, prev));
        return "[" + String.join((CharSequence)",", ranges) + "]";
    }

    private static String formatRange(long start, long end) {
        return start == end ? "(" + start + ")" : "(" + start + "-" + end + ")";
    }

    static class PruneResult {
        static final PruneResult NIL = new PruneResult(0L, 0, 0, 0, 0L, 0L);
        final long idealMaxBufferSize;
        final int numBlocksChecked;
        final int numBlocksPendingAck;
        final int numBlocksPruned;
        final long oldestBlockNumber;
        final long newestBlockNumber;
        final double saturationPercent;
        final boolean isSaturated;

        PruneResult(long idealMaxBufferSize, int numBlocksChecked, int numBlocksPendingAck, int numBlocksPruned, long oldestBlockNumber, long newestBlockNumber) {
            this.idealMaxBufferSize = idealMaxBufferSize;
            this.numBlocksChecked = numBlocksChecked;
            this.numBlocksPendingAck = numBlocksPendingAck;
            this.numBlocksPruned = numBlocksPruned;
            this.oldestBlockNumber = oldestBlockNumber;
            this.newestBlockNumber = newestBlockNumber;
            boolean bl = this.isSaturated = idealMaxBufferSize != 0L && (long)numBlocksPendingAck >= idealMaxBufferSize;
            if (idealMaxBufferSize == 0L) {
                this.saturationPercent = 0.0;
            } else {
                BigDecimal size = BigDecimal.valueOf(idealMaxBufferSize);
                BigDecimal pending = BigDecimal.valueOf(numBlocksPendingAck);
                this.saturationPercent = pending.divide(size, 6, RoundingMode.HALF_EVEN).multiply(BigDecimal.valueOf(100L)).doubleValue();
            }
        }

        public String toString() {
            return "PruneResult{idealMaxBufferSize=" + this.idealMaxBufferSize + ", numBlocksChecked=" + this.numBlocksChecked + ", numBlocksPendingAck=" + this.numBlocksPendingAck + ", numBlocksPruned=" + this.numBlocksPruned + ", saturationPercent=" + this.saturationPercent + ", isSaturated=" + this.isSaturated + "}";
        }
    }

    private class BufferWorkerTask
    implements Runnable {
        private BufferWorkerTask() {
        }

        @Override
        public void run() {
            if (!BlockBufferService.this.isStarted.get()) {
                logger.debug("Buffer service shutdown; aborting worker task");
                return;
            }
            try {
                BlockBufferService.this.checkBuffer();
            }
            catch (RuntimeException e) {
                logger.warn("Periodic buffer worker task failed", (Throwable)e);
            }
            finally {
                BlockBufferService.this.scheduleNextWorkerTask();
            }
        }
    }
}

