/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.platform.crypto;

import com.hedera.hapi.node.state.roster.RosterEntry;
import com.swirlds.config.api.Configuration;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.platform.config.PathsConfig;
import com.swirlds.platform.crypto.CryptoStatic;
import com.swirlds.platform.crypto.KeyLoadingException;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.lang.runtime.SwitchBootstraps;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMDecryptorProvider;
import org.bouncycastle.openssl.PEMEncryptedKeyPair;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcePEMDecryptorProviderBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.operator.jcajce.JceInputDecryptorProviderBuilder;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.util.encoders.DecoderException;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.bouncycastle.util.io.pem.PemWriter;
import org.hiero.base.crypto.KeystorePasswordPolicy;
import org.hiero.base.crypto.config.CryptoConfig;
import org.hiero.consensus.crypto.CertificateUtils;
import org.hiero.consensus.crypto.KeyCertPurpose;
import org.hiero.consensus.crypto.KeyGeneratingException;
import org.hiero.consensus.crypto.KeysAndCertsGenerator;
import org.hiero.consensus.model.node.KeysAndCerts;
import org.hiero.consensus.model.node.NodeId;
import org.hiero.consensus.node.NodeUtilities;
import org.hiero.consensus.roster.RosterUtils;

public class EnhancedKeyStoreLoader {
    private static final String MSG_NODE_ID_NON_NULL = "nodeId must not be null";
    private static final String MSG_NODE_ALIAS_NON_NULL = "nodeAlias must not be null";
    private static final String MSG_LOCATION_NON_NULL = "location must not be null";
    private static final String MSG_ENTRY_NON_NULL = "entry must not be null";
    private static final String MSG_KEY_STORE_DIRECTORY_NON_NULL = "keyStoreDirectory must not be null";
    private static final String MSG_KEY_STORE_PASSPHRASE_NON_NULL = "keyStorePassphrase must not be null";
    private static final String MSG_NODES_TO_START_NON_NULL = "the local nodes must not be null";
    private static final String MSG_ROSTER_ENTRIES_NON_NULL = "rosterEntries must not be null";
    private static final Logger logger = LogManager.getLogger(EnhancedKeyStoreLoader.class);
    private final Path keyStoreDirectory;
    private final char[] keyStorePassphrase;
    private final Map<NodeId, PrivateKey> sigPrivateKeys;
    private final Map<NodeId, Certificate> sigCertificates;
    private final Map<NodeId, PrivateKey> agrPrivateKeys;
    private final Map<NodeId, Certificate> agrCertificates;
    private final Set<NodeId> nodeIds;
    private final List<RosterEntry> rosterEntries;

    private EnhancedKeyStoreLoader(@NonNull Path keyStoreDirectory, @NonNull char[] keyStorePassphrase, @NonNull Set<NodeId> nodeIds, @NonNull List<RosterEntry> rosterEntries) {
        this.keyStoreDirectory = Objects.requireNonNull(keyStoreDirectory, MSG_KEY_STORE_DIRECTORY_NON_NULL);
        this.keyStorePassphrase = Objects.requireNonNull(keyStorePassphrase, MSG_KEY_STORE_PASSPHRASE_NON_NULL);
        this.sigPrivateKeys = HashMap.newHashMap(nodeIds.size());
        this.sigCertificates = HashMap.newHashMap(nodeIds.size());
        this.agrPrivateKeys = HashMap.newHashMap(nodeIds.size());
        this.agrCertificates = HashMap.newHashMap(nodeIds.size());
        this.nodeIds = Collections.unmodifiableSet(Objects.requireNonNull(nodeIds, MSG_NODES_TO_START_NON_NULL));
        this.rosterEntries = Collections.unmodifiableList(Objects.requireNonNull(rosterEntries, MSG_ROSTER_ENTRIES_NON_NULL));
    }

    @NonNull
    public static EnhancedKeyStoreLoader using(@NonNull Configuration configuration, @NonNull Set<NodeId> localNodes, @NonNull List<RosterEntry> rosterEntries) {
        Objects.requireNonNull(configuration, "configuration must not be null");
        Objects.requireNonNull(localNodes, MSG_NODES_TO_START_NON_NULL);
        String keyStorePassphrase = ((CryptoConfig)configuration.getConfigData(CryptoConfig.class)).keystorePassword();
        Path keyStoreDirectory = ((PathsConfig)configuration.getConfigData(PathsConfig.class)).getKeysDirPath();
        if (keyStorePassphrase == null || keyStorePassphrase.isBlank()) {
            throw new IllegalArgumentException("keyStorePassphrase must not be null or blank");
        }
        KeystorePasswordPolicy.warnIfNonCompliant((Logger)logger, (String)"crypto.keystorePassword", (String)keyStorePassphrase);
        return new EnhancedKeyStoreLoader(keyStoreDirectory, keyStorePassphrase.toCharArray(), localNodes, rosterEntries);
    }

    @NonNull
    public EnhancedKeyStoreLoader scan() throws KeyLoadingException, KeyStoreException {
        logger.debug(LogMarker.STARTUP.getMarker(), "Starting key store enumeration");
        for (NodeId nodeId : this.nodeIds) {
            logger.debug(LogMarker.STARTUP.getMarker(), "Attempting to locate key stores for nodeId {}", (Object)nodeId);
            if (this.nodeIds.contains(nodeId)) {
                this.sigPrivateKeys.compute(nodeId, (k, v) -> this.resolveNodePrivateKey(nodeId));
            }
            this.sigCertificates.compute(nodeId, (k, v) -> this.resolveNodeCertificate(nodeId));
        }
        logger.trace(LogMarker.STARTUP.getMarker(), "Completed key store enumeration");
        return this;
    }

    public EnhancedKeyStoreLoader generate() throws NoSuchAlgorithmException, NoSuchProviderException, KeyGeneratingException {
        for (NodeId nodeId : this.nodeIds) {
            if (this.agrPrivateKeys.containsKey(nodeId)) continue;
            logger.info(LogMarker.STARTUP.getMarker(), "Generating agreement key pair for local nodeId {}", (Object)nodeId);
            KeyPair agrKeyPair = KeysAndCertsGenerator.generateAgreementKeyPair();
            this.agrPrivateKeys.put(nodeId, agrKeyPair.getPrivate());
            PrivateKey privateSigningKey = this.sigPrivateKeys.get(nodeId);
            X509Certificate signingCert = (X509Certificate)this.sigCertificates.get(nodeId);
            if (privateSigningKey == null || signingCert == null) continue;
            PublicKey publicSigningKey = signingCert.getPublicKey();
            KeyPair signingKeyPair = new KeyPair(publicSigningKey, privateSigningKey);
            String dnA = CertificateUtils.distinguishedName((String)KeyCertPurpose.AGREEMENT.storeName(nodeId));
            X509Certificate agrCert = CertificateUtils.generateCertificate((String)dnA, (KeyPair)agrKeyPair, (String)signingCert.getSubjectX500Principal().getName(), (KeyPair)signingKeyPair, (SecureRandom)SecureRandom.getInstanceStrong(), (String)"SHA384withRSA");
            this.agrCertificates.put(nodeId, agrCert);
        }
        return this;
    }

    @NonNull
    public EnhancedKeyStoreLoader verify() throws KeyLoadingException, KeyStoreException {
        for (NodeId nodeId : this.nodeIds) {
            try {
                if (!this.sigPrivateKeys.containsKey(nodeId)) {
                    throw new KeyLoadingException("No private key found for nodeId %s [ purpose = %s ]".formatted(nodeId, KeyCertPurpose.SIGNING));
                }
                if (!this.agrPrivateKeys.containsKey(nodeId)) {
                    throw new KeyLoadingException("No private key found for nodeId %s [purpose = %s ]".formatted(nodeId, KeyCertPurpose.AGREEMENT));
                }
                if (!this.agrCertificates.containsKey(nodeId)) {
                    throw new KeyLoadingException("No certificate found for nodeId %s [purpose = %s ]".formatted(nodeId, KeyCertPurpose.AGREEMENT));
                }
                if (this.sigCertificates.containsKey(nodeId)) continue;
                throw new KeyLoadingException("No certificate found for nodeId %s [purpose = %s ]".formatted(nodeId, KeyCertPurpose.SIGNING));
            }
            catch (KeyLoadingException e) {
                logger.warn(LogMarker.STARTUP.getMarker(), e.getMessage());
                throw e;
            }
        }
        return this;
    }

    @NonNull
    public Map<NodeId, KeysAndCerts> keysAndCerts() throws KeyStoreException, KeyLoadingException {
        HashMap<NodeId, KeysAndCerts> keysAndCerts = HashMap.newHashMap(this.nodeIds.size());
        Map<NodeId, X509Certificate> signing = this.signingCertificates();
        for (NodeId nodeId : this.nodeIds) {
            Certificate agrCert = this.agrCertificates.get(nodeId);
            PrivateKey sigPrivateKey = this.sigPrivateKeys.get(nodeId);
            PrivateKey agrPrivateKey = this.agrPrivateKeys.get(nodeId);
            if (sigPrivateKey == null) {
                throw new KeyLoadingException("No signing private key found for nodeId: %s".formatted(nodeId));
            }
            if (agrPrivateKey == null) {
                throw new KeyLoadingException("No agreement private key found for nodeId: %s".formatted(nodeId));
            }
            if (agrCert == null) {
                throw new KeyLoadingException("No agreement certificate found for nodeId: %s".formatted(nodeId));
            }
            if (!(agrCert instanceof X509Certificate)) {
                throw new KeyLoadingException("Illegal agreement certificate type for nodeId: %s [ purpose = %s ]".formatted(nodeId, KeyCertPurpose.AGREEMENT));
            }
            X509Certificate x509AgrCert = (X509Certificate)agrCert;
            X509Certificate sigCert = signing.get(nodeId);
            KeyPair sigKeyPair = new KeyPair(sigCert.getPublicKey(), sigPrivateKey);
            KeyPair agrKeyPair = new KeyPair(agrCert.getPublicKey(), agrPrivateKey);
            KeysAndCerts kc = new KeysAndCerts(sigKeyPair, agrKeyPair, sigCert, x509AgrCert);
            keysAndCerts.put(nodeId, kc);
        }
        return keysAndCerts;
    }

    @NonNull
    private Map<NodeId, X509Certificate> signingCertificates() throws KeyLoadingException {
        HashMap<NodeId, X509Certificate> certs = HashMap.newHashMap(this.nodeIds.size());
        for (NodeId nodeId : this.nodeIds) {
            Certificate sigCert = this.sigCertificates.get(nodeId);
            if (sigCert == null) {
                throw new KeyLoadingException("No signing certificate found for nodeId: %s".formatted(nodeId));
            }
            if (!(sigCert instanceof X509Certificate)) {
                throw new KeyLoadingException("Illegal signing certificate type for nodeId: %s [ purpose = %s ]".formatted(nodeId, KeyCertPurpose.SIGNING));
            }
            X509Certificate x509SigCert = (X509Certificate)sigCert;
            certs.put(nodeId, x509SigCert);
        }
        return certs;
    }

    @Nullable
    private PrivateKey resolveNodePrivateKey(@NonNull NodeId nodeId) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Path ksLocation = this.privateKeyStore(nodeId);
        if (Files.exists(ksLocation, new LinkOption[0])) {
            logger.trace(LogMarker.STARTUP.getMarker(), "Found enhanced private key store for nodeId: {} [ purpose = {}, fileName = {} ]", (Object)nodeId, (Object)KeyCertPurpose.SIGNING, (Object)ksLocation.getFileName());
            return this.readPrivateKey(nodeId, ksLocation);
        }
        ksLocation = this.legacyPrivateKeyStore(nodeId);
        if (Files.exists(ksLocation, new LinkOption[0])) {
            logger.trace(LogMarker.STARTUP.getMarker(), "Found legacy private key store for nodeId: {} [ purpose = {}, fileName = {} ]", (Object)nodeId, (Object)KeyCertPurpose.SIGNING, (Object)ksLocation.getFileName());
            return this.readLegacyPrivateKey(nodeId, ksLocation, KeyCertPurpose.SIGNING.storeName(nodeId));
        }
        logger.warn(LogMarker.STARTUP.getMarker(), "No private key store found for nodeId: {} [ purpose = {} ]", (Object)nodeId, (Object)KeyCertPurpose.SIGNING);
        return null;
    }

    @Nullable
    private Certificate resolveNodeCertificate(@NonNull NodeId nodeId) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        return this.rosterEntries.stream().filter(e -> e.nodeId() == nodeId.id()).map(RosterUtils::fetchGossipCaCertificate).filter(Objects::nonNull).findFirst().orElse(null);
    }

    @Nullable
    PrivateKey readPrivateKey(@NonNull NodeId nodeId, @NonNull Path location) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        try {
            return this.readEnhancedStore(location);
        }
        catch (KeyLoadingException e) {
            logger.warn(LogMarker.STARTUP.getMarker(), "Unable to load the enhanced private key store for nodeId: {} [ fileName = {} ]", (Object)nodeId, (Object)location.getFileName(), (Object)e);
            return null;
        }
    }

    @Nullable
    private PrivateKey readLegacyPrivateKey(@NonNull NodeId nodeId, @NonNull Path location, @NonNull String entryName) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        try {
            PrivateKey pk;
            KeyStore ks = CryptoStatic.loadKeys(location, this.keyStorePassphrase);
            Key k = ks.getKey(entryName, this.keyStorePassphrase);
            if (!(k instanceof PrivateKey)) {
                logger.warn(LogMarker.STARTUP.getMarker(), "No private key found for nodeId: {} [ entryName = {} ]", (Object)nodeId, (Object)entryName);
            }
            return k instanceof PrivateKey ? (pk = (PrivateKey)k) : null;
        }
        catch (KeyLoadingException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            logger.warn(LogMarker.STARTUP.getMarker(), "Unable to load the legacy private key store [ fileName = {} ]", (Object)location.getFileName(), (Object)e);
            return null;
        }
    }

    @NonNull
    private Path privateKeyStore(@NonNull NodeId nodeId) {
        return this.keyStoreDirectory.resolve(String.format("%s-private-%s.pem", KeyCertPurpose.SIGNING.prefix(), NodeUtilities.formatNodeName((NodeId)nodeId)));
    }

    @NonNull
    private Path legacyPrivateKeyStore(@NonNull NodeId nodeId) {
        Objects.requireNonNull(nodeId, MSG_NODE_ALIAS_NON_NULL);
        return this.keyStoreDirectory.resolve(String.format("private-%s.pfx", NodeUtilities.formatNodeName((NodeId)nodeId)));
    }

    @NonNull
    private PrivateKey readEnhancedStore(@NonNull Path location) throws KeyLoadingException {
        PrivateKey privateKey;
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        PEMParser parser = new PEMParser((Reader)new InputStreamReader(Files.newInputStream(location, new OpenOption[0]), StandardCharsets.UTF_8));
        try {
            Object entry;
            while (!((entry = parser.readObject()) == null || entry instanceof PEMKeyPair || entry instanceof PrivateKeyInfo || entry instanceof PKCS8EncryptedPrivateKeyInfo || entry instanceof PEMEncryptedKeyPair)) {
            }
            if (entry == null) {
                throw new KeyLoadingException("No entry of the requested Private Key found [ fileName = %s ]".formatted(location.getFileName()));
            }
            privateKey = this.extractPrivateKeyEntity(entry);
        }
        catch (Throwable throwable) {
            try {
                try {
                    parser.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException | DecoderException e) {
                throw new KeyLoadingException("Unable to read enhanced store [ fileName = %s ]".formatted(location.getFileName()), e);
            }
        }
        parser.close();
        return privateKey;
    }

    @NonNull
    private PrivateKey extractPrivateKeyEntity(@NonNull Object entry) throws KeyLoadingException {
        Objects.requireNonNull(entry, MSG_ENTRY_NON_NULL);
        try {
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter();
            PEMDecryptorProvider decrypter = new JcePEMDecryptorProviderBuilder().build(this.keyStorePassphrase);
            InputDecryptorProvider inputDecrypter = new JceInputDecryptorProviderBuilder().build(new String(this.keyStorePassphrase).getBytes());
            Object object = entry;
            Objects.requireNonNull(object);
            Object object2 = object;
            int n = 0;
            return switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{PrivateKeyInfo.class, PKCS8EncryptedPrivateKeyInfo.class, PEMKeyPair.class, PEMEncryptedKeyPair.class}, (Object)object2, n)) {
                case 0 -> {
                    PrivateKeyInfo pki = (PrivateKeyInfo)object2;
                    yield converter.getPrivateKey(pki);
                }
                case 1 -> {
                    PKCS8EncryptedPrivateKeyInfo epki = (PKCS8EncryptedPrivateKeyInfo)object2;
                    yield converter.getPrivateKey(epki.decryptPrivateKeyInfo(inputDecrypter));
                }
                case 2 -> {
                    PEMKeyPair kp = (PEMKeyPair)object2;
                    yield converter.getPrivateKey(kp.getPrivateKeyInfo());
                }
                case 3 -> {
                    PEMEncryptedKeyPair ekp = (PEMEncryptedKeyPair)object2;
                    yield converter.getPrivateKey(ekp.decryptKeyPair(decrypter).getPrivateKeyInfo());
                }
                default -> throw new KeyLoadingException("Unsupported entry type [ entryType = %s ]".formatted(entry.getClass().getName()));
            };
        }
        catch (IOException | PKCSException e) {
            throw new KeyLoadingException("Unable to extract a private key from the specified entry [ entryType = %s ]".formatted(entry.getClass().getName()), e);
        }
    }

    @NonNull
    public EnhancedKeyStoreLoader migrate() throws KeyLoadingException, KeyStoreException {
        logger.info(LogMarker.STARTUP.getMarker(), "Starting key store migration");
        HashMap<NodeId, PrivateKey> pfxPrivateKeys = new HashMap<NodeId, PrivateKey>();
        this.deleteAgreementKeys();
        long errorCount = this.extractPrivateKeysFromPfxFiles(pfxPrivateKeys);
        if (errorCount == 0L) {
            errorCount = this.validateKeysAreLoadableFromPemFiles(pfxPrivateKeys);
        }
        if (errorCount > 0L) {
            logger.error(LogMarker.STARTUP.getMarker(), "Due to {} errors, reverting pem file creation.", (Object)errorCount);
            this.rollBackSigningKeysChanges(pfxPrivateKeys);
        } else {
            this.cleanupByMovingPfxFilesToSubDirectory();
            logger.info(LogMarker.STARTUP.getMarker(), "Finished key store migration.");
        }
        return this;
    }

    private void deleteAgreementKeys() {
        File[] agreementKeyFiles = this.keyStoreDirectory.toFile().listFiles((dir, name) -> name.startsWith("a-"));
        if (agreementKeyFiles != null) {
            for (File agreementKeyFile : agreementKeyFiles) {
                if (!agreementKeyFile.isFile()) continue;
                try {
                    Files.delete(agreementKeyFile.toPath());
                    logger.debug(LogMarker.STARTUP.getMarker(), "Deleted agreement key file {}", (Object)agreementKeyFile.getName());
                }
                catch (IOException e) {
                    logger.error(LogMarker.ERROR.getMarker(), "Failed to delete agreement key file {}", (Object)agreementKeyFile.getName());
                }
            }
        }
    }

    private long extractPrivateKeysFromPfxFiles(Map<NodeId, PrivateKey> pfxPrivateKeys) {
        AtomicLong errorCount = new AtomicLong(0L);
        for (NodeId nodeId : this.nodeIds) {
            Path sPrivateKeyLocation = this.keyStoreDirectory.resolve(String.format("s-private-%s.pem", NodeUtilities.formatNodeName((NodeId)nodeId)));
            Path privateKs = this.legacyPrivateKeyStore(nodeId);
            if (Files.exists(sPrivateKeyLocation, new LinkOption[0]) || !Files.exists(privateKs, new LinkOption[0])) continue;
            logger.info(LogMarker.STARTUP.getMarker(), "Extracting private signing key for nodeId: {} from file {}", (Object)nodeId, (Object)privateKs.getFileName());
            PrivateKey privateKey = this.readLegacyPrivateKey(nodeId, privateKs, KeyCertPurpose.SIGNING.storeName(nodeId));
            pfxPrivateKeys.put(nodeId, privateKey);
            if (privateKey == null) {
                logger.error(LogMarker.ERROR.getMarker(), "Failed to extract private signing key for nodeId: {} from file {}", (Object)nodeId, (Object)privateKs.getFileName());
                errorCount.incrementAndGet();
                continue;
            }
            logger.info(LogMarker.STARTUP.getMarker(), "Writing private signing key for nodeId: {} to PEM file {}", (Object)nodeId, (Object)sPrivateKeyLocation.getFileName());
            try {
                EnhancedKeyStoreLoader.writePemFile(true, sPrivateKeyLocation, privateKey.getEncoded());
            }
            catch (IOException e) {
                logger.error(LogMarker.ERROR.getMarker(), "Failed to write private key for nodeId: {} to PEM file {}", (Object)nodeId, (Object)sPrivateKeyLocation.getFileName());
                errorCount.incrementAndGet();
            }
        }
        return errorCount.get();
    }

    private long validateKeysAreLoadableFromPemFiles(Map<NodeId, PrivateKey> pfxPrivateKeys) {
        AtomicLong errorCount = new AtomicLong(0L);
        this.rosterEntries.stream().map(e -> NodeId.of((long)e.nodeId())).filter(this.nodeIds::contains).forEach(nodeId -> {
            Path ksLocation = this.privateKeyStore((NodeId)nodeId);
            PrivateKey pemPrivateKey = this.readPrivateKey((NodeId)nodeId, ksLocation);
            if (pemPrivateKey == null || pfxPrivateKeys.get(nodeId) != null && !Arrays.equals(pemPrivateKey.getEncoded(), ((PrivateKey)pfxPrivateKeys.get(nodeId)).getEncoded())) {
                logger.error(LogMarker.ERROR.getMarker(), "Private key for nodeId: {} does not match the migrated key", nodeId);
                errorCount.incrementAndGet();
            }
        });
        return errorCount.get();
    }

    private void rollBackSigningKeysChanges(Map<NodeId, PrivateKey> pfxPrivateKeys) {
        AtomicLong cleanupErrorCount = new AtomicLong(0L);
        for (NodeId nodeId : this.nodeIds) {
            if (!pfxPrivateKeys.containsKey(nodeId)) continue;
            try {
                Files.deleteIfExists(this.privateKeyStore(nodeId));
            }
            catch (IOException e) {
                cleanupErrorCount.incrementAndGet();
            }
        }
        if (cleanupErrorCount.get() > 0L) {
            logger.error(LogMarker.ERROR.getMarker(), "Failed to rollback {} pem files created. Manual cleanup required.", (Object)cleanupErrorCount.get());
            throw new IllegalStateException("Cryptography Migration failed to generate or validate PEM files.");
        }
    }

    private void cleanupByMovingPfxFilesToSubDirectory() {
        AtomicLong cleanupErrorCount = new AtomicLong(0L);
        AtomicBoolean doCleanup = new AtomicBoolean(false);
        for (NodeId nodeId : this.nodeIds) {
            File sPrivatePfx = this.legacyPrivateKeyStore(nodeId).toFile();
            if (!sPrivatePfx.exists() || !sPrivatePfx.isFile()) continue;
            doCleanup.set(true);
        }
        if (!doCleanup.get()) {
            return;
        }
        String archiveDirectory = ".archive";
        String now = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss").format(LocalDateTime.now());
        String newDirectory = ".archive" + File.pathSeparator + now;
        Path pfxArchiveDirectory = this.keyStoreDirectory.resolve(".archive");
        Path pfxDateDirectory = pfxArchiveDirectory.resolve(now);
        logger.info(LogMarker.STARTUP.getMarker(), "Cryptography Migration Cleanup: Moving PFX files to {}", (Object)pfxDateDirectory);
        if (!Files.exists(pfxDateDirectory, new LinkOption[0])) {
            try {
                if (!Files.exists(pfxArchiveDirectory, new LinkOption[0])) {
                    Files.createDirectory(pfxArchiveDirectory, new FileAttribute[0]);
                }
                Files.createDirectory(pfxDateDirectory, new FileAttribute[0]);
            }
            catch (IOException e) {
                logger.error(LogMarker.ERROR.getMarker(), "Failed to create [{}] subdirectory. Manual cleanup required.", (Object)newDirectory);
                return;
            }
        }
        for (NodeId nodeId : this.nodeIds) {
            File sPrivatePfx = this.legacyPrivateKeyStore(nodeId).toFile();
            if (!sPrivatePfx.exists() || !sPrivatePfx.isFile() || sPrivatePfx.renameTo(pfxDateDirectory.resolve(sPrivatePfx.getName()).toFile())) continue;
            cleanupErrorCount.incrementAndGet();
        }
        if (cleanupErrorCount.get() > 0L) {
            logger.error(LogMarker.ERROR.getMarker(), "Failed to move {} PFX files to [{}] subdirectory. Manual cleanup required.", (Object)cleanupErrorCount.get(), (Object)newDirectory);
            throw new IllegalStateException("Cryptography Migration failed to move PFX files to [" + newDirectory + "] subdirectory.");
        }
    }

    public static void writePemFile(boolean isPrivateKey, @NonNull Path location, @NonNull byte[] encoded) throws IOException {
        PemObject pemObj = new PemObject(isPrivateKey ? "PRIVATE KEY" : "CERTIFICATE", encoded);
        try (FileOutputStream file = new FileOutputStream(location.toFile(), false);
             OutputStreamWriter out = new OutputStreamWriter(file);
             PemWriter writer = new PemWriter((Writer)out);){
            writer.writeObject((PemObjectGenerator)pemObj);
            file.getFD().sync();
        }
    }

    static {
        if (Arrays.stream(Security.getProviders()).noneMatch(p -> p instanceof BouncyCastleProvider)) {
            Security.addProvider((Provider)new BouncyCastleProvider());
        }
    }
}

