/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.files;

import com.hedera.pbj.runtime.io.buffer.BufferedData;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.merkledb.KeyRange;
import com.swirlds.merkledb.collections.CASableLongIndex;
import com.swirlds.merkledb.config.MerkleDbConfig;
import com.swirlds.merkledb.files.DataFileCollection;
import com.swirlds.merkledb.files.DataFileCommon;
import com.swirlds.merkledb.files.DataFileMetadata;
import com.swirlds.merkledb.files.DataFileReader;
import com.swirlds.merkledb.files.DataFileWriter;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class DataFileCompactor {
    private static final Logger logger = LogManager.getLogger(DataFileCompactor.class);
    public static final String HASH_STORE_DISK = "HashStoreDisk";
    public static final String OBJECT_KEY_TO_PATH = "ObjectKeyToPath";
    public static final String PATH_TO_KEY_VALUE = "PathToKeyValue";
    public static final int INITIAL_COMPACTION_LEVEL = 0;
    private final MerkleDbConfig dbConfig;
    private final String storeName;
    private final DataFileCollection dataFileCollection;
    private final CASableLongIndex index;
    @Nullable
    private final BiConsumer<Integer, Long> reportDurationMetricFunction;
    @Nullable
    private final BiConsumer<Integer, Double> reportSavedSpaceMetricFunction;
    private final BiConsumer<Integer, Double> reportFileSizeByLevelMetricFunction;
    @Nullable
    private final Runnable updateTotalStatsFunction;
    private final Lock snapshotCompactionLock = new ReentrantLock();
    private final AtomicReference<Instant> currentCompactionStartTime = new AtomicReference();
    private final AtomicReference<DataFileWriter> currentWriter = new AtomicReference();
    private final AtomicReference<DataFileReader> currentReader = new AtomicReference();
    private final List<Path> newCompactedFiles = new ArrayList<Path>();
    private boolean compactionWasInProgress = false;
    private int compactionLevelInProgress = 0;
    private volatile boolean interruptFlag = false;

    public DataFileCompactor(MerkleDbConfig dbConfig, String storeName, DataFileCollection dataFileCollection, CASableLongIndex index, @Nullable BiConsumer<Integer, Long> reportDurationMetricFunction, @Nullable BiConsumer<Integer, Double> reportSavedSpaceMetricFunction, @Nullable BiConsumer<Integer, Double> reportFileSizeByLevelMetricFunction, @Nullable Runnable updateTotalStatsFunction) {
        this.dbConfig = dbConfig;
        this.storeName = storeName;
        this.dataFileCollection = dataFileCollection;
        this.index = index;
        this.reportDurationMetricFunction = reportDurationMetricFunction;
        this.reportSavedSpaceMetricFunction = reportSavedSpaceMetricFunction;
        this.reportFileSizeByLevelMetricFunction = reportFileSizeByLevelMetricFunction;
        this.updateTotalStatsFunction = updateTotalStatsFunction;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    List<Path> compactFiles(CASableLongIndex index, List<? extends DataFileReader> filesToCompact, int targetCompactionLevel) throws IOException, InterruptedException {
        if (filesToCompact.size() < this.getMinNumberOfFilesToCompact()) {
            logger.debug(LogMarker.MERKLE_DB.getMarker(), "No files were available for merging [{}]", (Object)this.storeName);
            return Collections.emptyList();
        }
        if (this.interruptFlag) {
            return Collections.emptyList();
        }
        Instant startTime = filesToCompact.stream().map(file -> file.getMetadata().getCreationDate()).max(Instant::compareTo).orElseGet(Instant::now);
        this.snapshotCompactionLock.lock();
        try {
            this.currentCompactionStartTime.set(startTime);
            this.newCompactedFiles.clear();
            this.startNewCompactionFile(targetCompactionLevel);
        }
        finally {
            this.snapshotCompactionLock.unlock();
        }
        int minFileIndex = Integer.MAX_VALUE;
        int maxFileIndex = 0;
        for (DataFileReader dataFileReader : filesToCompact) {
            minFileIndex = Math.min(minFileIndex, dataFileReader.getIndex());
            maxFileIndex = Math.max(maxFileIndex, dataFileReader.getIndex());
        }
        int firstIndexInc = minFileIndex;
        int n = maxFileIndex + 1;
        DataFileReader[] readers = new DataFileReader[n - firstIndexInc];
        Iterator<? extends DataFileReader> iterator = filesToCompact.iterator();
        while (iterator.hasNext()) {
            DataFileReader r;
            readers[r.getIndex() - firstIndexInc] = r = iterator.next();
        }
        boolean allDataItemsProcessed = false;
        try {
            KeyRange keyRange = this.dataFileCollection.getValidKeyRange();
            allDataItemsProcessed = index.forEach((path, dataLocation) -> {
                if (!keyRange.withinRange(path)) {
                    return;
                }
                int fileIndex = DataFileCommon.fileIndexFromDataLocation(dataLocation);
                if (fileIndex < firstIndexInc || fileIndex >= lastIndexExc) {
                    return;
                }
                DataFileReader reader = readers[fileIndex - firstIndexInc];
                if (reader == null) {
                    return;
                }
                long fileOffset = DataFileCommon.byteOffsetFromDataLocation(dataLocation);
                this.snapshotCompactionLock.lock();
                try {
                    DataFileWriter newFileWriter = this.currentWriter.get();
                    BufferedData itemBytesWithTag = reader.readDataItemWithTag(fileOffset);
                    assert (itemBytesWithTag != null);
                    long newLocation = newFileWriter.storeDataItemWithTag(itemBytesWithTag);
                    index.putIfEqual(path, dataLocation, newLocation);
                }
                catch (IOException z) {
                    logger.error(LogMarker.EXCEPTION.getMarker(), "[{}] Failed to copy data item {} / {}", (Object)this.storeName, (Object)fileIndex, (Object)fileOffset, (Object)z);
                    throw z;
                }
                finally {
                    this.snapshotCompactionLock.unlock();
                }
            }, this::notInterrupted);
        }
        finally {
            this.snapshotCompactionLock.lock();
            try {
                this.finishCurrentCompactionFile();
                this.currentCompactionStartTime.set(null);
                if (allDataItemsProcessed) {
                    logger.info(LogMarker.MERKLE_DB.getMarker(), "All files to compact have been processed, they will be deleted");
                    this.dataFileCollection.deleteFiles(filesToCompact);
                } else {
                    logger.info(LogMarker.MERKLE_DB.getMarker(), "Some files to compact haven't been processed, they will be compacted later");
                }
            }
            finally {
                this.snapshotCompactionLock.unlock();
            }
        }
        return this.newCompactedFiles;
    }

    int getMinNumberOfFilesToCompact() {
        return this.dbConfig.minNumberOfFilesInCompaction();
    }

    private void startNewCompactionFile(int compactionLevel) throws IOException {
        Instant startTime = this.currentCompactionStartTime.get();
        assert (startTime != null);
        DataFileWriter newFileWriter = this.dataFileCollection.newDataFile(startTime, compactionLevel);
        this.currentWriter.set(newFileWriter);
        Path newFileCreated = newFileWriter.getPath();
        this.newCompactedFiles.add(newFileCreated);
        DataFileMetadata newFileMetadata = newFileWriter.getMetadata();
        DataFileReader newFileReader = this.dataFileCollection.addNewDataFileReader(newFileCreated, newFileMetadata);
        this.currentReader.set(newFileReader);
        logger.info(LogMarker.MERKLE_DB.getMarker(), "[{}] New compaction file, newFile={}", (Object)this.storeName, (Object)newFileReader.getIndex());
    }

    private void finishCurrentCompactionFile() throws IOException {
        this.currentWriter.get().close();
        this.currentWriter.set(null);
        this.currentReader.get().setFileCompleted();
        logger.info(LogMarker.MERKLE_DB.getMarker(), "[{}] Compaction file written, fileNum={}", (Object)this.storeName, (Object)this.currentReader.get().getIndex());
        this.currentReader.set(null);
    }

    public void pauseCompaction() throws IOException {
        this.snapshotCompactionLock.lock();
        DataFileWriter compactionWriter = this.currentWriter.get();
        if (compactionWriter != null) {
            this.compactionWasInProgress = true;
            this.compactionLevelInProgress = compactionWriter.getMetadata().getCompactionLevel();
            this.finishCurrentCompactionFile();
        }
    }

    public void resumeCompaction() throws IOException {
        try {
            if (this.compactionWasInProgress) {
                this.compactionWasInProgress = false;
                assert (this.currentWriter.get() == null);
                assert (this.currentReader.get() == null);
                this.startNewCompactionFile(this.compactionLevelInProgress);
                this.compactionLevelInProgress = 0;
            }
        }
        finally {
            this.snapshotCompactionLock.unlock();
        }
    }

    public void interruptCompaction() {
        this.interruptFlag = true;
    }

    public boolean notInterrupted() {
        return !this.interruptFlag;
    }

    public boolean isCompactionRunning() {
        this.snapshotCompactionLock.lock();
        try {
            boolean bl = this.currentCompactionStartTime.get() != null;
            return bl;
        }
        finally {
            this.snapshotCompactionLock.unlock();
        }
    }

    public boolean isCompactionComplete() {
        this.snapshotCompactionLock.lock();
        try {
            boolean bl = this.currentCompactionStartTime.get() == null && !this.newCompactedFiles.isEmpty();
            return bl;
        }
        finally {
            this.snapshotCompactionLock.unlock();
        }
    }

    public boolean compact() throws IOException, InterruptedException {
        List<DataFileReader> completedFiles = this.dataFileCollection.getAllCompletedFiles();
        this.reportFileSizeByLevel(completedFiles);
        List<DataFileReader> filesToCompact = DataFileCompactor.compactionPlan(completedFiles, this.getMinNumberOfFilesToCompact(), this.dbConfig.maxCompactionLevel());
        if (filesToCompact.isEmpty()) {
            logger.debug(LogMarker.MERKLE_DB.getMarker(), "[{}] No need to compact, as the compaction plan is empty", (Object)this.storeName);
            return false;
        }
        int filesCount = filesToCompact.size();
        logger.info(LogMarker.MERKLE_DB.getMarker(), "[{}] Starting compaction", (Object)this.storeName);
        int targetCompactionLevel = this.getTargetCompactionLevel(filesToCompact, filesCount);
        long start = System.currentTimeMillis();
        long filesToCompactSize = DataFileCommon.getSizeOfFiles(filesToCompact);
        logger.debug(LogMarker.MERKLE_DB.getMarker(), "[{}] Starting merging {} files / {}", (Object)this.storeName, (Object)filesCount, (Object)DataFileCommon.formatSizeBytes(filesToCompactSize));
        List<Path> newFilesCreated = this.compactFiles(this.index, filesToCompact, targetCompactionLevel);
        long end = System.currentTimeMillis();
        long tookMillis = end - start;
        if (this.reportDurationMetricFunction != null) {
            this.reportDurationMetricFunction.accept(targetCompactionLevel, tookMillis);
        }
        long compactedFilesSize = DataFileCommon.getSizeOfFilesByPath(newFilesCreated);
        if (this.reportSavedSpaceMetricFunction != null) {
            this.reportSavedSpaceMetricFunction.accept(targetCompactionLevel, (double)(filesToCompactSize - compactedFilesSize) * 9.5367431640625E-7);
        }
        this.reportFileSizeByLevel(this.dataFileCollection.getAllCompletedFiles());
        DataFileCommon.logCompactStats(this.storeName, tookMillis, filesToCompact, filesToCompactSize, newFilesCreated, targetCompactionLevel, this.dataFileCollection);
        logger.info(LogMarker.MERKLE_DB.getMarker(), "[{}] Finished compaction {} files / {} in {} ms", (Object)this.storeName, (Object)filesCount, (Object)DataFileCommon.formatSizeBytes(filesToCompactSize), (Object)tookMillis);
        if (this.updateTotalStatsFunction != null) {
            this.updateTotalStatsFunction.run();
        }
        return true;
    }

    private void reportFileSizeByLevel(List<DataFileReader> allCompletedFiles) {
        if (this.reportFileSizeByLevelMetricFunction != null) {
            Map<Integer, List<DataFileReader>> readersByLevel = DataFileCompactor.getReadersByLevel(allCompletedFiles);
            for (int i = 0; i < readersByLevel.size(); ++i) {
                List<DataFileReader> readers = readersByLevel.get(i);
                if (readers == null) continue;
                this.reportFileSizeByLevelMetricFunction.accept(i, (double)DataFileCommon.getSizeOfFiles(readers) * 9.5367431640625E-7);
            }
        }
    }

    private int getTargetCompactionLevel(List<? extends DataFileReader> filesToCompact, int filesCount) {
        int highestExistingCompactionLevel = filesToCompact.get(filesCount - 1).getMetadata().getCompactionLevel();
        return Math.min(highestExistingCompactionLevel + 1, this.dbConfig.maxCompactionLevel());
    }

    static List<DataFileReader> compactionPlan(List<DataFileReader> dataFileReaders, int minNumberOfFilesToCompact, int maxCompactionLevel) {
        List<DataFileReader> readers;
        if (dataFileReaders.isEmpty()) {
            return dataFileReaders;
        }
        Map<Integer, List<DataFileReader>> readersByLevel = DataFileCompactor.getReadersByLevel(dataFileReaders);
        List<DataFileReader> nonCompactedReaders = readersByLevel.get(0);
        if (nonCompactedReaders == null || nonCompactedReaders.size() < minNumberOfFilesToCompact) {
            return Collections.emptyList();
        }
        ArrayList<DataFileReader> readersToCompact = new ArrayList<DataFileReader>(nonCompactedReaders);
        for (int i = 1; i <= maxCompactionLevel && (readers = readersByLevel.get(i)) != null && readers.size() >= minNumberOfFilesToCompact - 1; ++i) {
            readersToCompact.addAll(readers);
        }
        return readersToCompact;
    }

    private static Map<Integer, List<DataFileReader>> getReadersByLevel(List<DataFileReader> dataFileReaders) {
        return dataFileReaders.stream().collect(Collectors.groupingBy(r -> r.getMetadata().getCompactionLevel()));
    }
}

