/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.common.io.utility;

import com.swirlds.common.io.utility.IOConsumer;
import com.swirlds.common.io.utility.IORunnable;
import com.swirlds.common.io.utility.IOSupplier;
import com.swirlds.common.io.utility.LegacyTemporaryFileBuilder;
import com.swirlds.config.api.Configuration;
import com.swirlds.logging.legacy.LogMarker;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.base.io.streams.SerializableDataOutputStream;

public final class FileUtils {
    private static final Logger logger = LogManager.getLogger(FileUtils.class);

    private FileUtils() {
    }

    public static void rethrowIO(@NonNull IORunnable runnable) {
        try {
            runnable.run();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @NonNull
    public static <T> T rethrowIO(@NonNull IOSupplier<T> supplier) {
        try {
            return supplier.get();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @NonNull
    public static Path getAbsolutePath() {
        return FileUtils.getAbsolutePath(".");
    }

    @NonNull
    public static Path getAbsolutePath(@NonNull String pathDescription) {
        String expandedPath = pathDescription.replaceFirst("^~", System.getProperty("user.home"));
        return FileSystems.getDefault().getPath(expandedPath, new String[0]).toAbsolutePath().normalize();
    }

    @NonNull
    public static Path getAbsolutePath(@NonNull Path path) {
        return FileUtils.getAbsolutePath(path.toString());
    }

    public static void delete(@NonNull Path pathToDelete) throws IOException {
        if (Files.isDirectory(pathToDelete, new LinkOption[0])) {
            FileUtils.deleteDirectory(pathToDelete);
        } else {
            Files.deleteIfExists(pathToDelete);
        }
    }

    public static void deleteDirectory(@NonNull Path directoryToBeDeleted) throws IOException {
        if (Files.isDirectory(directoryToBeDeleted, new LinkOption[0])) {
            try (Stream<Path> list = Files.list(directoryToBeDeleted);){
                Iterator children = list.iterator();
                while (children.hasNext()) {
                    FileUtils.deleteDirectory((Path)children.next());
                }
            }
            Files.delete(directoryToBeDeleted);
        } else {
            Files.deleteIfExists(directoryToBeDeleted);
        }
    }

    public static void deleteDirectoryAndLog(@NonNull Path directoryToBeDeleted) throws IOException {
        logger.info(LogMarker.STATE_TO_DISK.getMarker(), "deleting directory {}", (Object)directoryToBeDeleted);
        try {
            FileUtils.deleteDirectory(directoryToBeDeleted);
        }
        catch (Exception e) {
            logger.error(LogMarker.EXCEPTION.getMarker(), "failed to delete directory {}", (Object)directoryToBeDeleted);
            throw e;
        }
    }

    public static void hardLinkTree(@NonNull Path source, @NonNull Path destination) throws IOException {
        if (!Files.exists(source, new LinkOption[0])) {
            throw new IOException(String.valueOf(source) + " does not exist or can not be accessed");
        }
        if (Files.exists(destination, new LinkOption[0])) {
            throw new IOException(String.valueOf(destination) + " already exists");
        }
        try (Stream<Path> files = Files.walk(source, new FileVisitOption[0]);){
            files.filter(x$0 -> Files.isDirectory(x$0, new LinkOption[0])).forEach(originalDirectoryPath -> {
                Path relativeDirectoryPath = source.relativize((Path)originalDirectoryPath);
                Path newDirectoryPath = destination.resolve(relativeDirectoryPath);
                try {
                    Files.createDirectories(newDirectoryPath, new FileAttribute[0]);
                }
                catch (IOException ex) {
                    throw new UncheckedIOException(ex);
                }
            });
        }
        files = Files.walk(source, new FileVisitOption[0]);
        try {
            files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).forEach(originalFilePath -> {
                Path relativeFilePath = source.relativize((Path)originalFilePath);
                Path newFilePath = destination.resolve(relativeFilePath);
                try {
                    Files.createLink(newFilePath, originalFilePath);
                }
                catch (IOException ex) {
                    throw new UncheckedIOException(ex);
                }
            });
        }
        finally {
            if (files != null) {
                files.close();
            }
        }
    }

    public static void throwIfFileExists(Path ... files) throws IOException {
        if (files == null) {
            return;
        }
        for (Path file : files) {
            if (!Files.exists(file, new LinkOption[0])) continue;
            throw new FileAlreadyExistsException("File " + String.valueOf(file) + " already exists");
        }
    }

    public static void executeAndRename(@NonNull Path directory, @NonNull IOConsumer<Path> operation, @NonNull Configuration configuration) throws IOException {
        Objects.requireNonNull(directory);
        Objects.requireNonNull(configuration);
        FileUtils.executeAndRename(directory, LegacyTemporaryFileBuilder.buildTemporaryDirectory(configuration), operation);
    }

    public static void executeAndRename(@NonNull Path directory, @NonNull Path tmpDirectory, @NonNull IOConsumer<Path> operation) throws IOException {
        try {
            FileUtils.throwIfFileExists(directory);
            if (!Files.exists(tmpDirectory, new LinkOption[0])) {
                Files.createDirectories(tmpDirectory, new FileAttribute[0]);
            }
            if (!Files.exists(directory.getParent(), new LinkOption[0])) {
                Files.createDirectories(directory.getParent(), new FileAttribute[0]);
            }
            operation.accept(tmpDirectory);
            Files.move(tmpDirectory, directory, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (Throwable ex) {
            logger.info(LogMarker.STATE_TO_DISK.getMarker(), "deleting temporary file due to exception");
            throw ex;
        }
        finally {
            try {
                FileUtils.deleteDirectory(tmpDirectory);
            }
            catch (Throwable t) {
                logger.warn(LogMarker.STATE_TO_DISK.getMarker(), "unable to delete temporary directory {} due to {}", (Object)tmpDirectory, (Object)t);
            }
        }
    }

    public static void writeAndFlush(@NonNull Path file, @NonNull IOConsumer<SerializableDataOutputStream> writeMethod) throws IOException {
        FileUtils.throwIfFileExists(file);
        try (FileOutputStream fileOut = new FileOutputStream(file.toFile());
             BufferedOutputStream bufOut = new BufferedOutputStream(fileOut);
             SerializableDataOutputStream out = new SerializableDataOutputStream((OutputStream)bufOut);){
            writeMethod.accept(out);
            out.flush();
            fileOut.getFD().sync();
        }
    }

    @NonNull
    public static List<Path> findFiles(@NonNull Path dir, @NonNull String suffix) throws IOException {
        try (Stream<Path> files = Files.walk(dir, new FileVisitOption[0]);){
            List<Path> list = files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(f -> f.toString().endsWith(suffix)).toList();
            return list;
        }
    }

    public static void deleteFiles(@NonNull Path dir, @NonNull String suffix) throws IOException {
        List<Path> files = FileUtils.findFiles(dir, suffix);
        for (Path file : files) {
            Files.delete(file);
        }
    }

    public static void moveDirectory(final @NonNull Path source, final @NonNull Path target) throws IOException {
        Objects.requireNonNull(source, "Source path cannot be null");
        Objects.requireNonNull(target, "Target path cannot be null");
        if (!Files.exists(source, new LinkOption[0])) {
            throw new IOException(String.valueOf(source) + " does not exist");
        }
        if (!Files.exists(target, new LinkOption[0])) {
            Files.createDirectories(target, new FileAttribute[0]);
        }
        Files.walkFileTree(source, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            @NonNull
            public FileVisitResult preVisitDirectory(@NonNull Path dir, @NonNull BasicFileAttributes attrs) throws IOException {
                Path relative = source.relativize(dir);
                Path targetDir = target.resolve(relative);
                Files.createDirectories(targetDir, new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }

            @Override
            @NonNull
            public FileVisitResult visitFile(@NonNull Path file, @NonNull BasicFileAttributes attrs) throws IOException {
                Path relative = source.relativize(file);
                Path targetFile = target.resolve(relative);
                Files.move(file, targetFile, StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }

            @Override
            @NonNull
            public FileVisitResult postVisitDirectory(@NonNull Path dir, @Nullable IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void copyDirectory(final @NonNull Path source, final @NonNull Path target) throws IOException {
        Objects.requireNonNull(source, "Source path cannot be null");
        Objects.requireNonNull(target, "Target path cannot be null");
        if (!Files.exists(source, new LinkOption[0])) {
            throw new IOException(String.valueOf(source) + " does not exist");
        }
        if (!Files.exists(target, new LinkOption[0])) {
            Files.createDirectories(target, new FileAttribute[0]);
        }
        Files.walkFileTree(source, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            @NonNull
            public FileVisitResult preVisitDirectory(@NonNull Path dir, @NonNull BasicFileAttributes attrs) throws IOException {
                Path relative = source.relativize(dir);
                Path targetDir = target.resolve(relative);
                Files.createDirectories(targetDir, new FileAttribute[0]);
                return FileVisitResult.CONTINUE;
            }

            @Override
            @NonNull
            public FileVisitResult visitFile(@NonNull Path file, @NonNull BasicFileAttributes attrs) throws IOException {
                Path relative = source.relativize(file);
                Path targetFile = target.resolve(relative);
                Files.copy(file, targetFile, StandardCopyOption.REPLACE_EXISTING);
                return FileVisitResult.CONTINUE;
            }
        });
    }

    public static void rename(@NonNull Path source, @NonNull String newName) throws IOException {
        Path target = Objects.requireNonNull(source).resolveSibling(Objects.requireNonNull(newName));
        if (Files.exists(target, new LinkOption[0])) {
            throw new FileAlreadyExistsException(target.toString());
        }
        try {
            Files.move(source, target, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            Files.move(source, target, new CopyOption[0]);
        }
    }

    public static Path searchFileUpwards(@NonNull String filename, int maxTries) {
        if (maxTries < 1) {
            throw new IllegalArgumentException("maxTries must be at least 1");
        }
        Path currentDir = Paths.get("", new String[0]).toAbsolutePath();
        for (int i = 0; i < maxTries; ++i) {
            Path filePath = currentDir.resolve(filename);
            if (Files.exists(filePath, new LinkOption[0])) {
                return filePath;
            }
            Path parent = currentDir.getParent();
            if (parent == null) break;
            currentDir = parent;
        }
        throw new RuntimeException("File '" + filename + "' not found after searching " + maxTries + " director" + (maxTries == 1 ? "y" : "ies") + " upwards from current working directory");
    }
}

