/*
 * Decompiled with CFR 0.152.
 */
package org.hiero.consensus.pcli;

import com.hedera.hapi.node.base.ServiceEndpoint;
import com.hedera.hapi.node.state.roster.Roster;
import com.hedera.hapi.node.state.roster.RosterEntry;
import com.hedera.hapi.platform.state.ConsensusSnapshot;
import com.hedera.pbj.runtime.ParseException;
import com.hedera.pbj.runtime.io.ReadableSequentialData;
import com.hedera.pbj.runtime.io.WritableSequentialData;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import com.hedera.pbj.runtime.io.stream.ReadableStreamingData;
import com.hedera.pbj.runtime.io.stream.WritableStreamingData;
import com.swirlds.common.context.PlatformContext;
import com.swirlds.common.io.utility.FileUtils;
import com.swirlds.config.api.Configuration;
import com.swirlds.config.api.ConfigurationBuilder;
import com.swirlds.platform.config.DefaultConfiguration;
import com.swirlds.platform.crypto.CryptoStatic;
import com.swirlds.platform.crypto.EnhancedKeyStoreLoader;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.Console;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.cert.CertificateEncodingException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.hiero.base.crypto.Signer;
import org.hiero.consensus.model.event.PlatformEvent;
import org.hiero.consensus.model.node.KeysAndCerts;
import org.hiero.consensus.model.node.NodeId;
import org.hiero.consensus.node.NodeUtilities;
import org.hiero.consensus.pcli.AbstractCommand;
import org.hiero.consensus.pcli.PcesCommand;
import org.hiero.consensus.pcli.SubcommandOf;
import org.hiero.consensus.pcli.graph.PcesGraphSlicer;
import picocli.CommandLine;

@CommandLine.Command(name="slice", mixinStandardHelpOptions=true, description={"Extract a section of an event graph from PCES files and transform it to be replayable from genesis. Events are filtered by birth round or ngen for each creator, birth rounds are adjusted, and all events are re-hashed and re-signed with newly generated keys."})
@SubcommandOf(value=PcesCommand.class)
public class PcesSliceCommand
extends AbstractCommand {
    private static final String ROSTER_JSON_FILENAME = "roster.json";
    private static final String KEYS_SUBDIRECTORY = "keys";
    private Path inputDirectory;
    private Path outputDirectory;
    private int nodeCount;
    private Long genesisBirthRound;
    private Path snapshotPath;
    private boolean forceOverwrite;
    private final Map<Long, Long> nodeMinBirthRound = new HashMap<Long, Long>();
    private final Map<Long, Long> nodeMinNgen = new HashMap<Long, Long>();
    @CommandLine.Option(names={"--node-min-birth-round"}, split=",", description={"Per-node minimum birth round filter. Format: nodeId:value,nodeId:value (e.g., 0:100,1:150). Events from the specified node with lower birth round are excluded."})
    private List<String> nodeMinBirthRoundRaw = new ArrayList<String>();
    @CommandLine.Option(names={"--node-min-ngen"}, split=",", description={"Per-node minimum ngen filter. Format: nodeId:value,nodeId:value (e.g., 0:150,1:200). Events from the specified node with lower ngen are excluded."})
    private List<String> nodeMinNgenRaw = new ArrayList<String>();

    @CommandLine.Parameters(index="0", description={"The input directory containing PCES files to slice."})
    private void setInputDirectory(Path inputDirectory) {
        this.inputDirectory = this.dirMustExist(inputDirectory);
    }

    @CommandLine.Parameters(index="1", description={"The output directory where sliced PCES files will be written."})
    private void setOutputDirectory(Path outputDirectory) {
        this.outputDirectory = outputDirectory;
    }

    @CommandLine.Option(names={"-n", "--nodes"}, description={"Number of nodes that generated the input PCES stream. The output will use the same number of nodes with newly generated keys."}, required=true)
    private void setNodeCount(int nodeCount) {
        if (nodeCount < 1) {
            throw this.buildParameterException("Node count must be at least 1");
        }
        this.nodeCount = nodeCount;
    }

    @CommandLine.Option(names={"--genesis-birth-round"}, description={"Value by which all events' birth rounds are adjusted. Events with br lower or equals to this value receive a new br of 1. The rest it's actual BR value minus this value."})
    private void setGenesisBirthRound(Long genesisBirthRound) {
        this.genesisBirthRound = genesisBirthRound;
    }

    @CommandLine.Option(names={"--snapshot"}, description={"Path to a consensus snapshot JSON file to initialize the consensus engine."})
    private void setSnapshotPath(Path snapshotPath) {
        this.snapshotPath = this.fileMustExist(snapshotPath).toPath();
    }

    @CommandLine.Option(names={"-f", "--force"}, description={"Force overwrite of output directory without prompting for confirmation."})
    private void setForceOverwrite(boolean forceOverwrite) {
        this.forceOverwrite = forceOverwrite;
    }

    @NonNull
    private NodeFilterValue parseNodeFilter(@NonNull String nodeFilter, @NonNull String optionName) {
        String[] parts = nodeFilter.split(":");
        if (parts.length != 2) {
            throw this.buildParameterException(optionName + " must be in format nodeId:value (e.g., 0:100), got: " + nodeFilter);
        }
        try {
            long nodeId = Long.parseLong(parts[0].trim());
            long value = Long.parseLong(parts[1].trim());
            if (nodeId < 0L) {
                throw this.buildParameterException(optionName + " nodeId must be non-negative, got: " + nodeId);
            }
            return new NodeFilterValue(nodeId, value);
        }
        catch (NumberFormatException e) {
            throw this.buildParameterException(optionName + " must contain valid numbers in format nodeId:value, got: " + nodeFilter);
        }
    }

    @Override
    public Integer call() throws Exception {
        ConsensusSnapshot consensusSnapshot;
        NodeFilterValue parsed;
        for (String filter : this.nodeMinBirthRoundRaw) {
            parsed = this.parseNodeFilter(filter, "--node-min-birth-round");
            this.nodeMinBirthRound.put(parsed.nodeId, parsed.value);
        }
        for (String filter : this.nodeMinNgenRaw) {
            parsed = this.parseNodeFilter(filter, "--node-min-ngen");
            this.nodeMinNgen.put(parsed.nodeId, parsed.value);
        }
        if (!this.prepareOutputDirectory()) {
            System.out.println("Operation cancelled by user.");
            return 1;
        }
        System.out.println("Slicing PCES files from: " + String.valueOf(this.inputDirectory));
        System.out.println("Output directory: " + String.valueOf(this.outputDirectory));
        System.out.println("Node count: " + this.nodeCount);
        if (this.genesisBirthRound != null) {
            System.out.println("Genesis birth round: " + this.genesisBirthRound);
        }
        if (!this.nodeMinBirthRound.isEmpty()) {
            System.out.println("Per-node min birth round filters:");
            this.nodeMinBirthRound.forEach((nodeId, value) -> System.out.println("  Node " + nodeId + ": " + value));
        }
        if (!this.nodeMinNgen.isEmpty()) {
            System.out.println("Per-node min ngen filters:");
            this.nodeMinNgen.forEach((nodeId, value) -> System.out.println("  Node " + nodeId + ": " + value));
        }
        if ((consensusSnapshot = this.parseConsensusSnapshot()) != null) {
            System.out.println("Using consensus snapshot from: " + String.valueOf(this.snapshotPath));
            System.out.println("Snapshot round: " + consensusSnapshot.round());
        } else {
            System.out.println("Using a genesys consensus snapshot");
        }
        List<NodeId> nodeIds = IntStream.range(0, this.nodeCount).mapToObj(NodeId::of).toList();
        Map keysAndCertsMap = CryptoStatic.generateKeysAndCerts(nodeIds);
        System.out.println("Generated keys for " + this.nodeCount + " nodes.");
        Roster roster = PcesSliceCommand.generateRoster(keysAndCertsMap);
        this.writeRosterJson(roster);
        System.out.println("Wrote roster to: " + String.valueOf(this.outputDirectory.resolve(ROSTER_JSON_FILENAME)));
        Path keysDirectory = this.outputDirectory.resolve(KEYS_SUBDIRECTORY);
        this.writeKeysPem(keysAndCertsMap, keysDirectory);
        System.out.println("Wrote keys to: " + String.valueOf(keysDirectory));
        PcesGraphSlicer slicer = PcesGraphSlicer.builder().context(PcesSliceCommand.createDefaultPlatformContext()).keysAndCertsMap(keysAndCertsMap).existingPcesFilesLocation(this.inputDirectory).exportPcesFileLocation(this.outputDirectory).graphEventFilter(this::filterEvent).graphEventCoreModifier(e -> e.copyBuilder().birthRound(Long.max(e.birthRound() - (this.genesisBirthRound == null ? 0L : this.genesisBirthRound), 1L)).build()).consensusSnapshot(consensusSnapshot).build();
        System.out.println("Starting slice operation...");
        slicer.slice();
        System.out.println("Slice operation completed successfully.");
        return 0;
    }

    private boolean filterEvent(@NonNull PlatformEvent event) {
        long birthRound = event.getBirthRound();
        long ngen = event.getNGen();
        long creatorNodeId = event.getCreatorId().id();
        Long nodeMinBR = this.nodeMinBirthRound.get(creatorNodeId);
        if (nodeMinBR != null && birthRound < nodeMinBR) {
            return false;
        }
        Long nodeMinNG = this.nodeMinNgen.get(creatorNodeId);
        return nodeMinNG == null || ngen >= nodeMinNG;
    }

    private boolean prepareOutputDirectory() throws IOException {
        if (Files.exists(this.outputDirectory, new LinkOption[0]) && this.isDirectoryNotEmpty(this.outputDirectory)) {
            if (!this.forceOverwrite) {
                System.out.println();
                System.out.println("WARNING: Output directory already exists and is not empty:");
                System.out.println("  " + String.valueOf(this.outputDirectory.toAbsolutePath()));
                System.out.println();
                System.out.println("All existing contents will be DELETED if you proceed.");
                System.out.println();
                if (!this.promptForConfirmation()) {
                    return false;
                }
            }
            System.out.println("Deleting existing output directory...");
            FileUtils.deleteDirectory((Path)this.outputDirectory);
        }
        Files.createDirectories(this.outputDirectory, new FileAttribute[0]);
        return true;
    }

    private boolean isDirectoryNotEmpty(Path directory) throws IOException {
        try (Stream<Path> entries = Files.list(directory);){
            boolean bl = entries.findFirst().isPresent();
            return bl;
        }
    }

    private boolean promptForConfirmation() {
        Console console = System.console();
        if (console == null) {
            System.out.println("No console available for confirmation. Use --force to skip confirmation.");
            return false;
        }
        System.out.print("Do you want to delete the existing directory and continue? (y/N): ");
        String response = console.readLine();
        return response != null && response.trim().equalsIgnoreCase("y");
    }

    @Nullable
    private ConsensusSnapshot parseConsensusSnapshot() throws IOException, ParseException {
        if (this.snapshotPath == null) {
            return null;
        }
        try (FileInputStream fis = new FileInputStream(this.snapshotPath.toFile());){
            ConsensusSnapshot consensusSnapshot = (ConsensusSnapshot)ConsensusSnapshot.JSON.parse((ReadableSequentialData)new ReadableStreamingData((InputStream)fis));
            return consensusSnapshot;
        }
    }

    private void writeRosterJson(@NonNull Roster roster) throws IOException {
        Path rosterPath = this.outputDirectory.resolve(ROSTER_JSON_FILENAME);
        try (FileOutputStream fos = new FileOutputStream(rosterPath.toFile());){
            Roster.JSON.write((Object)roster, (WritableSequentialData)new WritableStreamingData((OutputStream)fos));
        }
    }

    private void writeKeysPem(@NonNull Map<NodeId, KeysAndCerts> keysAndCertsMap, @NonNull Path keysDirectory) throws IOException, CertificateEncodingException {
        Files.createDirectories(keysDirectory, new FileAttribute[0]);
        for (Map.Entry<NodeId, KeysAndCerts> entry : keysAndCertsMap.entrySet()) {
            NodeId nodeId = entry.getKey();
            KeysAndCerts keysAndCerts = entry.getValue();
            String nodeName = NodeUtilities.formatNodeName((NodeId)nodeId);
            Path privateKeyPath = keysDirectory.resolve(String.format("s-private-%s.pem", nodeName));
            EnhancedKeyStoreLoader.writePemFile((boolean)true, (Path)privateKeyPath, (byte[])keysAndCerts.sigKeyPair().getPrivate().getEncoded());
            Path publicCertPath = keysDirectory.resolve(String.format("s-public-%s.pem", nodeName));
            EnhancedKeyStoreLoader.writePemFile((boolean)false, (Path)publicCertPath, (byte[])keysAndCerts.sigCert().getEncoded());
        }
    }

    @NonNull
    public static PlatformContext createDefaultPlatformContext() {
        try {
            Configuration configuration = DefaultConfiguration.buildBasicConfiguration((ConfigurationBuilder)ConfigurationBuilder.create());
            return PlatformContext.create((Configuration)configuration);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to create platform context", e);
        }
    }

    @NonNull
    public static <S extends Signer> Map<NodeId, S> generateSigners(@NonNull Map<NodeId, KeysAndCerts> keysAndCertsMap, @NonNull Function<KeysAndCerts, S> toSigner) {
        HashMap signers = new HashMap();
        keysAndCertsMap.forEach((nodeId, keysAndCerts) -> signers.put(nodeId, (Signer)toSigner.apply((KeysAndCerts)keysAndCerts)));
        return signers;
    }

    @NonNull
    private static Roster generateRoster(@NonNull Map<NodeId, KeysAndCerts> keysAndCertsMap) {
        ArrayList<RosterEntry> rosterEntries = new ArrayList<RosterEntry>();
        for (Map.Entry<NodeId, KeysAndCerts> entry : keysAndCertsMap.entrySet()) {
            rosterEntries.add(PcesSliceCommand.createRosterEntry(entry.getKey(), entry.getValue()));
        }
        rosterEntries.sort(Comparator.comparingLong(RosterEntry::nodeId));
        return Roster.newBuilder().rosterEntries(rosterEntries).build();
    }

    @NonNull
    private static RosterEntry createRosterEntry(@NonNull NodeId nodeId, @NonNull KeysAndCerts keysAndCerts) {
        try {
            long id = nodeId.id();
            byte[] certificate = keysAndCerts.sigCert().getEncoded();
            return RosterEntry.newBuilder().nodeId(id).weight(500L).gossipCaCertificate(Bytes.wrap((byte[])certificate)).gossipEndpoint(new ServiceEndpoint[]{ServiceEndpoint.newBuilder().ipAddressV4(Bytes.wrap((byte[])new byte[]{127, 0, 0, 1})).port(50000 + (int)id).build()}).build();
        }
        catch (CertificateEncodingException e) {
            throw new RuntimeException("Failed to encode certificate for node " + String.valueOf(nodeId), e);
        }
    }

    private record NodeFilterValue(long nodeId, long value) {
    }
}

