/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnectionManager;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeStats;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.app.util.LoggingUtilities;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.node.internal.network.BlockNodeConfig;
import com.hedera.pbj.runtime.grpc.GrpcException;
import com.hedera.pbj.runtime.grpc.Pipeline;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.Flow;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.BlockStreamPublishServiceInterface;
import org.hiero.block.api.PublishStreamRequest;
import org.hiero.block.api.PublishStreamResponse;

public class BlockNodeConnection
implements Pipeline<PublishStreamResponse> {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnection.class);
    private static final AtomicLong connectionIdCounter = new AtomicLong(0L);
    public static final Duration THIRTY_SECONDS = Duration.ofSeconds(30L);
    private final BlockNodeConfig blockNodeConfig;
    private final BlockNodeConnectionManager blockNodeConnectionManager;
    private final BlockBufferService blockBufferService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final Duration streamResetPeriod;
    private final AtomicBoolean streamShutdownInProgress = new AtomicBoolean(false);
    private final BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient blockStreamPublishServiceClient;
    private final AtomicReference<Pipeline<? super PublishStreamRequest>> requestPipelineRef = new AtomicReference();
    private final AtomicReference<ConnectionState> connectionState;
    private final ScheduledExecutorService executorService;
    private ScheduledFuture<?> streamResetTask;
    private final String connectionId;

    private void logWithContext(Level level, String message, Object ... args) {
        if (logger.isEnabled(level)) {
            message = String.format("%s %s %s", LoggingUtilities.threadInfo(), this, message);
            logger.atLevel(level).log(message, args);
        }
    }

    public BlockNodeConnection(@NonNull ConfigProvider configProvider, @NonNull BlockNodeConfig nodeConfig, @NonNull BlockNodeConnectionManager blockNodeConnectionManager, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient grpcServiceClient, @NonNull BlockStreamMetrics blockStreamMetrics, @NonNull ScheduledExecutorService executorService) {
        Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.blockNodeConfig = Objects.requireNonNull(nodeConfig, "nodeConfig must not be null");
        this.blockNodeConnectionManager = Objects.requireNonNull(blockNodeConnectionManager, "blockNodeConnectionManager must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.blockStreamPublishServiceClient = Objects.requireNonNull(grpcServiceClient, "grpcServiceClient must not be null");
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.connectionState = new AtomicReference<ConnectionState>(ConnectionState.UNINITIALIZED);
        this.executorService = Objects.requireNonNull(executorService, "executorService must not be null");
        BlockNodeConnectionConfig blockNodeConnectionConfig = (BlockNodeConnectionConfig)configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class);
        this.streamResetPeriod = blockNodeConnectionConfig.streamResetPeriod();
        this.connectionId = String.format("%04d", connectionIdCounter.incrementAndGet());
    }

    public synchronized void createRequestPipeline() {
        if (this.requestPipelineRef.get() == null) {
            Pipeline pipeline = this.blockStreamPublishServiceClient.publishBlockStream((Pipeline)this);
            this.requestPipelineRef.set((Pipeline<? super PublishStreamRequest>)pipeline);
            this.logWithContext(Level.DEBUG, "Request pipeline initialized.", new Object[0]);
            this.updateConnectionState(ConnectionState.PENDING);
            this.blockStreamMetrics.recordConnectionOpened();
        } else {
            this.logWithContext(Level.DEBUG, "Request pipeline already available.", new Object[0]);
        }
    }

    public boolean updateConnectionState(@NonNull ConnectionState newState) {
        return this.updateConnectionState(null, newState);
    }

    /*
     * Enabled aggressive block sorting
     */
    private boolean updateConnectionState(@Nullable ConnectionState expectedCurrentState, @NonNull ConnectionState newState) {
        Objects.requireNonNull(newState, "newState must not be null");
        if (expectedCurrentState != null) {
            if (!this.connectionState.compareAndSet(expectedCurrentState, newState)) {
                this.logWithContext(Level.DEBUG, "Failed to transition state from {} to {} because current state does not match expected state.", new Object[]{expectedCurrentState, newState});
                return false;
            }
            this.logWithContext(Level.DEBUG, "Connection state transitioned from {} to {}.", new Object[]{expectedCurrentState, newState});
        } else {
            ConnectionState oldState = this.connectionState.getAndSet(newState);
            this.logWithContext(Level.DEBUG, "Connection state transitioned from {} to {}.", new Object[]{oldState, newState});
        }
        if (newState == ConnectionState.ACTIVE) {
            this.scheduleStreamReset();
            return true;
        }
        this.cancelStreamReset();
        return true;
    }

    private void scheduleStreamReset() {
        if (this.streamResetTask != null && !this.streamResetTask.isDone()) {
            this.streamResetTask.cancel(false);
        }
        this.streamResetTask = this.executorService.scheduleAtFixedRate(this::performStreamReset, this.streamResetPeriod.toMillis(), this.streamResetPeriod.toMillis(), TimeUnit.MILLISECONDS);
        this.logWithContext(Level.DEBUG, "Scheduled periodic stream reset every {}.", this.streamResetPeriod);
    }

    private void performStreamReset() {
        if (this.getConnectionState() == ConnectionState.ACTIVE) {
            this.logWithContext(Level.DEBUG, "Performing scheduled stream reset.", new Object[0]);
            this.endTheStreamWith(PublishStreamRequest.EndStream.Code.RESET);
            this.blockNodeConnectionManager.connectionResetsTheStream(this);
        }
    }

    private void cancelStreamReset() {
        if (this.streamResetTask != null) {
            this.streamResetTask.cancel(false);
            this.streamResetTask = null;
            this.logWithContext(Level.DEBUG, "Cancelled periodic stream reset.", new Object[0]);
        }
    }

    private void closeAndReschedule(@Nullable Duration delay, boolean callOnComplete) {
        this.close(callOnComplete);
        this.blockNodeConnectionManager.rescheduleConnection(this, delay, null, true);
    }

    private void endStreamAndReschedule(@NonNull PublishStreamRequest.EndStream.Code code) {
        Objects.requireNonNull(code, "code must not be null");
        this.endTheStreamWith(code);
        this.blockNodeConnectionManager.rescheduleConnection(this, THIRTY_SECONDS, null, true);
    }

    private void closeAndRestart(long blockNumber) {
        this.close(true);
        this.blockNodeConnectionManager.rescheduleConnection(this, null, blockNumber, false);
    }

    public void handleStreamFailure() {
        this.logWithContext(Level.DEBUG, "Handling failed stream.", new Object[0]);
        this.closeAndReschedule(THIRTY_SECONDS, true);
    }

    public void handleStreamFailureWithoutOnComplete() {
        this.logWithContext(Level.DEBUG, "Handling failed stream without onComplete.", new Object[0]);
        this.closeAndReschedule(THIRTY_SECONDS, false);
    }

    private void handleAcknowledgement(@NonNull PublishStreamResponse.BlockAcknowledgement acknowledgement) {
        long acknowledgedBlockNumber = acknowledgement.blockNumber();
        this.logWithContext(Level.DEBUG, "BlockAcknowledgement received for block {}", acknowledgedBlockNumber);
        this.acknowledgeBlocks(acknowledgedBlockNumber, true);
        BlockNodeStats.HighLatencyResult result = this.blockNodeConnectionManager.recordBlockAckAndCheckLatency(this.blockNodeConfig, acknowledgedBlockNumber, Instant.now());
        if (result.shouldSwitch() && !this.blockNodeConnectionManager.isOnlyOneBlockNodeConfigured()) {
            this.logWithContext(Level.DEBUG, "Block node has exceeded high latency threshold {} times consecutively.", result.consecutiveHighLatencyEvents());
            this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TIMEOUT);
        }
    }

    private void acknowledgeBlocks(long acknowledgedBlockNumber, boolean maybeJumpToBlock) {
        this.logWithContext(Level.DEBUG, "Acknowledging blocks <= {}.", acknowledgedBlockNumber);
        long currentBlockStreaming = this.blockNodeConnectionManager.currentStreamingBlockNumber();
        long currentBlockProducing = this.blockBufferService.getLastBlockNumberProduced();
        this.blockNodeConnectionManager.updateLastVerifiedBlock(this.blockNodeConfig, acknowledgedBlockNumber);
        if (maybeJumpToBlock && (acknowledgedBlockNumber > currentBlockProducing || acknowledgedBlockNumber > currentBlockStreaming)) {
            long blockToJumpTo = acknowledgedBlockNumber + 1L;
            this.logWithContext(Level.DEBUG, "Received acknowledgement for block {}, later than current streamed ({}) or produced ({}).", acknowledgedBlockNumber, currentBlockStreaming, currentBlockProducing);
            this.jumpToBlock(blockToJumpTo);
        }
    }

    private void handleEndOfStream(@NonNull PublishStreamResponse.EndOfStream endOfStream) {
        Objects.requireNonNull(endOfStream, "endOfStream must not be null");
        long blockNumber = endOfStream.blockNumber();
        PublishStreamResponse.EndOfStream.Code responseCode = endOfStream.status();
        this.logWithContext(Level.DEBUG, "Received EndOfStream response (block={}, responseCode={}).", blockNumber, responseCode);
        this.acknowledgeBlocks(blockNumber, false);
        if (this.blockNodeConnectionManager.recordEndOfStreamAndCheckLimit(this.blockNodeConfig, Instant.now())) {
            this.logWithContext(Level.DEBUG, "Block node has exceeded the allowed number of EndOfStream responses (received={}, permitted={}, timeWindow={}). Reconnection scheduled for {}.", this.blockNodeConnectionManager.getEndOfStreamCount(this.blockNodeConfig), this.blockNodeConnectionManager.getMaxEndOfStreamsAllowed(), this.blockNodeConnectionManager.getEndOfStreamTimeframe(), this.blockNodeConnectionManager.getEndOfStreamScheduleDelay());
            this.blockStreamMetrics.recordEndOfStreamLimitExceeded();
            this.closeAndReschedule(this.blockNodeConnectionManager.getEndOfStreamScheduleDelay(), true);
            return;
        }
        switch (responseCode) {
            case ERROR: 
            case PERSISTENCE_FAILED: {
                this.logWithContext(Level.DEBUG, "Block node reported an error at block {}. Will attempt to reestablish the stream later.", blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case TIMEOUT: 
            case DUPLICATE_BLOCK: 
            case BAD_BLOCK_PROOF: 
            case INVALID_REQUEST: {
                long restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                this.logWithContext(Level.DEBUG, "Block node reported status indicating immediate restart should be attempted. Will restart stream at block {}.", restartBlockNumber);
                this.closeAndRestart(restartBlockNumber);
                break;
            }
            case SUCCESS: {
                this.logWithContext(Level.DEBUG, "Block node orderly ended the stream at block {}.", blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case BEHIND: {
                long restartBlockNumber;
                long l = restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                if (this.blockBufferService.getBlockState(restartBlockNumber) != null) {
                    this.logWithContext(Level.DEBUG, "Block node reported it is behind. Will restart stream at block {}.", restartBlockNumber);
                    this.closeAndRestart(restartBlockNumber);
                    break;
                }
                this.logWithContext(Level.DEBUG, "Block node is behind and block state is not available. Ending the stream.", new Object[0]);
                this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
                break;
            }
            case UNKNOWN: {
                this.logWithContext(Level.WARN, "Block node reported an unknown error at block {}.", blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
            }
        }
    }

    private void handleSkipBlock(@NonNull PublishStreamResponse.SkipBlock skipBlock) {
        Objects.requireNonNull(skipBlock, "skipBlock must not be null");
        long skipBlockNumber = skipBlock.blockNumber();
        long streamingBlockNumber = this.blockNodeConnectionManager.currentStreamingBlockNumber();
        if (skipBlockNumber == streamingBlockNumber) {
            long nextBlock = skipBlockNumber + 1L;
            this.logWithContext(Level.DEBUG, "Received SkipBlock response.", new Object[0]);
            this.jumpToBlock(nextBlock);
        } else {
            this.logWithContext(Level.DEBUG, "Received SkipBlock response for block {}, but we are streaming block {} so it will be ignored.", skipBlockNumber, streamingBlockNumber);
        }
    }

    private void handleResendBlock(@NonNull PublishStreamResponse.ResendBlock resendBlock) {
        Objects.requireNonNull(resendBlock, "resendBlock must not be null");
        long resendBlockNumber = resendBlock.blockNumber();
        this.logWithContext(Level.DEBUG, "Received ResendBlock response for block {}.", resendBlockNumber);
        if (this.blockBufferService.getBlockState(resendBlockNumber) != null) {
            this.jumpToBlock(resendBlockNumber);
        } else {
            this.logWithContext(Level.DEBUG, "Block node requested a ResendBlock for block {} but that block does not exist on this consensus node. Closing connection and will retry later.", resendBlockNumber);
            this.closeAndReschedule(THIRTY_SECONDS, true);
        }
    }

    public void endTheStreamWith(PublishStreamRequest.EndStream.Code code) {
        long earliestBlockNumber = this.blockBufferService.getEarliestAvailableBlockNumber();
        long highestAckedBlockNumber = this.blockBufferService.getHighestAckedBlockNumber();
        PublishStreamRequest endStream = PublishStreamRequest.newBuilder().endStream(PublishStreamRequest.EndStream.newBuilder().endCode(code).earliestBlockNumber(earliestBlockNumber).latestBlockNumber(highestAckedBlockNumber)).build();
        this.logWithContext(Level.DEBUG, "Sending EndStream (code={}, earliestBlock={}, latestAcked={}).", code, earliestBlockNumber, highestAckedBlockNumber);
        this.sendRequest(endStream);
        this.close(true);
    }

    public void sendRequest(@NonNull PublishStreamRequest request) {
        block9: {
            Objects.requireNonNull(request, "request must not be null");
            Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
            if (this.getConnectionState() == ConnectionState.ACTIVE && pipeline != null) {
                try {
                    BlockItem firstItem;
                    if (logger.isDebugEnabled()) {
                        this.logWithContext(Level.DEBUG, "Sending request to block node (type={}).", request.request().kind());
                    } else if (logger.isTraceEnabled()) {
                        this.logWithContext(Level.TRACE, "[{}] Sending request to block node (type={}, bytes={})", this, request.request().kind(), request.protobufSize());
                    }
                    long startMs = System.currentTimeMillis();
                    pipeline.onNext((Object)request);
                    long durationMs = System.currentTimeMillis() - startMs;
                    this.blockStreamMetrics.recordRequestLatency(durationMs);
                    this.logWithContext(Level.TRACE, "[{}] Request took {}ms to send", this, durationMs);
                    if (request.hasEndStream()) {
                        this.blockStreamMetrics.recordRequestEndStreamSent(request.endStream().endCode());
                    } else {
                        this.blockStreamMetrics.recordRequestSent((PublishStreamRequest.RequestOneOfType)request.request().kind());
                        this.blockStreamMetrics.recordBlockItemsSent(request.blockItems().blockItems().size());
                    }
                    if (request.blockItems() != null && !request.blockItems().blockItems().isEmpty() && (firstItem = (BlockItem)request.blockItems().blockItems().getFirst()).hasBlockProof()) {
                        this.blockNodeConnectionManager.recordBlockProofSent(this.blockNodeConfig, firstItem.blockProof().block(), Instant.now());
                    }
                }
                catch (RuntimeException e) {
                    if (this.getConnectionState() != ConnectionState.ACTIVE) break block9;
                    this.blockStreamMetrics.recordRequestSendFailure();
                    throw e;
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close(boolean callOnComplete) {
        ConnectionState connState = this.getConnectionState();
        if (connState.isTerminal()) {
            this.logWithContext(Level.DEBUG, "Connection already in terminal state ({}).", new Object[]{connState});
            return;
        }
        if (!this.updateConnectionState(connState, ConnectionState.CLOSING)) {
            this.logWithContext(Level.DEBUG, "State changed while trying to close connection. Aborting close attempt.", new Object[0]);
            return;
        }
        this.logWithContext(Level.DEBUG, "Closing connection.", new Object[0]);
        try {
            this.closePipeline(callOnComplete);
            this.jumpToBlock(-1L);
            this.blockStreamMetrics.recordConnectionClosed();
            this.logWithContext(Level.DEBUG, "Connection successfully closed.", new Object[0]);
        }
        catch (RuntimeException e) {
            this.logWithContext(Level.WARN, "Error occurred while attempting to close connection.", e);
        }
        finally {
            this.updateConnectionState(ConnectionState.CLOSED);
        }
    }

    private void closePipeline(boolean callOnComplete) {
        Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
        if (pipeline != null) {
            this.logWithContext(Level.DEBUG, "Closing request pipeline for block node.", new Object[0]);
            this.streamShutdownInProgress.set(true);
            try {
                ConnectionState state = this.getConnectionState();
                if (state == ConnectionState.CLOSING && callOnComplete) {
                    pipeline.onComplete();
                    this.logWithContext(Level.DEBUG, "Request pipeline successfully closed.", new Object[0]);
                }
            }
            catch (Exception e) {
                this.logWithContext(Level.DEBUG, "Error while completing request pipeline.", e);
            }
            this.logWithContext(Level.DEBUG, "Request pipeline removed.", this);
            this.requestPipelineRef.compareAndSet(pipeline, null);
        }
    }

    public BlockNodeConfig getNodeConfig() {
        return this.blockNodeConfig;
    }

    private void jumpToBlock(long blockNumber) {
        this.blockNodeConnectionManager.jumpToBlock(blockNumber);
    }

    public void onSubscribe(Flow.Subscription subscription) {
        this.logWithContext(Level.DEBUG, "OnSubscribe invoked.", new Object[0]);
        subscription.request(Long.MAX_VALUE);
    }

    public void clientEndStreamReceived() {
        this.logWithContext(Level.DEBUG, "Client End Stream received.", new Object[0]);
        super.clientEndStreamReceived();
    }

    public void onNext(@NonNull PublishStreamResponse response) {
        Objects.requireNonNull(response, "response must not be null");
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            this.logWithContext(Level.DEBUG, "onNext invoked but connection is already closed ({}).", response);
            return;
        }
        if (response.hasAcknowledgement()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.handleAcknowledgement(response.acknowledgement());
        } else if (response.hasEndStream()) {
            this.blockStreamMetrics.recordResponseEndOfStreamReceived(response.endStream().status());
            this.blockStreamMetrics.recordLatestBlockEndOfStream(response.endStream().blockNumber());
            this.handleEndOfStream(response.endStream());
        } else if (response.hasSkipBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.blockStreamMetrics.recordLatestBlockSkipBlock(response.skipBlock().blockNumber());
            this.handleSkipBlock(response.skipBlock());
        } else if (response.hasResendBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.blockStreamMetrics.recordLatestBlockResendBlock(response.resendBlock().blockNumber());
            this.handleResendBlock(response.resendBlock());
        } else {
            this.blockStreamMetrics.recordUnknownResponseReceived();
            this.logWithContext(Level.WARN, "Unexpected response received: {}.", response);
        }
    }

    public void onError(Throwable error) {
        if (!this.getConnectionState().isTerminal()) {
            this.blockStreamMetrics.recordConnectionOnError();
            if (error instanceof GrpcException) {
                GrpcException grpcException = (GrpcException)error;
                this.logWithContext(Level.DEBUG, "Error received (grpcStatus={}).", grpcException.status(), grpcException);
            } else {
                this.logWithContext(Level.DEBUG, "Error received.", error);
            }
            this.handleStreamFailure();
        }
    }

    public void onComplete() {
        this.blockStreamMetrics.recordConnectionOnComplete();
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            this.logWithContext(Level.DEBUG, "onComplete invoked but connection is already closed.", new Object[0]);
            return;
        }
        if (this.streamShutdownInProgress.getAndSet(false)) {
            this.logWithContext(Level.DEBUG, "Stream completed (stream close was in progress).", new Object[0]);
        } else {
            this.logWithContext(Level.DEBUG, "Stream completed unexpectedly.", new Object[0]);
            this.handleStreamFailure();
        }
    }

    @NonNull
    public ConnectionState getConnectionState() {
        return this.connectionState.get();
    }

    public String toString() {
        return "[" + this.connectionId + "/" + this.blockNodeConfig.address() + ":" + this.blockNodeConfig.port() + "/" + String.valueOf((Object)this.getConnectionState()) + "]";
    }

    public boolean equals(Object o) {
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BlockNodeConnection that = (BlockNodeConnection)o;
        return this.connectionId == that.connectionId && Objects.equals(this.blockNodeConfig, that.blockNodeConfig);
    }

    public int hashCode() {
        return Objects.hash(this.blockNodeConfig, this.connectionId);
    }

    public static enum ConnectionState {
        UNINITIALIZED(false),
        PENDING(false),
        ACTIVE(false),
        CLOSING(true),
        CLOSED(true);

        private final boolean isTerminal;

        private ConnectionState(boolean isTerminal) {
            this.isTerminal = isTerminal;
        }

        boolean isTerminal() {
            return this.isTerminal;
        }
    }
}

