/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.http.http2;

import io.helidon.common.buffers.BufferData;
import io.helidon.http.Header;
import io.helidon.http.HeaderName;
import io.helidon.http.HeaderNames;
import io.helidon.http.HeaderValues;
import io.helidon.http.Headers;
import io.helidon.http.Method;
import io.helidon.http.ServerRequestHeaders;
import io.helidon.http.Status;
import io.helidon.http.WritableHeaders;
import io.helidon.http.http2.Http2ErrorCode;
import io.helidon.http.http2.Http2Exception;
import io.helidon.http.http2.Http2Flag;
import io.helidon.http.http2.Http2FrameData;
import io.helidon.http.http2.Http2FrameTypes;
import io.helidon.http.http2.Http2HuffmanDecoder;
import io.helidon.http.http2.Http2HuffmanEncoder;
import io.helidon.http.http2.Http2Priority;
import io.helidon.http.http2.Http2Setting;
import io.helidon.http.http2.Http2Settings;
import io.helidon.http.http2.Http2Stream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Supplier;

public class Http2Headers {
    static final String AUTHORITY = ":authority";
    public static final HeaderName AUTHORITY_NAME = HeaderNames.create((String)":authority");
    static final String METHOD = ":method";
    public static final HeaderName METHOD_NAME = HeaderNames.create((String)":method");
    static final String PATH = ":path";
    public static final HeaderName PATH_NAME = HeaderNames.create((String)":path");
    static final String SCHEME = ":scheme";
    public static final HeaderName SCHEME_NAME = HeaderNames.create((String)":scheme");
    static final String STATUS = ":status";
    public static final HeaderName STATUS_NAME = HeaderNames.create((String)":status");
    static final DynamicHeader EMPTY_HEADER_RECORD = new DynamicHeader(null, null, 0);
    private static final System.Logger LOGGER = System.getLogger(Http2Headers.class.getName());
    private static final String TRAILERS = "trailers";
    private static final String HTTP = "http";
    private static final String HTTPS = "https";
    private static final String PATH_SLASH = "/";
    private static final String PATH_INDEX = "/index.html";
    private final Headers headers;
    private final PseudoHeaders pseudoHeaders;

    private Http2Headers(Headers httpHeaders, PseudoHeaders pseudoHeaders) {
        this.headers = httpHeaders;
        this.pseudoHeaders = pseudoHeaders;
    }

    public static Http2Headers create(Http2Stream stream, DynamicTable table, Http2HuffmanDecoder huffman, Http2Headers headers, Http2FrameData ... frames) {
        if (frames.length == 0) {
            return Http2Headers.create(ServerRequestHeaders.create((Headers)WritableHeaders.create()), new PseudoHeaders());
        }
        Http2FrameData firstFrame = frames[0];
        BufferData firstFrameData = firstFrame.data();
        Http2Flag.HeaderFlags flags = firstFrame.header().flags(Http2FrameTypes.HEADERS);
        int padLength = 0;
        if (flags.padded()) {
            padLength = firstFrameData.read();
        }
        if (flags.priority()) {
            Http2Priority priority = Http2Priority.create(firstFrameData);
            stream.priority(priority);
        }
        WritableHeaders writableHeaders = WritableHeaders.create((Headers)headers.httpHeaders());
        BufferData[] buffers = new BufferData[frames.length];
        for (int i = 0; i < frames.length; ++i) {
            Http2FrameData frame = frames[i];
            buffers[i] = frame.data();
        }
        BufferData data = BufferData.create((BufferData[])buffers);
        PseudoHeaders pseudoHeaders = new PseudoHeaders();
        boolean lastIsPseudoHeader = true;
        while (true) {
            if (data.available() == padLength) {
                if (padLength > 0) {
                    data.skip(padLength);
                }
                return Http2Headers.create(ServerRequestHeaders.create((Headers)writableHeaders), pseudoHeaders);
            }
            if (data.available() == 0) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Expecting more header bytes");
            }
            lastIsPseudoHeader = Http2Headers.readHeader(writableHeaders, pseudoHeaders, table, huffman, data, lastIsPseudoHeader);
        }
    }

    public static Http2Headers create(Http2Stream stream, DynamicTable table, Http2HuffmanDecoder huffman, Http2FrameData ... frames) {
        return Http2Headers.create(stream, table, huffman, Http2Headers.create(WritableHeaders.create()), frames);
    }

    public static Http2Headers create(Headers headers) {
        if (headers instanceof WritableHeaders) {
            return Http2Headers.createFromWritable((WritableHeaders)headers);
        }
        return Http2Headers.createFromWritable(WritableHeaders.create((Headers)headers));
    }

    public static Http2Headers create(WritableHeaders<?> writableHeaders) {
        return Http2Headers.createFromWritable(writableHeaders);
    }

    public Status status() {
        return this.pseudoHeaders.status();
    }

    public String path() {
        return this.pseudoHeaders.path();
    }

    public Method method() {
        return this.pseudoHeaders.method();
    }

    public String scheme() {
        return this.pseudoHeaders.scheme();
    }

    public String authority() {
        return this.pseudoHeaders.authority();
    }

    public Http2Headers path(String path) {
        this.pseudoHeaders.path(path);
        return this;
    }

    public Http2Headers method(Method method) {
        this.pseudoHeaders.method(method);
        return this;
    }

    public Http2Headers authority(String authority) {
        this.pseudoHeaders.authority(authority);
        return this;
    }

    public Http2Headers scheme(String scheme) {
        this.pseudoHeaders.scheme(scheme);
        return this;
    }

    public Headers httpHeaders() {
        return this.headers;
    }

    public void validateResponse() throws Http2Exception {
        if (!this.pseudoHeaders.hasStatus()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Missing :status pseudo header");
        }
        if (this.headers.contains(HeaderNames.CONNECTION)) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Connection in response headers");
        }
        if (this.pseudoHeaders.hasScheme()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, ":scheme in response headers");
        }
        if (this.pseudoHeaders.hasPath()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, ":path in response headers");
        }
        if (this.pseudoHeaders.hasMethod()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, ":method in response headers");
        }
    }

    public void validateRequest() throws Http2Exception {
        List values;
        if (this.pseudoHeaders.hasStatus()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, ":status in request headers");
        }
        if (this.headers.contains(HeaderNames.CONNECTION)) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Connection in request headers");
        }
        if (this.headers.contains(HeaderNames.TE) && !(values = this.headers.all(HeaderNames.TE, List::of)).equals(List.of(TRAILERS))) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "te in headers with other value than trailers: \n" + BufferData.create((String)values.toString()).debugDataHex());
        }
        if (!this.pseudoHeaders.hasScheme()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Missing :scheme pseudo header");
        }
        if (!this.pseudoHeaders.hasPath()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Missing :path pseudo header");
        }
        if (!this.pseudoHeaders.hasMethod()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Missing :method pseudo header");
        }
    }

    public String toString() {
        return this.pseudoHeaders.toString() + "\n" + this.headers.toString();
    }

    public Http2Headers status(Status status) {
        this.pseudoHeaders.status(status);
        return this;
    }

    public void write(DynamicTable table, Http2HuffmanEncoder huffman, BufferData growingBuffer) {
        if (this.pseudoHeaders.hasStatus()) {
            StaticHeader indexed = null;
            Status status = this.pseudoHeaders.status();
            if (status == Status.OK_200) {
                indexed = StaticHeader.STATUS_200;
            } else if (status == Status.NO_CONTENT_204) {
                indexed = StaticHeader.STATUS_204;
            } else if (status == Status.PARTIAL_CONTENT_206) {
                indexed = StaticHeader.STATUS_206;
            } else if (status == Status.NOT_MODIFIED_304) {
                indexed = StaticHeader.STATUS_304;
            } else if (status == Status.BAD_REQUEST_400) {
                indexed = StaticHeader.STATUS_400;
            } else if (status == Status.NOT_FOUND_404) {
                indexed = StaticHeader.STATUS_404;
            } else if (status == Status.INTERNAL_SERVER_ERROR_500) {
                indexed = StaticHeader.STATUS_500;
            }
            if (indexed == null) {
                this.writeHeader(huffman, table, growingBuffer, STATUS_NAME, this.status().codeText(), true, false);
            } else {
                this.writeHeader(growingBuffer, indexed);
            }
        }
        if (this.pseudoHeaders.hasMethod()) {
            Method method = this.pseudoHeaders.method();
            StaticHeader indexed = null;
            if (method == Method.GET) {
                indexed = StaticHeader.METHOD_GET;
            } else if (method == Method.POST) {
                indexed = StaticHeader.METHOD_POST;
            }
            if (indexed == null) {
                this.writeHeader(huffman, table, growingBuffer, METHOD_NAME, method.text(), true, false);
            } else {
                this.writeHeader(growingBuffer, indexed);
            }
        }
        if (this.pseudoHeaders.hasScheme()) {
            String scheme = this.pseudoHeaders.scheme();
            if (scheme.equals(HTTP)) {
                this.writeHeader(growingBuffer, StaticHeader.SCHEME_HTTP);
            } else if (scheme.equals(HTTPS)) {
                this.writeHeader(growingBuffer, StaticHeader.SCHEME_HTTPS);
            } else {
                this.writeHeader(huffman, table, growingBuffer, SCHEME_NAME, scheme, true, false);
            }
        }
        if (this.pseudoHeaders.hasPath()) {
            String path = this.pseudoHeaders.path();
            if (path.equals(PATH_SLASH)) {
                this.writeHeader(growingBuffer, StaticHeader.PATH_ROOT);
            } else if (path.equals(PATH_INDEX)) {
                this.writeHeader(growingBuffer, StaticHeader.PATH_INDEX);
            } else {
                this.writeHeader(huffman, table, growingBuffer, PATH_NAME, path, true, false);
            }
        }
        if (this.pseudoHeaders.hasAuthority()) {
            this.writeHeader(huffman, table, growingBuffer, AUTHORITY_NAME, this.pseudoHeaders.authority, true, false);
        }
        for (Header header : this.headers) {
            String value = (String)header.get();
            boolean shouldIndex = !header.changing();
            boolean neverIndex = header.sensitive();
            this.writeHeader(huffman, table, growingBuffer, header.headerName(), value, shouldIndex, neverIndex);
        }
    }

    static BufferData[] split(BufferData bufferData, int size) {
        int length = bufferData.available();
        if (length <= size) {
            return new BufferData[]{bufferData};
        }
        int lastFragmentSize = length % size;
        int allFrames = length / size + (lastFragmentSize != 0 ? 1 : 0);
        BufferData[] result = new BufferData[allFrames];
        for (int i = 0; i < allFrames; ++i) {
            boolean lastFrame;
            boolean bl = lastFrame = allFrames == i + 1;
            byte[] frag = new byte[lastFrame ? (lastFragmentSize != 0 ? lastFragmentSize : size) : size];
            bufferData.read(frag);
            result[i] = BufferData.create((byte[])frag);
        }
        return result;
    }

    private static Http2Headers create(ServerRequestHeaders httpHeaders, PseudoHeaders pseudoHeaders) {
        return new Http2Headers((Headers)httpHeaders, pseudoHeaders);
    }

    private static boolean readHeader(WritableHeaders<?> headers, PseudoHeaders pseudoHeaders, DynamicTable table, Http2HuffmanDecoder huffman, BufferData data, boolean lastIsPseudoHeader) {
        String value;
        HeaderName headerName;
        HeaderApproach approach = HeaderApproach.resolve(data);
        if (approach.tableSizeUpdate) {
            table.maxTableSize(approach.number);
            if (headers.size() > 0 || pseudoHeaders.size() > 0) {
                throw new Http2Exception(Http2ErrorCode.COMPRESSION, "Table size update is after headers");
            }
            return lastIsPseudoHeader;
        }
        HeaderRecord record = EMPTY_HEADER_RECORD;
        if (approach.number != 0) {
            record = table.get(approach.number);
        }
        if (approach.hasName) {
            String name;
            try {
                name = Http2Headers.readString(huffman, data);
            }
            catch (IllegalArgumentException e) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received a header with non ASCII character(s)\n" + data.debugDataHex(true));
            }
            if (!name.toLowerCase(Locale.ROOT).equals(name)) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received a header with uppercase letters\n" + BufferData.create((String)name).debugDataHex());
            }
            if (name.charAt(0) == ':') {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received invalid pseudo-header field (or explicit value instead of indexed)\n" + BufferData.create((String)name).debugDataHex());
            }
            headerName = HeaderNames.create((String)name);
        } else {
            headerName = record.headerName();
        }
        boolean isPseudoHeader = false;
        if (headerName != null && headerName.isPseudoHeader()) {
            isPseudoHeader = true;
            if (!lastIsPseudoHeader) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Pseudo header field appears after regular header");
            }
        }
        if (approach.hasValue) {
            value = Http2Headers.readString(huffman, data);
        } else {
            value = record.value();
            if (value == null) {
                value = "";
            }
        }
        if (isPseudoHeader) {
            if (value == null) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Value of a pseudo header must not be null");
            }
            if (headerName.equals((Object)PATH_NAME) && value.length() == 0) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, ":path pseudo header has empty value");
            }
            if (headerName.equals((Object)PATH_NAME)) {
                Http2Headers.validateAndSetPseudoHeader(headerName, pseudoHeaders::hasPath, pseudoHeaders::path, value);
            } else if (headerName.equals((Object)AUTHORITY_NAME)) {
                Http2Headers.validateAndSetPseudoHeader(headerName, pseudoHeaders::hasAuthority, pseudoHeaders::authority, value);
            } else if (headerName.equals((Object)METHOD_NAME)) {
                Http2Headers.validateAndSetPseudoHeader(headerName, pseudoHeaders::hasMethod, pseudoHeaders::method, value);
            } else if (headerName.equals((Object)SCHEME_NAME)) {
                Http2Headers.validateAndSetPseudoHeader(headerName, pseudoHeaders::hasScheme, pseudoHeaders::scheme, value);
            } else if (headerName.equals((Object)STATUS_NAME)) {
                Http2Headers.validateAndSetPseudoHeader(headerName, pseudoHeaders::hasStatus, pseudoHeaders::status, value);
            }
        } else if (headerName == null || value == null) {
            String tHeaderName = headerName == null ? "null" : headerName.lowerCase();
            String tValue = value == null ? "null" : BufferData.create((byte[])value.getBytes(StandardCharsets.US_ASCII)).debugDataHex();
            throw new Http2Exception(Http2ErrorCode.COMPRESSION, "Failed to get name or value. Name: " + tHeaderName + ", value " + tValue);
        }
        if (approach.addToIndex) {
            table.add(headerName, value);
        }
        if (!isPseudoHeader) {
            headers.add(HeaderValues.create((HeaderName)headerName, (!approach.addToIndex ? 1 : 0) != 0, (boolean)approach.neverIndex, (String[])new String[]{value}));
        }
        return isPseudoHeader;
    }

    private static String readString(Http2HuffmanDecoder huffman, BufferData data) {
        if (data.available() < 1) {
            throw new Http2Exception(Http2ErrorCode.COMPRESSION, "No data available to read header");
        }
        int first = data.read();
        boolean isHuffman = (first & 0x80) != 0;
        int length = data.readHpackInt(first, 7);
        if (isHuffman) {
            return huffman.decodeString(data, length);
        }
        return data.readString(length);
    }

    private static void validateAndSetPseudoHeader(HeaderName name, Supplier<Boolean> isSet, Consumer<String> setter, String value) {
        if (isSet.get().booleanValue()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Duplicated pseudo header: " + String.valueOf(name));
        }
        setter.accept(value);
    }

    private static Http2Headers createFromWritable(WritableHeaders<?> headers) {
        PseudoHeaders pseudoHeaders = new PseudoHeaders();
        Http2Headers.removeFromHeadersAddToPseudo(headers, pseudoHeaders::status, STATUS_NAME);
        Http2Headers.removeFromHeadersAddToPseudo(headers, pseudoHeaders::path, PATH_NAME);
        Http2Headers.removeFromHeadersAddToPseudo(headers, pseudoHeaders::authority, AUTHORITY_NAME);
        Http2Headers.removeFromHeadersAddToPseudo(headers, pseudoHeaders::scheme, SCHEME_NAME);
        Http2Headers.removeFromHeadersAddToPseudo(headers, pseudoHeaders::method, METHOD_NAME);
        headers.remove(HeaderNames.HOST, it -> {
            if (!pseudoHeaders.hasAuthority()) {
                pseudoHeaders.authority((String)it.get());
            }
        });
        return new Http2Headers((Headers)headers, pseudoHeaders);
    }

    private static void removeFromHeadersAddToPseudo(WritableHeaders<?> headers, Consumer<String> valueConsumer, HeaderName pseudoHeader) {
        headers.remove(pseudoHeader, it -> valueConsumer.accept((String)it.get()));
    }

    private void writeHeader(Http2HuffmanEncoder huffman, DynamicTable table, BufferData buffer, HeaderName name, String value, boolean shouldIndex, boolean neverIndex) {
        HeaderApproach approach;
        IndexedHeaderRecord record = table.find(name, value);
        if (record == null) {
            if (shouldIndex) {
                table.add(name, value);
            }
            approach = new HeaderApproach(shouldIndex, neverIndex, true, true, 0);
        } else if (value.equals(record.value())) {
            approach = new HeaderApproach(false, neverIndex, false, false, record.index());
        } else {
            if (shouldIndex) {
                table.add(name, value);
            }
            approach = new HeaderApproach(shouldIndex, neverIndex, false, true, record.index());
        }
        approach.write(huffman, buffer, name, value);
    }

    private void writeHeader(BufferData buffer, StaticHeader header) {
        new HeaderApproach(false, false, false, false, header.index).write(buffer);
    }

    private static class PseudoHeaders {
        private String authority;
        private Method method;
        private String path;
        private String scheme;
        private Status status;
        private int size;

        private PseudoHeaders() {
        }

        public int size() {
            return this.size;
        }

        public String toString() {
            return "PseudoHeaders{authority='" + this.authority + "', method=" + String.valueOf(this.method) + ", path='" + this.path + "', scheme='" + this.scheme + "', status=" + String.valueOf(this.status) + "}";
        }

        PseudoHeaders authority(String authority) {
            this.authority = authority;
            ++this.size;
            return this;
        }

        void method(String method) {
            this.method(Method.create((String)method));
        }

        PseudoHeaders method(Method method) {
            this.method = method;
            ++this.size;
            return this;
        }

        PseudoHeaders path(String path) {
            this.path = path;
            ++this.size;
            return this;
        }

        PseudoHeaders scheme(String scheme) {
            this.scheme = scheme;
            ++this.size;
            return this;
        }

        void status(String status) {
            this.status(Status.create((int)Integer.parseInt(status)));
        }

        PseudoHeaders status(Status status) {
            this.status = status;
            ++this.size;
            return this;
        }

        boolean hasAuthority() {
            return this.authority != null;
        }

        String authority() {
            return this.authority;
        }

        boolean hasMethod() {
            return this.method != null;
        }

        Method method() {
            return this.method;
        }

        boolean hasPath() {
            return this.path != null;
        }

        String path() {
            return this.path;
        }

        boolean hasScheme() {
            return this.scheme != null;
        }

        String scheme() {
            return this.scheme;
        }

        boolean hasStatus() {
            return this.status != null;
        }

        Status status() {
            return this.status;
        }
    }

    public static class DynamicTable {
        private final List<DynamicHeader> headers = new ArrayList<DynamicHeader>();
        private volatile long protocolMaxTableSize;
        private long maxTableSize;
        private int currentTableSize;

        private DynamicTable(long protocolMaxTableSize) {
            this.protocolMaxTableSize = protocolMaxTableSize;
            this.maxTableSize = protocolMaxTableSize;
        }

        public static DynamicTable create(long maxTableSize) {
            return new DynamicTable(maxTableSize);
        }

        static DynamicTable create(Http2Settings settings) {
            return DynamicTable.create(settings.value(Http2Setting.HEADER_TABLE_SIZE));
        }

        public void protocolMaxTableSize(long number) {
            this.protocolMaxTableSize = number;
        }

        HeaderRecord get(int index) {
            if (index > StaticHeader.MAX_INDEX) {
                return this.doGet(index - StaticHeader.MAX_INDEX);
            }
            return StaticHeader.get(index);
        }

        void maxTableSize(long number) {
            if (number > this.protocolMaxTableSize) {
                throw new Http2Exception(Http2ErrorCode.COMPRESSION, "Attempt to set larger size than protocol max");
            }
            this.maxTableSize = number;
            if (this.maxTableSize == 0L) {
                this.headers.clear();
            }
            while (this.maxTableSize < (long)this.currentTableSize) {
                this.evict();
            }
        }

        int add(HeaderName headerName, String headerValue) {
            String name = headerName.lowerCase();
            int size = name.length() + headerValue.getBytes(StandardCharsets.US_ASCII).length + 32;
            if ((long)(this.currentTableSize + size) <= this.maxTableSize) {
                return this.add(headerName, headerValue, size);
            }
            while ((long)(this.currentTableSize + size) > this.maxTableSize) {
                this.evict();
                if (this.currentTableSize > 0) continue;
                throw new Http2Exception(Http2ErrorCode.COMPRESSION, "Cannot add header record, max table size too low. current size: " + this.currentTableSize + ", max size: " + this.maxTableSize + ", header size: " + size);
            }
            return this.add(headerName, headerValue, size);
        }

        long protocolMaxTableSize() {
            return this.protocolMaxTableSize;
        }

        long maxTableSize() {
            return this.maxTableSize;
        }

        int currentTableSize() {
            return this.currentTableSize;
        }

        private IndexedHeaderRecord find(HeaderName headerName, String headerValue) {
            StaticHeader staticHeader = StaticHeader.find(headerName, headerValue);
            IndexedHeaderRecord candidate = null;
            if (staticHeader != null) {
                if (staticHeader.name.equals((Object)headerName) && staticHeader.hasValue && staticHeader.value().equals(headerValue)) {
                    return staticHeader;
                }
                candidate = staticHeader;
            }
            for (int i = 0; i < this.headers.size(); ++i) {
                DynamicHeader header = this.headers.get(i);
                if (!header.headerName.equals((Object)headerName)) continue;
                if (header.value().equals(headerValue)) {
                    return new IndexedHeader(header, StaticHeader.MAX_INDEX + i + 1);
                }
                if (candidate != null) continue;
                candidate = new IndexedHeader(header, StaticHeader.MAX_INDEX + i + 1);
            }
            return candidate;
        }

        private void evict() {
            if (this.headers.isEmpty()) {
                return;
            }
            DynamicHeader removed = this.headers.remove(this.headers.size() - 1);
            if (removed != null) {
                this.currentTableSize -= removed.size();
            }
        }

        private int add(HeaderName name, String value, int size) {
            this.headers.add(0, new DynamicHeader(name, value, size));
            this.currentTableSize += size;
            return 0;
        }

        private HeaderRecord doGet(int index) {
            try {
                return this.headers.get(index - 1);
            }
            catch (IndexOutOfBoundsException e) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Dynamic table does not contain required header at index " + index);
            }
        }
    }

    static enum StaticHeader implements IndexedHeaderRecord
    {
        AUTHORITY(1, AUTHORITY_NAME, true),
        METHOD_GET(2, METHOD_NAME, "GET"),
        METHOD_POST(3, METHOD_NAME, "POST"),
        PATH_ROOT(4, PATH_NAME, "/"),
        PATH_INDEX(5, PATH_NAME, "/index.html"),
        SCHEME_HTTP(6, SCHEME_NAME, "http"),
        SCHEME_HTTPS(7, SCHEME_NAME, "https"),
        STATUS_200(8, STATUS_NAME, "200"),
        STATUS_204(9, STATUS_NAME, "204"),
        STATUS_206(10, STATUS_NAME, "206"),
        STATUS_304(11, STATUS_NAME, "304"),
        STATUS_400(12, STATUS_NAME, "400"),
        STATUS_404(13, STATUS_NAME, "404"),
        STATUS_500(14, STATUS_NAME, "500"),
        ACCEPT_CHARSET(15, HeaderNames.ACCEPT_CHARSET),
        ACCEPT_ENCODING(16, HeaderNames.ACCEPT_ENCODING, "gzip, deflate", false),
        ACCEPT_LANGUAGE(17, HeaderNames.ACCEPT_LANGUAGE),
        ACCEPT_RANGES(18, HeaderNames.ACCEPT_RANGES),
        ACCEPT(19, HeaderNames.ACCEPT),
        ACCESS_CONTROL_ALLOW_ORIGIN(20, HeaderNames.ACCESS_CONTROL_ALLOW_ORIGIN),
        AGE(21, HeaderNames.AGE),
        ALLOW(22, HeaderNames.ALLOW),
        AUTHORIZATION(23, HeaderNames.AUTHORIZATION),
        CACHE_CONTROL(24, HeaderNames.CACHE_CONTROL),
        CONTENT_DISPOSITION(25, HeaderNames.CONTENT_DISPOSITION),
        CONTENT_ENCODING(26, HeaderNames.CONTENT_ENCODING),
        CONTENT_LANGUAGE(27, HeaderNames.CONTENT_LANGUAGE),
        CONTENT_LENGTH(28, HeaderNames.CONTENT_LENGTH),
        CONTENT_LOCATION(29, HeaderNames.CONTENT_LOCATION),
        CONTENT_RANGE(30, HeaderNames.CONTENT_RANGE),
        CONTENT_TYPE(31, HeaderNames.CONTENT_TYPE),
        COOKIE(32, HeaderNames.COOKIE),
        DATE(33, HeaderNames.DATE),
        ETAG(34, HeaderNames.ETAG),
        EXPECT(35, HeaderNames.EXPECT),
        EXPIRES(36, HeaderNames.EXPIRES),
        FROM(37, HeaderNames.FROM),
        HOST(38, HeaderNames.HOST),
        IF_MATCH(39, HeaderNames.IF_MATCH),
        IF_MODIFIED_SINCE(40, HeaderNames.IF_MODIFIED_SINCE),
        IF_NONE_MATCH(41, HeaderNames.IF_NONE_MATCH),
        IF_RANGE(42, HeaderNames.IF_RANGE),
        IF_UNMODIFIED_SINCE(43, HeaderNames.IF_UNMODIFIED_SINCE),
        LAST_MODIFIED(44, HeaderNames.LAST_MODIFIED),
        LINK(45, HeaderNames.LINK),
        LOCATION(46, HeaderNames.LOCATION),
        MAX_FORWARDS(47, HeaderNames.MAX_FORWARDS),
        PROXY_AUTHENTICATE(48, HeaderNames.PROXY_AUTHENTICATE),
        PROXY_AUTHORIZATION(49, HeaderNames.PROXY_AUTHORIZATION),
        RANGE(50, HeaderNames.CONTENT_LOCATION),
        REFERER(51, HeaderNames.REFERER),
        REFRESH(52, HeaderNames.REFRESH),
        RETRY_AFTER(53, HeaderNames.RETRY_AFTER),
        SERVER(54, HeaderNames.SERVER),
        SET_COOKIE(55, HeaderNames.SET_COOKIE),
        STRICT_TRANSPORT_SECURITY(56, HeaderNames.STRICT_TRANSPORT_SECURITY),
        TRANSFER_ENCODING(57, HeaderNames.TRANSFER_ENCODING),
        USER_AGENT(58, HeaderNames.USER_AGENT),
        VARY(59, HeaderNames.VARY),
        VIA(60, HeaderNames.VIA),
        WWW_AUTHENTICATE(61, HeaderNames.WWW_AUTHENTICATE);

        public static final int MAX_INDEX;
        private static final Map<Integer, StaticHeader> BY_INDEX;
        private static final Map<String, StaticHeader> BY_NAME_NO_VALUE;
        private static final Map<String, Map<String, StaticHeader>> BY_NAME_VALUE;
        private final boolean isPseudoHeader;
        private final int index;
        private final HeaderName name;
        private final String value;
        private final boolean hasValue;

        private StaticHeader(int index, HeaderName name) {
            this(index, name, false);
        }

        private StaticHeader(int index, HeaderName name, boolean isPseudoHeader) {
            this.index = index;
            this.name = name;
            this.value = null;
            this.hasValue = false;
            this.isPseudoHeader = isPseudoHeader;
        }

        private StaticHeader(int index, HeaderName name, String value) {
            this(index, name, value, true);
        }

        private StaticHeader(int index, HeaderName name, String value, boolean isPseudoHeader) {
            this.index = index;
            this.name = name;
            this.value = value;
            this.hasValue = true;
            this.isPseudoHeader = isPseudoHeader;
        }

        static StaticHeader get(int index) {
            if (index > MAX_INDEX) {
                throw new IllegalArgumentException("Max index for predefined headers is " + MAX_INDEX + ", but requested " + index);
            }
            return BY_INDEX.get(index);
        }

        static StaticHeader find(HeaderName headerName, String headerValue) {
            Map<String, StaticHeader> map = BY_NAME_VALUE.get(headerName.lowerCase());
            if (map == null) {
                return BY_NAME_NO_VALUE.get(headerName.lowerCase());
            }
            StaticHeader staticHeader = map.get(headerValue);
            if (staticHeader == null) {
                return BY_NAME_NO_VALUE.get(headerName.lowerCase());
            }
            return staticHeader;
        }

        @Override
        public int index() {
            return this.index;
        }

        @Override
        public HeaderName headerName() {
            return this.name;
        }

        @Override
        public String value() {
            return this.value;
        }

        boolean hasValue() {
            return this.hasValue;
        }

        static {
            BY_INDEX = new HashMap<Integer, StaticHeader>();
            BY_NAME_NO_VALUE = new HashMap<String, StaticHeader>();
            BY_NAME_VALUE = new HashMap<String, Map<String, StaticHeader>>();
            int maxIndex = 0;
            for (StaticHeader predefinedHeader : StaticHeader.values()) {
                BY_INDEX.put(predefinedHeader.index(), predefinedHeader);
                maxIndex = Math.max(maxIndex, predefinedHeader.index);
                if (predefinedHeader.hasValue()) {
                    BY_NAME_VALUE.computeIfAbsent(predefinedHeader.headerName().lowerCase(), it -> new HashMap()).put(predefinedHeader.value(), predefinedHeader);
                }
                BY_NAME_NO_VALUE.putIfAbsent(predefinedHeader.headerName().lowerCase(), predefinedHeader);
            }
            MAX_INDEX = maxIndex;
        }
    }

    private static class HeaderApproach {
        private final boolean addToIndex;
        private final boolean neverIndex;
        private final boolean hasName;
        private final boolean hasValue;
        private final boolean tableSizeUpdate;
        private final int number;

        private HeaderApproach(boolean addToIndex, boolean neverIndex, boolean hasName, boolean hasValue, int number) {
            this.addToIndex = addToIndex;
            this.neverIndex = neverIndex;
            this.hasName = hasName;
            this.hasValue = hasValue;
            this.tableSizeUpdate = false;
            this.number = number;
        }

        HeaderApproach(int size) {
            this.tableSizeUpdate = true;
            this.number = size;
            this.addToIndex = false;
            this.neverIndex = false;
            this.hasValue = false;
            this.hasName = false;
        }

        static HeaderApproach resolve(BufferData data) {
            int value = data.read();
            HeaderApproach approach = HeaderApproach.resolve(data, value);
            return approach;
        }

        static HeaderApproach resolve(BufferData data, int value) {
            if (value == 0) {
                return new HeaderApproach(false, false, true, true, 0);
            }
            if (value == 64) {
                return new HeaderApproach(true, false, true, true, 0);
            }
            if (value == 16) {
                return new HeaderApproach(false, true, true, true, 0);
            }
            if ((value & 0x80) != 0) {
                int indexPart = data.readHpackInt(value, 7);
                return new HeaderApproach(false, false, false, false, indexPart);
            }
            if ((value & 0xC0) == 64) {
                int indexPart = data.readHpackInt(value, 6);
                return new HeaderApproach(true, false, false, true, indexPart);
            }
            if ((value & 0xE0) == 32) {
                int size = data.readHpackInt(value, 5);
                return new HeaderApproach(size);
            }
            if ((value & 0xF0) == 0) {
                int indexPart = data.readHpackInt(value, 4);
                return new HeaderApproach(false, false, false, true, indexPart);
            }
            if ((value & 0xF0) == 16) {
                int indexPart = data.readHpackInt(value, 4);
                return new HeaderApproach(false, true, false, true, indexPart);
            }
            throw new Http2Exception(Http2ErrorCode.COMPRESSION, "Header approach cannot be determined from value " + value);
        }

        public String toString() {
            if (this.tableSizeUpdate) {
                return "table_size_update: " + this.number;
            }
            return (this.addToIndex ? "do_index" : "do_not_index") + ", " + (this.neverIndex ? "never_index" : "may_index") + ", " + (this.hasName ? "name_from_stream" : "name_from_indexed") + ", " + (this.hasValue ? "value_from_stream" : "value_from_indexed") + ", " + this.number;
        }

        public boolean nameFromIndex() {
            return !this.hasName;
        }

        public void write(Http2HuffmanEncoder huffman, BufferData buffer, HeaderName headerName, String value) {
            boolean hasValue = this.hasValue();
            if (this.neverIndex()) {
                if (this.hasName()) {
                    buffer.writeInt8(16);
                } else {
                    if (!hasValue) {
                        if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                            LOGGER.log(System.Logger.Level.DEBUG, "Never index on field with indexed value: " + String.valueOf(headerName) + ": " + value);
                        }
                        hasValue = true;
                    }
                    buffer.writeHpackInt(this.number, 16, 4);
                }
            } else if (this.addToIndex()) {
                if (this.hasName) {
                    buffer.writeInt8(64);
                } else {
                    if (!hasValue) {
                        if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                            LOGGER.log(System.Logger.Level.DEBUG, "Index on field with indexed value: " + String.valueOf(headerName) + ": " + value);
                        }
                        hasValue = true;
                    }
                    buffer.writeHpackInt(this.number, 64, 6);
                }
            } else if (this.hasName) {
                buffer.write(0);
            } else if (hasValue) {
                buffer.writeHpackInt(this.number, 0, 4);
            } else {
                buffer.writeHpackInt(this.number, 128, 7);
            }
            if (this.hasName) {
                String name = headerName.lowerCase();
                if (name.length() > 3) {
                    huffman.encode(buffer, name);
                } else {
                    byte[] nameBytes = name.getBytes(StandardCharsets.US_ASCII);
                    buffer.writeHpackInt(nameBytes.length, 0, 7);
                    buffer.write(nameBytes);
                }
            }
            if (hasValue) {
                if (value.length() > 3) {
                    huffman.encode(buffer, value);
                } else {
                    byte[] valueBytes = value.getBytes(StandardCharsets.US_ASCII);
                    buffer.writeHpackInt(valueBytes.length, 0, 7);
                    buffer.write(valueBytes);
                }
            }
        }

        public boolean hasValue() {
            return this.hasValue;
        }

        public boolean neverIndex() {
            return this.neverIndex;
        }

        public boolean hasName() {
            return this.hasName;
        }

        public boolean addToIndex() {
            return this.addToIndex;
        }

        void write(BufferData buffer) {
            buffer.writeHpackInt(this.number, 128, 7);
        }
    }

    private record DynamicHeader(HeaderName headerName, String value, int size) implements HeaderRecord
    {
    }

    static interface HeaderRecord {
        public HeaderName headerName();

        public String value();
    }

    static interface IndexedHeaderRecord
    extends HeaderRecord {
        public int index();
    }

    private record IndexedHeader(HeaderRecord delegate, int index) implements IndexedHeaderRecord
    {
        @Override
        public HeaderName headerName() {
            return this.delegate().headerName();
        }

        @Override
        public String value() {
            return this.delegate.value();
        }
    }
}

