/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.service.networkadmin.impl.handlers;

import com.hedera.hapi.node.base.FileID;
import com.hedera.hapi.node.base.ResponseCodeEnum;
import com.hedera.hapi.node.base.Timestamp;
import com.hedera.hapi.node.freeze.FreezeTransactionBody;
import com.hedera.hapi.node.freeze.FreezeType;
import com.hedera.hapi.node.transaction.TransactionBody;
import com.hedera.node.app.service.addressbook.ReadableNodeStore;
import com.hedera.node.app.service.file.ReadableUpgradeFileStore;
import com.hedera.node.app.service.networkadmin.ReadableFreezeStore;
import com.hedera.node.app.service.networkadmin.impl.WritableFreezeStore;
import com.hedera.node.app.service.networkadmin.impl.handlers.FreezeUpgradeActions;
import com.hedera.node.app.service.token.ReadableStakingInfoStore;
import com.hedera.node.app.spi.fees.FeeContext;
import com.hedera.node.app.spi.fees.Fees;
import com.hedera.node.app.spi.ids.EntityIdFactory;
import com.hedera.node.app.spi.store.StoreFactory;
import com.hedera.node.app.spi.workflows.HandleContext;
import com.hedera.node.app.spi.workflows.HandleException;
import com.hedera.node.app.spi.workflows.PreCheckException;
import com.hedera.node.app.spi.workflows.PreHandleContext;
import com.hedera.node.app.spi.workflows.PureChecksContext;
import com.hedera.node.app.spi.workflows.TransactionHandler;
import com.hedera.node.config.data.FilesConfig;
import com.hedera.node.config.types.LongPair;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.IOException;
import java.time.Instant;
import java.util.Arrays;
import java.util.Objects;
import java.util.concurrent.Executor;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Singleton
public class FreezeHandler
implements TransactionHandler {
    private static final Logger log = LogManager.getLogger(FreezeHandler.class);
    private static final int UPDATE_FILE_HASH_LEN = 48;
    private final Executor freezeExecutor;
    private final EntityIdFactory entityIdFactory;

    @Inject
    public FreezeHandler(@Named(value="FreezeService") @NonNull Executor freezeExecutor, @NonNull EntityIdFactory entityIdFactory) {
        this.freezeExecutor = Objects.requireNonNull(freezeExecutor);
        this.entityIdFactory = Objects.requireNonNull(entityIdFactory);
    }

    public void preHandle(@NonNull PreHandleContext context) throws PreCheckException {
        Objects.requireNonNull(context);
        FreezeTransactionBody freezeTxn = context.body().freezeOrThrow();
        FreezeType freezeType = freezeTxn.freezeType();
        if (Arrays.asList(FreezeType.FREEZE_UPGRADE, FreezeType.TELEMETRY_UPGRADE, FreezeType.PREPARE_UPGRADE).contains(freezeType)) {
            ReadableUpgradeFileStore upgradeStore = (ReadableUpgradeFileStore)context.createStore(ReadableUpgradeFileStore.class);
            FilesConfig filesConfig = (FilesConfig)context.configuration().getConfigData(FilesConfig.class);
            FreezeHandler.verifyUpdateFile(freezeTxn, upgradeStore, filesConfig.softwareUpdateRange());
        }
    }

    public void pureChecks(@NonNull PureChecksContext context) throws PreCheckException {
        Bytes fileHash;
        FileID updateFileID;
        Objects.requireNonNull(context);
        TransactionBody txn = context.body();
        Objects.requireNonNull(txn);
        FreezeTransactionBody freezeTxn = FreezeHandler.getFreezeTransactionBody(txn);
        FreezeType freezeType = freezeTxn.freezeType();
        if (freezeType == FreezeType.UNKNOWN_FREEZE_TYPE) {
            throw new PreCheckException(ResponseCodeEnum.INVALID_FREEZE_TRANSACTION_BODY);
        }
        if (Arrays.asList(FreezeType.FREEZE_ONLY, FreezeType.FREEZE_UPGRADE, FreezeType.TELEMETRY_UPGRADE).contains(freezeType)) {
            Timestamp txValidStart = txn.transactionIDOrThrow().transactionValidStartOrThrow();
            FreezeHandler.verifyFreezeStartTimeIsInFuture(freezeTxn, txValidStart);
        }
        if ((freezeTxn.freezeType() == FreezeType.PREPARE_UPGRADE || freezeTxn.freezeType() == FreezeType.TELEMETRY_UPGRADE) && (updateFileID = freezeTxn.updateFile()) == null) {
            throw new PreCheckException(ResponseCodeEnum.INVALID_FREEZE_TRANSACTION_BODY);
        }
        if (Arrays.asList(FreezeType.FREEZE_UPGRADE, FreezeType.TELEMETRY_UPGRADE, FreezeType.PREPARE_UPGRADE).contains(freezeType) && ((fileHash = freezeTxn.fileHash()) == null || Bytes.EMPTY.equals((Object)fileHash) || fileHash.length() != 48L)) {
            throw new PreCheckException(ResponseCodeEnum.FREEZE_UPDATE_FILE_HASH_DOES_NOT_MATCH);
        }
    }

    @NonNull
    private static FreezeTransactionBody getFreezeTransactionBody(TransactionBody txn) throws PreCheckException {
        FreezeTransactionBody freezeTxn = txn.freezeOrThrow();
        if (freezeTxn.startHour() != 0 || freezeTxn.startMin() != 0 || freezeTxn.endHour() != 0 || freezeTxn.endMin() != 0) {
            throw new PreCheckException(ResponseCodeEnum.INVALID_FREEZE_TRANSACTION_BODY);
        }
        return freezeTxn;
    }

    public void handle(@NonNull HandleContext context) throws HandleException {
        Objects.requireNonNull(context);
        TransactionBody txn = context.body();
        StoreFactory storeFactory = context.storeFactory();
        ReadableUpgradeFileStore upgradeFileStore = (ReadableUpgradeFileStore)storeFactory.readableStore(ReadableUpgradeFileStore.class);
        ReadableNodeStore nodeStore = (ReadableNodeStore)storeFactory.readableStore(ReadableNodeStore.class);
        ReadableStakingInfoStore stakingInfoStore = (ReadableStakingInfoStore)storeFactory.readableStore(ReadableStakingInfoStore.class);
        WritableFreezeStore freezeStore = (WritableFreezeStore)storeFactory.writableStore(WritableFreezeStore.class);
        FreezeTransactionBody freezeTxn = txn.freezeOrThrow();
        FreezeHandler.validateSemantics(freezeTxn, freezeStore, upgradeFileStore);
        FileID updateFileID = freezeTxn.updateFile();
        FilesConfig filesConfig = (FilesConfig)context.configuration().getConfigData(FilesConfig.class);
        FreezeUpgradeActions upgradeActions = new FreezeUpgradeActions(context.configuration(), freezeStore, this.freezeExecutor, upgradeFileStore, nodeStore, stakingInfoStore, this.entityIdFactory);
        Timestamp freezeStartTime = freezeTxn.startTime();
        switch (freezeTxn.freezeType()) {
            case PREPARE_UPGRADE: {
                freezeStore.updateFileHash(freezeTxn.fileHash());
                log.info("Preparing upgrade with file {}, hash {}", (Object)updateFileID, (Object)freezeTxn.fileHash());
                try {
                    if (updateFileID == null || updateFileID.fileNum() < filesConfig.softwareUpdateRange().left() || updateFileID.fileNum() > filesConfig.softwareUpdateRange().right()) break;
                    upgradeActions.extractSoftwareUpgrade(upgradeFileStore.getFull(updateFileID));
                    break;
                }
                catch (IOException e) {
                    throw new IllegalStateException("Error extracting upgrade file", e);
                }
            }
            case FREEZE_UPGRADE: {
                upgradeActions.scheduleFreezeUpgradeAt(Objects.requireNonNull(freezeStartTime));
                break;
            }
            case FREEZE_ABORT: {
                upgradeActions.abortScheduledFreeze();
                freezeStore.updateFileHash(Bytes.EMPTY);
                log.info("Preparing freeze abort with file {}, hash null", (Object)updateFileID);
                break;
            }
            case TELEMETRY_UPGRADE: {
                try {
                    if (updateFileID == null || updateFileID.fileNum() < filesConfig.softwareUpdateRange().left() || updateFileID.fileNum() > filesConfig.softwareUpdateRange().right()) break;
                    upgradeActions.extractTelemetryUpgrade(upgradeFileStore.getFull(updateFileID), Objects.requireNonNull(freezeStartTime));
                    break;
                }
                catch (IOException e) {
                    throw new IllegalStateException("Error extracting upgrade file", e);
                }
            }
            case FREEZE_ONLY: {
                upgradeActions.scheduleFreezeOnlyAt(Objects.requireNonNull(freezeStartTime));
                break;
            }
            case UNKNOWN_FREEZE_TYPE: {
                throw new HandleException(ResponseCodeEnum.INVALID_FREEZE_TRANSACTION_BODY);
            }
        }
    }

    @NonNull
    public Fees calculateFees(@NonNull FeeContext feeContext) {
        Objects.requireNonNull(feeContext);
        return Fees.FREE;
    }

    private static void validateSemantics(@NonNull FreezeTransactionBody freezeTxn, @NonNull ReadableFreezeStore freezeStore, @NonNull ReadableUpgradeFileStore upgradeStore) {
        Objects.requireNonNull(freezeTxn);
        Objects.requireNonNull(freezeStore);
        Objects.requireNonNull(upgradeStore);
        FileID updateFileID = freezeTxn.updateFile();
        if (!(freezeTxn.freezeType() != FreezeType.PREPARE_UPGRADE && freezeTxn.freezeType() != FreezeType.TELEMETRY_UPGRADE || updateFileID != null && upgradeStore.peek(updateFileID) != null)) {
            throw new IllegalStateException("Update file not found");
        }
    }

    private static void verifyFreezeStartTimeIsInFuture(@NonNull FreezeTransactionBody freezeTxn, @NonNull Timestamp curConsensusTime) throws PreCheckException {
        Instant effectiveNowInstant;
        Objects.requireNonNull(freezeTxn);
        Objects.requireNonNull(curConsensusTime);
        Timestamp freezeStartTime = freezeTxn.startTime();
        if (freezeStartTime == null || freezeStartTime.seconds() == 0L && freezeStartTime.nanos() == 0) {
            throw new PreCheckException(ResponseCodeEnum.INVALID_FREEZE_TRANSACTION_BODY);
        }
        Instant freezeStartTimeInstant = Instant.ofEpochSecond(freezeStartTime.seconds(), freezeStartTime.nanos());
        boolean freezeStartTimeIsInFuture = freezeStartTimeInstant.isAfter(effectiveNowInstant = Instant.ofEpochSecond(curConsensusTime.seconds(), curConsensusTime.nanos()));
        if (!freezeStartTimeIsInFuture) {
            throw new PreCheckException(ResponseCodeEnum.FREEZE_START_TIME_MUST_BE_FUTURE);
        }
    }

    private static void verifyUpdateFile(@NonNull FreezeTransactionBody freezeTxn, @NonNull ReadableUpgradeFileStore upgradeStore, LongPair softwareUpdateRange) throws PreCheckException {
        Objects.requireNonNull(freezeTxn);
        Objects.requireNonNull(upgradeStore);
        FileID updateFileID = freezeTxn.updateFile();
        if (updateFileID == null || updateFileID.fileNum() < softwareUpdateRange.left() || updateFileID.fileNum() > softwareUpdateRange.right()) {
            throw new PreCheckException(ResponseCodeEnum.INVALID_FREEZE_TRANSACTION_BODY);
        }
        if (upgradeStore.peek(updateFileID) == null) {
            throw new PreCheckException(ResponseCodeEnum.FREEZE_UPDATE_FILE_DOES_NOT_EXIST);
        }
    }
}

