/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.hapi.block.stream.output.StateChange;
import com.hedera.hapi.block.stream.output.StateChanges;
import com.hedera.hapi.node.state.blockstream.BlockStreamInfo;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Objects;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.BlockItemSet;
import org.hiero.block.api.PublishStreamRequest;

public class BlockState {
    private static final Logger logger = LogManager.getLogger(BlockState.class);
    private final long blockNumber;
    private final Queue<BlockItem> pendingItems = new ConcurrentLinkedQueue<BlockItem>();
    private final ConcurrentMap<Integer, RequestWrapper> requestsByIndex = new ConcurrentHashMap<Integer, RequestWrapper>();
    private final AtomicInteger requestIdxCtr = new AtomicInteger(0);
    private final AtomicReference<Instant> closedTimestamp = new AtomicReference();
    private final ItemInfo headerItemInfo = new ItemInfo();
    private final ItemInfo proofItemInfo = new ItemInfo();
    private final ItemInfo preProofItemInfo = new ItemInfo();

    public BlockState(long blockNumber) {
        this.blockNumber = blockNumber;
    }

    public long blockNumber() {
        return this.blockNumber;
    }

    public void addItem(BlockItem item) {
        if (item == null) {
            return;
        }
        if (this.closedTimestamp.get() != null) {
            throw new IllegalStateException("Block is closed; adding more items is not permitted");
        }
        if (item.hasBlockHeader() && !this.headerItemInfo.addedInBlockState()) {
            logger.warn("[Block {}] Block header item added, but block header already encountered (state={})", (Object)this.blockNumber, (Object)this.headerItemInfo.state.get());
        } else if (item.hasBlockProof() && !this.proofItemInfo.addedInBlockState()) {
            logger.warn("[Block {}] Block proof item added, but block proof already encountered (state={})", (Object)this.blockNumber, (Object)this.proofItemInfo.state.get());
        } else if (item.hasStateChanges() && this.isPreProofItemReceived(item.stateChangesOrElse(StateChanges.DEFAULT)) && !this.preProofItemInfo.addedInBlockState()) {
            logger.warn("[Block {}] Pre-proof state change item added, but pre-proof state change already encountered (state={})", (Object)this.blockNumber, (Object)this.preProofItemInfo.state.get());
        }
        this.pendingItems.add(item);
    }

    public int numRequestsCreated() {
        return this.requestsByIndex.size();
    }

    @Nullable
    public PublishStreamRequest getRequest(int index) {
        RequestWrapper rs = (RequestWrapper)this.requestsByIndex.get(index);
        return rs == null ? null : rs.request;
    }

    public void closeBlock() {
        Instant now = Instant.now();
        if (this.closedTimestamp.compareAndSet(null, now)) {
            logger.debug("[Block {}] closed at {}", (Object)this.blockNumber, (Object)now);
        } else {
            logger.warn("[Block {}] Attempted to close block at {}, but this block was already closed at {}. Ignoring new close attempt.", (Object)this.blockNumber, (Object)now, (Object)this.closedTimestamp.get());
        }
    }

    public void closeBlock(Instant timestamp) {
        this.closedTimestamp.set(timestamp);
    }

    @Nullable
    public Instant closedTimestamp() {
        return this.closedTimestamp.get();
    }

    public boolean isClosed() {
        return this.closedTimestamp.get() != null;
    }

    public synchronized void processPendingItems(int batchSize) {
        boolean preProofNeedsToBeSent;
        if (this.pendingItems.isEmpty()) {
            return;
        }
        int maxItems = Math.max(1, batchSize);
        boolean hasEnoughItemsForBatch = this.pendingItems.size() >= maxItems;
        boolean headerNeedsToBeSent = ItemState.ADDED == this.headerItemInfo.state.get();
        boolean proofNeedsToBeSent = ItemState.ADDED == this.proofItemInfo.state.get();
        boolean bl = preProofNeedsToBeSent = ItemState.ADDED == this.preProofItemInfo.state.get();
        if (!(hasEnoughItemsForBatch || headerNeedsToBeSent || proofNeedsToBeSent || preProofNeedsToBeSent)) {
            return;
        }
        ArrayList<BlockItem> blockItems = new ArrayList<BlockItem>(maxItems);
        int index = this.requestIdxCtr.getAndIncrement();
        Iterator it = this.pendingItems.iterator();
        boolean forceCreation = false;
        while (it.hasNext()) {
            BlockItem item = (BlockItem)it.next();
            blockItems.add(item);
            it.remove();
            if (item.hasBlockHeader()) {
                if (this.headerItemInfo.packedInRequest(index)) {
                    logger.trace("[Block {}] Block header packed in request #{}", (Object)this.blockNumber, (Object)index);
                } else {
                    logger.warn("[Block {}] Block header item was not yet added (state={})", (Object)this.blockNumber, (Object)this.headerItemInfo.state.get());
                }
            } else if (item.hasStateChanges() && this.isPreProofItemReceived(item.stateChangesOrElse(StateChanges.DEFAULT))) {
                if (this.preProofItemInfo.packedInRequest(index)) {
                    forceCreation = true;
                    logger.trace("[Block {}] Pre-proof block state change packed in request #{}", (Object)this.blockNumber, (Object)index);
                } else {
                    logger.warn("[Block {}] Pre-proof block state change was not yet added (state={})", (Object)this.blockNumber, (Object)this.preProofItemInfo.state.get());
                }
            } else if (item.hasBlockProof()) {
                if (this.proofItemInfo.packedInRequest(index)) {
                    forceCreation = true;
                    logger.trace("[Block {}] Block proof packed in request #{}", (Object)this.blockNumber, (Object)index);
                } else {
                    logger.warn("[Block {}] Block proof was not yet added (state={})", (Object)this.blockNumber, (Object)this.proofItemInfo.state.get());
                }
            }
            if (it.hasNext() && blockItems.size() != maxItems && !forceCreation) continue;
            break;
        }
        BlockItemSet bis = BlockItemSet.newBuilder().blockItems(blockItems).build();
        PublishStreamRequest psr = PublishStreamRequest.newBuilder().blockItems(bis).build();
        RequestWrapper rs = new RequestWrapper(index, psr, new AtomicBoolean(false));
        this.requestsByIndex.put(index, rs);
        logger.trace("[Block {}] Created new request (index={}, numItems={})", (Object)this.blockNumber, (Object)index, (Object)blockItems.size());
        if (!this.pendingItems.isEmpty()) {
            this.processPendingItems(batchSize);
        }
    }

    public boolean isBlockProofSent() {
        return ItemState.SENT == this.proofItemInfo.state.get();
    }

    public void markRequestSent(int requestIndex) {
        RequestWrapper wrapper = (RequestWrapper)this.requestsByIndex.get(requestIndex);
        if (wrapper == null) {
            throw new IllegalArgumentException("Invalid request index: " + requestIndex);
        }
        wrapper.isSent.set(true);
        if (requestIndex == this.proofItemInfo.requestIndex.get()) {
            this.proofItemInfo.state.set(ItemState.SENT);
        }
    }

    private boolean isPreProofItemReceived(@NonNull StateChanges stateChanges) {
        return stateChanges.stateChanges().stream().map(StateChange::singletonUpdate).filter(Objects::nonNull).anyMatch(update -> update.hasBlockStreamInfoValue() && update.blockStreamInfoValueOrElse(BlockStreamInfo.DEFAULT).blockNumber() != -1L);
    }

    public String toString() {
        return "BlockState {blockNumber=" + this.blockNumber + ", closedTimestamp=" + String.valueOf(this.closedTimestamp.get()) + ", numPendingItems=" + this.pendingItems.size() + ", numRequests=" + this.requestsByIndex.size() + ", blockHeader=" + String.valueOf(this.headerItemInfo) + ", blockPreProof=" + String.valueOf(this.preProofItemInfo) + ", blockProof=" + String.valueOf(this.proofItemInfo) + "}";
    }

    record ItemInfo(AtomicReference<ItemState> state, AtomicInteger requestIndex) {
        ItemInfo() {
            this(new AtomicReference<ItemState>(ItemState.NIL), new AtomicInteger(-1));
        }

        boolean addedInBlockState() {
            return this.state.compareAndSet(ItemState.NIL, ItemState.ADDED);
        }

        boolean packedInRequest(int requestIdx) {
            if (this.state.compareAndSet(ItemState.ADDED, ItemState.PACKED)) {
                this.requestIndex.set(requestIdx);
                return true;
            }
            return false;
        }
    }

    record RequestWrapper(int index, PublishStreamRequest request, AtomicBoolean isSent) {
    }

    static enum ItemState {
        NIL,
        ADDED,
        PACKED,
        SENT;

    }
}

