/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.files;

import com.hedera.pbj.runtime.ProtoConstants;
import com.hedera.pbj.runtime.ProtoWriterTools;
import com.hedera.pbj.runtime.io.buffer.BufferedData;
import com.swirlds.merkledb.collections.IndexedObject;
import com.swirlds.merkledb.config.MerkleDbConfig;
import com.swirlds.merkledb.files.DataFileCommon;
import com.swirlds.merkledb.files.DataFileIterator;
import com.swirlds.merkledb.files.DataFileMetadata;
import com.swirlds.merkledb.utilities.MerkleDbFileUtils;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;

public final class DataFileReader
implements AutoCloseable,
Comparable<DataFileReader>,
IndexedObject {
    private static final ThreadLocal<ByteBuffer> BUFFER_CACHE = new ThreadLocal();
    private static final ThreadLocal<BufferedData> BUFFEREDDATA_CACHE = new ThreadLocal();
    private final MerkleDbConfig dbConfig;
    private final int maxFileChannels;
    private final int threadsPerFileChannel;
    private final AtomicReferenceArray<FileChannel> fileChannels;
    private final AtomicInteger fileChannelsCount = new AtomicInteger(0);
    private final AtomicInteger fileChannelsInUse = new AtomicInteger(0);
    private final AtomicBoolean open = new AtomicBoolean(true);
    private final Path path;
    private final DataFileMetadata metadata;
    private final AtomicBoolean fileCompleted = new AtomicBoolean(false);
    private final AtomicLong fileSizeBytes = new AtomicLong(0L);

    public DataFileReader(MerkleDbConfig dbConfig, Path path) throws IOException {
        this(dbConfig, path, DataFileMetadata.readFromFile(path));
    }

    DataFileReader(MerkleDbConfig dbConfig, Path path, DataFileMetadata metadata) throws IOException {
        this.dbConfig = dbConfig;
        this.maxFileChannels = dbConfig.maxFileChannelsPerFileReader();
        this.threadsPerFileChannel = dbConfig.maxThreadsPerFileChannel();
        this.fileChannels = new AtomicReferenceArray(this.maxFileChannels);
        if (!Files.exists(path, new LinkOption[0])) {
            throw new IllegalArgumentException("Tried to open a non existent data file [" + String.valueOf(path.toAbsolutePath()) + "].");
        }
        this.path = path;
        this.metadata = metadata;
        this.openNewFileChannel(0);
    }

    public boolean isFileCompleted() {
        return this.fileCompleted.get();
    }

    void setFileCompleted() {
        try {
            this.fileSizeBytes.set(this.fileChannels.get(0).size());
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to update data file reader size", e);
        }
        finally {
            this.fileCompleted.set(true);
        }
    }

    @Override
    public int getIndex() {
        return this.metadata.getIndex();
    }

    public DataFileMetadata getMetadata() {
        return this.metadata;
    }

    public Path getPath() {
        return this.path;
    }

    public DataFileIterator createIterator() throws IOException {
        return new DataFileIterator(this.dbConfig, this.path, this.metadata);
    }

    public BufferedData readDataItem(long dataLocation) throws IOException {
        long byteOffset = DataFileCommon.byteOffsetFromDataLocation(dataLocation);
        return this.read(byteOffset);
    }

    public long getSize() {
        return this.fileSizeBytes.get();
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DataFileReader that = (DataFileReader)o;
        return this.path.equals(that.getPath());
    }

    public int hashCode() {
        return this.path.hashCode();
    }

    @Override
    public int compareTo(@NonNull DataFileReader o) {
        Objects.requireNonNull(o);
        if (this == o) {
            return 0;
        }
        int res = this.metadata.getCreationDate().compareTo(o.getMetadata().getCreationDate());
        return res != 0 ? res : Integer.compare(this.metadata.getIndex(), o.getMetadata().getIndex());
    }

    public String toString() {
        return Integer.toString(this.metadata.getIndex());
    }

    int getMaxFileChannels() {
        return this.maxFileChannels;
    }

    int getThreadsPerFileChannel() {
        return this.threadsPerFileChannel;
    }

    public boolean isOpen() {
        return this.open.get();
    }

    @Override
    public void close() throws IOException {
        if (!this.open.compareAndSet(true, false)) {
            return;
        }
        for (int i = 0; i < this.maxFileChannels; ++i) {
            FileChannel fileChannel = this.fileChannels.getAndSet(i, null);
            if (fileChannel == null) continue;
            fileChannel.close();
        }
    }

    private void openNewFileChannel(int index) throws IOException {
        if (index >= this.maxFileChannels) {
            return;
        }
        FileChannel fileChannel = FileChannel.open(this.path, StandardOpenOption.READ);
        if (this.fileChannels.compareAndSet(index, null, fileChannel)) {
            this.fileChannelsCount.incrementAndGet();
        } else {
            fileChannel.close();
        }
    }

    private void reopenFileChannel(int index, FileChannel closedChannel) throws IOException {
        assert (index < this.maxFileChannels);
        assert (!closedChannel.isOpen());
        FileChannel fileChannel = FileChannel.open(this.path, StandardOpenOption.READ);
        if (!this.fileChannels.compareAndSet(index, closedChannel, fileChannel)) {
            fileChannel.close();
        }
    }

    int leaseFileChannel() throws IOException {
        int count = this.fileChannelsCount.get();
        int inUse = this.fileChannelsInUse.incrementAndGet();
        if (inUse / count > this.threadsPerFileChannel && count < this.maxFileChannels) {
            this.openNewFileChannel(count);
            count = this.fileChannelsCount.get();
        }
        return inUse % count;
    }

    void releaseFileChannel() {
        this.fileChannelsInUse.decrementAndGet();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private BufferedData read(long byteOffsetInFile) throws IOException {
        int PRE_READ_BUF_SIZE = 2048;
        ByteBuffer readBB = BUFFER_CACHE.get();
        BufferedData readBuf = BUFFEREDDATA_CACHE.get();
        if (readBuf == null) {
            readBB = ByteBuffer.allocate(2048);
            BUFFER_CACHE.set(readBB);
            readBuf = BufferedData.wrap((ByteBuffer)readBB);
            BUFFEREDDATA_CACHE.set(readBuf);
        }
        for (int retries = 3; retries > 0; --retries) {
            int fcIndex = this.leaseFileChannel();
            FileChannel fileChannel = this.fileChannels.get(fcIndex);
            if (fileChannel == null) {
                return null;
            }
            try {
                int size;
                int sizeOfSize;
                readBB.clear();
                readBB.limit(2048);
                int bytesRead = MerkleDbFileUtils.completelyRead(fileChannel, readBB, byteOffsetInFile);
                if (this.isFileCompleted() && (long)bytesRead != Math.min(2048L, this.getSize() - byteOffsetInFile)) {
                    throw new IOException("Failed to read all bytes: toread=" + Math.min(2048L, this.getSize() - byteOffsetInFile) + " read=" + bytesRead + " file=" + this.getIndex() + " off=" + byteOffsetInFile);
                }
                readBuf.reset();
                int tag = readBuf.getVarInt(0L, false);
                if (tag != (DataFileCommon.FIELD_DATAFILE_ITEMS.number() << 3 | ProtoConstants.WIRE_TYPE_DELIMITED.ordinal())) {
                    throw new IOException("Unknown data item tag: tag=" + tag + " file=" + this.getIndex() + " off=" + byteOffsetInFile);
                }
                int sizeOfTag = ProtoWriterTools.sizeOfUnsignedVarInt32((int)tag);
                if (bytesRead >= sizeOfTag + (sizeOfSize = ProtoWriterTools.sizeOfUnsignedVarInt32((int)(size = readBuf.getVarInt((long)sizeOfTag, false)))) + size) {
                    readBuf.position((long)(sizeOfTag + sizeOfSize));
                    readBuf.limit((long)(sizeOfTag + sizeOfSize + size));
                    BufferedData bufferedData = readBuf;
                    return bufferedData;
                }
                if (readBB.capacity() <= size) {
                    readBB = ByteBuffer.allocate(size);
                    BUFFER_CACHE.set(readBB);
                    readBuf = BufferedData.wrap((ByteBuffer)readBB);
                    BUFFEREDDATA_CACHE.set(readBuf);
                } else {
                    readBB.position(0);
                    readBB.limit(size);
                }
                bytesRead = MerkleDbFileUtils.completelyRead(fileChannel, readBB, byteOffsetInFile + (long)sizeOfTag + (long)sizeOfSize);
                if (bytesRead != size) {
                    throw new IOException("Failed to read all bytes: toread=" + size + " read=" + bytesRead + " file=" + this.getIndex() + " off=" + byteOffsetInFile);
                }
                readBuf.position(0L);
                readBuf.limit((long)bytesRead);
                BufferedData bufferedData = readBuf;
                return bufferedData;
            }
            catch (ClosedByInterruptException e) {
                throw e;
            }
            catch (ClosedChannelException e) {
                this.reopenFileChannel(fcIndex, fileChannel);
                continue;
            }
            finally {
                this.releaseFileChannel();
            }
        }
        throw new IOException("Failed to read from file, file channel keeps getting closed");
    }

    int getFileChannelsCount() {
        return this.fileChannelsCount.get();
    }
}

