/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.webserver.http2;

import io.helidon.common.buffers.BufferData;
import io.helidon.common.buffers.DataReader;
import io.helidon.common.concurrency.limits.FixedLimit;
import io.helidon.common.concurrency.limits.Limit;
import io.helidon.common.socket.SocketContext;
import io.helidon.common.task.InterruptableTask;
import io.helidon.common.tls.TlsUtils;
import io.helidon.http.DateTime;
import io.helidon.http.HeaderNames;
import io.helidon.http.HeaderValues;
import io.helidon.http.Headers;
import io.helidon.http.HttpPrologue;
import io.helidon.http.Method;
import io.helidon.http.WritableHeaders;
import io.helidon.http.http2.ConnectionFlowControl;
import io.helidon.http.http2.Http2ConnectionWriter;
import io.helidon.http.http2.Http2ErrorCode;
import io.helidon.http.http2.Http2Exception;
import io.helidon.http.http2.Http2Flag;
import io.helidon.http.http2.Http2FrameData;
import io.helidon.http.http2.Http2FrameHeader;
import io.helidon.http.http2.Http2FrameListener;
import io.helidon.http.http2.Http2FrameType;
import io.helidon.http.http2.Http2FrameTypes;
import io.helidon.http.http2.Http2GoAway;
import io.helidon.http.http2.Http2Headers;
import io.helidon.http.http2.Http2HuffmanDecoder;
import io.helidon.http.http2.Http2LoggingFrameListener;
import io.helidon.http.http2.Http2Ping;
import io.helidon.http.http2.Http2Priority;
import io.helidon.http.http2.Http2RstStream;
import io.helidon.http.http2.Http2Setting;
import io.helidon.http.http2.Http2Settings;
import io.helidon.http.http2.Http2Stream;
import io.helidon.http.http2.Http2StreamState;
import io.helidon.http.http2.Http2StreamWriter;
import io.helidon.http.http2.Http2Util;
import io.helidon.http.http2.Http2WindowUpdate;
import io.helidon.webserver.CloseConnectionException;
import io.helidon.webserver.ConnectionContext;
import io.helidon.webserver.Routing;
import io.helidon.webserver.http.HttpRouting;
import io.helidon.webserver.http2.Http2Config;
import io.helidon.webserver.http2.Http2ConnectionStreams;
import io.helidon.webserver.http2.Http2ServerStream;
import io.helidon.webserver.http2.spi.Http2SubProtocolSelector;
import io.helidon.webserver.spi.ServerConnection;
import java.io.UncheckedIOException;
import java.net.SocketException;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Semaphore;

public class Http2Connection
implements ServerConnection,
InterruptableTask<Void> {
    static final String FULL_PROTOCOL = "HTTP/2.0";
    static final String PROTOCOL = "HTTP";
    static final String PROTOCOL_VERSION = "2.0";
    private static final System.Logger LOGGER = System.getLogger(Http2Connection.class.getName());
    private static final int FRAME_HEADER_LENGTH = 9;
    private static final Set<Http2StreamState> REMOVABLE_STREAMS = Set.of(Http2StreamState.CLOSED, Http2StreamState.HALF_CLOSED_LOCAL);
    private final Http2ConnectionStreams streams = new Http2ConnectionStreams();
    private final ConnectionContext ctx;
    private final Http2Config http2Config;
    private final HttpRouting routing;
    private final Http2Headers.DynamicTable requestDynamicTable;
    private final Http2HuffmanDecoder requestHuffman;
    private final Http2FrameListener receiveFrameListener = Http2FrameListener.create(List.of(new Http2LoggingFrameListener("recv")));
    private final Http2ConnectionWriter connectionWriter;
    private final List<Http2SubProtocolSelector> subProviders;
    private final DataReader reader;
    private final Http2Settings serverSettings;
    private final boolean sendErrorDetails;
    private final ConnectionFlowControl flowControl;
    private final WritableHeaders<?> connectionHeaders;
    private final long rapidResetCheckPeriod;
    private final int maxRapidResets;
    private final int maxEmptyFrames;
    private final long maxClientConcurrentStreams;
    private int rapidResetCnt = 0;
    private long rapidResetPeriodStart = 0L;
    private int emptyFrames = 0;
    private Http2Settings clientSettings = Http2Settings.builder().build();
    private Http2FrameHeader frameHeader;
    private BufferData frameInProgress;
    private Http2Ping ping;
    private boolean expectPreface;
    private HttpPrologue upgradePrologue;
    private Http2Headers upgradeHeaders;
    private State state = State.WRITE_SERVER_SETTINGS;
    private int continuationExpectedStreamId;
    private int lastStreamId;
    private boolean initConnectionHeaders;
    private volatile ZonedDateTime lastRequestTimestamp;
    private volatile Thread myThread;
    private volatile boolean canRun = true;

    Http2Connection(ConnectionContext ctx, Http2Config http2Config, List<Http2SubProtocolSelector> subProviders) {
        this.ctx = ctx;
        this.http2Config = http2Config;
        this.rapidResetCheckPeriod = http2Config.rapidResetCheckPeriod().toNanos();
        this.maxRapidResets = http2Config.maxRapidResets();
        this.maxEmptyFrames = http2Config.maxEmptyFrames();
        this.serverSettings = ((Http2Settings.Builder)Http2Settings.builder().update(builder -> Http2Connection.settingsUpdate(http2Config, builder))).add(Http2Setting.ENABLE_PUSH, (Object)false).build();
        this.connectionWriter = new Http2ConnectionWriter((SocketContext)ctx, ctx.dataWriter(), List.of(new Http2LoggingFrameListener("send")));
        this.subProviders = subProviders;
        this.requestDynamicTable = Http2Headers.DynamicTable.create((long)((Long)this.serverSettings.value(Http2Setting.HEADER_TABLE_SIZE)));
        this.requestHuffman = Http2HuffmanDecoder.create();
        this.routing = (HttpRouting)ctx.router().routing(HttpRouting.class, (Routing)HttpRouting.empty());
        this.reader = ctx.dataReader();
        this.sendErrorDetails = http2Config.sendErrorDetails();
        this.maxClientConcurrentStreams = http2Config.maxConcurrentStreams();
        this.flowControl = ConnectionFlowControl.serverBuilder(this::writeWindowUpdateFrame).initialWindowSize(http2Config.initialWindowSize()).blockTimeout(http2Config.flowControlTimeout()).maxFrameSize(http2Config.maxFrameSize()).build();
        this.lastRequestTimestamp = DateTime.timestamp();
        this.connectionHeaders = WritableHeaders.create();
        this.initConnectionHeaders = true;
    }

    private static void settingsUpdate(Http2Config config, Http2Settings.Builder builder) {
        Http2Connection.applySetting(builder, config.maxFrameSize(), (Http2Setting<Long>)Http2Setting.MAX_FRAME_SIZE);
        Http2Connection.applySetting(builder, config.maxHeaderListSize(), (Http2Setting<Long>)Http2Setting.MAX_HEADER_LIST_SIZE);
        Http2Connection.applySetting(builder, config.maxConcurrentStreams(), (Http2Setting<Long>)Http2Setting.MAX_CONCURRENT_STREAMS);
        Http2Connection.applySetting(builder, config.initialWindowSize(), (Http2Setting<Long>)Http2Setting.INITIAL_WINDOW_SIZE);
    }

    private static void applySetting(Http2Settings.Builder builder, long value, Http2Setting<Long> settings) {
        if (value != (Long)settings.defaultValue()) {
            builder.add(settings, (Object)value);
        }
    }

    public void handle(Limit limit) throws InterruptedException {
        try {
            this.doHandle(limit);
        }
        catch (Http2Exception e) {
            if (this.state == State.FINISHED) {
                return;
            }
            this.ctx.log(LOGGER, System.Logger.Level.DEBUG, "Intentional HTTP/2 exception, code: %s, message: %s", new Object[]{e.code(), e.getMessage()});
            this.ctx.log(LOGGER, System.Logger.Level.TRACE, "Stacktrace of HTTP/2 exception", (Throwable)e, new Object[0]);
            Http2GoAway frame = new Http2GoAway(0, e.code(), this.sendErrorDetails ? e.getMessage() : "");
            this.connectionWriter.write(frame.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
            this.state = State.FINISHED;
        }
        catch (CloseConnectionException | UncheckedIOException | InterruptedException e) {
            throw e;
        }
        catch (Throwable e) {
            if (this.state == State.FINISHED) {
                return;
            }
            Http2GoAway frame = new Http2GoAway(0, Http2ErrorCode.INTERNAL, (String)(this.sendErrorDetails ? e.getClass().getName() + ": " + e.getMessage() : ""));
            this.connectionWriter.write(frame.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
            this.state = State.FINISHED;
            throw e;
        }
    }

    public void handle(Semaphore requestSemaphore) throws InterruptedException {
        this.handle((Limit)FixedLimit.create((Semaphore)requestSemaphore));
    }

    public void clientSettings(Http2Settings http2Settings) {
        this.clientSettings = http2Settings;
        this.receiveFrameListener.frame((SocketContext)this.ctx, 0, this.clientSettings);
        if (this.clientSettings.hasValue(Http2Setting.HEADER_TABLE_SIZE)) {
            this.updateHeaderTableSize((Long)this.clientSettings.value(Http2Setting.HEADER_TABLE_SIZE));
        }
        if (this.clientSettings.hasValue(Http2Setting.INITIAL_WINDOW_SIZE)) {
            Long initialWindowSize = (Long)this.clientSettings.value(Http2Setting.INITIAL_WINDOW_SIZE);
            if (initialWindowSize > Integer.MAX_VALUE) {
                Http2GoAway frame = new Http2GoAway(0, Http2ErrorCode.FLOW_CONTROL, "Window " + initialWindowSize + " size too large");
                this.connectionWriter.write(frame.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
            }
            this.flowControl.resetInitialWindowSize(initialWindowSize.intValue());
            for (StreamContext sctx : this.streams.contexts()) {
                Http2StreamState streamState = sctx.stream.streamState();
                if (streamState != Http2StreamState.OPEN && streamState != Http2StreamState.HALF_CLOSED_REMOTE) continue;
                sctx.stream.flowControl().outbound().resetStreamWindowSize(initialWindowSize.intValue());
            }
            this.flowControl.outbound().triggerUpdate();
        }
        if (this.clientSettings.hasValue(Http2Setting.MAX_FRAME_SIZE)) {
            Long maxFrameSize = (Long)this.clientSettings.value(Http2Setting.MAX_FRAME_SIZE);
            if (maxFrameSize < 16384L || maxFrameSize > 0xFFFFFFL) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Frame size must be between 2^14 and 2^24-1, but is: " + maxFrameSize);
            }
            this.flowControl.resetMaxFrameSize(maxFrameSize.intValue());
        }
    }

    public void upgradeConnectionData(HttpPrologue prologue, Http2Headers headers) {
        this.upgradePrologue = prologue;
        this.upgradeHeaders = headers;
    }

    public void expectPreface() {
        this.expectPreface = true;
    }

    public boolean canInterrupt() {
        return this.streams.isEmpty();
    }

    public String toString() {
        return "[" + this.ctx.socketId() + " " + this.ctx.childSocketId() + "]";
    }

    public Duration idleTime() {
        return Duration.between(this.lastRequestTimestamp, DateTime.timestamp());
    }

    public void close(boolean interrupt) {
        this.canRun = false;
        if (interrupt) {
            if (this.myThread != null) {
                this.myThread.interrupt();
            }
        } else if (this.canInterrupt()) {
            this.myThread.interrupt();
        }
    }

    Http2Config config() {
        return this.http2Config;
    }

    Http2Settings serverSettings() {
        return this.serverSettings;
    }

    Http2Settings clientSettings() {
        return this.clientSettings;
    }

    private void doHandle(Limit limit) throws InterruptedException {
        this.myThread = Thread.currentThread();
        while (this.canRun && this.state != State.FINISHED) {
            if (this.expectPreface && this.state != State.WRITE_SERVER_SETTINGS) {
                this.readPreface();
                this.expectPreface = false;
            }
            if (this.state == State.READ_FRAME) {
                try {
                    this.readFrame();
                }
                catch (DataReader.InsufficientDataAvailableException e) {
                    throw new CloseConnectionException("Connection closed by client", (Throwable)e);
                }
            }
            this.dispatchHandler(limit);
        }
        if (this.state != State.FINISHED) {
            Http2GoAway frame = new Http2GoAway(0, Http2ErrorCode.NO_ERROR, "Idle timeout");
            this.connectionWriter.write(frame.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
        }
    }

    private void dispatchHandler(Limit limit) {
        switch (this.state.ordinal()) {
            case 14: {
                this.doContinuation();
                break;
            }
            case 11: {
                this.writeServerSettings();
                break;
            }
            case 9: {
                this.readWindowUpdateFrame();
                break;
            }
            case 5: {
                this.doSettings();
                break;
            }
            case 10: {
                this.ackSettings();
                break;
            }
            case 1: {
                this.dataFrame();
                break;
            }
            case 2: {
                this.doHeaders(limit);
                break;
            }
            case 3: {
                this.doPriority();
                break;
            }
            case 6: {
                throw new Http2Exception(Http2ErrorCode.REFUSED_STREAM, "Push promise not supported");
            }
            case 7: {
                this.pingFrame();
                break;
            }
            case 13: {
                this.writePingAck();
                break;
            }
            case 8: {
                this.goAwayFrame();
                break;
            }
            case 4: {
                this.rstStream();
                break;
            }
            default: {
                this.unknownFrame();
            }
        }
    }

    private void readPreface() {
        BufferData preface = this.reader.readBuffer(Http2Util.PREFACE_LENGTH);
        byte[] bytes = new byte[Http2Util.PREFACE_LENGTH];
        preface.read(bytes);
        if (!Http2Util.isPreface((byte[])bytes)) {
            throw new IllegalStateException("Invalid HTTP/2 connection preface: \n" + preface.debugDataHex(true));
        }
        this.ctx.log(LOGGER, System.Logger.Level.TRACE, "Processed preface data", new Object[0]);
        this.state = State.READ_FRAME;
    }

    private void readFrame() {
        int streamId;
        BufferData frameHeaderBuffer = this.reader.readBuffer(9);
        try {
            this.frameHeader = Http2FrameHeader.create((BufferData)frameHeaderBuffer);
            streamId = this.frameHeader.streamId();
            this.receiveFrameListener.frameHeader((SocketContext)this.ctx, streamId, frameHeaderBuffer);
        }
        catch (Throwable e) {
            this.receiveFrameListener.frameHeader((SocketContext)this.ctx, 0, frameHeaderBuffer);
            throw e;
        }
        this.receiveFrameListener.frameHeader((SocketContext)this.ctx, this.frameHeader.streamId(), this.frameHeader);
        if ((Long)this.serverSettings.value(Http2Setting.MAX_FRAME_SIZE) < (long)this.frameHeader.length()) {
            throw new Http2Exception(Http2ErrorCode.FRAME_SIZE, "Frame size " + this.frameHeader.length() + " is too big");
        }
        this.frameHeader.type().checkLength(this.frameHeader.length());
        this.frameInProgress = this.frameHeader.length() == 0 ? BufferData.empty() : this.reader.readBuffer(this.frameHeader.length());
        this.receiveFrameListener.frame((SocketContext)this.ctx, streamId, this.frameInProgress);
        switch (this.frameHeader.type()) {
            default: {
                throw new MatchException(null, null);
            }
            case DATA: {
                State state = State.DATA;
                break;
            }
            case HEADERS: {
                State state = State.HEADERS;
                break;
            }
            case PRIORITY: {
                State state = State.PRIORITY;
                break;
            }
            case RST_STREAM: {
                State state = State.RST_STREAM;
                break;
            }
            case SETTINGS: {
                State state = State.SETTINGS;
                break;
            }
            case PUSH_PROMISE: {
                State state = State.READ_PUSH_PROMISE;
                break;
            }
            case PING: {
                State state = State.PING;
                break;
            }
            case GO_AWAY: {
                State state = State.GO_AWAY;
                break;
            }
            case WINDOW_UPDATE: {
                State state = State.WINDOW_UPDATE;
                break;
            }
            case CONTINUATION: {
                State state = State.CONTINUATION;
                break;
            }
            case UNKNOWN: {
                State state = this.state = State.UNKNOWN;
            }
        }
        if (this.continuationExpectedStreamId != 0) {
            if (this.state == State.CONTINUATION) {
                if (this.continuationExpectedStreamId != streamId) {
                    throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received CONTINUATION for stream " + streamId + ", expected for " + this.continuationExpectedStreamId);
                }
            } else {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Expecting CONTINUATION for stream " + this.continuationExpectedStreamId + ", received " + String.valueOf((Object)this.state) + " for " + streamId);
            }
        }
    }

    private void doContinuation() {
        Http2Flag.ContinuationFlags flags = (Http2Flag.ContinuationFlags)this.frameHeader.flags(Http2FrameTypes.CONTINUATION);
        this.stream(this.frameHeader.streamId()).addContinuation(new Http2FrameData(this.frameHeader, this.inProgressFrame()));
        this.state = flags.endOfHeaders() ? State.HEADERS : State.READ_FRAME;
    }

    private void writeServerSettings() {
        this.connectionWriter.write(this.serverSettings.toFrameData(this.serverSettings, 0, Http2Flag.SettingsFlags.create((int)0)));
        int connectionWinSizeUpd = this.http2Config.initialWindowSize() - 65535;
        if (connectionWinSizeUpd > 0) {
            Http2WindowUpdate windowUpdate = new Http2WindowUpdate(this.http2Config.initialWindowSize() - 65535);
            this.connectionWriter.write(windowUpdate.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
        }
        this.state = State.READ_FRAME;
    }

    private void readWindowUpdateFrame() {
        Http2WindowUpdate windowUpdate = Http2WindowUpdate.create((BufferData)this.inProgressFrame());
        int streamId = this.frameHeader.streamId();
        this.receiveFrameListener.frame((SocketContext)this.ctx, streamId, windowUpdate);
        this.state = State.READ_FRAME;
        int increment = windowUpdate.windowSizeIncrement();
        if (streamId == 0) {
            long size;
            if (increment == 0) {
                Http2GoAway frame = new Http2GoAway(0, Http2ErrorCode.PROTOCOL, "Window size 0");
                this.connectionWriter.write(frame.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
            }
            if ((size = this.flowControl.incrementOutboundConnectionWindowSize(increment)) > Integer.MAX_VALUE || size < 0L) {
                Http2GoAway frame = new Http2GoAway(0, Http2ErrorCode.FLOW_CONTROL, "Window size too big.");
                this.connectionWriter.write(frame.toFrameData(this.clientSettings, 0, Http2Flag.NoFlags.create()));
            }
        } else {
            try {
                StreamContext stream = this.stream(streamId);
                stream.stream().windowUpdate(windowUpdate);
            }
            catch (Http2Exception http2Exception) {
                // empty catch block
            }
        }
    }

    private void writeWindowUpdateFrame(int streamId, Http2WindowUpdate windowUpdateFrame) {
        this.connectionWriter.write(windowUpdateFrame.toFrameData(this.clientSettings, streamId, Http2Flag.NoFlags.create()));
    }

    private void doSettings() {
        if (this.frameHeader.streamId() != 0) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Settings must use stream ID 0, but use " + this.frameHeader.streamId());
        }
        Http2Flag.SettingsFlags flags = (Http2Flag.SettingsFlags)this.frameHeader.flags(Http2FrameTypes.SETTINGS);
        if (flags.ack()) {
            this.state = State.READ_FRAME;
            if (this.frameHeader.length() > 0) {
                throw new Http2Exception(Http2ErrorCode.FRAME_SIZE, "Settings with ACK should not have payload.");
            }
        } else {
            this.clientSettings(Http2Settings.create((BufferData)this.inProgressFrame()));
            this.state = State.ACK_SETTINGS;
        }
    }

    private void ackSettings() {
        Http2Flag.SettingsFlags flags = Http2Flag.SettingsFlags.create((int)1);
        Http2FrameHeader header = Http2FrameHeader.create((int)0, (Http2FrameTypes)Http2FrameTypes.SETTINGS, (Http2Flag)flags, (int)0);
        this.connectionWriter.write(new Http2FrameData(header, BufferData.empty()));
        this.state = State.READ_FRAME;
        if (this.upgradeHeaders != null) {
            Headers httpHeaders = this.upgradeHeaders.httpHeaders();
            boolean hasEntity = httpHeaders.contains(HeaderNames.CONTENT_LENGTH) || httpHeaders.contains(HeaderValues.TRANSFER_ENCODING_CHUNKED);
            Http2ServerStream stream = this.stream(1).stream();
            stream.prologue(this.upgradePrologue);
            stream.headers(this.upgradeHeaders, !hasEntity);
            this.upgradeHeaders = null;
            this.ctx.executor().submit(new StreamRunnable(this.streams, stream, Thread.currentThread()));
        }
    }

    private void dataFrame() {
        BufferData buffer;
        int streamId = this.frameHeader.streamId();
        StreamContext stream = this.stream(streamId);
        stream.stream().checkDataReceivable();
        boolean endOfStream = ((Http2Flag.DataFlags)this.frameHeader.flags(Http2FrameTypes.DATA)).endOfStream();
        int length = this.frameHeader.length();
        if (length > 0) {
            this.emptyFrames = 0;
            if (streamId > 0 && this.frameHeader.type() != Http2FrameType.HEADERS) {
                stream.stream().flowControl().inbound().decrementWindowSize(length);
            }
        } else if (this.emptyFrames++ > this.maxEmptyFrames && !endOfStream) {
            throw new Http2Exception(Http2ErrorCode.ENHANCE_YOUR_CALM, "Too much subsequent empty frames received.");
        }
        if (((Http2Flag.DataFlags)this.frameHeader.flags(Http2FrameTypes.DATA)).padded()) {
            BufferData frameData = this.inProgressFrame();
            int padLength = frameData.read();
            if (this.frameHeader.length() == padLength) {
                buffer = null;
            } else if (this.frameHeader.length() - padLength > 0) {
                buffer = BufferData.create((int)(this.frameHeader.length() - padLength));
                buffer.write(frameData);
            } else {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Invalid pad length");
            }
            frameData.skip(padLength);
        } else {
            buffer = this.inProgressFrame();
        }
        stream.stream().data(this.frameHeader, buffer, endOfStream);
        if (REMOVABLE_STREAMS.contains(stream.stream.streamState()) && endOfStream) {
            this.streams.remove(streamId);
        }
        this.state = State.READ_FRAME;
    }

    private void doHeaders(Limit limit) {
        boolean endOfStream;
        Http2Headers headers;
        int streamId = this.frameHeader.streamId();
        StreamContext streamContext = this.stream(streamId);
        boolean trailers = streamContext.stream().checkHeadersReceivable();
        if (this.frameHeader.type() == Http2FrameType.HEADERS && !((Http2Flag.HeaderFlags)this.frameHeader.flags(Http2FrameTypes.HEADERS)).endOfHeaders()) {
            streamContext.addHeadersToBeContinued(this.frameHeader, this.inProgressFrame().copy());
            this.continuationExpectedStreamId = streamId;
            this.state = State.READ_FRAME;
            return;
        }
        Http2ServerStream stream = streamContext.stream();
        if (this.initConnectionHeaders) {
            this.ctx.remotePeer().tlsCertificates().flatMap(TlsUtils::parseCn).ifPresent(cn -> this.connectionHeaders.add(HeaderNames.X_HELIDON_CN, new String[]{cn}));
            this.ctx.proxyProtocolData().ifPresent(proxyProtocolData -> {
                int sourcePort;
                String sourceAddress = proxyProtocolData.sourceAddress();
                if (!sourceAddress.isEmpty()) {
                    this.connectionHeaders.add(HeaderNames.X_FORWARDED_FOR, new String[]{sourceAddress});
                }
                if ((sourcePort = proxyProtocolData.sourcePort()) != -1) {
                    this.connectionHeaders.set(HeaderNames.X_FORWARDED_PORT, sourcePort);
                }
            });
            this.initConnectionHeaders = false;
        }
        if (this.frameHeader.type() == Http2FrameType.CONTINUATION) {
            headers = Http2Headers.create((Http2Stream)stream, (Http2Headers.DynamicTable)this.requestDynamicTable, (Http2HuffmanDecoder)this.requestHuffman, (Http2Headers)Http2Headers.create(this.connectionHeaders), (Http2FrameData[])streamContext.contData());
            endOfStream = ((Http2Flag.HeaderFlags)streamContext.contHeader().flags(Http2FrameTypes.HEADERS)).endOfStream();
            streamContext.clearContinuations();
            this.continuationExpectedStreamId = 0;
        } else {
            endOfStream = ((Http2Flag.HeaderFlags)this.frameHeader.flags(Http2FrameTypes.HEADERS)).endOfStream();
            headers = Http2Headers.create((Http2Stream)stream, (Http2Headers.DynamicTable)this.requestDynamicTable, (Http2HuffmanDecoder)this.requestHuffman, (Http2Headers)Http2Headers.create(this.connectionHeaders), (Http2FrameData[])new Http2FrameData[]{new Http2FrameData(this.frameHeader, this.inProgressFrame())});
        }
        this.receiveFrameListener.headers((SocketContext)this.ctx, streamId, headers);
        if (trailers) {
            if (!endOfStream) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received trailers without endOfStream flag " + streamId);
            }
            stream.closeFromRemote();
            this.state = State.READ_FRAME;
            return;
        }
        headers.validateRequest();
        String path = headers.path();
        Method method = headers.method();
        HttpPrologue httpPrologue = HttpPrologue.create((String)FULL_PROTOCOL, (String)PROTOCOL, (String)PROTOCOL_VERSION, (Method)method, (String)path, (boolean)this.http2Config.validatePath());
        stream.prologue(httpPrologue);
        stream.requestLimit(limit);
        stream.headers(headers, endOfStream);
        this.state = State.READ_FRAME;
        this.lastRequestTimestamp = DateTime.timestamp();
        this.ctx.executor().submit(new StreamRunnable(this.streams, stream, Thread.currentThread()));
    }

    private void pingFrame() {
        if (this.frameHeader.streamId() != 0) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received ping for a stream " + this.frameHeader.streamId());
        }
        if (this.frameHeader.length() != 8) {
            throw new Http2Exception(Http2ErrorCode.FRAME_SIZE, "Received ping with wrong size. Should be 8 bytes, is " + this.frameHeader.length());
        }
        if (((Http2Flag.PingFlags)this.frameHeader.flags(Http2FrameTypes.PING)).ack()) {
            this.state = State.READ_FRAME;
        } else {
            this.ping = Http2Ping.create((BufferData)this.inProgressFrame());
            this.receiveFrameListener.frame((SocketContext)this.ctx, 0, this.ping);
            this.state = State.SEND_PING_ACK;
        }
    }

    private void doPriority() {
        StreamContext stream;
        Http2Priority http2Priority = Http2Priority.create((BufferData)this.inProgressFrame());
        this.receiveFrameListener.frame((SocketContext)this.ctx, http2Priority.streamId(), http2Priority);
        if (this.frameHeader.streamId() == 0) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Priority with stream id 0");
        }
        if (http2Priority.streamId() != 0) {
            try {
                this.stream(http2Priority.streamId());
            }
            catch (Http2Exception http2Exception) {
                // empty catch block
            }
        }
        try {
            stream = this.stream(this.frameHeader.streamId());
        }
        catch (Http2Exception ignored) {
            this.state = State.READ_FRAME;
            return;
        }
        if (!stream.continuationData.isEmpty()) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received priority while processing headers");
        }
        stream.stream().priority(http2Priority);
        this.state = State.READ_FRAME;
    }

    private void writePingAck() {
        BufferData frame = this.ping.data();
        Http2FrameHeader header = Http2FrameHeader.create((int)frame.available(), (Http2FrameTypes)Http2FrameTypes.PING, (Http2Flag)Http2Flag.PingFlags.create((int)1), (int)0);
        this.ping = null;
        this.connectionWriter.write(new Http2FrameData(header, frame));
        this.state = State.READ_FRAME;
    }

    private void goAwayFrame() {
        Http2GoAway go = Http2GoAway.create((BufferData)this.inProgressFrame());
        this.receiveFrameListener.frame((SocketContext)this.ctx, 0, go);
        this.state = State.FINISHED;
        if (go.errorCode() != Http2ErrorCode.NO_ERROR) {
            this.ctx.log(LOGGER, System.Logger.Level.DEBUG, "Received go away. Error code: %s, message: %s", new Object[]{go.errorCode(), go.details()});
        }
    }

    private void rstStream() {
        Http2RstStream rstStream = Http2RstStream.create((BufferData)this.inProgressFrame());
        int streamId = this.frameHeader.streamId();
        this.receiveFrameListener.frame((SocketContext)this.ctx, streamId, rstStream);
        try {
            StreamContext streamContext = this.stream(streamId);
            boolean rapidReset = streamContext.stream().rstStream(rstStream);
            if (rapidReset && this.maxRapidResets != -1) {
                long currentTime = System.nanoTime();
                if (this.rapidResetCheckPeriod >= currentTime - this.rapidResetPeriodStart) {
                    this.rapidResetCnt = 1;
                    this.rapidResetPeriodStart = currentTime;
                } else {
                    if (this.maxRapidResets < this.rapidResetCnt) {
                        throw new Http2Exception(Http2ErrorCode.ENHANCE_YOUR_CALM, "Rapid reset attack detected!");
                    }
                    ++this.rapidResetCnt;
                }
            }
            this.state = State.READ_FRAME;
        }
        catch (Http2Exception e) {
            if (e.getMessage().startsWith("Stream closed")) {
                this.state = State.READ_FRAME;
                return;
            }
            throw e;
        }
        finally {
            this.streams.remove(streamId);
        }
    }

    private void unknownFrame() {
        this.state = State.READ_FRAME;
    }

    private StreamContext stream(int streamId) {
        if (streamId % 2 == 0) {
            throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Stream " + streamId + " is even, only odd numbers allowed");
        }
        boolean same = streamId == this.lastStreamId;
        this.lastStreamId = Math.max(streamId, this.lastStreamId);
        StreamContext streamContext = this.streams.get(streamId);
        if (streamContext == null) {
            if (same) {
                throw new Http2Exception(Http2ErrorCode.STREAM_CLOSED, "Stream closed");
            }
            if (streamId < this.lastStreamId) {
                for (StreamContext context : this.streams.contexts()) {
                    if (context.streamId <= streamId || context.stream().streamState() == Http2StreamState.IDLE) continue;
                    throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Stream " + streamId + " was never created and has lower ID than last: " + this.lastStreamId);
                }
            }
            if ((long)(this.streams.size() + 1) > this.maxClientConcurrentStreams) {
                throw new Http2Exception(Http2ErrorCode.REFUSED_STREAM, "Maximum concurrent streams limit " + this.maxClientConcurrentStreams + " exceeded");
            }
            streamContext = new StreamContext(streamId, this.http2Config.maxHeaderListSize(), new Http2ServerStream(this.ctx, this.streams, this.routing, this.http2Config, this.subProviders, streamId, this.serverSettings, this.clientSettings, (Http2StreamWriter)this.connectionWriter, this.flowControl));
            this.streams.put(streamContext);
            this.streams.doMaintenance(this.maxClientConcurrentStreams);
        }
        this.lastRequestTimestamp = DateTime.timestamp();
        return streamContext;
    }

    private BufferData inProgressFrame() {
        BufferData inProgress = this.frameInProgress;
        this.frameInProgress = null;
        if (inProgress == null) {
            throw new Http2Exception(Http2ErrorCode.INTERNAL, "In progress frame is null for state: " + String.valueOf((Object)this.state));
        }
        return inProgress;
    }

    private void updateHeaderTableSize(long value) {
        try {
            this.connectionWriter.updateHeaderTableSize(value);
        }
        catch (InterruptedException e) {
            throw new CloseConnectionException("Failed to update header table size, interrupted", (Throwable)e);
        }
    }

    private static enum State {
        READ_FRAME,
        DATA,
        HEADERS,
        PRIORITY,
        RST_STREAM,
        SETTINGS,
        READ_PUSH_PROMISE,
        PING,
        GO_AWAY,
        WINDOW_UPDATE,
        ACK_SETTINGS,
        WRITE_SERVER_SETTINGS,
        FINISHED,
        SEND_PING_ACK,
        CONTINUATION,
        UNKNOWN;

    }

    static class StreamContext {
        private final List<Http2FrameData> continuationData = new ArrayList<Http2FrameData>();
        private final long maxHeaderListSize;
        private final int streamId;
        private final Http2ServerStream stream;
        private long headerListSize = 0L;
        private Http2FrameHeader continuationHeader;

        StreamContext(int streamId, long maxHeaderListSize, Http2ServerStream stream) {
            this.streamId = streamId;
            this.maxHeaderListSize = maxHeaderListSize;
            this.stream = stream;
        }

        public Http2ServerStream stream() {
            return this.stream;
        }

        Http2FrameData[] contData() {
            return this.continuationData.toArray(new Http2FrameData[0]);
        }

        Http2FrameHeader contHeader() {
            return this.continuationHeader;
        }

        void addContinuation(Http2FrameData frameData) {
            if (this.continuationData.isEmpty()) {
                throw new Http2Exception(Http2ErrorCode.PROTOCOL, "Received continuation without headers.");
            }
            this.continuationData.add(frameData);
            this.addAndValidateHeaderListSize(frameData.header().length());
        }

        void addHeadersToBeContinued(Http2FrameHeader frameHeader, BufferData bufferData) {
            this.clearContinuations();
            this.continuationHeader = frameHeader;
            this.continuationData.add(new Http2FrameData(frameHeader, bufferData));
            this.addAndValidateHeaderListSize(frameHeader.length());
        }

        private void addAndValidateHeaderListSize(int headerSizeIncrement) {
            this.headerListSize += (long)headerSizeIncrement;
            if (this.headerListSize > this.maxHeaderListSize) {
                throw new Http2Exception(Http2ErrorCode.REQUEST_HEADER_FIELDS_TOO_LARGE, "Request Header Fields Too Large");
            }
        }

        private void clearContinuations() {
            this.continuationData.clear();
            this.headerListSize = 0L;
        }
    }

    private record StreamRunnable(Http2ConnectionStreams streams, Http2ServerStream stream, Thread handlerThread) implements Runnable
    {
        @Override
        public void run() {
            block7: {
                try {
                    this.stream.run();
                }
                catch (UncheckedIOException e) {
                    if (e.getCause() instanceof SocketException) {
                        this.handlerThread.interrupt();
                        LOGGER.log(System.Logger.Level.DEBUG, "Socket error on writer thread", (Throwable)e);
                        break block7;
                    }
                    throw e;
                }
                finally {
                    if (this.stream.streamState() == Http2StreamState.CLOSED) {
                        this.streams.remove(this.stream.streamId());
                    }
                }
            }
        }
    }
}

