/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.besu.crypto;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.io.Resources;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import org.apache.tuweni.bytes.Bytes32;
import org.hyperledger.besu.crypto.KeyPair;
import org.hyperledger.besu.crypto.SECPPrivateKey;
import org.hyperledger.besu.crypto.SignatureAlgorithm;
import org.hyperledger.besu.crypto.SignatureAlgorithmFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyPairUtil {
    private static final Logger LOG = LoggerFactory.getLogger(KeyPairUtil.class);
    private static final Supplier<SignatureAlgorithm> SIGNATURE_ALGORITHM = Suppliers.memoize(SignatureAlgorithmFactory::getInstance);

    public static String loadResourceFile(String resourcePath) {
        try {
            URL path = KeyPairUtil.class.getClassLoader().getResource(resourcePath);
            return Resources.toString((URL)path, (Charset)StandardCharsets.UTF_8).trim();
        }
        catch (Exception e) {
            throw new RuntimeException("Unable to load resource: " + resourcePath, e);
        }
    }

    public static KeyPair loadKeyPairFromResource(String resourcePath) {
        String keyData = KeyPairUtil.loadResourceFile(resourcePath);
        if (keyData == null || keyData.isEmpty()) {
            throw new IllegalArgumentException("Unable to load resource: " + resourcePath);
        }
        SECPPrivateKey privateKey = ((SignatureAlgorithm)SIGNATURE_ALGORITHM.get()).createPrivateKey(Bytes32.fromHexString((CharSequence)keyData));
        KeyPair keyPair = ((SignatureAlgorithm)SIGNATURE_ALGORITHM.get()).createKeyPair(privateKey);
        LOG.info("Loaded keyPair {} from {}", (Object)keyPair.getPublicKey().toString(), (Object)resourcePath);
        return keyPair;
    }

    public static KeyPair loadKeyPair(File keyFile) {
        KeyPair key;
        if (keyFile.exists()) {
            LOG.info("Attempting to load public key from {}", (Object)keyFile.getAbsolutePath());
            key = KeyPairUtil.load(keyFile);
            LOG.info("Loaded public key {} from {}", (Object)key.getPublicKey().toString(), (Object)keyFile.getAbsolutePath());
        } else {
            SignatureAlgorithm signatureAlgorithm = (SignatureAlgorithm)SIGNATURE_ALGORITHM.get();
            key = signatureAlgorithm.generateKeyPair();
            KeyPairUtil.storeKeyFile(key, keyFile.getParentFile().toPath());
            LOG.info("Generated new {} public key {} and stored it to {}", new Object[]{signatureAlgorithm.getCurveName(), key.getPublicKey().toString(), keyFile.getAbsolutePath()});
        }
        return key;
    }

    public static KeyPair loadKeyPair(Path directory) {
        return KeyPairUtil.loadKeyPair(KeyPairUtil.getDefaultKeyFile(directory));
    }

    public static void storeKeyFile(KeyPair keyPair, Path homeDirectory) {
        try {
            KeyPairUtil.storeKeyPair(keyPair, KeyPairUtil.getDefaultKeyFile(homeDirectory));
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Cannot store generated private key.");
        }
    }

    public static File getDefaultKeyFile(Path directory) {
        return directory.resolve("key").toFile();
    }

    public static KeyPair load(File file) {
        return ((SignatureAlgorithm)SIGNATURE_ALGORITHM.get()).createKeyPair(KeyPairUtil.loadPrivateKey(file));
    }

    static SECPPrivateKey loadPrivateKey(File file) {
        try {
            List<String> info = Files.readAllLines(file.toPath());
            if (info.size() != 1) {
                throw new IllegalArgumentException("Supplied file does not contain valid keyPair pair.");
            }
            return ((SignatureAlgorithm)SIGNATURE_ALGORITHM.get()).createPrivateKey(Bytes32.fromHexString((CharSequence)info.get(0)));
        }
        catch (IOException ex) {
            throw new IllegalArgumentException("Supplied file does not contain valid keyPair pair.");
        }
    }

    static void storeKeyPair(KeyPair keyPair, File file) throws IOException {
        File privateKeyDir = file.getParentFile();
        privateKeyDir.mkdirs();
        Path tempPath = Files.createTempFile(privateKeyDir.toPath(), ".tmp", "", new FileAttribute[0]);
        Files.write(tempPath, keyPair.getPrivateKey().getEncodedBytes().toString().getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        Files.move(tempPath, file.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
    }
}

