/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeClientFactory;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnection;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeStats;
import com.hedera.node.app.blocks.impl.streaming.config.BlockNodeConfiguration;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.node.config.data.BlockStreamConfig;
import com.hedera.node.internal.network.BlockNodeConfig;
import com.hedera.node.internal.network.BlockNodeConnectionInfo;
import com.hedera.pbj.runtime.ParseException;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import com.swirlds.common.io.utility.FileUtils;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URL;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Singleton
public class BlockNodeConnectionManager {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnectionManager.class);
    public static final Duration INITIAL_RETRY_DELAY = Duration.ofSeconds(1L);
    private static final long RETRY_BACKOFF_MULTIPLIER = 2L;
    private final BlockBufferService blockBufferService;
    private ScheduledExecutorService sharedExecutorService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final ConfigProvider configProvider;
    private final List<BlockNodeConfiguration> availableBlockNodes = new ArrayList<BlockNodeConfiguration>();
    private final AtomicBoolean isConnectionManagerActive = new AtomicBoolean(false);
    private final AtomicReference<WatchService> configWatchServiceRef = new AtomicReference();
    private final AtomicReference<Thread> configWatcherThreadRef = new AtomicReference();
    private Path blockNodeConfigDirectory;
    private static final String BLOCK_NODES_FILE_NAME = "block-nodes.json";
    private final Map<BlockNodeConfiguration, BlockNodeConnection> connections = new ConcurrentHashMap<BlockNodeConfiguration, BlockNodeConnection>();
    private final AtomicReference<BlockNodeConnection> activeConnectionRef = new AtomicReference();
    private final Map<BlockNodeConfiguration, BlockNodeStats> nodeStats;
    private final Map<BlockNodeConfiguration, RetryState> retryStates = new ConcurrentHashMap<BlockNodeConfiguration, RetryState>();
    private final BlockNodeClientFactory clientFactory;

    @Inject
    public BlockNodeConnectionManager(@NonNull ConfigProvider configProvider, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamMetrics blockStreamMetrics) {
        this.configProvider = Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.nodeStats = new ConcurrentHashMap<BlockNodeConfiguration, BlockNodeStats>();
        this.blockNodeConfigDirectory = FileUtils.getAbsolutePath((String)this.blockNodeConnectionFileDir());
        this.clientFactory = new BlockNodeClientFactory();
    }

    private boolean isStreamingEnabled() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).streamToBlockNodes();
    }

    private String blockNodeConnectionFileDir() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).blockNodeConnectionFileDir();
    }

    private Duration expBackoffTimeframeReset() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).protocolExpBackoffTimeframeReset();
    }

    private Duration maxBackoffDelay() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).maxBackoffDelay();
    }

    private List<BlockNodeConfiguration> extractBlockNodesConfigurations(@NonNull String blockNodeConfigPath) {
        BlockNodeConnectionInfo connectionInfo;
        Path configPath = Paths.get(blockNodeConfigPath, BLOCK_NODES_FILE_NAME);
        ArrayList<BlockNodeConfiguration> nodes = new ArrayList<BlockNodeConfiguration>();
        try {
            if (!Files.exists(configPath, new LinkOption[0])) {
                logger.info("Block node configuration file does not exist: {}", (Object)configPath);
                return nodes;
            }
            byte[] jsonConfig = Files.readAllBytes(configPath);
            connectionInfo = (BlockNodeConnectionInfo)BlockNodeConnectionInfo.JSON.parse(Bytes.wrap((byte[])jsonConfig));
        }
        catch (ParseException | IOException e) {
            logger.warn("Failed to read or parse block node configuration from {}. Continuing without block node connections.", (Object)configPath, (Object)e);
            return nodes;
        }
        for (BlockNodeConfig nodeConfig : connectionInfo.nodes()) {
            try {
                BlockNodeConfiguration cfg = BlockNodeConfiguration.from(nodeConfig);
                nodes.add(cfg);
            }
            catch (RuntimeException e) {
                logger.warn("Failed to parse block node configuration; skipping block node (config={})", (Object)nodeConfig, (Object)e);
            }
        }
        return nodes;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isOnlyOneBlockNodeConfigured() {
        int size;
        List<BlockNodeConfiguration> list = this.availableBlockNodes;
        synchronized (list) {
            size = this.availableBlockNodes.size();
        }
        return size == 1;
    }

    public void rescheduleConnection(@NonNull BlockNodeConnection connection, @Nullable Duration delay, @Nullable Long blockNumber, boolean selectNewBlockNode) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(connection, "connection must not be null");
        logger.debug("{} Closing and rescheduling connection for reconnect attempt.", (Object)connection);
        this.handleConnectionCleanupAndReschedule(connection, delay, blockNumber, selectNewBlockNode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleConnectionCleanupAndReschedule(@NonNull BlockNodeConnection connection, @Nullable Duration delay, @Nullable Long blockNumber, boolean selectNewBlockNode) {
        long delayMs;
        int retryAttempt;
        RetryState retryState;
        RetryState retryState2 = retryState = this.retryStates.computeIfAbsent(connection.getNodeConfig(), k -> new RetryState());
        synchronized (retryState2) {
            retryState.updateRetryTime();
            retryAttempt = retryState.getRetryAttempt();
            delayMs = delay == null ? this.calculateJitteredDelayMs(retryAttempt) : delay.toMillis();
            retryState.increment();
        }
        logger.info("{} Apply exponential backoff and reschedule in {} ms (attempt={}).", (Object)connection, (Object)delayMs, (Object)retryAttempt);
        this.scheduleConnectionAttempt(connection.getNodeConfig(), Duration.ofMillis(delayMs), blockNumber, false);
        if (!this.isOnlyOneBlockNodeConfigured() && selectNewBlockNode) {
            this.selectNewBlockNodeForStreaming(false);
        }
    }

    private void scheduleConnectionAttempt(@NonNull BlockNodeConfiguration blockNodeConfig, @NonNull Duration initialDelay, @Nullable Long initialBlockToStream, boolean force) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(blockNodeConfig);
        Objects.requireNonNull(initialDelay);
        long delayMillis = Math.max(0L, initialDelay.toMillis());
        BlockNodeConnection newConnection = this.createConnection(blockNodeConfig, initialBlockToStream);
        logger.debug("{} Scheduling reconnection for node in {} ms (force={}).", (Object)newConnection, (Object)delayMillis, (Object)force);
        try {
            this.sharedExecutorService.schedule(new BlockNodeConnectionTask(newConnection, initialDelay, force), delayMillis, TimeUnit.MILLISECONDS);
            logger.debug("{} Successfully scheduled reconnection task.", (Object)newConnection);
        }
        catch (Exception e) {
            logger.warn("{} Failed to schedule connection task for block node.", (Object)newConnection, (Object)e);
            newConnection.closeAtBlockBoundary();
        }
    }

    public void shutdown() {
        if (!this.isConnectionManagerActive.compareAndSet(true, false)) {
            logger.info("Connection Manager already shutdown.");
            return;
        }
        logger.info("Shutting down block node connection manager.");
        this.stopConfigWatcher();
        this.blockBufferService.shutdown();
        this.shutdownScheduledExecutorService();
        this.closeAllConnections();
        this.clearManagerMetadata();
    }

    private void shutdownScheduledExecutorService() {
        if (this.sharedExecutorService != null) {
            this.sharedExecutorService.shutdownNow();
        }
    }

    private void clearManagerMetadata() {
        this.activeConnectionRef.set(null);
        this.nodeStats.clear();
        this.availableBlockNodes.clear();
    }

    private void closeAllConnections() {
        logger.info("Stopping block node connections");
        Iterator<Map.Entry<BlockNodeConfiguration, BlockNodeConnection>> iterator = this.connections.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<BlockNodeConfiguration, BlockNodeConnection> entry = iterator.next();
            BlockNodeConnection connection = entry.getValue();
            try {
                connection.close(true);
            }
            catch (RuntimeException e) {
                logger.debug("{} Error while closing connection during connection manager shutdown. Ignoring.", (Object)connection, (Object)e);
            }
            iterator.remove();
        }
    }

    public void start() {
        if (!this.isStreamingEnabled()) {
            logger.warn("Cannot start the connection manager, streaming is not enabled.");
            return;
        }
        if (!this.isConnectionManagerActive.compareAndSet(false, true)) {
            logger.info("Connection Manager already started.");
            return;
        }
        logger.info("Starting connection manager.");
        this.blockBufferService.start();
        this.startConfigWatcher();
        this.refreshAvailableBlockNodes();
    }

    private void createScheduledExecutorService() {
        logger.debug("Creating scheduled executor service for the Block Node connection manager.");
        this.sharedExecutorService = Executors.newSingleThreadScheduledExecutor();
    }

    public boolean selectNewBlockNodeForStreaming(boolean force) {
        if (!this.isStreamingEnabled()) {
            logger.debug("Cannot select block node, streaming is not enabled.");
            return false;
        }
        logger.debug("Selecting highest priority available block node for connection attempt.");
        BlockNodeConfiguration selectedNode = this.getNextPriorityBlockNode();
        if (selectedNode == null) {
            logger.info("No available block nodes found for streaming.");
            return false;
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Selected block node {}:{} for connection attempt", (Object)selectedNode.address(), (Object)selectedNode.port());
        }
        this.scheduleConnectionAttempt(selectedNode, Duration.ZERO, null, force);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    private BlockNodeConfiguration getNextPriorityBlockNode() {
        ArrayList<BlockNodeConfiguration> snapshot;
        logger.debug("Searching for new block node connection based on node priorities.");
        List<BlockNodeConfiguration> list = this.availableBlockNodes;
        synchronized (list) {
            snapshot = new ArrayList<BlockNodeConfiguration>(this.availableBlockNodes);
        }
        SortedMap priorityGroups = snapshot.stream().collect(Collectors.groupingBy(BlockNodeConfiguration::priority, TreeMap::new, Collectors.toList()));
        BlockNodeConfiguration selectedNode = null;
        for (Map.Entry entry : priorityGroups.entrySet()) {
            int priority = (Integer)entry.getKey();
            List nodesInGroup = (List)entry.getValue();
            selectedNode = this.findAvailableNode(nodesInGroup);
            if (selectedNode == null) {
                logger.debug("No available node found in priority group {}.", (Object)priority);
                continue;
            }
            logger.debug("Found available node in priority group {}.", (Object)priority);
            return selectedNode;
        }
        return selectedNode;
    }

    @Nullable
    private BlockNodeConfiguration findAvailableNode(@NonNull List<BlockNodeConfiguration> nodes) {
        Objects.requireNonNull(nodes, "nodes must not be null");
        return nodes.stream().filter(nodeConfig -> !this.connections.containsKey(nodeConfig)).collect(Collectors.collectingAndThen(Collectors.toList(), collected -> {
            Collections.shuffle(collected);
            return collected.stream();
        })).findFirst().orElse(null);
    }

    @NonNull
    private BlockNodeConnection createConnection(@NonNull BlockNodeConfiguration nodeConfig, @Nullable Long initialBlockToStream) {
        Objects.requireNonNull(nodeConfig);
        BlockNodeConnection connection = new BlockNodeConnection(this.configProvider, nodeConfig, this, this.blockBufferService, this.blockStreamMetrics, this.sharedExecutorService, Executors.newVirtualThreadPerTaskExecutor(), initialBlockToStream, this.clientFactory);
        this.connections.put(nodeConfig, connection);
        return connection;
    }

    private void startConfigWatcher() {
        if (this.configWatchServiceRef.get() != null) {
            logger.debug("Configuration watcher already running.");
            return;
        }
        try {
            WatchService watchService = this.blockNodeConfigDirectory.getFileSystem().newWatchService();
            this.configWatchServiceRef.set(watchService);
            this.blockNodeConfigDirectory.register(watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
            Thread watcherThread = Thread.ofPlatform().name("BlockNodesConfigWatcher").start(() -> {
                while (!Thread.currentThread().isInterrupted()) {
                    WatchKey key = null;
                    try {
                        key = watchService.take();
                        for (WatchEvent<?> event : key.pollEvents()) {
                            Path changed;
                            WatchEvent.Kind<?> kind = event.kind();
                            Object ctx = event.context();
                            if (!(ctx instanceof Path) || !BLOCK_NODES_FILE_NAME.equals((changed = (Path)ctx).toString())) continue;
                            if (logger.isInfoEnabled()) {
                                logger.info("Detected {} event for {}.", (Object)kind.name(), (Object)changed);
                            }
                            try {
                                this.refreshAvailableBlockNodes();
                            }
                            catch (Exception e) {
                                logger.info("Exception in BlockNodesConfigWatcher config file change handler.", (Throwable)e);
                            }
                        }
                    }
                    catch (InterruptedException | ClosedWatchServiceException e) {
                        if (key == null || key.reset()) break;
                        logger.info("WatchKey could not be reset. Exiting config watcher loop.");
                        break;
                    }
                    catch (Exception e) {
                        logger.info("Exception in config watcher loop.", (Throwable)e);
                        if (Thread.currentThread().isInterrupted()) {
                            logger.debug("Config watcher thread interrupted, exiting.");
                            return;
                        }
                    }
                    finally {
                        if (key == null || key.reset()) continue;
                        logger.info("WatchKey could not be reset. Exiting config watcher loop.");
                        break;
                    }
                }
            });
            this.configWatcherThreadRef.set(watcherThread);
            logger.info("Started block-nodes.json configuration watcher thread.");
        }
        catch (IOException e) {
            logger.info("Failed to start block-nodes.json configuration watcher ({}). Dynamic updates disabled.", (Object)e.getMessage());
        }
    }

    private void stopConfigWatcher() {
        WatchService ws;
        Thread watcherThread = this.configWatcherThreadRef.getAndSet(null);
        if (watcherThread != null) {
            watcherThread.interrupt();
            try {
                watcherThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if ((ws = (WatchService)this.configWatchServiceRef.getAndSet(null)) != null) {
            try {
                ws.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshAvailableBlockNodes() {
        String configDir = this.blockNodeConfigDirectory.toString();
        List<BlockNodeConfiguration> newConfigs = this.extractBlockNodesConfigurations(configDir);
        List<BlockNodeConfiguration> list = this.availableBlockNodes;
        synchronized (list) {
            if (newConfigs.equals(this.availableBlockNodes)) {
                logger.info("Block node configuration unchanged. No action taken.");
                return;
            }
        }
        this.shutdownScheduledExecutorService();
        this.closeAllConnections();
        this.clearManagerMetadata();
        list = this.availableBlockNodes;
        synchronized (list) {
            this.availableBlockNodes.addAll(newConfigs);
        }
        if (!newConfigs.isEmpty()) {
            logger.info("Reloaded block node configurations ({})", newConfigs);
            this.createScheduledExecutorService();
            this.selectNewBlockNodeForStreaming(false);
        } else {
            logger.info("No valid block node configurations available after file change. Connections remain stopped.");
        }
    }

    private long calculateJitteredDelayMs(int retryAttempt) {
        Duration maxBackoff;
        Duration nextDelay = INITIAL_RETRY_DELAY.multipliedBy((long)Math.pow(2.0, retryAttempt));
        if (nextDelay.compareTo(maxBackoff = this.maxBackoffDelay()) > 0) {
            nextDelay = maxBackoff;
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        return nextDelay.toMillis() / 2L + random.nextLong(nextDelay.toMillis() / 2L + 1L);
    }

    public boolean recordEndOfStreamAndCheckLimit(@NonNull BlockNodeConfiguration blockNodeConfig, @NonNull Instant timestamp) {
        if (!this.isStreamingEnabled()) {
            return false;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        return stats.addEndOfStreamAndCheckLimit(timestamp, this.getMaxEndOfStreamsAllowed(), this.getEndOfStreamTimeframe());
    }

    public Duration getEndOfStreamScheduleDelay() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).endOfStreamScheduleDelay();
    }

    public Duration getEndOfStreamTimeframe() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).endOfStreamTimeFrame();
    }

    private Duration getForcedSwitchRescheduleDelay() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).forcedSwitchRescheduleDelay();
    }

    public int getMaxEndOfStreamsAllowed() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).maxEndOfStreamsAllowed();
    }

    public int getEndOfStreamCount(@NonNull BlockNodeConfiguration blockNodeConfig) {
        if (!this.isStreamingEnabled()) {
            return 0;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.get(blockNodeConfig);
        return stats != null ? stats.getEndOfStreamCount() : 0;
    }

    private Duration getHighLatencyThreshold() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).highLatencyThreshold();
    }

    private int getHighLatencyEventsBeforeSwitching() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).highLatencyEventsBeforeSwitching();
    }

    private static long calculateIpAsInteger(@NonNull InetAddress address) {
        Objects.requireNonNull(address);
        byte[] bytes = address.getAddress();
        if (bytes.length != 4) {
            throw new IllegalArgumentException("Only IPv4 addresses are supported");
        }
        long octet1 = 0x1000000L * (long)(bytes[0] & 0xFF);
        long octet2 = 65536L * (long)(bytes[1] & 0xFF);
        long octet3 = 256L * (long)(bytes[2] & 0xFF);
        long octet4 = 1L * (long)(bytes[3] & 0xFF);
        return octet1 + octet2 + octet3 + octet4;
    }

    private void recordActiveConnectionIp(BlockNodeConfiguration nodeConfig) {
        long ipAsInteger;
        try {
            URL blockNodeUrl = URI.create("http://" + nodeConfig.address() + ":" + nodeConfig.port()).toURL();
            InetAddress blockAddress = InetAddress.getByName(blockNodeUrl.getHost());
            ipAsInteger = BlockNodeConnectionManager.calculateIpAsInteger(blockAddress);
            if (logger.isInfoEnabled()) {
                logger.info("Active block node connection updated to: {}:{} (resolvedIp: {}, resolvedIpAsInt={})", (Object)nodeConfig.address(), (Object)nodeConfig.port(), (Object)blockAddress.getHostAddress(), (Object)ipAsInteger);
            }
        }
        catch (IOException e) {
            logger.debug("Failed to resolve block node host ({}:{})", (Object)nodeConfig.address(), (Object)nodeConfig.port(), (Object)e);
            ipAsInteger = -1L;
        }
        this.blockStreamMetrics.recordActiveConnectionIp(ipAsInteger);
    }

    public void recordBlockProofSent(@NonNull BlockNodeConfiguration blockNodeConfig, long blockNumber, @NonNull Instant timestamp) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        stats.recordBlockProofSent(blockNumber, timestamp);
    }

    public BlockNodeStats.HighLatencyResult recordBlockAckAndCheckLatency(@NonNull BlockNodeConfiguration blockNodeConfig, long blockNumber, @NonNull Instant timestamp) {
        if (!this.isStreamingEnabled()) {
            return new BlockNodeStats.HighLatencyResult(0L, 0, false, false);
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        BlockNodeStats.HighLatencyResult result = stats.recordAcknowledgementAndEvaluate(blockNumber, timestamp, this.getHighLatencyThreshold(), this.getHighLatencyEventsBeforeSwitching());
        long latencyMs = result.latencyMs();
        if (result.isHighLatency()) {
            if (logger.isDebugEnabled()) {
                logger.debug("[{}] A high latency event ({}ms) has occurred. A total of {} consecutive events", (Object)blockNodeConfig, (Object)latencyMs, (Object)result.consecutiveHighLatencyEvents());
            }
            this.blockStreamMetrics.recordHighLatencyEvent();
        }
        return result;
    }

    public void notifyConnectionClosed(@NonNull BlockNodeConnection connection) {
        this.activeConnectionRef.compareAndSet(connection, null);
        this.connections.remove(connection.getNodeConfig());
    }

    class RetryState {
        private int retryAttempt = 0;
        private Instant lastRetryTime;

        RetryState() {
        }

        public int getRetryAttempt() {
            return this.retryAttempt;
        }

        public void increment() {
            ++this.retryAttempt;
        }

        public void updateRetryTime() {
            Duration timeSinceLastRetry;
            Instant now = Instant.now();
            if (this.lastRetryTime != null && (timeSinceLastRetry = Duration.between(this.lastRetryTime, now)).compareTo(BlockNodeConnectionManager.this.expBackoffTimeframeReset()) > 0) {
                this.retryAttempt = 0;
                this.lastRetryTime = now;
                return;
            }
            this.lastRetryTime = now;
        }
    }

    class BlockNodeConnectionTask
    implements Runnable {
        private final BlockNodeConnection connection;
        private Duration currentBackoffDelayMs;
        private final boolean force;

        BlockNodeConnectionTask(@NonNull BlockNodeConnection connection, Duration initialDelay, boolean force) {
            this.connection = Objects.requireNonNull(connection);
            this.currentBackoffDelayMs = initialDelay.isNegative() ? Duration.ZERO : initialDelay;
            this.force = force;
        }

        @Override
        public void run() {
            block16: {
                if (!BlockNodeConnectionManager.this.isStreamingEnabled()) {
                    logger.debug("{} Cannot run connection task, streaming is not enabled.", (Object)this.connection);
                    return;
                }
                if (!BlockNodeConnectionManager.this.isConnectionManagerActive.get()) {
                    logger.debug("{} Cannot run connection task, connection manager has shutdown.", (Object)this.connection);
                    return;
                }
                try {
                    logger.debug("{} Running connection task.", (Object)this.connection);
                    BlockNodeConnection activeConnection = BlockNodeConnectionManager.this.activeConnectionRef.get();
                    if (activeConnection != null) {
                        if (activeConnection.equals(this.connection)) {
                            logger.debug("{} The current connection is the active connection, ignoring task.", (Object)this.connection);
                            return;
                        }
                        if (this.force) {
                            BlockNodeConfiguration newConnConfig = this.connection.getNodeConfig();
                            BlockNodeConfiguration oldConnConfig = activeConnection.getNodeConfig();
                            if (logger.isDebugEnabled()) {
                                logger.debug("{} Promoting forced connection with priority={} over active ({}:{} priority={}).", (Object)this.connection, (Object)newConnConfig.priority(), (Object)oldConnConfig.address(), (Object)oldConnConfig.port(), (Object)oldConnConfig.priority());
                            }
                        } else if (activeConnection.getNodeConfig().priority() <= this.connection.getNodeConfig().priority()) {
                            logger.info("{} Active connection has equal/higher priority. Ignoring candidate. Active: {}.", (Object)this.connection, (Object)activeConnection);
                            this.connection.close(false);
                            return;
                        }
                    }
                    this.connection.createRequestPipeline();
                    if (BlockNodeConnectionManager.this.activeConnectionRef.compareAndSet(activeConnection, this.connection)) {
                        this.connection.updateConnectionState(BlockNodeConnection.ConnectionState.ACTIVE);
                        BlockNodeConnectionManager.this.recordActiveConnectionIp(this.connection.getNodeConfig());
                    } else {
                        logger.info("{} Current connection task was preempted, rescheduling.", (Object)this.connection);
                        this.reschedule();
                    }
                    if (activeConnection == null) break block16;
                    try {
                        logger.info("{} Closing current active connection {}.", (Object)this.connection, (Object)activeConnection);
                        activeConnection.closeAtBlockBoundary();
                        if (!this.force) break block16;
                        try {
                            Duration delay = BlockNodeConnectionManager.this.getForcedSwitchRescheduleDelay();
                            BlockNodeConnectionManager.this.scheduleConnectionAttempt(activeConnection.getNodeConfig(), delay, null, false);
                            logger.info("Scheduled previously active connection {} in {} ms due to forced switch.", (Object)activeConnection, (Object)delay.toMillis());
                        }
                        catch (Exception e) {
                            logger.warn("Failed to schedule reschedule for previous active connection after forced switch.", (Throwable)e);
                            BlockNodeConnectionManager.this.connections.remove(activeConnection.getNodeConfig());
                        }
                    }
                    catch (RuntimeException e) {
                        logger.info("Failed to shutdown current active connection {} (shutdown reason: another connection was elevated to active).", (Object)activeConnection, (Object)e);
                    }
                }
                catch (Exception e) {
                    logger.warn("{} Failed to establish connection to block node. Will schedule a retry.", (Object)this.connection, (Object)e);
                    BlockNodeConnectionManager.this.blockStreamMetrics.recordConnectionCreateFailure();
                    this.reschedule();
                    BlockNodeConnectionManager.this.selectNewBlockNodeForStreaming(false);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void reschedule() {
            long jitteredDelayMs;
            Duration maxBackoff;
            Duration nextDelay = this.currentBackoffDelayMs.isZero() ? INITIAL_RETRY_DELAY : this.currentBackoffDelayMs.multipliedBy(2L);
            if (nextDelay.compareTo(maxBackoff = BlockNodeConnectionManager.this.maxBackoffDelay()) > 0) {
                nextDelay = maxBackoff;
            }
            ThreadLocalRandom random = ThreadLocalRandom.current();
            if (nextDelay.toMillis() > 0L) {
                jitteredDelayMs = nextDelay.toMillis() / 2L + random.nextLong(nextDelay.toMillis() / 2L + 1L);
            } else {
                jitteredDelayMs = INITIAL_RETRY_DELAY.toMillis() / 2L + random.nextLong(INITIAL_RETRY_DELAY.toMillis() / 2L + 1L);
                jitteredDelayMs = Math.max(1L, jitteredDelayMs);
            }
            this.currentBackoffDelayMs = Duration.ofMillis(jitteredDelayMs);
            try {
                List<BlockNodeConfiguration> list = BlockNodeConnectionManager.this.availableBlockNodes;
                synchronized (list) {
                    if (!BlockNodeConnectionManager.this.availableBlockNodes.contains(this.connection.getNodeConfig())) {
                        logger.debug("{} Node no longer available, skipping reschedule.", (Object)this.connection);
                        BlockNodeConnectionManager.this.connections.remove(this.connection.getNodeConfig());
                        return;
                    }
                }
                BlockNodeConnectionManager.this.sharedExecutorService.schedule(this, jitteredDelayMs, TimeUnit.MILLISECONDS);
                logger.info("{} Rescheduled connection attempt (delayMillis={}).", (Object)this.connection, (Object)jitteredDelayMs);
            }
            catch (Exception e) {
                logger.warn("{} Failed to reschedule connection attempt. Removing from retry map.", (Object)this.connection, (Object)e);
                this.connection.closeAtBlockBoundary();
            }
        }
    }
}

