/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnection;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeStats;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.node.app.blocks.impl.streaming.NoBlockNodesAvailableException;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.node.config.data.BlockStreamConfig;
import com.hedera.node.internal.network.BlockNodeConfig;
import com.hedera.node.internal.network.BlockNodeConnectionInfo;
import com.hedera.pbj.grpc.client.helidon.PbjGrpcClient;
import com.hedera.pbj.grpc.client.helidon.PbjGrpcClientConfig;
import com.hedera.pbj.runtime.ParseException;
import com.hedera.pbj.runtime.grpc.GrpcClient;
import com.hedera.pbj.runtime.grpc.ServiceInterface;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import io.helidon.common.tls.Tls;
import io.helidon.common.tls.TlsConfig;
import io.helidon.webclient.api.WebClient;
import io.helidon.webclient.api.WebClientConfig;
import io.helidon.webclient.grpc.GrpcClientProtocolConfig;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.BlockStreamPublishServiceInterface;
import org.hiero.block.api.PublishStreamRequest;

@Singleton
public class BlockNodeConnectionManager {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnectionManager.class);
    private static final Options OPTIONS = new Options(Optional.empty(), "application/grpc");
    public static final Duration INITIAL_RETRY_DELAY = Duration.ofSeconds(1L);
    private static final long RETRY_BACKOFF_MULTIPLIER = 2L;
    private static final Duration MAX_RETRY_DELAY = Duration.ofSeconds(10L);
    private final Map<BlockNodeConfig, Long> lastVerifiedBlockPerConnection;
    private final BlockBufferService blockBufferService;
    private final ScheduledExecutorService sharedExecutorService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final ConfigProvider configProvider;
    private final List<BlockNodeConfig> availableBlockNodes;
    private final AtomicBoolean isConnectionManagerActive = new AtomicBoolean(false);
    private final AtomicLong jumpTargetBlock = new AtomicLong(-1L);
    private final AtomicLong streamingBlockNumber = new AtomicLong(-1L);
    private int requestIndex = 0;
    private final AtomicReference<Thread> blockStreamWorkerThreadRef = new AtomicReference();
    private final Map<BlockNodeConfig, BlockNodeConnection> connections = new ConcurrentHashMap<BlockNodeConfig, BlockNodeConnection>();
    private final AtomicReference<BlockNodeConnection> activeConnectionRef = new AtomicReference();
    private final AtomicBoolean isStreamingEnabled = new AtomicBoolean(false);
    private final Map<BlockNodeConfig, BlockNodeStats> nodeStats;
    private final int maxEndOfStreamsAllowed;
    private final Duration endOfStreamTimeFrame;
    private final Duration endOfStreamScheduleDelay;
    private final Map<BlockNodeConfig, RetryState> retryStates = new ConcurrentHashMap<BlockNodeConfig, RetryState>();

    @Inject
    public BlockNodeConnectionManager(@NonNull ConfigProvider configProvider, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamMetrics blockStreamMetrics, @NonNull ScheduledExecutorService sharedExecutorService) {
        this.configProvider = Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.lastVerifiedBlockPerConnection = new ConcurrentHashMap<BlockNodeConfig, Long>();
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.sharedExecutorService = Objects.requireNonNull(sharedExecutorService, "sharedExecutorService must not be null");
        this.nodeStats = new ConcurrentHashMap<BlockNodeConfig, BlockNodeStats>();
        BlockNodeConnectionConfig blockNodeConnectionConfig = (BlockNodeConnectionConfig)configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class);
        this.maxEndOfStreamsAllowed = blockNodeConnectionConfig.maxEndOfStreamsAllowed();
        this.endOfStreamTimeFrame = blockNodeConnectionConfig.endOfStreamTimeFrame();
        this.endOfStreamScheduleDelay = blockNodeConnectionConfig.endOfStreamScheduleDelay();
        this.isStreamingEnabled.set(this.isStreamingEnabled());
        if (this.isStreamingEnabled.get()) {
            String blockNodeConnectionConfigPath = this.blockNodeConnectionFileDir();
            this.availableBlockNodes = new ArrayList<BlockNodeConfig>(this.extractBlockNodesConfigurations(blockNodeConnectionConfigPath));
            logger.info("Loaded block node configuration from {}", (Object)blockNodeConnectionConfigPath);
            logger.info("Block node configuration: {}", this.availableBlockNodes);
        } else {
            logger.info("Block node streaming is disabled; will not setup connections to block nodes");
            this.availableBlockNodes = new ArrayList<BlockNodeConfig>();
        }
    }

    private boolean isStreamingEnabled() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).streamToBlockNodes();
    }

    private String blockNodeConnectionFileDir() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).blockNodeConnectionFileDir();
    }

    private Duration expBackoffTimeframeReset() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).protocolExpBackoffTimeframeReset();
    }

    private int blockItemBatchSize() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).blockItemBatchSize();
    }

    private Duration workerLoopSleepDuration() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).workerLoopSleepDuration();
    }

    private List<BlockNodeConfig> extractBlockNodesConfigurations(@NonNull String blockNodeConfigPath) {
        Path configPath = Paths.get(blockNodeConfigPath, "block-nodes.json");
        try {
            byte[] jsonConfig = Files.readAllBytes(configPath);
            BlockNodeConnectionInfo protoConfig = (BlockNodeConnectionInfo)BlockNodeConnectionInfo.JSON.parse(Bytes.wrap((byte[])jsonConfig));
            return protoConfig.nodes().stream().map(node -> new BlockNodeConfig(node.address(), node.port(), node.priority())).toList();
        }
        catch (ParseException | IOException e) {
            logger.error("Failed to read block node configuration from {}", (Object)configPath, (Object)e);
            throw new RuntimeException("Failed to read block node configuration from " + String.valueOf(configPath), e);
        }
    }

    private boolean isOnlyOneBlockNodeConfigured() {
        return this.availableBlockNodes.size() == 1;
    }

    @NonNull
    private BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient createNewGrpcClient(@NonNull BlockNodeConfig nodeConfig) {
        Objects.requireNonNull(nodeConfig);
        Tls tls = ((TlsConfig.Builder)Tls.builder().enabled(false)).build();
        PbjGrpcClientConfig grpcConfig = new PbjGrpcClientConfig(Duration.ofSeconds(30L), tls, Optional.of(""), "application/grpc");
        WebClient webClient = ((WebClientConfig.Builder)((WebClientConfig.Builder)((WebClientConfig.Builder)((WebClientConfig.Builder)WebClient.builder().baseUri("http://" + nodeConfig.address() + ":" + nodeConfig.port())).tls(tls)).protocolConfigs(List.of(((GrpcClientProtocolConfig.Builder)((GrpcClientProtocolConfig.Builder)GrpcClientProtocolConfig.builder().abortPollTimeExpired(false)).pollWaitTime(Duration.ofSeconds(30L))).build()))).connectTimeout(Duration.ofSeconds(10L))).build();
        return new BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient((GrpcClient)new PbjGrpcClient(webClient, grpcConfig), (ServiceInterface.RequestOptions)OPTIONS);
    }

    public void rescheduleConnection(@NonNull BlockNodeConnection connection, @Nullable Duration delay, @Nullable Long blockNumber, boolean selectNewBlockNode) {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        Objects.requireNonNull(connection, "connection must not be null");
        logger.warn("[{}] Closing and rescheduling connection for reconnect attempt", (Object)connection);
        this.handleConnectionCleanupAndReschedule(connection, delay, blockNumber, selectNewBlockNode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleConnectionCleanupAndReschedule(@NonNull BlockNodeConnection connection, @Nullable Duration delay, @Nullable Long blockNumber, boolean selectNewBlockNode) {
        long delayMs;
        this.removeConnectionAndClearActive(connection);
        RetryState retryState = this.retryStates.computeIfAbsent(connection.getNodeConfig(), k -> new RetryState());
        int retryAttempt = 0;
        RetryState retryState2 = retryState;
        synchronized (retryState2) {
            retryState.updateRetryTime();
            retryAttempt = retryState.getRetryAttempt();
            delayMs = delay == null ? BlockNodeConnectionManager.calculateJitteredDelayMs(retryAttempt) : delay.toMillis();
            retryState.increment();
        }
        logger.debug("[{}] Apply exponential backoff and reschedule in {} ms (attempt={})", (Object)connection, (Object)delayMs, (Object)retryAttempt);
        this.scheduleConnectionAttempt(connection.getNodeConfig(), Duration.ofMillis(delayMs), blockNumber, false);
        if (!this.isOnlyOneBlockNodeConfigured() && selectNewBlockNode) {
            this.selectNewBlockNodeForStreaming(false);
        }
    }

    public void connectionResetsTheStream(@NonNull BlockNodeConnection connection) {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        Objects.requireNonNull(connection);
        this.removeConnectionAndClearActive(connection);
        this.selectNewBlockNodeForStreaming(false);
    }

    private void removeConnectionAndClearActive(@NonNull BlockNodeConnection connection) {
        Objects.requireNonNull(connection);
        this.connections.remove(connection.getNodeConfig(), connection);
        this.activeConnectionRef.compareAndSet(connection, null);
    }

    public void scheduleConnectionAttempt(@NonNull BlockNodeConfig blockNodeConfig, @NonNull Duration initialDelay, @Nullable Long blockNumber) {
        this.scheduleConnectionAttempt(blockNodeConfig, initialDelay, blockNumber, false);
    }

    private void scheduleConnectionAttempt(@NonNull BlockNodeConfig blockNodeConfig, @NonNull Duration initialDelay, @Nullable Long blockNumber, boolean force) {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        Objects.requireNonNull(blockNodeConfig);
        Objects.requireNonNull(initialDelay);
        long delayMillis = Math.max(0L, initialDelay.toMillis());
        BlockNodeConnection newConnection = this.createConnection(blockNodeConfig);
        if (blockNumber == null) {
            logger.debug("[{}] Scheduling reconnection for node in {} ms", (Object)newConnection, (Object)delayMillis);
        } else {
            logger.debug("[{}] Scheduling reconnection for node at block {} in {} ms", (Object)newConnection, (Object)blockNumber, (Object)delayMillis);
        }
        try {
            this.sharedExecutorService.schedule(new BlockNodeConnectionTask(newConnection, initialDelay, blockNumber, force), delayMillis, TimeUnit.MILLISECONDS);
            logger.debug("[{}] Successfully scheduled reconnection task", (Object)newConnection);
        }
        catch (Exception e) {
            logger.error("[{}] Failed to schedule connection task for block node", (Object)newConnection, (Object)e);
            this.connections.remove(newConnection.getNodeConfig());
            newConnection.close(true);
        }
    }

    public void shutdown() {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        this.blockBufferService.shutdown();
        logger.info("Shutting down connection manager!");
        if (!this.isConnectionManagerActive.compareAndSet(true, false)) {
            logger.debug("Connection Manager already shutdown");
            return;
        }
        Thread workerThread = this.blockStreamWorkerThreadRef.get();
        if (workerThread != null) {
            workerThread.interrupt();
            try {
                workerThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.error("Interrupted while waiting for block stream worker thread to terminate", (Throwable)e);
            }
        }
        this.blockStreamWorkerThreadRef.set(null);
        Iterator<Map.Entry<BlockNodeConfig, BlockNodeConnection>> it = this.connections.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<BlockNodeConfig, BlockNodeConnection> entry = it.next();
            BlockNodeConnection connection = entry.getValue();
            try {
                connection.close(true);
            }
            catch (RuntimeException e) {
                logger.debug("[{}] Error while closing connection during connection manager shutdown; ignoring", (Object)connection, (Object)e);
            }
            it.remove();
        }
        this.streamingBlockNumber.set(-1L);
        this.requestIndex = 0;
        this.activeConnectionRef.set(null);
        this.nodeStats.clear();
    }

    public void start() {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        if (!this.isConnectionManagerActive.compareAndSet(false, true)) {
            return;
        }
        Thread t = Thread.ofPlatform().name("BlockStreamWorkerLoop").start(this::blockStreamWorkerLoop);
        this.blockStreamWorkerThreadRef.set(t);
        if (!this.selectNewBlockNodeForStreaming(false)) {
            this.isConnectionManagerActive.set(false);
            throw new NoBlockNodesAvailableException();
        }
    }

    public boolean selectNewBlockNodeForStreaming(boolean force) {
        if (!this.isStreamingEnabled.get()) {
            return false;
        }
        BlockNodeConfig selectedNode = this.getNextPriorityBlockNode();
        if (selectedNode == null) {
            logger.debug("No block nodes found for attempted streaming");
            return false;
        }
        logger.debug("Selected block node {}:{} for connection attempt", (Object)selectedNode.address(), (Object)selectedNode.port());
        this.scheduleConnectionAttempt(selectedNode, Duration.ZERO, null, force);
        return true;
    }

    @Nullable
    private BlockNodeConfig getNextPriorityBlockNode() {
        logger.debug("Searching for new block node connection based on node priorities...");
        SortedMap priorityGroups = this.availableBlockNodes.stream().collect(Collectors.groupingBy(BlockNodeConfig::priority, TreeMap::new, Collectors.toList()));
        BlockNodeConfig selectedNode = null;
        for (Map.Entry entry : priorityGroups.entrySet()) {
            int priority = (Integer)entry.getKey();
            List nodesInGroup = (List)entry.getValue();
            selectedNode = this.findAvailableNode(nodesInGroup);
            if (selectedNode == null) {
                logger.trace("No available node found in priority group {}", (Object)priority);
                continue;
            }
            logger.trace("Found available node in priority group {}", (Object)priority);
            return selectedNode;
        }
        return selectedNode;
    }

    @Nullable
    private BlockNodeConfig findAvailableNode(@NonNull List<BlockNodeConfig> nodes) {
        Objects.requireNonNull(nodes, "nodes must not be null");
        return nodes.stream().filter(nodeConfig -> !this.connections.containsKey(nodeConfig)).collect(Collectors.collectingAndThen(Collectors.toList(), collected -> {
            Collections.shuffle(collected);
            return collected.stream();
        })).findFirst().orElse(null);
    }

    @NonNull
    private BlockNodeConnection createConnection(@NonNull BlockNodeConfig nodeConfig) {
        Objects.requireNonNull(nodeConfig);
        BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient grpcClient = this.createNewGrpcClient(nodeConfig);
        BlockNodeConnection connection = new BlockNodeConnection(this.configProvider, nodeConfig, this, this.blockBufferService, grpcClient, this.blockStreamMetrics, this.sharedExecutorService);
        this.connections.put(nodeConfig, connection);
        return connection;
    }

    public void openBlock(long blockNumber) {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        BlockNodeConnection activeConnection = this.activeConnectionRef.get();
        if (activeConnection == null) {
            this.blockStreamMetrics.recordNoActiveConnection();
            logger.debug("No active connections available for streaming block {}", (Object)blockNumber);
            return;
        }
        if (this.streamingBlockNumber.get() == -1L) {
            this.jumpTargetBlock.set(blockNumber);
        }
    }

    public void updateLastVerifiedBlock(@NonNull BlockNodeConfig blockNodeConfig, long blockNumber) {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        Objects.requireNonNull(blockNodeConfig);
        this.lastVerifiedBlockPerConnection.compute(blockNodeConfig, (cfg, lastVerifiedBlockNumber) -> lastVerifiedBlockNumber == null ? blockNumber : Math.max(lastVerifiedBlockNumber, blockNumber));
        this.blockBufferService.setLatestAcknowledgedBlock(blockNumber);
    }

    private void sleep(Duration duration) {
        try {
            Thread.sleep(duration);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    private void blockStreamWorkerLoop() {
        while (this.isConnectionManagerActive.get()) {
            BlockNodeConnection connection = this.activeConnectionRef.get();
            if (connection == null) {
                this.sleep(this.workerLoopSleepDuration());
                continue;
            }
            try {
                this.jumpToBlockIfNeeded();
                boolean shouldSleep = this.processStreamingToBlockNode(connection);
                if (!shouldSleep) continue;
                this.sleep(this.workerLoopSleepDuration());
            }
            catch (UncheckedIOException e) {
                logger.debug("UncheckedIOException caught in block stream worker loop", (Throwable)e);
                connection.handleStreamFailureWithoutOnComplete();
            }
            catch (Exception e) {
                logger.debug("Exception caught in block stream worker loop", (Throwable)e);
                connection.handleStreamFailure();
            }
        }
    }

    private boolean processStreamingToBlockNode(BlockNodeConnection connection) {
        if (connection == null || BlockNodeConnection.ConnectionState.ACTIVE != connection.getConnectionState()) {
            return true;
        }
        long currentStreamingBlockNumber = this.streamingBlockNumber.get();
        BlockState blockState = this.blockBufferService.getBlockState(currentStreamingBlockNumber);
        long latestBlockNumber = this.blockBufferService.getLastBlockNumberProduced();
        if (blockState == null && latestBlockNumber > currentStreamingBlockNumber) {
            logger.debug("[{}] Block {} not found in buffer (latestBlock={}); connection will be closed", (Object)connection, (Object)currentStreamingBlockNumber, (Object)latestBlockNumber);
            connection.close(true);
            this.rescheduleConnection(connection, BlockNodeConnection.THIRTY_SECONDS, null, true);
            return true;
        }
        if (blockState == null) {
            return true;
        }
        blockState.processPendingItems(this.blockItemBatchSize());
        if (blockState.numRequestsCreated() == 0) {
            return true;
        }
        if (this.requestIndex < blockState.numRequestsCreated()) {
            logger.debug("[{}] Processing block {} (isBlockProofSent={}, totalBlockRequests={}, currentRequestIndex={})", (Object)connection, (Object)this.streamingBlockNumber, (Object)blockState.isBlockProofSent(), (Object)blockState.numRequestsCreated(), (Object)this.requestIndex);
            PublishStreamRequest publishStreamRequest = blockState.getRequest(this.requestIndex);
            if (publishStreamRequest != null) {
                connection.sendRequest(publishStreamRequest);
                blockState.markRequestSent(this.requestIndex);
                ++this.requestIndex;
            }
        }
        if (this.requestIndex == blockState.numRequestsCreated() && blockState.isBlockProofSent()) {
            long nextBlockNumber = this.streamingBlockNumber.incrementAndGet();
            this.requestIndex = 0;
            logger.trace("[{}] Moving to next block number: {}", (Object)connection, (Object)nextBlockNumber);
            return false;
        }
        return this.requestIndex >= blockState.numRequestsCreated();
    }

    private void jumpToBlockIfNeeded() {
        long targetBlock = this.jumpTargetBlock.getAndSet(-1L);
        if (targetBlock < 0L) {
            return;
        }
        logger.debug("Jumping to block {}", (Object)targetBlock);
        this.streamingBlockNumber.set(targetBlock);
        this.requestIndex = 0;
    }

    public long currentStreamingBlockNumber() {
        return this.streamingBlockNumber.get();
    }

    public void jumpToBlock(long blockNumberToJumpTo) {
        if (!this.isStreamingEnabled.get()) {
            return;
        }
        logger.debug("Marking request to jump to block {}", (Object)blockNumberToJumpTo);
        this.jumpTargetBlock.set(blockNumberToJumpTo);
    }

    private static long calculateJitteredDelayMs(int retryAttempt) {
        Duration nextDelay = INITIAL_RETRY_DELAY.multipliedBy((long)Math.pow(2.0, retryAttempt));
        if (nextDelay.compareTo(MAX_RETRY_DELAY) > 0) {
            nextDelay = MAX_RETRY_DELAY;
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        return nextDelay.toMillis() / 2L + random.nextLong(nextDelay.toMillis() / 2L + 1L);
    }

    public boolean recordEndOfStreamAndCheckLimit(@NonNull BlockNodeConfig blockNodeConfig) {
        if (!this.isStreamingEnabled.get()) {
            return false;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        Instant now = Instant.now();
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        return stats.addEndOfStreamAndCheckLimit(now, this.maxEndOfStreamsAllowed, this.endOfStreamTimeFrame);
    }

    public Duration getEndOfStreamScheduleDelay() {
        return this.endOfStreamScheduleDelay;
    }

    public Duration getEndOfStreamTimeframe() {
        return this.endOfStreamTimeFrame;
    }

    public int getMaxEndOfStreamsAllowed() {
        return this.maxEndOfStreamsAllowed;
    }

    public int getEndOfStreamCount(@NonNull BlockNodeConfig blockNodeConfig) {
        if (!this.isStreamingEnabled.get()) {
            return 0;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.get(blockNodeConfig);
        return stats != null ? stats.getEndOfStreamCount() : 0;
    }

    private static long calculateIpAsInteger(@NonNull InetAddress address) {
        Objects.requireNonNull(address);
        byte[] bytes = address.getAddress();
        if (bytes.length != 4) {
            throw new IllegalArgumentException("Only IPv4 addresses are supported");
        }
        long octet1 = 0x1000000L * (long)(bytes[0] & 0xFF);
        long octet2 = 65536L * (long)(bytes[1] & 0xFF);
        long octet3 = 256L * (long)(bytes[2] & 0xFF);
        long octet4 = 1L * (long)(bytes[3] & 0xFF);
        return octet1 + octet2 + octet3 + octet4;
    }

    private void recordActiveConnectionIp(BlockNodeConfig nodeConfig) {
        long ipAsInteger;
        try {
            URL blockNodeUrl = URI.create("http://" + nodeConfig.address() + ":" + nodeConfig.port()).toURL();
            InetAddress blockAddress = InetAddress.getByName(blockNodeUrl.getHost());
            ipAsInteger = BlockNodeConnectionManager.calculateIpAsInteger(blockAddress);
            logger.info("Active block node connection updated to: {}:{} (resolvedIp: {}, resolvedIpAsInt={})", (Object)nodeConfig.address(), (Object)nodeConfig.port(), (Object)blockAddress.getHostAddress(), (Object)ipAsInteger);
        }
        catch (IOException e) {
            logger.error("Failed to resolve block node host ({}:{})", (Object)nodeConfig.address(), (Object)nodeConfig.port(), (Object)e);
            ipAsInteger = -1L;
        }
        this.blockStreamMetrics.recordActiveConnectionIp(ipAsInteger);
    }

    private record Options(Optional<String> authority, String contentType) implements ServiceInterface.RequestOptions
    {
    }

    class RetryState {
        private int retryAttempt = 0;
        private Instant lastRetryTime;

        RetryState() {
        }

        public int getRetryAttempt() {
            return this.retryAttempt;
        }

        public void increment() {
            ++this.retryAttempt;
        }

        public void updateRetryTime() {
            Duration timeSinceLastRetry;
            Instant now = Instant.now();
            if (this.lastRetryTime != null && (timeSinceLastRetry = Duration.between(this.lastRetryTime, now)).compareTo(BlockNodeConnectionManager.this.expBackoffTimeframeReset()) > 0) {
                this.retryAttempt = 0;
                this.lastRetryTime = now;
                return;
            }
            this.lastRetryTime = now;
        }
    }

    class BlockNodeConnectionTask
    implements Runnable {
        private final BlockNodeConnection connection;
        private Duration currentBackoffDelayMs;
        private final Long blockNumber;
        private final boolean force;

        BlockNodeConnectionTask(@NonNull BlockNodeConnection connection, @Nullable Duration initialDelay, Long blockNumber, boolean force) {
            this.connection = Objects.requireNonNull(connection);
            this.currentBackoffDelayMs = initialDelay.isNegative() ? Duration.ZERO : initialDelay;
            this.blockNumber = blockNumber;
            this.force = force;
        }

        @Override
        public void run() {
            if (!BlockNodeConnectionManager.this.isStreamingEnabled.get()) {
                return;
            }
            if (!BlockNodeConnectionManager.this.isConnectionManagerActive.get()) {
                logger.info("Connection task will not run because the connection manager has shutdown");
                return;
            }
            try {
                logger.debug("[{}] Running connection task...", (Object)this.connection);
                BlockNodeConnection activeConnection = BlockNodeConnectionManager.this.activeConnectionRef.get();
                if (activeConnection != null) {
                    if (activeConnection.equals(this.connection)) {
                        return;
                    }
                    if (this.force) {
                        BlockNodeConfig newConnConfig = this.connection.getNodeConfig();
                        BlockNodeConfig oldConnConfig = activeConnection.getNodeConfig();
                        logger.debug("New connection ({}:{} priority={}) is being forced as the new connection (old: {}:{} priority={})", (Object)newConnConfig.address(), (Object)newConnConfig.port(), (Object)newConnConfig.priority(), (Object)oldConnConfig.address(), (Object)oldConnConfig.port(), (Object)oldConnConfig.priority());
                    } else if (activeConnection.getNodeConfig().priority() <= this.connection.getNodeConfig().priority()) {
                        logger.debug("The existing active connection ({}) has an equal or higher priority than the connection ({}) we are attempting to connect to and this new connection attempt will be ignored", (Object)activeConnection, (Object)this.connection);
                        this.connection.close(true);
                        return;
                    }
                }
                this.connection.createRequestPipeline();
                if (BlockNodeConnectionManager.this.activeConnectionRef.compareAndSet(activeConnection, this.connection)) {
                    this.connection.updateConnectionState(BlockNodeConnection.ConnectionState.ACTIVE);
                    long blockToJumpTo = this.blockNumber != null ? this.blockNumber.longValue() : BlockNodeConnectionManager.this.blockBufferService.getLastBlockNumberProduced();
                    BlockNodeConnectionManager.this.jumpTargetBlock.set(blockToJumpTo);
                    BlockNodeConnectionManager.this.recordActiveConnectionIp(this.connection.getNodeConfig());
                } else {
                    this.reschedule();
                }
                if (activeConnection != null) {
                    try {
                        activeConnection.close(true);
                    }
                    catch (RuntimeException e) {
                        logger.debug("[{}] Failed to shutdown connection (shutdown reason: another connection was elevated to active)", (Object)activeConnection, (Object)e);
                    }
                }
            }
            catch (Exception e) {
                logger.debug("[{}] Failed to establish connection to block node; will schedule a retry", (Object)this.connection, (Object)e);
                BlockNodeConnectionManager.this.blockStreamMetrics.recordConnectionCreateFailure();
                this.reschedule();
            }
        }

        private void reschedule() {
            long jitteredDelayMs;
            Duration nextDelay;
            Duration duration = nextDelay = this.currentBackoffDelayMs.isZero() ? INITIAL_RETRY_DELAY : this.currentBackoffDelayMs.multipliedBy(2L);
            if (nextDelay.compareTo(MAX_RETRY_DELAY) > 0) {
                nextDelay = MAX_RETRY_DELAY;
            }
            ThreadLocalRandom random = ThreadLocalRandom.current();
            if (nextDelay.toMillis() > 0L) {
                jitteredDelayMs = nextDelay.toMillis() / 2L + random.nextLong(nextDelay.toMillis() / 2L + 1L);
            } else {
                jitteredDelayMs = INITIAL_RETRY_DELAY.toMillis() / 2L + random.nextLong(INITIAL_RETRY_DELAY.toMillis() / 2L + 1L);
                jitteredDelayMs = Math.max(1L, jitteredDelayMs);
            }
            this.currentBackoffDelayMs = Duration.ofMillis(jitteredDelayMs);
            try {
                BlockNodeConnectionManager.this.sharedExecutorService.schedule(this, jitteredDelayMs, TimeUnit.MILLISECONDS);
                logger.debug("[{}] Rescheduled connection attempt (delayMillis={})", (Object)this.connection, (Object)jitteredDelayMs);
            }
            catch (Exception e) {
                logger.error("[{}] Failed to reschedule connection attempt; removing from retry map", (Object)this.connection, (Object)e);
                BlockNodeConnectionManager.this.connections.remove(this.connection.getNodeConfig());
                this.connection.close(true);
            }
        }
    }
}

