/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnectionManager;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.node.internal.network.BlockNodeConfig;
import com.hedera.pbj.runtime.grpc.GrpcException;
import com.hedera.pbj.runtime.grpc.Pipeline;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.Flow;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.BlockStreamPublishServiceInterface;
import org.hiero.block.api.PublishStreamRequest;
import org.hiero.block.api.PublishStreamResponse;

public class BlockNodeConnection
implements Pipeline<PublishStreamResponse> {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnection.class);
    private static final AtomicLong connectionIdCounter = new AtomicLong(0L);
    public static final Duration THIRTY_SECONDS = Duration.ofSeconds(30L);
    private final BlockNodeConfig blockNodeConfig;
    private final BlockNodeConnectionManager blockNodeConnectionManager;
    private final BlockBufferService blockBufferService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final Duration streamResetPeriod;
    private final AtomicBoolean streamShutdownInProgress = new AtomicBoolean(false);
    private final BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient blockStreamPublishServiceClient;
    private final AtomicReference<Pipeline<? super PublishStreamRequest>> requestPipelineRef = new AtomicReference();
    private final AtomicReference<ConnectionState> connectionState;
    private final ScheduledExecutorService executorService;
    private ScheduledFuture<?> streamResetTask;
    private final String connectionId;

    public BlockNodeConnection(@NonNull ConfigProvider configProvider, @NonNull BlockNodeConfig nodeConfig, @NonNull BlockNodeConnectionManager blockNodeConnectionManager, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient grpcServiceClient, @NonNull BlockStreamMetrics blockStreamMetrics, @NonNull ScheduledExecutorService executorService) {
        Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.blockNodeConfig = Objects.requireNonNull(nodeConfig, "nodeConfig must not be null");
        this.blockNodeConnectionManager = Objects.requireNonNull(blockNodeConnectionManager, "blockNodeConnectionManager must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.blockStreamPublishServiceClient = Objects.requireNonNull(grpcServiceClient, "grpcServiceClient must not be null");
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.connectionState = new AtomicReference<ConnectionState>(ConnectionState.UNINITIALIZED);
        this.executorService = Objects.requireNonNull(executorService, "executorService must not be null");
        BlockNodeConnectionConfig blockNodeConnectionConfig = (BlockNodeConnectionConfig)configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class);
        this.streamResetPeriod = blockNodeConnectionConfig.streamResetPeriod();
        this.connectionId = String.format("%04d", connectionIdCounter.incrementAndGet());
    }

    public synchronized void createRequestPipeline() {
        if (this.requestPipelineRef.get() == null) {
            Pipeline pipeline = this.blockStreamPublishServiceClient.publishBlockStream((Pipeline)this);
            this.requestPipelineRef.set((Pipeline<? super PublishStreamRequest>)pipeline);
            this.updateConnectionState(ConnectionState.PENDING);
            this.blockStreamMetrics.recordConnectionOpened();
        }
    }

    public boolean updateConnectionState(@NonNull ConnectionState newState) {
        return this.updateConnectionState(null, newState);
    }

    /*
     * Enabled aggressive block sorting
     */
    private boolean updateConnectionState(@Nullable ConnectionState expectedCurrentState, @NonNull ConnectionState newState) {
        Objects.requireNonNull(newState, "newState must not be null");
        if (expectedCurrentState != null) {
            if (!this.connectionState.compareAndSet(expectedCurrentState, newState)) {
                logger.warn("[{}] Failed to transition state from {} to {} because current state does not match expected state", (Object)this, (Object)expectedCurrentState, (Object)newState);
                return false;
            }
            logger.debug("[{}] Connection state transitioned from {} to {}", (Object)this, (Object)expectedCurrentState, (Object)newState);
        } else {
            ConnectionState oldState = this.connectionState.getAndSet(newState);
            logger.debug("[{}] Connection state transitioned from {} to {}", (Object)this, (Object)oldState, (Object)newState);
        }
        if (newState == ConnectionState.ACTIVE) {
            this.scheduleStreamReset();
            return true;
        }
        this.cancelStreamReset();
        return true;
    }

    private void scheduleStreamReset() {
        if (this.streamResetTask != null && !this.streamResetTask.isDone()) {
            this.streamResetTask.cancel(false);
        }
        this.streamResetTask = this.executorService.scheduleAtFixedRate(this::performStreamReset, this.streamResetPeriod.toMillis(), this.streamResetPeriod.toMillis(), TimeUnit.MILLISECONDS);
        logger.debug("[{}] Scheduled periodic stream reset every {}", (Object)this, (Object)this.streamResetPeriod);
    }

    private void performStreamReset() {
        if (this.getConnectionState() == ConnectionState.ACTIVE) {
            logger.debug("[{}] Performing scheduled stream reset", (Object)this);
            this.endTheStreamWith(PublishStreamRequest.EndStream.Code.RESET);
            this.blockNodeConnectionManager.connectionResetsTheStream(this);
        }
    }

    private void cancelStreamReset() {
        if (this.streamResetTask != null) {
            this.streamResetTask.cancel(false);
            this.streamResetTask = null;
            logger.debug("[{}] Cancelled periodic stream reset", (Object)this);
        }
    }

    private void closeAndReschedule(@Nullable Duration delay, boolean callOnComplete) {
        this.close(callOnComplete);
        this.blockNodeConnectionManager.rescheduleConnection(this, delay, null, true);
    }

    private void endStreamAndReschedule(@NonNull PublishStreamRequest.EndStream.Code code) {
        Objects.requireNonNull(code, "code must not be null");
        this.endTheStreamWith(code);
        this.blockNodeConnectionManager.rescheduleConnection(this, THIRTY_SECONDS, null, true);
    }

    private void closeAndRestart(long blockNumber) {
        this.close(true);
        this.blockNodeConnectionManager.rescheduleConnection(this, null, blockNumber, false);
    }

    public void handleStreamFailure() {
        this.closeAndReschedule(THIRTY_SECONDS, true);
    }

    public void handleStreamFailureWithoutOnComplete() {
        this.closeAndReschedule(THIRTY_SECONDS, false);
    }

    private void handleAcknowledgement(@NonNull PublishStreamResponse.BlockAcknowledgement acknowledgement) {
        long acknowledgedBlockNumber = acknowledgement.blockNumber();
        logger.debug("[{}] BlockAcknowledgement received for block {}", (Object)this, (Object)acknowledgedBlockNumber);
        this.acknowledgeBlocks(acknowledgedBlockNumber, true);
    }

    private void acknowledgeBlocks(long acknowledgedBlockNumber, boolean maybeJumpToBlock) {
        long currentBlockStreaming = this.blockNodeConnectionManager.currentStreamingBlockNumber();
        long currentBlockProducing = this.blockBufferService.getLastBlockNumberProduced();
        this.blockNodeConnectionManager.updateLastVerifiedBlock(this.blockNodeConfig, acknowledgedBlockNumber);
        if (maybeJumpToBlock && (acknowledgedBlockNumber > currentBlockProducing || acknowledgedBlockNumber > currentBlockStreaming)) {
            long blockToJumpTo = acknowledgedBlockNumber + 1L;
            logger.debug("[{}] Received acknowledgement for block {}, however this is later than the current block being streamed ({}) or the block being currently produced ({}); skipping ahead to block {}", (Object)this, (Object)acknowledgedBlockNumber, (Object)currentBlockStreaming, (Object)currentBlockProducing, (Object)blockToJumpTo);
            this.jumpToBlock(blockToJumpTo);
        }
    }

    private void handleEndOfStream(@NonNull PublishStreamResponse.EndOfStream endOfStream) {
        Objects.requireNonNull(endOfStream, "endOfStream must not be null");
        long blockNumber = endOfStream.blockNumber();
        PublishStreamResponse.EndOfStream.Code responseCode = endOfStream.status();
        logger.debug("[{}] Received EndOfStream response (block={}, responseCode={})", (Object)this, (Object)blockNumber, (Object)responseCode);
        this.acknowledgeBlocks(blockNumber, false);
        if (this.blockNodeConnectionManager.recordEndOfStreamAndCheckLimit(this.blockNodeConfig)) {
            logger.debug("[{}] Block node has exceeded the allowed number of EndOfStream responses (received={}, permitted={}, timeWindow={}); reconnection scheduled for {}", (Object)this, (Object)this.blockNodeConnectionManager.getEndOfStreamCount(this.blockNodeConfig), (Object)this.blockNodeConnectionManager.getMaxEndOfStreamsAllowed(), (Object)this.blockNodeConnectionManager.getEndOfStreamTimeframe(), (Object)this.blockNodeConnectionManager.getEndOfStreamScheduleDelay());
            this.blockStreamMetrics.recordEndOfStreamLimitExceeded();
            this.closeAndReschedule(this.blockNodeConnectionManager.getEndOfStreamScheduleDelay(), true);
            return;
        }
        switch (responseCode) {
            case ERROR: 
            case PERSISTENCE_FAILED: {
                logger.debug("[{}] Block node reported an error at block {}. Will attempt to reestablish the stream later.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case TIMEOUT: 
            case DUPLICATE_BLOCK: 
            case BAD_BLOCK_PROOF: 
            case INVALID_REQUEST: {
                long restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                logger.debug("[{}] Block node reported status indicating immediate restart should be attempted. Will restart stream at block {}.", (Object)this, (Object)restartBlockNumber);
                this.closeAndRestart(restartBlockNumber);
                break;
            }
            case SUCCESS: {
                logger.debug("[{}] Block node orderly ended the stream at block {}", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case BEHIND: {
                long restartBlockNumber;
                long l = restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                if (this.blockBufferService.getBlockState(restartBlockNumber) != null) {
                    logger.debug("[{}] Block node reported it is behind. Will restart stream at block {}.", (Object)this, (Object)restartBlockNumber);
                    this.closeAndRestart(restartBlockNumber);
                    break;
                }
                logger.debug("[{}] Block node is behind and block state is not available.", (Object)this);
                this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
                break;
            }
            case UNKNOWN: {
                logger.error("[{}] Block node reported an unknown error at block {}.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
            }
        }
    }

    private void handleSkipBlock(@NonNull PublishStreamResponse.SkipBlock skipBlock) {
        Objects.requireNonNull(skipBlock, "skipBlock must not be null");
        long skipBlockNumber = skipBlock.blockNumber();
        long streamingBlockNumber = this.blockNodeConnectionManager.currentStreamingBlockNumber();
        if (skipBlockNumber == streamingBlockNumber) {
            long nextBlock = skipBlockNumber + 1L;
            logger.debug("[{}] Received SkipBlock response; skipping to block {}", (Object)this, (Object)nextBlock);
            this.jumpToBlock(nextBlock);
        } else {
            logger.debug("[{}] Received SkipBlock response for block {}, but we are not streaming that block so it will be ignored", (Object)this, (Object)skipBlockNumber);
        }
    }

    private void handleResendBlock(@NonNull PublishStreamResponse.ResendBlock resendBlock) {
        Objects.requireNonNull(resendBlock, "resendBlock must not be null");
        long resendBlockNumber = resendBlock.blockNumber();
        logger.debug("[{}] Received ResendBlock response for block {}", (Object)this, (Object)resendBlockNumber);
        if (this.blockBufferService.getBlockState(resendBlockNumber) != null) {
            this.jumpToBlock(resendBlockNumber);
        } else {
            logger.debug("[{}] Block node requested a ResendBlock for block {} but that block does not exist on this consensus node. Closing connection and will retry later", (Object)this, (Object)resendBlockNumber);
            this.closeAndReschedule(THIRTY_SECONDS, true);
        }
    }

    public void endTheStreamWith(PublishStreamRequest.EndStream.Code code) {
        long earliestBlockNumber = this.blockBufferService.getEarliestAvailableBlockNumber();
        long highestAckedBlockNumber = this.blockBufferService.getHighestAckedBlockNumber();
        PublishStreamRequest endStream = PublishStreamRequest.newBuilder().endStream(PublishStreamRequest.EndStream.newBuilder().endCode(code).earliestBlockNumber(earliestBlockNumber).latestBlockNumber(highestAckedBlockNumber)).build();
        logger.debug("[{}] Sending EndStream request with code {} (earliestBlockNumber={}, highestAckedBlockNumber={})", (Object)this, (Object)code, (Object)earliestBlockNumber, (Object)highestAckedBlockNumber);
        this.sendRequest(endStream);
        this.close(true);
    }

    public void sendRequest(@NonNull PublishStreamRequest request) {
        block8: {
            Objects.requireNonNull(request, "request must not be null");
            Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
            if (this.getConnectionState() == ConnectionState.ACTIVE && pipeline != null) {
                try {
                    if (logger.isDebugEnabled()) {
                        logger.debug("[{}] Sending request to block node (type={})", (Object)this, (Object)request.request().kind());
                    } else if (logger.isTraceEnabled()) {
                        logger.trace("[{}] Sending request to block node (type={}, bytes={})", (Object)this, (Object)request.request().kind(), (Object)request.protobufSize());
                    }
                    pipeline.onNext((Object)request);
                    if (request.hasEndStream()) {
                        this.blockStreamMetrics.recordRequestEndStreamSent(request.endStream().endCode());
                    } else {
                        this.blockStreamMetrics.recordRequestSent((PublishStreamRequest.RequestOneOfType)request.request().kind());
                        this.blockStreamMetrics.recordBlockItemsSent(request.blockItems().blockItems().size());
                    }
                }
                catch (RuntimeException e) {
                    if (this.getConnectionState() != ConnectionState.ACTIVE) break block8;
                    this.blockStreamMetrics.recordRequestSendFailure();
                    throw e;
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close(boolean callOnComplete) {
        ConnectionState connState = this.getConnectionState();
        if (connState.isTerminal()) {
            logger.debug("[{}] Connection already in terminal state ({})", (Object)this, (Object)connState);
            return;
        }
        if (!this.updateConnectionState(connState, ConnectionState.CLOSING)) {
            logger.debug("[{}] State changed while trying to close connection; aborting close attempt", (Object)this);
            return;
        }
        logger.debug("[{}] Closing connection...", (Object)this);
        try {
            this.closePipeline(callOnComplete);
            this.jumpToBlock(-1L);
            this.blockStreamMetrics.recordConnectionClosed();
            logger.debug("[{}] Connection successfully closed", (Object)this);
        }
        catch (RuntimeException e) {
            logger.warn("[{}] Error occurred while attempting to close connection", (Object)this);
        }
        finally {
            this.updateConnectionState(ConnectionState.CLOSED);
        }
    }

    private void closePipeline(boolean callOnComplete) {
        Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
        if (pipeline != null) {
            logger.debug("[{}] Closing request pipeline for block node", (Object)this);
            this.streamShutdownInProgress.set(true);
            try {
                ConnectionState state = this.getConnectionState();
                if (state == ConnectionState.CLOSING && callOnComplete) {
                    pipeline.onComplete();
                    logger.debug("[{}] Request pipeline successfully closed", (Object)this);
                }
            }
            catch (Exception e) {
                logger.warn("[{}] Error while completing request pipeline", (Object)this, (Object)e);
            }
            logger.debug("[{}] Request pipeline removed", (Object)this);
            this.requestPipelineRef.compareAndSet(pipeline, null);
        }
    }

    public BlockNodeConfig getNodeConfig() {
        return this.blockNodeConfig;
    }

    private void jumpToBlock(long blockNumber) {
        logger.debug("[{}] Jumping to block {}", (Object)this, (Object)blockNumber);
        this.blockNodeConnectionManager.jumpToBlock(blockNumber);
    }

    public void onSubscribe(Flow.Subscription subscription) {
        logger.debug("[{}] onSubscribe", (Object)this);
        subscription.request(Long.MAX_VALUE);
    }

    public void clientEndStreamReceived() {
        logger.debug("[{}] Client end stream received", (Object)this);
        super.clientEndStreamReceived();
    }

    public void onNext(@NonNull PublishStreamResponse response) {
        Objects.requireNonNull(response, "response must not be null");
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            logger.debug("[{}] onNext invoked but connection is already closed", (Object)this);
            return;
        }
        if (response.hasAcknowledgement()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.handleAcknowledgement(response.acknowledgement());
        } else if (response.hasEndStream()) {
            this.blockStreamMetrics.recordResponseEndOfStreamReceived(response.endStream().status());
            this.handleEndOfStream(response.endStream());
        } else if (response.hasSkipBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.handleSkipBlock(response.skipBlock());
        } else if (response.hasResendBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.handleResendBlock(response.resendBlock());
        } else {
            this.blockStreamMetrics.recordUnknownResponseReceived();
            logger.debug("[{}] Unexpected response received: {}", (Object)this, (Object)response);
        }
    }

    public void onError(Throwable error) {
        if (!this.getConnectionState().isTerminal()) {
            this.blockStreamMetrics.recordConnectionOnError();
            if (error instanceof GrpcException) {
                GrpcException grpcException = (GrpcException)error;
                logger.debug("[{}] Error received (grpcStatus={})", (Object)this, (Object)grpcException.status(), (Object)grpcException);
            } else {
                logger.debug("[{}] Error received", (Object)this, (Object)error);
            }
            this.handleStreamFailure();
        }
    }

    public void onComplete() {
        this.blockStreamMetrics.recordConnectionOnComplete();
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            logger.debug("[{}] onComplete invoked but connection is already closed", (Object)this);
            return;
        }
        if (this.streamShutdownInProgress.getAndSet(false)) {
            logger.debug("[{}] Stream completed (stream close was in progress)", (Object)this);
        } else {
            logger.debug("[{}] Stream completed unexpectedly", (Object)this);
            this.handleStreamFailure();
        }
    }

    @NonNull
    public ConnectionState getConnectionState() {
        return this.connectionState.get();
    }

    public String toString() {
        return this.connectionId + "/" + this.blockNodeConfig.address() + ":" + this.blockNodeConfig.port() + "/" + String.valueOf((Object)this.getConnectionState());
    }

    public boolean equals(Object o) {
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BlockNodeConnection that = (BlockNodeConnection)o;
        return this.connectionId == that.connectionId && Objects.equals(this.blockNodeConfig, that.blockNodeConfig);
    }

    public int hashCode() {
        return Objects.hash(this.blockNodeConfig, this.connectionId);
    }

    public static enum ConnectionState {
        UNINITIALIZED(false),
        PENDING(false),
        ACTIVE(false),
        CLOSING(true),
        CLOSED(true);

        private final boolean isTerminal;

        private ConnectionState(boolean isTerminal) {
            this.isTerminal = isTerminal;
        }

        boolean isTerminal() {
            return this.isTerminal;
        }
    }
}

