/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeClientFactory;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnection;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeStats;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.app.util.LoggingUtilities;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.node.config.data.BlockStreamConfig;
import com.hedera.node.internal.network.BlockNodeConfig;
import com.hedera.node.internal.network.BlockNodeConnectionInfo;
import com.hedera.pbj.runtime.ParseException;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import com.swirlds.common.io.utility.FileUtils;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.InetAddress;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.PublishStreamRequest;

@Singleton
public class BlockNodeConnectionManager {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnectionManager.class);
    public static final Duration INITIAL_RETRY_DELAY = Duration.ofSeconds(1L);
    private static final long RETRY_BACKOFF_MULTIPLIER = 2L;
    private static final Duration MAX_RETRY_DELAY = Duration.ofSeconds(10L);
    private final Map<BlockNodeConfig, Long> lastVerifiedBlockPerConnection;
    private final BlockBufferService blockBufferService;
    private ScheduledExecutorService sharedExecutorService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final ConfigProvider configProvider;
    private final List<BlockNodeConfig> availableBlockNodes = new ArrayList<BlockNodeConfig>();
    private final AtomicBoolean isConnectionManagerActive = new AtomicBoolean(false);
    private final AtomicLong jumpTargetBlock = new AtomicLong(-1L);
    private final AtomicLong streamingBlockNumber = new AtomicLong(-1L);
    private int requestIndex = 0;
    private final AtomicReference<Thread> blockStreamWorkerThreadRef = new AtomicReference();
    private final AtomicReference<WatchService> configWatchServiceRef = new AtomicReference();
    private final AtomicReference<Thread> configWatcherThreadRef = new AtomicReference();
    private Path blockNodeConfigDirectory;
    private static final String BLOCK_NODES_FILE_NAME = "block-nodes.json";
    private final Map<BlockNodeConfig, BlockNodeConnection> connections = new ConcurrentHashMap<BlockNodeConfig, BlockNodeConnection>();
    private final AtomicReference<BlockNodeConnection> activeConnectionRef = new AtomicReference();
    private final Map<BlockNodeConfig, BlockNodeStats> nodeStats;
    private final Map<BlockNodeConfig, RetryState> retryStates = new ConcurrentHashMap<BlockNodeConfig, RetryState>();
    private final BlockNodeClientFactory clientFactory;

    @Inject
    public BlockNodeConnectionManager(@NonNull ConfigProvider configProvider, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamMetrics blockStreamMetrics) {
        this.configProvider = Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.lastVerifiedBlockPerConnection = new ConcurrentHashMap<BlockNodeConfig, Long>();
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.nodeStats = new ConcurrentHashMap<BlockNodeConfig, BlockNodeStats>();
        this.blockNodeConfigDirectory = FileUtils.getAbsolutePath((String)this.blockNodeConnectionFileDir());
        this.clientFactory = new BlockNodeClientFactory();
    }

    private boolean isStreamingEnabled() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).streamToBlockNodes();
    }

    private String blockNodeConnectionFileDir() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).blockNodeConnectionFileDir();
    }

    private Duration expBackoffTimeframeReset() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).protocolExpBackoffTimeframeReset();
    }

    private int blockItemBatchSize() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).blockItemBatchSize();
    }

    private Duration workerLoopSleepDuration() {
        return ((BlockStreamConfig)this.configProvider.getConfiguration().getConfigData(BlockStreamConfig.class)).workerLoopSleepDuration();
    }

    private List<BlockNodeConfig> extractBlockNodesConfigurations(@NonNull String blockNodeConfigPath) {
        Path configPath = Paths.get(blockNodeConfigPath, BLOCK_NODES_FILE_NAME);
        try {
            if (!Files.exists(configPath, new LinkOption[0])) {
                LoggingUtilities.logWithContext(logger, Level.INFO, "Block node configuration file does not exist: {}", configPath);
                return List.of();
            }
            byte[] jsonConfig = Files.readAllBytes(configPath);
            BlockNodeConnectionInfo protoConfig = (BlockNodeConnectionInfo)BlockNodeConnectionInfo.JSON.parse(Bytes.wrap((byte[])jsonConfig));
            return protoConfig.nodes().stream().map(node -> new BlockNodeConfig(node.address(), node.port(), node.priority())).toList();
        }
        catch (ParseException | IOException e) {
            LoggingUtilities.logWithContext(logger, Level.INFO, "Failed to read or parse block node configuration from {}. Continuing without block node connections.", configPath, e);
            return List.of();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean isOnlyOneBlockNodeConfigured() {
        int size;
        List<BlockNodeConfig> list = this.availableBlockNodes;
        synchronized (list) {
            size = this.availableBlockNodes.size();
        }
        return size == 1;
    }

    public void rescheduleConnection(@NonNull BlockNodeConnection connection, @Nullable Duration delay, @Nullable Long blockNumber, boolean selectNewBlockNode) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(connection, "connection must not be null");
        LoggingUtilities.logWithContext(logger, Level.DEBUG, connection, "Closing and rescheduling connection for reconnect attempt.", new Object[0]);
        this.handleConnectionCleanupAndReschedule(connection, delay, blockNumber, selectNewBlockNode);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleConnectionCleanupAndReschedule(@NonNull BlockNodeConnection connection, @Nullable Duration delay, @Nullable Long blockNumber, boolean selectNewBlockNode) {
        long delayMs;
        this.removeConnectionAndClearActive(connection);
        RetryState retryState = this.retryStates.computeIfAbsent(connection.getNodeConfig(), k -> new RetryState());
        int retryAttempt = 0;
        RetryState retryState2 = retryState;
        synchronized (retryState2) {
            retryState.updateRetryTime();
            retryAttempt = retryState.getRetryAttempt();
            delayMs = delay == null ? BlockNodeConnectionManager.calculateJitteredDelayMs(retryAttempt) : delay.toMillis();
            retryState.increment();
        }
        LoggingUtilities.logWithContext(logger, Level.INFO, connection, "Apply exponential backoff and reschedule in {} ms (attempt={}).", delayMs, retryAttempt);
        this.scheduleConnectionAttempt(connection.getNodeConfig(), Duration.ofMillis(delayMs), blockNumber, false);
        if (!this.isOnlyOneBlockNodeConfigured() && selectNewBlockNode) {
            this.selectNewBlockNodeForStreaming(false);
        }
    }

    public void connectionResetsTheStream(@NonNull BlockNodeConnection connection) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(connection);
        this.removeConnectionAndClearActive(connection);
        this.selectNewBlockNodeForStreaming(false);
    }

    private void removeConnectionAndClearActive(@NonNull BlockNodeConnection connection) {
        Objects.requireNonNull(connection);
        this.connections.remove(connection.getNodeConfig(), connection);
        this.activeConnectionRef.compareAndSet(connection, null);
    }

    public void scheduleConnectionAttempt(@NonNull BlockNodeConfig blockNodeConfig, @NonNull Duration initialDelay, @Nullable Long blockNumber) {
        this.scheduleConnectionAttempt(blockNodeConfig, initialDelay, blockNumber, false);
    }

    private void scheduleConnectionAttempt(@NonNull BlockNodeConfig blockNodeConfig, @NonNull Duration initialDelay, @Nullable Long blockNumber, boolean force) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(blockNodeConfig);
        Objects.requireNonNull(initialDelay);
        long delayMillis = Math.max(0L, initialDelay.toMillis());
        BlockNodeConnection newConnection = this.createConnection(blockNodeConfig);
        if (blockNumber == null) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, newConnection, "Scheduling reconnection for node in {} ms (force={}).", delayMillis, force);
        } else {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Scheduling reconnection for node at block {} in {} ms (force={}).", newConnection, blockNumber, delayMillis, force);
        }
        try {
            this.sharedExecutorService.schedule(new BlockNodeConnectionTask(newConnection, initialDelay, blockNumber, force), delayMillis, TimeUnit.MILLISECONDS);
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Successfully scheduled reconnection task.", newConnection);
        }
        catch (Exception e) {
            logger.error(LoggingUtilities.formatLogMessage("Failed to schedule connection task for block node.", newConnection), (Throwable)e);
            this.connections.remove(newConnection.getNodeConfig());
            newConnection.close(true);
        }
    }

    public void shutdown() {
        if (!this.isConnectionManagerActive.compareAndSet(true, false)) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Connection Manager already shutdown.", new Object[0]);
            return;
        }
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Shutting down block node connection manager.", new Object[0]);
        this.stopConfigWatcher();
        this.blockBufferService.shutdown();
        this.shutdownScheduledExecutorService();
        this.shutdownBlockStreamWorkerThread();
        this.closeAllConnections();
        this.clearManagerMetadata();
    }

    private void shutdownScheduledExecutorService() {
        if (this.sharedExecutorService != null) {
            this.sharedExecutorService.shutdownNow();
        }
    }

    private void clearManagerMetadata() {
        this.streamingBlockNumber.set(-1L);
        this.requestIndex = 0;
        this.activeConnectionRef.set(null);
        this.nodeStats.clear();
        this.availableBlockNodes.clear();
    }

    private void closeAllConnections() {
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Stopping block node connections", new Object[0]);
        Iterator<Map.Entry<BlockNodeConfig, BlockNodeConnection>> it = this.connections.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<BlockNodeConfig, BlockNodeConnection> entry = it.next();
            BlockNodeConnection connection = entry.getValue();
            try {
                connection.close(true);
            }
            catch (RuntimeException e) {
                LoggingUtilities.logWithContext(logger, Level.DEBUG, "Error while closing connection during connection manager shutdown. Ignoring.", connection, e);
            }
            it.remove();
        }
    }

    private void shutdownBlockStreamWorkerThread() {
        Thread workerThread = this.blockStreamWorkerThreadRef.get();
        if (workerThread != null) {
            workerThread.interrupt();
            try {
                workerThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                LoggingUtilities.logWithContext(logger, Level.DEBUG, "Interrupted while waiting for block stream worker thread to terminate.", e);
            }
        }
        this.blockStreamWorkerThreadRef.set(null);
    }

    public void start() {
        if (!this.isStreamingEnabled()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Cannot start the connection manager, streaming is not enabled.", new Object[0]);
            return;
        }
        if (!this.isConnectionManagerActive.compareAndSet(false, true)) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Connection Manager already started.", new Object[0]);
            return;
        }
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Starting connection manager.", new Object[0]);
        this.blockBufferService.start();
        this.startConfigWatcher();
        this.refreshAvailableBlockNodes();
    }

    private void startBlockStreamWorkerThread() {
        if (this.blockStreamWorkerThreadRef.get() == null) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Starting block stream worker loop thread.", new Object[0]);
            Thread t = Thread.ofPlatform().name("BlockStreamWorkerLoop").start(this::blockStreamWorkerLoop);
            this.blockStreamWorkerThreadRef.set(t);
        }
    }

    private void createScheduledExectorService() {
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Creating scheduled executor service for the Block Node connection manager.", new Object[0]);
        this.sharedExecutorService = Executors.newSingleThreadScheduledExecutor();
    }

    public boolean selectNewBlockNodeForStreaming(boolean force) {
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Selecting highest priority available block node for connection attempt.", new Object[0]);
        if (!this.isStreamingEnabled()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Cannot select block node, streaming is not enabled.", new Object[0]);
            return false;
        }
        BlockNodeConfig selectedNode = this.getNextPriorityBlockNode();
        if (selectedNode == null) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "No available block nodes found for streaming.", new Object[0]);
            return false;
        }
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Selected block node {}:{} for connection attempt", selectedNode.address(), selectedNode.port());
        this.scheduleConnectionAttempt(selectedNode, Duration.ZERO, null, force);
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    private BlockNodeConfig getNextPriorityBlockNode() {
        ArrayList<BlockNodeConfig> snapshot;
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Searching for new block node connection based on node priorities.", new Object[0]);
        List<BlockNodeConfig> list = this.availableBlockNodes;
        synchronized (list) {
            snapshot = new ArrayList<BlockNodeConfig>(this.availableBlockNodes);
        }
        SortedMap priorityGroups = snapshot.stream().collect(Collectors.groupingBy(BlockNodeConfig::priority, TreeMap::new, Collectors.toList()));
        BlockNodeConfig selectedNode = null;
        for (Map.Entry entry : priorityGroups.entrySet()) {
            int priority = (Integer)entry.getKey();
            List nodesInGroup = (List)entry.getValue();
            selectedNode = this.findAvailableNode(nodesInGroup);
            if (selectedNode == null) {
                LoggingUtilities.logWithContext(logger, Level.DEBUG, "No available node found in priority group {}.", priority);
                continue;
            }
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Found available node in priority group {}.", priority);
            return selectedNode;
        }
        return selectedNode;
    }

    @Nullable
    private BlockNodeConfig findAvailableNode(@NonNull List<BlockNodeConfig> nodes) {
        Objects.requireNonNull(nodes, "nodes must not be null");
        return nodes.stream().filter(nodeConfig -> !this.connections.containsKey(nodeConfig)).collect(Collectors.collectingAndThen(Collectors.toList(), collected -> {
            Collections.shuffle(collected);
            return collected.stream();
        })).findFirst().orElse(null);
    }

    @NonNull
    private BlockNodeConnection createConnection(@NonNull BlockNodeConfig nodeConfig) {
        Objects.requireNonNull(nodeConfig);
        BlockNodeConnection connection = new BlockNodeConnection(this.configProvider, nodeConfig, this, this.blockBufferService, this.blockStreamMetrics, this.sharedExecutorService, this.clientFactory);
        this.connections.put(nodeConfig, connection);
        return connection;
    }

    public void openBlock(long blockNumber) {
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Opening block with number {}.", blockNumber);
        if (!this.isStreamingEnabled()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Cannot open block, streaming is not enabled.", new Object[0]);
            return;
        }
        BlockNodeConnection activeConnection = this.activeConnectionRef.get();
        if (activeConnection == null) {
            this.blockStreamMetrics.recordNoActiveConnection();
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "No active connections available for streaming block {}", blockNumber);
            return;
        }
        if (this.streamingBlockNumber.get() == -1L) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Current streaming block number is -1, jumping to {}.", blockNumber);
            this.jumpTargetBlock.set(blockNumber);
        }
    }

    public void updateLastVerifiedBlock(@NonNull BlockNodeConfig blockNodeConfig, long blockNumber) {
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Updating last verified block for {}:{} to {}.", blockNodeConfig.address(), blockNodeConfig.port(), blockNumber);
        if (!this.isStreamingEnabled()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Cannot update last verified block, streaming is not enabled.", new Object[0]);
            return;
        }
        Objects.requireNonNull(blockNodeConfig);
        this.lastVerifiedBlockPerConnection.compute(blockNodeConfig, (cfg, lastVerifiedBlockNumber) -> lastVerifiedBlockNumber == null ? blockNumber : Math.max(lastVerifiedBlockNumber, blockNumber));
        this.blockBufferService.setLatestAcknowledgedBlock(blockNumber);
    }

    private void sleep(Duration duration) {
        try {
            Thread.sleep(duration);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    private void blockStreamWorkerLoop() {
        while (this.isConnectionManagerActive.get() && !Thread.currentThread().isInterrupted()) {
            BlockNodeConnection connection = this.activeConnectionRef.get();
            if (connection == null) {
                this.sleep(this.workerLoopSleepDuration());
                continue;
            }
            try {
                this.jumpToBlockIfNeeded();
                boolean shouldSleep = this.processStreamingToBlockNode(connection);
                if (!shouldSleep) continue;
                this.sleep(this.workerLoopSleepDuration());
            }
            catch (UncheckedIOException e) {
                logger.warn("UncheckedIOException caught in block stream worker loop.", (Throwable)e);
                connection.handleStreamFailureWithoutOnComplete();
            }
            catch (Exception e) {
                if (Thread.currentThread().isInterrupted()) {
                    logger.info("Block stream worker loop interrupted, exiting.");
                    return;
                }
                logger.warn("Exception caught in block stream worker loop", (Throwable)e);
                connection.handleStreamFailure();
            }
        }
    }

    private void startConfigWatcher() {
        if (this.configWatchServiceRef.get() != null) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Configuration watcher already running.", new Object[0]);
            return;
        }
        try {
            WatchService watchService = this.blockNodeConfigDirectory.getFileSystem().newWatchService();
            this.configWatchServiceRef.set(watchService);
            this.blockNodeConfigDirectory.register(watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_DELETE);
            Thread watcherThread = Thread.ofPlatform().name("BlockNodesConfigWatcher").start(() -> {
                while (!Thread.currentThread().isInterrupted()) {
                    WatchKey key = null;
                    try {
                        key = watchService.take();
                        for (WatchEvent<?> event : key.pollEvents()) {
                            Path changed;
                            WatchEvent.Kind<?> kind = event.kind();
                            Object ctx = event.context();
                            if (!(ctx instanceof Path) || !BLOCK_NODES_FILE_NAME.equals((changed = (Path)ctx).toString())) continue;
                            LoggingUtilities.logWithContext(logger, Level.INFO, "Detected {} event for {}.", kind.name(), changed);
                            try {
                                this.refreshAvailableBlockNodes();
                            }
                            catch (Exception e) {
                                LoggingUtilities.logWithContext(logger, Level.INFO, "Exception in BlockNodesConfigWatcher config file change handler. {}", e);
                            }
                        }
                    }
                    catch (InterruptedException e) {
                        if (key == null || key.reset()) break;
                        LoggingUtilities.logWithContext(logger, Level.INFO, "WatchKey could not be reset. Exiting config watcher loop.", new Object[0]);
                        break;
                    }
                    catch (Exception e) {
                        LoggingUtilities.logWithContext(logger, Level.INFO, "Exception in config watcher loop.", e);
                        if (Thread.currentThread().isInterrupted()) {
                            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Config watcher thread interrupted, exiting.", new Object[0]);
                            return;
                        }
                    }
                    finally {
                        if (key == null || key.reset()) continue;
                        LoggingUtilities.logWithContext(logger, Level.INFO, "WatchKey could not be reset. Exiting config watcher loop.", new Object[0]);
                        break;
                    }
                }
            });
            this.configWatcherThreadRef.set(watcherThread);
            LoggingUtilities.logWithContext(logger, Level.INFO, "Started block-nodes.json configuration watcher thread.", new Object[0]);
        }
        catch (IOException e) {
            logger.info("Failed to start block-nodes.json configuration watcher. Dynamic updates disabled.", (Throwable)e);
        }
    }

    private void stopConfigWatcher() {
        WatchService ws;
        Thread watcherThread = this.configWatcherThreadRef.getAndSet(null);
        if (watcherThread != null) {
            watcherThread.interrupt();
            try {
                watcherThread.join();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        if ((ws = (WatchService)this.configWatchServiceRef.getAndSet(null)) != null) {
            try {
                ws.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void refreshAvailableBlockNodes() {
        String configDir = this.blockNodeConfigDirectory.toString();
        List<BlockNodeConfig> newConfigs = this.extractBlockNodesConfigurations(configDir);
        List<BlockNodeConfig> list = this.availableBlockNodes;
        synchronized (list) {
            if (newConfigs.equals(this.availableBlockNodes)) {
                LoggingUtilities.logWithContext(logger, Level.INFO, "Block node configuration unchanged. No action taken.", new Object[0]);
                return;
            }
        }
        this.shutdownScheduledExecutorService();
        this.shutdownBlockStreamWorkerThread();
        this.closeAllConnections();
        this.clearManagerMetadata();
        list = this.availableBlockNodes;
        synchronized (list) {
            this.availableBlockNodes.addAll(newConfigs);
        }
        if (!newConfigs.isEmpty()) {
            LoggingUtilities.logWithContext(logger, Level.INFO, "Reloaded block node configurations ({})", newConfigs);
            this.createScheduledExectorService();
            this.startBlockStreamWorkerThread();
            this.selectNewBlockNodeForStreaming(false);
        } else {
            LoggingUtilities.logWithContext(logger, Level.INFO, "No valid block node configurations available after file change. Connections remain stopped.", new Object[0]);
        }
    }

    private boolean processStreamingToBlockNode(BlockNodeConnection connection) {
        if (connection == null || BlockNodeConnection.ConnectionState.ACTIVE != connection.getConnectionState()) {
            return true;
        }
        long currentStreamingBlockNumber = this.streamingBlockNumber.get();
        BlockState blockState = this.blockBufferService.getBlockState(currentStreamingBlockNumber);
        long latestBlockNumber = this.blockBufferService.getLastBlockNumberProduced();
        if (blockState == null && latestBlockNumber > currentStreamingBlockNumber && currentStreamingBlockNumber != -1L) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Block {} not found in buffer (latestBlock={}). Closing and rescheduling.", connection, currentStreamingBlockNumber, latestBlockNumber);
            connection.close(true);
            this.rescheduleConnection(connection, BlockNodeConnection.THIRTY_SECONDS, null, true);
            return true;
        }
        if (blockState == null) {
            return true;
        }
        blockState.processPendingItems(this.blockItemBatchSize());
        if (blockState.numRequestsCreated() == 0) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Idle: no requests for block {}.", connection, currentStreamingBlockNumber);
            return true;
        }
        if (this.requestIndex < blockState.numRequestsCreated()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, connection, "Processing block {} (isBlockProofSent={}, totalBlockRequests={}, currentRequestIndex={}).", this.streamingBlockNumber, blockState.isBlockProofSent(), blockState.numRequestsCreated(), this.requestIndex);
            PublishStreamRequest publishStreamRequest = blockState.getRequest(this.requestIndex);
            if (publishStreamRequest != null) {
                connection.sendRequest(publishStreamRequest);
                LoggingUtilities.logWithContext(logger, Level.TRACE, connection, "Sent request {} for block {}.", this.requestIndex, currentStreamingBlockNumber);
                blockState.markRequestSent(this.requestIndex);
                ++this.requestIndex;
            }
        }
        if (this.requestIndex == blockState.numRequestsCreated() && blockState.isBlockProofSent()) {
            long nextBlockNumber = this.streamingBlockNumber.incrementAndGet();
            this.requestIndex = 0;
            LoggingUtilities.logWithContext(logger, Level.DEBUG, connection, "Moving to next block number: {}.", nextBlockNumber);
            return false;
        }
        return this.requestIndex >= blockState.numRequestsCreated();
    }

    private void jumpToBlockIfNeeded() {
        long targetBlock = this.jumpTargetBlock.getAndSet(-1L);
        if (targetBlock < 0L) {
            return;
        }
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Jumping to block {}.", targetBlock);
        this.streamingBlockNumber.set(targetBlock);
        this.requestIndex = 0;
    }

    public long currentStreamingBlockNumber() {
        return this.streamingBlockNumber.get();
    }

    public void jumpToBlock(long blockNumberToJumpTo) {
        if (!this.isStreamingEnabled()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "Cannot jump to block, streaming is not enabled.", new Object[0]);
            return;
        }
        LoggingUtilities.logWithContext(logger, Level.DEBUG, "Marking request to jump to block {}.", blockNumberToJumpTo);
        this.jumpTargetBlock.set(blockNumberToJumpTo);
    }

    private static long calculateJitteredDelayMs(int retryAttempt) {
        Duration nextDelay = INITIAL_RETRY_DELAY.multipliedBy((long)Math.pow(2.0, retryAttempt));
        if (nextDelay.compareTo(MAX_RETRY_DELAY) > 0) {
            nextDelay = MAX_RETRY_DELAY;
        }
        ThreadLocalRandom random = ThreadLocalRandom.current();
        return nextDelay.toMillis() / 2L + random.nextLong(nextDelay.toMillis() / 2L + 1L);
    }

    public boolean recordEndOfStreamAndCheckLimit(@NonNull BlockNodeConfig blockNodeConfig, @NonNull Instant timestamp) {
        if (!this.isStreamingEnabled()) {
            return false;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        return stats.addEndOfStreamAndCheckLimit(timestamp, this.getMaxEndOfStreamsAllowed(), this.getEndOfStreamTimeframe());
    }

    public Duration getEndOfStreamScheduleDelay() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).endOfStreamScheduleDelay();
    }

    public Duration getEndOfStreamTimeframe() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).endOfStreamTimeFrame();
    }

    public int getMaxEndOfStreamsAllowed() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).maxEndOfStreamsAllowed();
    }

    public int getEndOfStreamCount(@NonNull BlockNodeConfig blockNodeConfig) {
        if (!this.isStreamingEnabled()) {
            return 0;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.get(blockNodeConfig);
        return stats != null ? stats.getEndOfStreamCount() : 0;
    }

    private Duration getHighLatencyThreshold() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).highLatencyThreshold();
    }

    private int getHighLatencyEventsBeforeSwitching() {
        return ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).highLatencyEventsBeforeSwitching();
    }

    private static long calculateIpAsInteger(@NonNull InetAddress address) {
        Objects.requireNonNull(address);
        byte[] bytes = address.getAddress();
        if (bytes.length != 4) {
            throw new IllegalArgumentException("Only IPv4 addresses are supported");
        }
        long octet1 = 0x1000000L * (long)(bytes[0] & 0xFF);
        long octet2 = 65536L * (long)(bytes[1] & 0xFF);
        long octet3 = 256L * (long)(bytes[2] & 0xFF);
        long octet4 = 1L * (long)(bytes[3] & 0xFF);
        return octet1 + octet2 + octet3 + octet4;
    }

    private void recordActiveConnectionIp(BlockNodeConfig nodeConfig) {
        long ipAsInteger;
        try {
            URL blockNodeUrl = URI.create("http://" + nodeConfig.address() + ":" + nodeConfig.port()).toURL();
            InetAddress blockAddress = InetAddress.getByName(blockNodeUrl.getHost());
            ipAsInteger = BlockNodeConnectionManager.calculateIpAsInteger(blockAddress);
            LoggingUtilities.logWithContext(logger, Level.INFO, "Active block node connection updated to: {}:{} (resolvedIp: {}, resolvedIpAsInt={})", nodeConfig.address(), nodeConfig.port(), blockAddress.getHostAddress(), ipAsInteger);
        }
        catch (IOException e) {
            logger.debug("Failed to resolve block node host ({}:{})", (Object)nodeConfig.address(), (Object)nodeConfig.port(), (Object)e);
            ipAsInteger = -1L;
        }
        this.blockStreamMetrics.recordActiveConnectionIp(ipAsInteger);
    }

    public void recordBlockProofSent(@NonNull BlockNodeConfig blockNodeConfig, long blockNumber, @NonNull Instant timestamp) {
        if (!this.isStreamingEnabled()) {
            return;
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        stats.recordBlockProofSent(blockNumber, timestamp);
    }

    public BlockNodeStats.HighLatencyResult recordBlockAckAndCheckLatency(@NonNull BlockNodeConfig blockNodeConfig, long blockNumber, @NonNull Instant timestamp) {
        if (!this.isStreamingEnabled()) {
            return new BlockNodeStats.HighLatencyResult(0L, 0, false, false);
        }
        Objects.requireNonNull(blockNodeConfig, "blockNodeConfig must not be null");
        BlockNodeStats stats = this.nodeStats.computeIfAbsent(blockNodeConfig, k -> new BlockNodeStats());
        BlockNodeStats.HighLatencyResult result = stats.recordAcknowledgementAndEvaluate(blockNumber, timestamp, this.getHighLatencyThreshold(), this.getHighLatencyEventsBeforeSwitching());
        long latencyMs = result.latencyMs();
        this.blockStreamMetrics.recordAcknowledgementLatency(latencyMs);
        if (result.isHighLatency()) {
            LoggingUtilities.logWithContext(logger, Level.DEBUG, "[{}] A high latency event ({}ms) has occurred. A total of {} consecutive events", blockNodeConfig, latencyMs, result.consecutiveHighLatencyEvents());
            this.blockStreamMetrics.recordHighLatencyEvent();
        }
        return result;
    }

    class RetryState {
        private int retryAttempt = 0;
        private Instant lastRetryTime;

        RetryState() {
        }

        public int getRetryAttempt() {
            return this.retryAttempt;
        }

        public void increment() {
            ++this.retryAttempt;
        }

        public void updateRetryTime() {
            Duration timeSinceLastRetry;
            Instant now = Instant.now();
            if (this.lastRetryTime != null && (timeSinceLastRetry = Duration.between(this.lastRetryTime, now)).compareTo(BlockNodeConnectionManager.this.expBackoffTimeframeReset()) > 0) {
                this.retryAttempt = 0;
                this.lastRetryTime = now;
                return;
            }
            this.lastRetryTime = now;
        }
    }

    class BlockNodeConnectionTask
    implements Runnable {
        private final BlockNodeConnection connection;
        private Duration currentBackoffDelayMs;
        private final Long blockNumber;
        private final boolean force;

        private void logWithContext(Level level, String message, Object ... args) {
            if (logger.isEnabled(level)) {
                message = String.format("%s %s %s", LoggingUtilities.threadInfo(), this.connection.toString(), message);
                logger.atLevel(level).log(message, args);
            }
        }

        BlockNodeConnectionTask(@NonNull BlockNodeConnection connection, @Nullable Duration initialDelay, Long blockNumber, boolean force) {
            this.connection = Objects.requireNonNull(connection);
            this.currentBackoffDelayMs = initialDelay.isNegative() ? Duration.ZERO : initialDelay;
            this.blockNumber = blockNumber;
            this.force = force;
        }

        @Override
        public void run() {
            if (!BlockNodeConnectionManager.this.isStreamingEnabled()) {
                this.logWithContext(Level.DEBUG, "Cannot run connection task, streaming is not enabled.", new Object[0]);
                return;
            }
            if (!BlockNodeConnectionManager.this.isConnectionManagerActive.get()) {
                this.logWithContext(Level.DEBUG, "Cannot run connection task, connection manager has shutdown.", new Object[0]);
                return;
            }
            try {
                this.logWithContext(Level.DEBUG, "Running connection task.", new Object[0]);
                BlockNodeConnection activeConnection = BlockNodeConnectionManager.this.activeConnectionRef.get();
                if (activeConnection != null) {
                    if (activeConnection.equals(this.connection)) {
                        this.logWithContext(Level.DEBUG, "The current connection is the active connection, ignoring task.", new Object[0]);
                        return;
                    }
                    if (this.force) {
                        BlockNodeConfig newConnConfig = this.connection.getNodeConfig();
                        BlockNodeConfig oldConnConfig = activeConnection.getNodeConfig();
                        this.logWithContext(Level.DEBUG, "Promoting forced connection with priority={} over active ({}:{} priority={}).", newConnConfig.priority(), oldConnConfig.address(), oldConnConfig.port(), oldConnConfig.priority());
                    } else if (activeConnection.getNodeConfig().priority() <= this.connection.getNodeConfig().priority()) {
                        this.logWithContext(Level.DEBUG, "Active connection has equal/higher priority. Ignoring candidate. Active: {}.", activeConnection);
                        this.connection.close(true);
                        return;
                    }
                }
                this.connection.createRequestPipeline();
                if (BlockNodeConnectionManager.this.activeConnectionRef.compareAndSet(activeConnection, this.connection)) {
                    this.connection.updateConnectionState(BlockNodeConnection.ConnectionState.ACTIVE);
                    long blockToJumpTo = this.blockNumber != null ? this.blockNumber.longValue() : BlockNodeConnectionManager.this.blockBufferService.getLastBlockNumberProduced();
                    BlockNodeConnectionManager.this.jumpTargetBlock.set(blockToJumpTo);
                    BlockNodeConnectionManager.this.recordActiveConnectionIp(this.connection.getNodeConfig());
                    this.logWithContext(Level.DEBUG, "Jump target block is set to {}.", blockToJumpTo);
                } else {
                    this.logWithContext(Level.DEBUG, "Current connection task was preempted, rescheduling.", new Object[0]);
                    this.reschedule();
                }
                if (activeConnection != null) {
                    try {
                        this.logWithContext(Level.DEBUG, "Closing current active connection {}.", activeConnection);
                        activeConnection.close(true);
                    }
                    catch (RuntimeException e) {
                        logger.info("Failed to shutdown current active connection {} (shutdown reason: another connection was elevated to active).", (Object)activeConnection, (Object)e);
                    }
                }
            }
            catch (Exception e) {
                this.logWithContext(Level.DEBUG, "Failed to establish connection to block node. Will schedule a retry.", new Object[0]);
                BlockNodeConnectionManager.this.blockStreamMetrics.recordConnectionCreateFailure();
                this.reschedule();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void reschedule() {
            long jitteredDelayMs;
            Duration nextDelay;
            Duration duration = nextDelay = this.currentBackoffDelayMs.isZero() ? INITIAL_RETRY_DELAY : this.currentBackoffDelayMs.multipliedBy(2L);
            if (nextDelay.compareTo(MAX_RETRY_DELAY) > 0) {
                nextDelay = MAX_RETRY_DELAY;
            }
            ThreadLocalRandom random = ThreadLocalRandom.current();
            if (nextDelay.toMillis() > 0L) {
                jitteredDelayMs = nextDelay.toMillis() / 2L + random.nextLong(nextDelay.toMillis() / 2L + 1L);
            } else {
                jitteredDelayMs = INITIAL_RETRY_DELAY.toMillis() / 2L + random.nextLong(INITIAL_RETRY_DELAY.toMillis() / 2L + 1L);
                jitteredDelayMs = Math.max(1L, jitteredDelayMs);
            }
            this.currentBackoffDelayMs = Duration.ofMillis(jitteredDelayMs);
            try {
                List<BlockNodeConfig> list = BlockNodeConnectionManager.this.availableBlockNodes;
                synchronized (list) {
                    if (!BlockNodeConnectionManager.this.availableBlockNodes.contains(this.connection.getNodeConfig())) {
                        this.logWithContext(Level.DEBUG, "Node no longer available, skipping reschedule.", new Object[0]);
                        BlockNodeConnectionManager.this.connections.remove(this.connection.getNodeConfig());
                        return;
                    }
                }
                BlockNodeConnectionManager.this.sharedExecutorService.schedule(this, jitteredDelayMs, TimeUnit.MILLISECONDS);
                this.logWithContext(Level.INFO, "Rescheduled connection attempt (delayMillis={}).", jitteredDelayMs);
            }
            catch (Exception e) {
                logger.error("Failed to reschedule connection attempt. Removing from retry map.", (Throwable)e);
                BlockNodeConnectionManager.this.connections.remove(this.connection.getNodeConfig());
                this.connection.close(true);
            }
        }
    }
}

