/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.service.contract.impl.exec;

import com.hedera.hapi.node.base.AccountID;
import com.hedera.hapi.node.base.ContractID;
import com.hedera.node.app.service.contract.impl.exec.ActionSidecarContentTracer;
import com.hedera.node.app.service.contract.impl.exec.failure.CustomExceptionalHaltReason;
import com.hedera.node.app.service.contract.impl.exec.gas.CustomGasCalculator;
import com.hedera.node.app.service.contract.impl.exec.processors.CustomMessageCallProcessor;
import com.hedera.node.app.service.contract.impl.exec.utils.FrameUtils;
import com.hedera.node.app.service.contract.impl.hevm.HederaEvmTransactionResult;
import com.hedera.node.app.service.contract.impl.hevm.HevmPropagatedCallFailure;
import com.hedera.node.app.service.contract.impl.state.ProxyWorldUpdater;
import com.hedera.node.app.service.contract.impl.utils.ConversionUtils;
import com.hedera.node.app.service.entityid.EntityIdFactory;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.util.Deque;
import java.util.Objects;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.hyperledger.besu.datatypes.Address;
import org.hyperledger.besu.evm.frame.ExceptionalHaltReason;
import org.hyperledger.besu.evm.frame.MessageFrame;
import org.hyperledger.besu.evm.operation.Operation;
import org.hyperledger.besu.evm.processor.ContractCreationProcessor;

@Singleton
public class FrameRunner {
    private final CustomGasCalculator gasCalculator;
    private final EntityIdFactory entityIdFactory;

    @Inject
    public FrameRunner(@NonNull CustomGasCalculator gasCalculator, @NonNull EntityIdFactory entityIdFactory) {
        this.gasCalculator = gasCalculator;
        this.entityIdFactory = entityIdFactory;
    }

    public HederaEvmTransactionResult runToCompletion(long gasLimit, @NonNull AccountID senderId, @NonNull MessageFrame frame, @NonNull ActionSidecarContentTracer tracer, @NonNull CustomMessageCallProcessor messageCall, @NonNull ContractCreationProcessor contractCreation) {
        Objects.requireNonNull(frame);
        Objects.requireNonNull(tracer);
        Objects.requireNonNull(senderId);
        Objects.requireNonNull(messageCall);
        Objects.requireNonNull(contractCreation);
        Address recipientAddress = frame.getRecipientAddress();
        RecipientMetadata recipientMetadata = this.computeRecipientMetadata(frame, recipientAddress);
        tracer.traceOriginAction(frame);
        Deque stack = frame.getMessageFrameStack();
        while (!stack.isEmpty()) {
            this.runToCompletion((MessageFrame)stack.peekFirst(), tracer, messageCall, contractCreation);
        }
        tracer.sanitizeTracedActions(frame);
        long gasUsed = this.effectiveGasUsed(gasLimit, frame);
        if (frame.getState() == MessageFrame.State.COMPLETED_SUCCESS) {
            return HederaEvmTransactionResult.successFrom(gasUsed, senderId, recipientMetadata.hederaId(), ConversionUtils.asEvmContractId(this.entityIdFactory, recipientAddress), frame, tracer, this.entityIdFactory);
        }
        return HederaEvmTransactionResult.failureFrom(gasUsed, senderId, frame, recipientMetadata.postFailureHederaId(), tracer);
    }

    private RecipientMetadata computeRecipientMetadata(@NonNull MessageFrame frame, @NonNull Address address) {
        if (ConversionUtils.isLongZero(address)) {
            return new RecipientMetadata(false, ConversionUtils.asNumberedContractId(this.entityIdFactory, address));
        }
        ProxyWorldUpdater updater = FrameUtils.proxyUpdaterFor(frame);
        return new RecipientMetadata(updater.getPendingCreation() != null, updater.getHederaContractId(address));
    }

    private void runToCompletion(@NonNull MessageFrame frame, @NonNull ActionSidecarContentTracer tracer, @NonNull CustomMessageCallProcessor messageCall, @NonNull ContractCreationProcessor contractCreation) {
        CustomMessageCallProcessor executor = switch (frame.getType()) {
            default -> throw new MatchException(null, null);
            case MessageFrame.Type.MESSAGE_CALL -> messageCall;
            case MessageFrame.Type.CONTRACT_CREATION -> contractCreation;
        };
        executor.process(frame, tracer);
        frame.getExceptionalHaltReason().ifPresent(haltReason -> this.propagateHaltException(frame, (ExceptionalHaltReason)haltReason));
        HevmPropagatedCallFailure maybeFailureToPropagate = FrameUtils.getAndClearPropagatedCallFailure(frame);
        if (maybeFailureToPropagate != HevmPropagatedCallFailure.NONE) {
            FrameUtils.maybeNext(frame).ifPresent(f -> {
                f.setState(MessageFrame.State.EXCEPTIONAL_HALT);
                f.setExceptionalHaltReason(maybeFailureToPropagate.exceptionalHaltReason());
                maybeFailureToPropagate.exceptionalHaltReason().ifPresent(reason -> tracer.tracePostExecution((MessageFrame)f, new Operation.OperationResult(frame.getRemainingGas(), reason)));
            });
        }
    }

    private long effectiveGasUsed(long gasLimit, @NonNull MessageFrame frame) {
        long nominalGasUsed = gasLimit - frame.getRemainingGas();
        long nominalRefund = frame.getGasRefund();
        long maxGasRefunded = nominalGasUsed / this.gasCalculator.getMaxRefundQuotient();
        long actualGasToRefund = Math.min(maxGasRefunded, nominalRefund);
        long gasUsedAfterRefund = nominalGasUsed - actualGasToRefund;
        int maxRefundPercentOfGasLimit = FrameUtils.contractsConfigOf(frame).maxRefundPercentOfGasLimit();
        long minimumGasUsed = gasLimit - gasLimit * (long)maxRefundPercentOfGasLimit / 100L;
        return Math.max(gasUsedAfterRefund, minimumGasUsed);
    }

    private void propagateHaltException(MessageFrame frame, ExceptionalHaltReason haltReason) {
        if (haltReason.equals((Object)CustomExceptionalHaltReason.INSUFFICIENT_CHILD_RECORDS)) {
            FrameUtils.setPropagatedCallFailure(frame, HevmPropagatedCallFailure.RESULT_CANNOT_BE_EXTERNALIZED);
        } else if (haltReason.equals((Object)CustomExceptionalHaltReason.INVALID_CONTRACT_ID)) {
            FrameUtils.setPropagatedCallFailure(frame, HevmPropagatedCallFailure.INVALID_CONTRACT_ID);
        }
    }

    private record RecipientMetadata(boolean isPendingCreation, @NonNull ContractID hederaId) {
        private RecipientMetadata {
            Objects.requireNonNull(hederaId);
        }

        @Nullable
        public ContractID postFailureHederaId() {
            return this.isPendingCreation ? null : this.hederaId;
        }
    }
}

