/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeClientFactory;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConfiguration;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnectionManager;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeStats;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.pbj.grpc.client.helidon.PbjGrpcClientConfig;
import com.hedera.pbj.runtime.grpc.GrpcException;
import com.hedera.pbj.runtime.grpc.Pipeline;
import com.hedera.pbj.runtime.grpc.ServiceInterface;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import io.helidon.common.tls.Tls;
import io.helidon.common.tls.TlsConfig;
import io.helidon.webclient.api.WebClient;
import io.helidon.webclient.api.WebClientConfig;
import io.helidon.webclient.grpc.GrpcClientProtocolConfig;
import java.io.UncheckedIOException;
import java.lang.runtime.SwitchBootstraps;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Flow;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.BlockEnd;
import org.hiero.block.api.BlockItemSet;
import org.hiero.block.api.BlockStreamPublishServiceInterface;
import org.hiero.block.api.PublishStreamRequest;
import org.hiero.block.api.PublishStreamResponse;

public class BlockNodeConnection
implements Pipeline<PublishStreamResponse> {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnection.class);
    private static final Options OPTIONS = new Options(Optional.empty(), "application/grpc");
    private static final AtomicLong connectionIdCounter = new AtomicLong(0L);
    public static final Duration THIRTY_SECONDS = Duration.ofSeconds(30L);
    private final BlockNodeConfiguration nodeConfig;
    private final BlockNodeConnectionManager blockNodeConnectionManager;
    private final BlockBufferService blockBufferService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final Duration streamResetPeriod;
    private final Duration pipelineOperationTimeout;
    private final AtomicBoolean streamShutdownInProgress = new AtomicBoolean(false);
    private BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient blockStreamPublishServiceClient;
    private final AtomicReference<Pipeline<? super PublishStreamRequest>> requestPipelineRef = new AtomicReference();
    private final AtomicReference<ConnectionState> connectionState;
    private final ScheduledExecutorService executorService;
    private final ExecutorService pipelineExecutor;
    private ScheduledFuture<?> streamResetTask;
    private final String connectionId;
    private final AtomicLong streamingBlockNumber = new AtomicLong(-1L);
    private final AtomicReference<Thread> workerThreadRef = new AtomicReference();
    private final ConfigProvider configProvider;
    private final BlockNodeClientFactory clientFactory;
    private final AtomicBoolean closeAtNextBlockBoundary = new AtomicBoolean(false);

    public BlockNodeConnection(@NonNull ConfigProvider configProvider, @NonNull BlockNodeConfiguration nodeConfig, @NonNull BlockNodeConnectionManager blockNodeConnectionManager, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamMetrics blockStreamMetrics, @NonNull ScheduledExecutorService executorService, @NonNull ExecutorService pipelineExecutor, @Nullable Long initialBlockToStream, @NonNull BlockNodeClientFactory clientFactory) {
        this.configProvider = Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.nodeConfig = Objects.requireNonNull(nodeConfig, "nodeConfig must not be null");
        this.blockNodeConnectionManager = Objects.requireNonNull(blockNodeConnectionManager, "blockNodeConnectionManager must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.connectionState = new AtomicReference<ConnectionState>(ConnectionState.UNINITIALIZED);
        this.executorService = Objects.requireNonNull(executorService, "executorService must not be null");
        this.pipelineExecutor = Objects.requireNonNull(pipelineExecutor, "pipelineExecutor must not be null");
        BlockNodeConnectionConfig blockNodeConnectionConfig = (BlockNodeConnectionConfig)configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class);
        this.streamResetPeriod = blockNodeConnectionConfig.streamResetPeriod();
        this.clientFactory = Objects.requireNonNull(clientFactory, "clientFactory must not be null");
        this.pipelineOperationTimeout = blockNodeConnectionConfig.pipelineOperationTimeout();
        this.connectionId = String.format("%04d", connectionIdCounter.incrementAndGet());
        if (initialBlockToStream != null) {
            this.streamingBlockNumber.set(initialBlockToStream);
            logger.info("{} Block node connection will initially stream with block {}", (Object)this, (Object)initialBlockToStream);
        }
    }

    public synchronized void createRequestPipeline() {
        if (this.requestPipelineRef.get() == null) {
            Future<?> future = this.pipelineExecutor.submit(() -> {
                this.blockStreamPublishServiceClient = this.createNewGrpcClient();
                Pipeline pipeline = this.blockStreamPublishServiceClient.publishBlockStream((Pipeline)this);
                this.requestPipelineRef.set((Pipeline<? super PublishStreamRequest>)pipeline);
            });
            try {
                future.get(this.pipelineOperationTimeout.toMillis(), TimeUnit.MILLISECONDS);
                logger.debug("{} Request pipeline initialized.", (Object)this);
                this.updateConnectionState(ConnectionState.PENDING);
                this.blockStreamMetrics.recordConnectionOpened();
            }
            catch (TimeoutException e) {
                future.cancel(true);
                logger.debug("{} Pipeline creation timed out after {}ms", (Object)this, (Object)this.pipelineOperationTimeout.toMillis());
                this.blockStreamMetrics.recordPipelineOperationTimeout();
                throw new RuntimeException("Pipeline creation timed out", e);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.debug("{} Interrupted while creating pipeline", (Object)this, (Object)e);
                throw new RuntimeException("Interrupted while creating pipeline", e);
            }
            catch (ExecutionException e) {
                logger.debug("{} Error creating pipeline", (Object)this, (Object)e.getCause());
                throw new RuntimeException("Error creating pipeline", e.getCause());
            }
        } else {
            logger.debug("{} Request pipeline already available.", (Object)this);
        }
    }

    @NonNull
    private BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient createNewGrpcClient() {
        Duration timeoutDuration = ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).grpcOverallTimeout();
        Tls tls = ((TlsConfig.Builder)Tls.builder().enabled(false)).build();
        PbjGrpcClientConfig grpcConfig = new PbjGrpcClientConfig(timeoutDuration, tls, Optional.of(""), "application/grpc");
        WebClient webClient = ((WebClientConfig.Builder)((WebClientConfig.Builder)((WebClientConfig.Builder)((WebClientConfig.Builder)WebClient.builder().baseUri("http://" + this.nodeConfig.address() + ":" + this.nodeConfig.port())).tls(tls)).protocolConfigs(List.of(((GrpcClientProtocolConfig.Builder)((GrpcClientProtocolConfig.Builder)GrpcClientProtocolConfig.builder().abortPollTimeExpired(false)).pollWaitTime(timeoutDuration)).build()))).connectTimeout(timeoutDuration)).build();
        logger.debug("{} Created BlockStreamPublishServiceClient for {}:{}.", (Object)this, (Object)this.nodeConfig.address(), (Object)this.nodeConfig.port());
        return this.clientFactory.createClient(webClient, grpcConfig, OPTIONS);
    }

    public void updateConnectionState(@NonNull ConnectionState newState) {
        this.updateConnectionState(null, newState);
    }

    /*
     * Enabled aggressive block sorting
     */
    private boolean updateConnectionState(@Nullable ConnectionState expectedCurrentState, @NonNull ConnectionState newState) {
        Objects.requireNonNull(newState, "newState must not be null");
        if (expectedCurrentState != null) {
            if (!this.connectionState.compareAndSet(expectedCurrentState, newState)) {
                logger.debug("{} Failed to transition state from {} to {} because current state does not match expected state.", (Object)this, (Object)expectedCurrentState, (Object)newState);
                return false;
            }
            logger.info("{} Connection state transitioned from {} to {}.", (Object)this, (Object)expectedCurrentState, (Object)newState);
        } else {
            ConnectionState oldState = this.connectionState.getAndSet(newState);
            logger.info("{} Connection state transitioned from {} to {}.", (Object)this, (Object)oldState, (Object)newState);
        }
        if (newState == ConnectionState.ACTIVE) {
            this.handleConnectionActive();
            return true;
        }
        this.cancelStreamReset();
        return true;
    }

    private void handleConnectionActive() {
        this.scheduleStreamReset();
        Thread workerThread = new Thread((Runnable)new ConnectionWorkerLoopTask(), "bn-conn-worker-" + this.connectionId);
        if (this.workerThreadRef.compareAndSet(null, workerThread)) {
            workerThread.start();
        }
    }

    private void scheduleStreamReset() {
        if (this.streamResetTask != null && !this.streamResetTask.isDone()) {
            this.streamResetTask.cancel(false);
        }
        this.streamResetTask = this.executorService.scheduleAtFixedRate(this::performStreamReset, this.streamResetPeriod.toMillis(), this.streamResetPeriod.toMillis(), TimeUnit.MILLISECONDS);
        logger.debug("{} Scheduled periodic stream reset every {}.", (Object)this, (Object)this.streamResetPeriod);
    }

    private void performStreamReset() {
        if (this.getConnectionState() == ConnectionState.ACTIVE) {
            logger.info("{} Performing scheduled stream reset.", (Object)this);
            this.endTheStreamWith(PublishStreamRequest.EndStream.Code.RESET);
            this.blockNodeConnectionManager.selectNewBlockNodeForStreaming(false);
        }
    }

    private void cancelStreamReset() {
        if (this.streamResetTask != null) {
            this.streamResetTask.cancel(false);
            this.streamResetTask = null;
            logger.debug("{} Cancelled periodic stream reset.", (Object)this);
        }
    }

    private void closeAndReschedule(@Nullable Duration delay, boolean callOnComplete) {
        this.close(callOnComplete);
        this.blockNodeConnectionManager.rescheduleConnection(this, delay, null, true);
    }

    private void endStreamAndReschedule(@NonNull PublishStreamRequest.EndStream.Code code) {
        Objects.requireNonNull(code, "code must not be null");
        this.endTheStreamWith(code);
        this.blockNodeConnectionManager.rescheduleConnection(this, THIRTY_SECONDS, null, true);
    }

    private void closeAndRestart(long blockNumber) {
        this.close(true);
        this.blockNodeConnectionManager.rescheduleConnection(this, null, blockNumber, false);
    }

    public void handleStreamFailure() {
        logger.info("{} Handling failed stream.", (Object)this);
        this.closeAndReschedule(THIRTY_SECONDS, true);
    }

    public void handleStreamFailureWithoutOnComplete() {
        logger.info("{} Handling failed stream without onComplete.", (Object)this);
        this.closeAndReschedule(THIRTY_SECONDS, false);
    }

    private void handleAcknowledgement(@NonNull PublishStreamResponse.BlockAcknowledgement acknowledgement) {
        long acknowledgedBlockNumber = acknowledgement.blockNumber();
        logger.debug("{} BlockAcknowledgement received for block {}.", (Object)this, (Object)acknowledgedBlockNumber);
        this.acknowledgeBlocks(acknowledgedBlockNumber, true);
        BlockNodeStats.HighLatencyResult result = this.blockNodeConnectionManager.recordBlockAckAndCheckLatency(this.nodeConfig, acknowledgedBlockNumber, Instant.now());
        if (result.shouldSwitch() && !this.blockNodeConnectionManager.isOnlyOneBlockNodeConfigured()) {
            if (logger.isInfoEnabled()) {
                logger.info("{} Block node has exceeded high latency threshold {} times consecutively.", (Object)this, (Object)result.consecutiveHighLatencyEvents());
            }
            this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TIMEOUT);
        }
    }

    private void updateAcknowledgementMetrics(long acknowledgedBlockNumber) {
        long currentBlockProducing = this.blockBufferService.getLastBlockNumberProduced();
        if (acknowledgedBlockNumber != Long.MAX_VALUE) {
            long end;
            long lowestAvailableBlockInBuffer;
            long nowMs = System.currentTimeMillis();
            long previousAcknowledgedBlockNumber = this.blockBufferService.getHighestAckedBlockNumber();
            long start = Math.max(previousAcknowledgedBlockNumber + 1L, lowestAvailableBlockInBuffer = this.blockBufferService.getEarliestAvailableBlockNumber());
            if (start <= (end = Math.min(acknowledgedBlockNumber, currentBlockProducing))) {
                for (long blkNum = start; blkNum <= end; ++blkNum) {
                    long latencyMs;
                    BlockState blockState = this.blockBufferService.getBlockState(blkNum);
                    if (blockState == null) continue;
                    if (blockState.openedTimestamp() != null) {
                        long headerProducedToAckMs = nowMs - blockState.openedTimestamp().toEpochMilli();
                        this.blockStreamMetrics.recordHeaderProducedToAckLatency(headerProducedToAckMs);
                    }
                    if (blockState.closedTimestamp() != null) {
                        long blockClosedToAckMs = nowMs - blockState.closedTimestamp().toEpochMilli();
                        this.blockStreamMetrics.recordBlockClosedToAckLatency(blockClosedToAckMs);
                    }
                    if (blockState.getHeaderSentMs() != null) {
                        latencyMs = nowMs - blockState.getHeaderSentMs();
                        this.blockStreamMetrics.recordHeaderSentAckLatency(latencyMs);
                    }
                    if (blockState.getBlockEndSentMs() == null) continue;
                    latencyMs = nowMs - blockState.getBlockEndSentMs();
                    this.blockStreamMetrics.recordBlockEndSentToAckLatency(latencyMs);
                }
            }
        }
    }

    private void acknowledgeBlocks(long acknowledgedBlockNumber, boolean maybeJumpToBlock) {
        logger.debug("{} Acknowledging blocks <= {}.", (Object)this, (Object)acknowledgedBlockNumber);
        long currentBlockStreaming = this.streamingBlockNumber.get();
        long currentBlockProducing = this.blockBufferService.getLastBlockNumberProduced();
        this.updateAcknowledgementMetrics(acknowledgedBlockNumber);
        this.blockBufferService.setLatestAcknowledgedBlock(acknowledgedBlockNumber);
        if (maybeJumpToBlock && (acknowledgedBlockNumber > currentBlockProducing || acknowledgedBlockNumber > currentBlockStreaming)) {
            long blockToJumpTo = acknowledgedBlockNumber + 1L;
            logger.debug("{} Received acknowledgement for block {}, later than current streamed ({}) or produced ({}).", (Object)this, (Object)acknowledgedBlockNumber, (Object)currentBlockStreaming, (Object)currentBlockProducing);
            this.streamingBlockNumber.updateAndGet(current -> Math.max(current, blockToJumpTo));
        }
    }

    private void handleEndOfStream(@NonNull PublishStreamResponse.EndOfStream endOfStream) {
        Objects.requireNonNull(endOfStream, "endOfStream must not be null");
        long blockNumber = endOfStream.blockNumber();
        PublishStreamResponse.EndOfStream.Code responseCode = endOfStream.status();
        logger.info("{} Received EndOfStream response (block={}, responseCode={}).", (Object)this, (Object)blockNumber, (Object)responseCode);
        this.acknowledgeBlocks(blockNumber, false);
        if (this.blockNodeConnectionManager.recordEndOfStreamAndCheckLimit(this.nodeConfig, Instant.now())) {
            if (logger.isInfoEnabled()) {
                logger.info("{} Block node has exceeded the allowed number of EndOfStream responses (received={}, permitted={}, timeWindow={}). Reconnection scheduled for {}.", (Object)this, (Object)this.blockNodeConnectionManager.getEndOfStreamCount(this.nodeConfig), (Object)this.blockNodeConnectionManager.getMaxEndOfStreamsAllowed(), (Object)this.blockNodeConnectionManager.getEndOfStreamTimeframe(), (Object)this.blockNodeConnectionManager.getEndOfStreamScheduleDelay());
            }
            this.blockStreamMetrics.recordEndOfStreamLimitExceeded();
            this.closeAndReschedule(this.blockNodeConnectionManager.getEndOfStreamScheduleDelay(), true);
            return;
        }
        switch (responseCode) {
            case ERROR: 
            case PERSISTENCE_FAILED: {
                logger.info("{} Block node reported an error at block {}. Will attempt to reestablish the stream later.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case TIMEOUT: 
            case DUPLICATE_BLOCK: 
            case BAD_BLOCK_PROOF: 
            case INVALID_REQUEST: {
                long restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                logger.info("{} Block node reported status indicating immediate restart should be attempted. Will restart stream at block {}.", (Object)this, (Object)restartBlockNumber);
                this.closeAndRestart(restartBlockNumber);
                break;
            }
            case SUCCESS: {
                logger.info("{} Block node orderly ended the stream at block {}.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case BEHIND: {
                long restartBlockNumber;
                long l = restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                if (this.blockBufferService.getBlockState(restartBlockNumber) != null) {
                    logger.info("{} Block node reported it is behind. Will restart stream at block {}.", (Object)this, (Object)restartBlockNumber);
                    this.closeAndRestart(restartBlockNumber);
                    break;
                }
                logger.info("{} Block node is behind and block state is not available. Ending the stream.", (Object)this);
                this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
                break;
            }
            case UNKNOWN: {
                logger.info("{} Block node reported an unknown error at block {}.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
            }
        }
    }

    private void handleSkipBlock(@NonNull PublishStreamResponse.SkipBlock skipBlock) {
        long nextBlock;
        Objects.requireNonNull(skipBlock, "skipBlock must not be null");
        long skipBlockNumber = skipBlock.blockNumber();
        long activeBlockNumber = this.streamingBlockNumber.get();
        if (skipBlockNumber == activeBlockNumber && this.streamingBlockNumber.compareAndSet(activeBlockNumber, nextBlock = skipBlockNumber + 1L)) {
            logger.debug("{} Received SkipBlock response; skipping to block {}", (Object)this, (Object)nextBlock);
            return;
        }
        logger.debug("{} Received SkipBlock response (blockToSkip={}), but we've moved on to another block. Ignoring skip request", (Object)this, (Object)skipBlockNumber);
    }

    private void handleResendBlock(@NonNull PublishStreamResponse.ResendBlock resendBlock) {
        Objects.requireNonNull(resendBlock, "resendBlock must not be null");
        long resendBlockNumber = resendBlock.blockNumber();
        logger.debug("{} Received ResendBlock response for block {}.", (Object)this, (Object)resendBlockNumber);
        if (this.blockBufferService.getBlockState(resendBlockNumber) != null) {
            this.streamingBlockNumber.set(resendBlockNumber);
        } else {
            logger.info("{} Block node requested a ResendBlock for block {} but that block does not exist on this consensus node. Closing connection and will retry later.", (Object)this, (Object)resendBlockNumber);
            if (resendBlockNumber < this.blockBufferService.getEarliestAvailableBlockNumber()) {
                this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
            } else if (resendBlockNumber > this.blockBufferService.getLastBlockNumberProduced()) {
                this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.ERROR);
            }
        }
    }

    public void endTheStreamWith(PublishStreamRequest.EndStream.Code code) {
        long earliestBlockNumber = this.blockBufferService.getEarliestAvailableBlockNumber();
        long highestAckedBlockNumber = this.blockBufferService.getHighestAckedBlockNumber();
        PublishStreamRequest endStream = PublishStreamRequest.newBuilder().endStream(PublishStreamRequest.EndStream.newBuilder().endCode(code).earliestBlockNumber(earliestBlockNumber).latestBlockNumber(highestAckedBlockNumber)).build();
        logger.info("{} Sending EndStream (code={}, earliestBlock={}, latestAcked={}).", (Object)this, (Object)code, (Object)earliestBlockNumber, (Object)highestAckedBlockNumber);
        try {
            this.sendRequest(new EndStreamRequest(endStream));
        }
        catch (RuntimeException e) {
            logger.warn("{} Error sending EndStream request", (Object)this, (Object)e);
        }
        this.close(true);
    }

    private boolean sendRequest(@NonNull StreamRequest request) {
        Objects.requireNonNull(request, "request must not be null");
        Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
        if (this.getConnectionState() != ConnectionState.ACTIVE || pipeline == null) {
            logger.debug("{} Tried to send a request but the connection is not active or initialized; ignoring request", (Object)this);
            return false;
        }
        if (request instanceof BlockRequest) {
            BlockRequest br = (BlockRequest)request;
            logger.debug("{} [block={}, request={}] Sending request to block node (type={})", (Object)this, (Object)br.blockNumber(), (Object)br.requestNumber(), (Object)br.streamRequestType());
        } else {
            logger.debug("{} Sending ad hoc request to block node (type={})", (Object)this, (Object)request.streamRequestType());
        }
        long startMs = System.currentTimeMillis();
        long sentMs = 0L;
        try {
            Future<?> future = this.pipelineExecutor.submit(() -> pipeline.onNext((Object)request.streamRequest()));
            try {
                future.get(this.pipelineOperationTimeout.toMillis(), TimeUnit.MILLISECONDS);
                sentMs = System.currentTimeMillis();
            }
            catch (TimeoutException e) {
                future.cancel(true);
                if (this.getConnectionState() == ConnectionState.ACTIVE) {
                    logger.debug("{} Pipeline onNext() timed out after {}ms", (Object)this, (Object)this.pipelineOperationTimeout.toMillis());
                    this.blockStreamMetrics.recordPipelineOperationTimeout();
                    this.handleStreamFailure();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                logger.debug("{} Interrupted while waiting for pipeline.onNext()", (Object)this, (Object)e);
                throw new RuntimeException("Interrupted while waiting for pipeline.onNext()", e);
            }
            catch (ExecutionException e) {
                logger.debug("{} Error executing pipeline.onNext()", (Object)this, (Object)e.getCause());
                throw new RuntimeException("Error executing pipeline.onNext()", e.getCause());
            }
        }
        catch (RuntimeException e) {
            if (this.getConnectionState() == ConnectionState.ACTIVE) {
                this.blockStreamMetrics.recordRequestSendFailure();
                throw e;
            }
            logger.debug("{} Error occurred while sending request, but the connection is no longer active; suppressing error", (Object)this, (Object)e);
            return false;
        }
        long durationMs = sentMs - startMs;
        this.blockStreamMetrics.recordRequestLatency(durationMs);
        if (request instanceof BlockRequest) {
            BlockRequest br = (BlockRequest)request;
            logger.trace("{} [block={}, request={}] Request took {}ms to send", (Object)this, (Object)br.blockNumber(), (Object)br.requestNumber(), (Object)durationMs);
        } else {
            logger.trace("{} Ad hoc request took {}ms to send", (Object)this, (Object)durationMs);
        }
        StreamRequest streamRequest = request;
        Objects.requireNonNull(streamRequest);
        StreamRequest streamRequest2 = streamRequest;
        int n = 0;
        block2 : switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{EndStreamRequest.class, BlockRequest.class}, (Object)streamRequest2, n)) {
            default: {
                throw new MatchException(null, null);
            }
            case 0: {
                EndStreamRequest r = (EndStreamRequest)streamRequest2;
                this.blockStreamMetrics.recordRequestEndStreamSent(r.code());
                break;
            }
            case 1: {
                BlockState blockState;
                BlockRequest br;
                BlockRequest blockRequest = br = (BlockRequest)streamRequest2;
                Objects.requireNonNull(blockRequest);
                BlockRequest blockRequest2 = blockRequest;
                int n2 = 0;
                switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{BlockEndRequest.class, BlockItemsStreamRequest.class}, (Object)blockRequest2, n2)) {
                    default: {
                        throw new MatchException(null, null);
                    }
                    case 0: {
                        BlockEndRequest r = (BlockEndRequest)blockRequest2;
                        this.blockStreamMetrics.recordRequestSent(r.streamRequestType());
                        break block2;
                    }
                    case 1: 
                }
                BlockItemsStreamRequest r = (BlockItemsStreamRequest)blockRequest2;
                this.blockStreamMetrics.recordRequestSent(r.streamRequestType());
                this.blockStreamMetrics.recordBlockItemsSent(r.numItems());
                if (r.hasBlockProof()) {
                    this.blockNodeConnectionManager.recordBlockProofSent(this.nodeConfig, r.blockNumber(), Instant.ofEpochMilli(sentMs));
                }
                if (r.hasBlockHeader() && (blockState = this.blockBufferService.getBlockState(r.blockNumber())) != null) {
                    blockState.setHeaderSentMs(sentMs);
                }
                this.blockStreamMetrics.recordRequestBlockItemCount(r.numItems());
                this.blockStreamMetrics.recordRequestBytes(r.streamRequest().protobufSize());
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void close(boolean callOnComplete) {
        ConnectionState connState = this.getConnectionState();
        if (connState.isTerminal()) {
            logger.debug("{} Connection already in terminal state ({}).", (Object)this, (Object)connState);
            return;
        }
        if (!this.updateConnectionState(connState, ConnectionState.CLOSING)) {
            logger.debug("{} State changed while trying to close connection. Aborting close attempt.", (Object)this);
            return;
        }
        logger.info("{} Closing connection.", (Object)this);
        try {
            this.closePipeline(callOnComplete);
            logger.debug("{} Connection successfully closed.", (Object)this);
        }
        catch (RuntimeException e) {
            logger.warn("{} Error occurred while attempting to close connection.", (Object)this, (Object)e);
        }
        finally {
            try {
                if (this.blockStreamPublishServiceClient != null) {
                    this.blockStreamPublishServiceClient.close();
                }
            }
            catch (Exception e) {
                logger.error("{} Error occurred while closing gRPC client.", (Object)this, (Object)e);
            }
            try {
                this.pipelineExecutor.shutdown();
                if (!this.pipelineExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.pipelineExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                this.pipelineExecutor.shutdownNow();
                logger.error("{} Error occurred while shutting down pipeline executor.", (Object)this, (Object)e);
            }
            this.blockStreamMetrics.recordConnectionClosed();
            this.blockStreamMetrics.recordActiveConnectionIp(-1L);
            this.blockNodeConnectionManager.notifyConnectionClosed(this);
            this.updateConnectionState(ConnectionState.CLOSED);
        }
    }

    private void closePipeline(boolean callOnComplete) {
        Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
        if (pipeline != null) {
            logger.debug("{} Closing request pipeline for block node.", (Object)this);
            this.streamShutdownInProgress.set(true);
            try {
                ConnectionState state = this.getConnectionState();
                if (state == ConnectionState.CLOSING && callOnComplete) {
                    Future<?> future = this.pipelineExecutor.submit(() -> pipeline.onComplete());
                    try {
                        future.get(this.pipelineOperationTimeout.toMillis(), TimeUnit.MILLISECONDS);
                        logger.debug("{} Request pipeline successfully closed.", (Object)this);
                    }
                    catch (TimeoutException e) {
                        future.cancel(true);
                        logger.debug("{} Pipeline onComplete() timed out after {}ms", (Object)this, (Object)this.pipelineOperationTimeout.toMillis());
                        this.blockStreamMetrics.recordPipelineOperationTimeout();
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        logger.debug("{} Interrupted while waiting for pipeline.onComplete()", (Object)this);
                    }
                    catch (ExecutionException e) {
                        logger.debug("{} Error executing pipeline.onComplete()", (Object)this, (Object)e.getCause());
                    }
                }
            }
            catch (Exception e) {
                logger.warn("{} Error while completing request pipeline.", (Object)this, (Object)e);
            }
            logger.debug("{} Request pipeline removed.", (Object)this);
            this.requestPipelineRef.compareAndSet(pipeline, null);
        }
    }

    public BlockNodeConfiguration getNodeConfig() {
        return this.nodeConfig;
    }

    public void onSubscribe(Flow.Subscription subscription) {
        logger.debug("{} OnSubscribe invoked.", (Object)this);
        subscription.request(Long.MAX_VALUE);
    }

    public void clientEndStreamReceived() {
        logger.debug("{} Client End Stream received.", (Object)this);
        super.clientEndStreamReceived();
    }

    public void onNext(@NonNull PublishStreamResponse response) {
        Objects.requireNonNull(response, "response must not be null");
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            logger.debug("{} onNext invoked but connection is already closed ({}).", (Object)this, (Object)response);
            return;
        }
        if (response.hasAcknowledgement()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.handleAcknowledgement(response.acknowledgement());
        } else if (response.hasEndStream()) {
            this.blockStreamMetrics.recordResponseEndOfStreamReceived(response.endStream().status());
            this.blockStreamMetrics.recordLatestBlockEndOfStream(response.endStream().blockNumber());
            this.handleEndOfStream(response.endStream());
        } else if (response.hasSkipBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.blockStreamMetrics.recordLatestBlockSkipBlock(response.skipBlock().blockNumber());
            this.handleSkipBlock(response.skipBlock());
        } else if (response.hasResendBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.blockStreamMetrics.recordLatestBlockResendBlock(response.resendBlock().blockNumber());
            this.handleResendBlock(response.resendBlock());
        } else {
            this.blockStreamMetrics.recordUnknownResponseReceived();
            logger.debug("{} Unexpected response received: {}.", (Object)this, (Object)response);
        }
    }

    public void onError(Throwable error) {
        if (!this.getConnectionState().isTerminal()) {
            this.blockStreamMetrics.recordConnectionOnError();
            if (error instanceof GrpcException) {
                GrpcException grpcException = (GrpcException)error;
                logger.warn("{} Error received (grpcStatus={}).", (Object)this, (Object)grpcException.status(), (Object)grpcException);
            } else {
                logger.warn("{} Error received.", (Object)this, (Object)error);
            }
            this.handleStreamFailure();
        }
    }

    public void onComplete() {
        this.blockStreamMetrics.recordConnectionOnComplete();
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            logger.debug("{} onComplete invoked but connection is already closed.", (Object)this);
            return;
        }
        if (this.streamShutdownInProgress.getAndSet(false)) {
            logger.debug("{} Stream completed (stream close was in progress).", (Object)this);
        } else {
            logger.debug("{} Stream completed unexpectedly.", (Object)this);
            this.handleStreamFailure();
        }
    }

    @NonNull
    public ConnectionState getConnectionState() {
        return this.connectionState.get();
    }

    public void closeAtBlockBoundary() {
        logger.info("{} Connection will be closed at the next block boundary", (Object)this);
        this.closeAtNextBlockBoundary.set(true);
    }

    public String toString() {
        return "[" + this.connectionId + "/" + this.nodeConfig.address() + ":" + this.nodeConfig.port() + "/" + String.valueOf((Object)this.getConnectionState()) + "]";
    }

    public boolean equals(Object o) {
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BlockNodeConnection that = (BlockNodeConnection)o;
        return Objects.equals(this.connectionId, that.connectionId) && Objects.equals(this.nodeConfig, that.nodeConfig);
    }

    public int hashCode() {
        return Objects.hash(this.nodeConfig, this.connectionId);
    }

    public static enum ConnectionState {
        UNINITIALIZED(false),
        PENDING(false),
        ACTIVE(false),
        CLOSING(true),
        CLOSED(true);

        private final boolean isTerminal;

        private ConnectionState(boolean isTerminal) {
            this.isTerminal = isTerminal;
        }

        boolean isTerminal() {
            return this.isTerminal;
        }
    }

    private record Options(Optional<String> authority, String contentType) implements ServiceInterface.RequestOptions
    {
    }

    private class ConnectionWorkerLoopTask
    implements Runnable {
        private final List<BlockItem> pendingRequestItems = new ArrayList<BlockItem>();
        private long pendingRequestBytes;
        private int itemIndex = 0;
        private boolean pendingRequestHasBlockProof = false;
        private boolean pendingRequestHasBlockHeader = false;
        private BlockState block;
        private long lastSendTimeMillis = -1L;
        private final AtomicInteger requestCtr = new AtomicInteger(1);
        private final long softLimitBytes;
        private final long hardLimitBytes;
        private final int requestBasePaddingBytes;
        private final int requestItemPaddingBytes;

        private ConnectionWorkerLoopTask() {
            this.softLimitBytes = BlockNodeConnection.this.nodeConfig.messageSizeSoftLimitBytes();
            this.hardLimitBytes = BlockNodeConnection.this.nodeConfig.messageSizeHardLimitBytes();
            this.requestBasePaddingBytes = this.requestPaddingBytes();
            this.requestItemPaddingBytes = this.requestItemPaddingBytes();
            this.pendingRequestBytes = this.requestBasePaddingBytes;
        }

        @Override
        public void run() {
            logger.info("{} Worker thread started (messageSizeSoftLimit={}, messageSizeHardLimit={}, requestPadding={}, itemPadding={})", (Object)BlockNodeConnection.this, (Object)this.softLimitBytes, (Object)this.hardLimitBytes, (Object)this.requestBasePaddingBytes, (Object)this.requestItemPaddingBytes);
            block3: while (true) {
                try {
                    while (!BlockNodeConnection.this.connectionState.get().isTerminal()) {
                        boolean shouldSleep = this.doWork();
                        if (BlockNodeConnection.this.connectionState.get().isTerminal()) break block3;
                        if (!shouldSleep) continue;
                        Thread.sleep(this.connectionWorkerSleepMillis());
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.warn("{} Worker loop was interrupted", (Object)BlockNodeConnection.this);
                    continue;
                }
                catch (Exception e) {
                    logger.warn("{} Error caught in connection worker loop", (Object)BlockNodeConnection.this, (Object)e);
                    continue;
                }
                break;
            }
            logger.info("{} Worker thread exiting", (Object)BlockNodeConnection.this);
            BlockNodeConnection.this.workerThreadRef.compareAndSet(Thread.currentThread(), null);
        }

        private boolean doWork() {
            BlockItem item;
            this.switchBlockIfNeeded();
            if (this.block == null) {
                if (BlockNodeConnection.this.closeAtNextBlockBoundary.get()) {
                    logger.info("{} Block boundary reached; closing connection (no block available)", (Object)BlockNodeConnection.this);
                    BlockNodeConnection.this.endTheStreamWith(PublishStreamRequest.EndStream.Code.RESET);
                }
                return true;
            }
            while ((item = this.block.blockItem(this.itemIndex)) != null) {
                if (this.itemIndex == 0) {
                    logger.trace("{} Starting to process items for block {}", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber());
                    BlockNodeConnection.this.blockStreamMetrics.recordStreamingBlockNumber(this.block.blockNumber());
                    if (this.lastSendTimeMillis == -1L) {
                        this.lastSendTimeMillis = System.currentTimeMillis();
                    }
                }
                int itemSize = item.protobufSize() + this.requestItemPaddingBytes;
                long newRequestBytes = this.pendingRequestBytes + (long)itemSize;
                if ((long)itemSize > this.hardLimitBytes) {
                    try {
                        this.trySendPendingRequest();
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    BlockNodeConnection.this.blockStreamMetrics.recordRequestExceedsHardLimit();
                    logger.error("{} !!! FATAL: Block item exceeds max message size hard limit; closing connection (block={}, itemIndex={}, itemSize={}, sizeHardLimit={})", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.itemIndex, (Object)itemSize, (Object)this.hardLimitBytes);
                    BlockNodeConnection.this.endTheStreamWith(PublishStreamRequest.EndStream.Code.ERROR);
                    return true;
                }
                if ((long)itemSize >= this.softLimitBytes) {
                    if (!this.pendingRequestItems.isEmpty() && !this.trySendPendingRequest()) {
                        return true;
                    }
                    this.pendingRequestItems.add(item);
                    this.pendingRequestBytes += (long)itemSize;
                    this.pendingRequestHasBlockProof |= item.hasBlockProof();
                    this.pendingRequestHasBlockHeader |= item.hasBlockHeader();
                    ++this.itemIndex;
                    if (this.trySendPendingRequest()) continue;
                    return true;
                }
                if (newRequestBytes > this.softLimitBytes) {
                    if (this.trySendPendingRequest()) continue;
                    return true;
                }
                this.pendingRequestItems.add(item);
                this.pendingRequestBytes += (long)itemSize;
                this.pendingRequestHasBlockProof |= item.hasBlockProof();
                this.pendingRequestHasBlockHeader |= item.hasBlockHeader();
                ++this.itemIndex;
            }
            this.maybeSendPendingRequest();
            this.maybeAdvanceBlock();
            return this.block == null || this.block.itemCount() == this.itemIndex;
        }

        private void maybeSendPendingRequest() {
            if (this.pendingRequestItems.isEmpty()) {
                return;
            }
            if (this.block.isClosed() && this.block.itemCount() == this.itemIndex) {
                this.trySendPendingRequest();
            } else {
                long maxDelayMillis;
                long diffMillis = System.currentTimeMillis() - this.lastSendTimeMillis;
                if (diffMillis >= (maxDelayMillis = this.maxRequestDelayMillis())) {
                    logger.trace("{} Max delay exceeded (target: {}ms, actual: {}ms) - sending {} item(s)", (Object)BlockNodeConnection.this, (Object)maxDelayMillis, (Object)diffMillis, (Object)this.pendingRequestItems.size());
                    this.trySendPendingRequest();
                }
            }
        }

        private void sendBlockEnd() {
            PublishStreamRequest endOfBlock = PublishStreamRequest.newBuilder().endOfBlock(BlockEnd.newBuilder().blockNumber(this.block.blockNumber())).build();
            try {
                if (BlockNodeConnection.this.sendRequest(new BlockEndRequest(endOfBlock, this.block.blockNumber(), this.requestCtr.get()))) {
                    BlockNodeConnection.this.blockStreamMetrics.recordLatestBlockEndOfBlockSent(this.block.blockNumber());
                    this.block.setBlockEndSentMs(System.currentTimeMillis());
                    if (this.block.getHeaderSentMs() != null) {
                        long latencyMs = this.block.getBlockEndSentMs() - this.block.getHeaderSentMs();
                        BlockNodeConnection.this.blockStreamMetrics.recordHeaderSentToBlockEndSentLatency(latencyMs);
                    }
                }
            }
            catch (RuntimeException e) {
                logger.warn("{} Error sending EndOfBlock request", (Object)BlockNodeConnection.this, (Object)e);
                BlockNodeConnection.this.handleStreamFailureWithoutOnComplete();
            }
        }

        private void maybeAdvanceBlock() {
            boolean finishedWithCurrentBlock;
            boolean bl = finishedWithCurrentBlock = this.pendingRequestItems.isEmpty() && this.block.isClosed() && this.block.itemCount() == this.itemIndex;
            if (!finishedWithCurrentBlock) {
                return;
            }
            this.sendBlockEnd();
            if (BlockNodeConnection.this.closeAtNextBlockBoundary.get()) {
                logger.info("{} Block boundary reached; closing connection (finished sending block)", (Object)BlockNodeConnection.this);
                BlockNodeConnection.this.endTheStreamWith(PublishStreamRequest.EndStream.Code.RESET);
            } else {
                long nextBlockNumber = this.block.blockNumber() + 1L;
                if (BlockNodeConnection.this.streamingBlockNumber.compareAndSet(this.block.blockNumber(), nextBlockNumber)) {
                    logger.trace("{} Advancing to block {}", (Object)BlockNodeConnection.this, (Object)nextBlockNumber);
                } else {
                    logger.trace("{} Tried to advance to block {} but the block to stream was updated externally", (Object)BlockNodeConnection.this, (Object)nextBlockNumber);
                }
                this.switchBlockIfNeeded();
            }
        }

        private boolean trySendPendingRequest() {
            BlockItemSet itemSet = BlockItemSet.newBuilder().blockItems(List.copyOf(this.pendingRequestItems)).build();
            PublishStreamRequest req = PublishStreamRequest.newBuilder().blockItems(itemSet).build();
            long reqBytes = req.protobufSize();
            if (reqBytes > this.softLimitBytes && this.pendingRequestItems.size() > 1) {
                BlockNodeConnection.this.blockStreamMetrics.recordMultiItemRequestExceedsSoftLimit();
                logger.trace("{} Multi-item request exceeds soft limit; will attempt to remove last item and send again (requestSize={}, items={})", (Object)BlockNodeConnection.this, (Object)reqBytes, (Object)this.pendingRequestItems.size());
                BlockItem item = this.pendingRequestItems.removeLast();
                --this.itemIndex;
                this.pendingRequestBytes -= (long)(item.protobufSize() + this.requestItemPaddingBytes);
                if (item.hasBlockProof()) {
                    this.pendingRequestHasBlockProof = false;
                }
                if (item.hasBlockHeader()) {
                    this.pendingRequestHasBlockHeader = false;
                }
                return this.trySendPendingRequest();
            }
            if (reqBytes > this.hardLimitBytes) {
                BlockNodeConnection.this.blockStreamMetrics.recordRequestExceedsHardLimit();
                logger.error("{} !!! FATAL: Request exceeds maximum size hard limit of {} bytes (block={}, requestSize={}); Closing connection", (Object)BlockNodeConnection.this, (Object)this.hardLimitBytes, (Object)this.block.blockNumber(), (Object)reqBytes);
                BlockNodeConnection.this.endTheStreamWith(PublishStreamRequest.EndStream.Code.ERROR);
                return false;
            }
            logger.trace("{} Attempting to send request (block={}, request={}, itemCount={}, bytes={})", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.requestCtr.get(), (Object)this.pendingRequestItems.size(), (Object)reqBytes);
            try {
                if (BlockNodeConnection.this.sendRequest(new BlockItemsStreamRequest(req, this.block.blockNumber(), this.requestCtr.get(), this.pendingRequestItems.size(), this.pendingRequestHasBlockProof, this.pendingRequestHasBlockHeader))) {
                    this.lastSendTimeMillis = System.currentTimeMillis();
                    this.pendingRequestBytes = this.requestBasePaddingBytes;
                    this.pendingRequestItems.clear();
                    this.requestCtr.incrementAndGet();
                    this.pendingRequestHasBlockProof = false;
                    this.pendingRequestHasBlockHeader = false;
                    return true;
                }
                logger.warn("{} Sending the request failed for a non-exceptional reason (block={}, request={})", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.requestCtr.get());
            }
            catch (UncheckedIOException e) {
                logger.warn("{} UncheckedIOException caught in connection worker thread (block={}, request={})", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.requestCtr.get(), (Object)e);
                BlockNodeConnection.this.handleStreamFailureWithoutOnComplete();
            }
            catch (Exception e) {
                logger.warn("{} Exception caught in connection worker thread (block={}, request={})", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.requestCtr.get(), (Object)e);
                BlockNodeConnection.this.handleStreamFailure();
            }
            return false;
        }

        private void switchBlockIfNeeded() {
            long latestActiveBlockNumber;
            long latestBlock;
            if (BlockNodeConnection.this.streamingBlockNumber.get() == -1L && BlockNodeConnection.this.streamingBlockNumber.compareAndSet(-1L, latestBlock = BlockNodeConnection.this.blockBufferService.getLastBlockNumberProduced())) {
                logger.info("{} Connection was not initialized with a starting block; defaulting to latest block produced ({})", (Object)BlockNodeConnection.this, (Object)latestBlock);
            }
            if ((latestActiveBlockNumber = BlockNodeConnection.this.streamingBlockNumber.get()) == -1L) {
                return;
            }
            if (this.block != null && this.block.blockNumber() == latestActiveBlockNumber) {
                return;
            }
            BlockState oldBlock = this.block;
            this.block = BlockNodeConnection.this.blockBufferService.getBlockState(latestActiveBlockNumber);
            if (this.block == null && latestActiveBlockNumber < BlockNodeConnection.this.blockBufferService.getEarliestAvailableBlockNumber()) {
                logger.warn("{} Wanted block ({}) is not obtainable; notifying block node it is too far behind and closing connection", (Object)BlockNodeConnection.this, (Object)latestActiveBlockNumber);
                BlockNodeConnection.this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
            }
            this.pendingRequestBytes = this.requestBasePaddingBytes;
            this.itemIndex = 0;
            this.pendingRequestItems.clear();
            this.requestCtr.set(1);
            this.pendingRequestHasBlockProof = false;
            this.pendingRequestHasBlockHeader = false;
            if (this.block == null) {
                logger.trace("{} Wanted to switch from block {} to block {}, but it is not available", (Object)BlockNodeConnection.this, (Object)(oldBlock == null ? -1L : oldBlock.blockNumber()), (Object)latestActiveBlockNumber);
            } else {
                logger.trace("{} Switched from block {} to block {}", (Object)BlockNodeConnection.this, (Object)(oldBlock == null ? -1L : oldBlock.blockNumber()), (Object)latestActiveBlockNumber);
            }
        }

        private long maxRequestDelayMillis() {
            return ((BlockNodeConnectionConfig)BlockNodeConnection.this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).maxRequestDelay().toMillis();
        }

        private long connectionWorkerSleepMillis() {
            return ((BlockNodeConnectionConfig)BlockNodeConnection.this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).connectionWorkerSleepDuration().toMillis();
        }

        private int requestPaddingBytes() {
            return ((BlockNodeConnectionConfig)BlockNodeConnection.this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).streamingRequestPaddingBytes();
        }

        private int requestItemPaddingBytes() {
            return ((BlockNodeConnectionConfig)BlockNodeConnection.this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).streamingRequestItemPaddingBytes();
        }
    }

    record EndStreamRequest(@NonNull PublishStreamRequest streamRequest) implements StreamRequest
    {
        EndStreamRequest {
            Objects.requireNonNull(streamRequest);
        }

        @NonNull
        PublishStreamRequest.EndStream.Code code() {
            return Objects.requireNonNull(this.streamRequest().endStream()).endCode();
        }
    }

    static sealed interface StreamRequest
    permits EndStreamRequest, BlockRequest {
        @NonNull
        public PublishStreamRequest streamRequest();

        default public PublishStreamRequest.RequestOneOfType streamRequestType() {
            return (PublishStreamRequest.RequestOneOfType)this.streamRequest().request().kind();
        }
    }

    static sealed interface BlockRequest
    extends StreamRequest
    permits BlockEndRequest, BlockItemsStreamRequest {
        public long blockNumber();

        public int requestNumber();
    }

    record BlockEndRequest(@NonNull PublishStreamRequest streamRequest, long blockNumber, int requestNumber) implements BlockRequest
    {
        BlockEndRequest {
            Objects.requireNonNull(streamRequest);
        }
    }

    record BlockItemsStreamRequest(@NonNull PublishStreamRequest streamRequest, long blockNumber, int requestNumber, int numItems, boolean hasBlockProof, boolean hasBlockHeader) implements BlockRequest
    {
        BlockItemsStreamRequest {
            Objects.requireNonNull(streamRequest);
        }
    }
}

