/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.collections;

import com.swirlds.common.io.utility.LegacyTemporaryFileBuilder;
import com.swirlds.config.api.Configuration;
import com.swirlds.merkledb.collections.AbstractLongList;
import com.swirlds.merkledb.utilities.MerkleDbFileUtils;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.Deque;
import java.util.Objects;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicBoolean;

public class LongListDisk
extends AbstractLongList<Long> {
    private static final String STORE_POSTFIX = "longListDisk";
    private static final String DEFAULT_FILE_NAME = "LongListDisk.ll";
    private static final ThreadLocal<ByteBuffer> TEMP_LONG_BUFFER_THREAD_LOCAL;
    private FileChannel currentFileChannel;
    private Path tempFile;
    private Path tempDir;
    private static final ThreadLocal<ByteBuffer> TRANSFER_BUFFER_THREAD_LOCAL;
    private final Deque<Long> freeChunks = new ConcurrentLinkedDeque<Long>();
    private final AtomicBoolean closed = new AtomicBoolean(false);

    public LongListDisk(long capacity, Configuration configuration) {
        super(capacity, configuration);
        this.initFileChannel(configuration);
        this.fillBufferWithZeroes(this.initOrGetTransferBuffer());
    }

    public LongListDisk(int longsPerChunk, long capacity, long reservedBufferSize, @NonNull Configuration configuration) {
        super(longsPerChunk, capacity, reservedBufferSize);
        this.initFileChannel(configuration);
        this.fillBufferWithZeroes(this.initOrGetTransferBuffer());
    }

    public LongListDisk(@NonNull Path file, long capacity, @NonNull Configuration configuration) throws IOException {
        super(file, capacity, configuration);
        if (this.tempFile == null) {
            throw new IllegalStateException("The temp file is not initialized");
        }
    }

    public LongListDisk(@NonNull Path path, int longsPerChunk, long capacity, long reservedBufferSize, @NonNull Configuration configuration) throws IOException {
        super(path, longsPerChunk, capacity, reservedBufferSize, configuration);
        if (this.tempFile == null) {
            throw new IllegalStateException("The temp file is not initialized");
        }
    }

    private void initFileChannel(Configuration configuration) {
        if (this.tempFile != null) {
            throw new IllegalStateException("The temp file has been already initialized");
        }
        try {
            this.tempFile = this.createTempFile(DEFAULT_FILE_NAME, configuration);
            this.currentFileChannel = FileChannel.open(this.tempFile, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    protected void readBodyFromFileChannelOnInit(String sourceFileName, FileChannel fileChannel, Configuration configuration) throws IOException {
        this.tempFile = this.createTempFile(sourceFileName, configuration);
        this.currentFileChannel = FileChannel.open(this.tempFile, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        if (this.minValidIndex.get() < 0L) {
            return;
        }
        int firstChunkIndex = Math.toIntExact(this.minValidIndex.get() / (long)this.longsPerChunk);
        int lastChunkIndex = Math.toIntExact(this.maxValidIndex.get() / (long)this.longsPerChunk);
        int minValidIndexInChunk = Math.toIntExact(this.minValidIndex.get() % (long)this.longsPerChunk);
        long bytesTransferred = MerkleDbFileUtils.completelyTransferFrom(this.currentFileChannel, fileChannel, (long)minValidIndexInChunk * 8L, fileChannel.size() - fileChannel.position());
        if (bytesTransferred != (this.maxValidIndex.get() - this.minValidIndex.get() + 1L) * 8L) {
            throw new IOException("Failed to init LongListDisk from file: " + sourceFileName);
        }
        for (int chunkIndex = firstChunkIndex; chunkIndex <= lastChunkIndex; ++chunkIndex) {
            long chunk = (long)(chunkIndex - firstChunkIndex) * (long)this.memoryChunkSize;
            this.setChunk(chunkIndex, chunk);
        }
    }

    @Override
    protected Long readChunkData(FileChannel fileChannel, int chunkIndex, int startIndex, int endIndex) {
        throw new UnsupportedOperationException("This method should not be called");
    }

    private void fillBufferWithZeroes(ByteBuffer transferBuffer) {
        Arrays.fill(transferBuffer.array(), (byte)0);
        transferBuffer.position(0);
        transferBuffer.limit(this.memoryChunkSize);
    }

    private ByteBuffer initOrGetTransferBuffer() {
        ByteBuffer buffer = TRANSFER_BUFFER_THREAD_LOCAL.get();
        if (buffer == null || buffer.capacity() < this.memoryChunkSize) {
            buffer = ByteBuffer.allocate(this.memoryChunkSize).order(ByteOrder.nativeOrder());
            TRANSFER_BUFFER_THREAD_LOCAL.set(buffer);
        } else {
            buffer.clear();
        }
        buffer.limit(this.memoryChunkSize);
        return buffer;
    }

    Path createTempFile(String sourceFileName, @NonNull Configuration configuration) throws IOException {
        Objects.requireNonNull(configuration);
        this.tempDir = LegacyTemporaryFileBuilder.buildTemporaryDirectory((String)STORE_POSTFIX, (Configuration)configuration);
        if (!Files.exists(this.tempDir, new LinkOption[0])) {
            Files.createDirectories(this.tempDir, new FileAttribute[0]);
        }
        return this.tempDir.resolve(sourceFileName);
    }

    @Override
    protected synchronized void putToChunk(Long chunk, int subIndex, long value) {
        try {
            ByteBuffer buf = TEMP_LONG_BUFFER_THREAD_LOCAL.get();
            long offset = chunk + (long)subIndex * 8L;
            buf.putLong(0, value);
            buf.position(0);
            MerkleDbFileUtils.completelyWrite(this.currentFileChannel, buf, offset);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    protected synchronized boolean putIfEqual(Long chunk, int subIndex, long oldValue, long newValue) {
        ByteBuffer buf = TEMP_LONG_BUFFER_THREAD_LOCAL.get();
        buf.position(0);
        try {
            long offset = chunk + (long)subIndex * 8L;
            MerkleDbFileUtils.completelyRead(this.currentFileChannel, buf, offset);
            long filesOldValue = buf.getLong(0);
            if (filesOldValue == oldValue) {
                buf.putLong(0, newValue);
                buf.position(0);
                MerkleDbFileUtils.completelyWrite(this.currentFileChannel, buf, offset);
                return true;
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return false;
    }

    private long calculateOffsetInChunk(long index) {
        return index % (long)this.longsPerChunk * 8L;
    }

    @Override
    protected void writeLongsData(FileChannel fc) throws IOException {
        int firstChunkWithDataIndex;
        ByteBuffer transferBuffer = this.initOrGetTransferBuffer();
        int totalNumOfChunks = this.calculateNumberOfChunks(this.size());
        long currentMinValidIndex = this.minValidIndex.get();
        for (int i = firstChunkWithDataIndex = Math.toIntExact(currentMinValidIndex / (long)this.longsPerChunk); i < totalNumOfChunks; ++i) {
            Long currentChunkStartOffset = (Long)this.chunkList.get(i);
            if (currentChunkStartOffset != null) {
                long chunkOffset;
                if (i == firstChunkWithDataIndex) {
                    int firstValidIndexInChunk = Math.toIntExact(currentMinValidIndex % (long)this.longsPerChunk);
                    transferBuffer.position(firstValidIndexInChunk * 8);
                    chunkOffset = currentChunkStartOffset + this.calculateOffsetInChunk(currentMinValidIndex);
                } else {
                    transferBuffer.position(0);
                    chunkOffset = currentChunkStartOffset;
                }
                if (i == totalNumOfChunks - 1) {
                    long bytesWrittenSoFar = (long)this.memoryChunkSize * (long)i;
                    long remainingBytes = this.size() * 8L - bytesWrittenSoFar;
                    transferBuffer.limit(Math.toIntExact(remainingBytes));
                } else {
                    transferBuffer.limit(this.memoryChunkSize);
                }
                int currentPosition = transferBuffer.position();
                int toRead = transferBuffer.remaining();
                int read = MerkleDbFileUtils.completelyRead(this.currentFileChannel, transferBuffer, chunkOffset);
                if (toRead != read) {
                    throw new IOException("Failed to read a chunk from the file, offset=" + chunkOffset + ", toRead=" + toRead + ", read=" + read + ", file size=" + this.currentFileChannel.size());
                }
                transferBuffer.position(currentPosition);
            } else {
                this.fillBufferWithZeroes(transferBuffer);
            }
            MerkleDbFileUtils.completelyWrite(fc, transferBuffer);
        }
    }

    @Override
    protected long lookupInChunk(@NonNull Long chunkOffset, long subIndex) {
        try {
            ByteBuffer buf = TEMP_LONG_BUFFER_THREAD_LOCAL.get();
            buf.putLong(0, 0L);
            buf.clear();
            long offset = chunkOffset + subIndex * 8L;
            MerkleDbFileUtils.completelyRead(this.currentFileChannel, buf, offset);
            return buf.getLong(0);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    public void close() {
        if (!this.closed.compareAndSet(false, true)) {
            return;
        }
        try {
            if (this.currentFileChannel.isOpen()) {
                this.currentFileChannel.force(false);
            }
            super.close();
            this.currentFileChannel.close();
            this.freeChunks.clear();
            Files.delete(this.tempFile);
            Files.delete(this.tempDir);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    protected void closeChunk(@NonNull Long chunk) {
        ByteBuffer transferBuffer = this.initOrGetTransferBuffer();
        this.fillBufferWithZeroes(transferBuffer);
        try {
            this.currentFileChannel.write(transferBuffer, chunk);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        this.freeChunks.add(chunk);
    }

    @Override
    protected void partialChunkCleanup(@NonNull Long chunkOffset, boolean leftSide, long entriesToCleanUp) {
        ByteBuffer transferBuffer = this.initOrGetTransferBuffer();
        this.fillBufferWithZeroes(transferBuffer);
        transferBuffer.limit(Math.toIntExact(entriesToCleanUp * 8L));
        if (leftSide) {
            try {
                this.currentFileChannel.write(transferBuffer, chunkOffset);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        long cleanUpOffset = (long)this.memoryChunkSize - entriesToCleanUp * 8L;
        try {
            this.currentFileChannel.write(transferBuffer, chunkOffset + cleanUpOffset);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @Override
    protected Long createChunk() {
        Long chunkOffset = this.freeChunks.poll();
        if (chunkOffset == null) {
            long maxOffset = -1L;
            for (int i = 0; i < this.chunkList.length(); ++i) {
                Long currentOffset = (Long)this.chunkList.get(i);
                maxOffset = Math.max(maxOffset, currentOffset == null ? -1L : currentOffset);
            }
            long chunk = maxOffset == -1L ? 0L : maxOffset + (long)this.memoryChunkSize;
            try {
                ByteBuffer tmp = this.initOrGetTransferBuffer();
                this.fillBufferWithZeroes(tmp);
                MerkleDbFileUtils.completelyWrite(this.currentFileChannel, tmp, chunk);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            return chunk;
        }
        return chunkOffset;
    }

    FileChannel getCurrentFileChannel() {
        return this.currentFileChannel;
    }

    LongListDisk resetTransferBuffer() {
        TRANSFER_BUFFER_THREAD_LOCAL.remove();
        return this;
    }

    static {
        TRANSFER_BUFFER_THREAD_LOCAL = new ThreadLocal();
        TEMP_LONG_BUFFER_THREAD_LOCAL = ThreadLocal.withInitial(() -> ByteBuffer.allocate(8).order(ByteOrder.nativeOrder()));
    }
}

