/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.files;

import com.hedera.pbj.runtime.FieldDefinition;
import com.hedera.pbj.runtime.FieldType;
import com.swirlds.base.formatting.HorizontalAlignment;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.merkledb.KeyRange;
import com.swirlds.merkledb.collections.IndexedObject;
import com.swirlds.merkledb.collections.LongList;
import com.swirlds.merkledb.files.DataFileCollection;
import com.swirlds.merkledb.files.DataFileReader;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;

public final class DataFileCommon {
    private static final Logger logger = LogManager.getLogger(DataFileCommon.class);
    private static final int ROUNDING_SCALE_FACTOR = 100;
    private static final int PRINTED_INDEX_FIELD_WIDTH = 10;
    public static final long NON_EXISTENT_DATA_LOCATION = 0L;
    private static final int DATA_ITEM_OFFSET_BITS = 40;
    private static final long MAX_ADDRESSABLE_DATA_FILE_SIZE_BYTES = 0x10000000000L;
    private static final long ITEM_OFFSET_MASK = 0xFFFFFFFFFFL;
    private static final DateTimeFormatter DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss-SSS").withZone(ZoneId.of("Z"));
    public static final String FILE_EXTENSION = ".pbj";
    public static final int PAGE_SIZE = 4096;
    static final FieldDefinition FIELD_DATAFILE_METADATA = new FieldDefinition("metadata", FieldType.MESSAGE, false, false, false, 1);
    static final FieldDefinition FIELD_DATAFILE_ITEMS = new FieldDefinition("items", FieldType.MESSAGE, true, true, false, 11);

    private DataFileCommon() {
        throw new IllegalStateException("Utility class; should not be instantiated.");
    }

    static Path createDataFilePath(String filePrefix, Path dataFileDir, int index, Instant creationInstant, String extension) {
        return dataFileDir.resolve(filePrefix + "_" + DATE_FORMAT.format(creationInstant) + "_" + HorizontalAlignment.ALIGNED_RIGHT.pad(Integer.toString(index), '_', 10, false) + extension);
    }

    public static long dataLocation(int fileIndex, long byteOffset) {
        long indexShifted = (long)(fileIndex + 1) << 40;
        long byteOffsetMasked = byteOffset & 0xFFFFFFFFFFL;
        return indexShifted | byteOffsetMasked;
    }

    public static String dataLocationToString(long dataLocation) {
        if (dataLocation <= 0L) {
            return String.valueOf(dataLocation);
        }
        return "{" + DataFileCommon.fileIndexFromDataLocation(dataLocation) + "," + DataFileCommon.byteOffsetFromDataLocation(dataLocation) + "}";
    }

    public static int fileIndexFromDataLocation(long dataLocation) {
        return (int)(dataLocation >> 40) - 1;
    }

    static long byteOffsetFromDataLocation(long dataLocation) {
        return dataLocation & 0xFFFFFFFFFFL;
    }

    static boolean isFullyWrittenDataFile(String filePrefix, Path path) {
        if (filePrefix == null) {
            return false;
        }
        String fileName = path.getFileName().toString();
        return fileName.startsWith(filePrefix) && !fileName.contains("metadata") && fileName.endsWith(FILE_EXTENSION);
    }

    static <D> void printDataLinkValidation(String storeName, LongList index, Set<Integer> newFileIndexes, List<DataFileReader> fileList, KeyRange validKeyRange) {
        TreeSet<Integer> validFileIds = new TreeSet<Integer>();
        int newestFileIndex = 0;
        for (DataFileReader file : fileList) {
            int fileIndex = file.getMetadata().getIndex();
            validFileIds.add(fileIndex);
            if (fileIndex <= newestFileIndex) continue;
            newestFileIndex = fileIndex;
        }
        int finalNewestFileIndex = newestFileIndex;
        ConcurrentMap<Integer, Long> missingFileCounts = LongStream.range(validKeyRange.getMinValidKey(), validKeyRange.getMaxValidKey() + 1L).parallel().map(key -> index.get(key, 0L)).filter(location -> location != 0L).filter(location -> {
            int fileIndex = DataFileCommon.fileIndexFromDataLocation(location);
            return !validFileIds.contains(fileIndex) && fileIndex <= finalNewestFileIndex && !newFileIndexes.contains(fileIndex);
        }).boxed().collect(Collectors.groupingByConcurrent(DataFileCommon::fileIndexFromDataLocation, Collectors.counting()));
        if (!missingFileCounts.isEmpty()) {
            Supplier[] supplierArray = new Supplier[5];
            supplierArray[0] = () -> storeName;
            supplierArray[1] = index::size;
            supplierArray[2] = fileList::size;
            supplierArray[3] = () -> Arrays.toString(fileList.stream().mapToInt(IndexedObject::getIndex).toArray());
            supplierArray[4] = () -> newFileIndexes;
            logger.trace(LogMarker.MERKLE_DB.getMarker(), "{}:printDataLinkValidation index size={} numOfFiles={}, fileIndexes={}, newFileIndexes={}", supplierArray);
            missingFileCounts.forEach((id, count) -> logger.trace(LogMarker.MERKLE_DB.getMarker(), "{}:       missing file {} has {} references", (Object)storeName, id, count));
            logger.error(LogMarker.EXCEPTION.getMarker(), "{} has references to files {} that don't exists in the index. Latest new files = {}", (Object)storeName, (Object)missingFileCounts.keySet().toString(), newFileIndexes);
        }
    }

    public static long getSizeOfFilesByPath(Iterable<Path> filePaths) throws IOException {
        long totalSize = 0L;
        for (Path path : filePaths) {
            totalSize += Files.size(path);
        }
        return totalSize;
    }

    public static long getSizeOfFiles(Iterable<? extends DataFileReader> fileReaders) {
        long totalSize = 0L;
        for (DataFileReader dataFileReader : fileReaders) {
            totalSize += dataFileReader.getSize();
        }
        return totalSize;
    }

    public static String formatSizeBytes(long numOfBytes) {
        if (numOfBytes <= 1024L) {
            return numOfBytes + " bytes";
        }
        return DataFileCommon.formatLargeDenomBytes(numOfBytes);
    }

    private static String formatLargeDenomBytes(long numOfBytes) {
        if (numOfBytes > 0x40000000L) {
            double numOfGb = (double)numOfBytes / 1.073741824E9;
            return DataFileCommon.roundTwoDecimals(numOfGb) + " GB";
        }
        if (numOfBytes > 0x100000L) {
            double numOfMb = (double)numOfBytes / 1048576.0;
            return DataFileCommon.roundTwoDecimals(numOfMb) + " MB";
        }
        double numOfKb = (double)numOfBytes / 1024.0;
        return DataFileCommon.roundTwoDecimals(numOfKb) + " KB";
    }

    public static double roundTwoDecimals(double d) {
        return (double)Math.round(d * 100.0) / 100.0;
    }

    public static void logCompactStats(String storeName, double tookMillis, Collection<? extends DataFileReader> filesToMerge, long filesToMergeSize, List<Path> mergedFiles, int targetCompactionLevel, DataFileCollection fileCollection) throws IOException {
        long mergedFilesCount = mergedFiles.size();
        long mergedFilesSize = DataFileCommon.getSizeOfFilesByPath(mergedFiles);
        double tookSeconds = tookMillis / 1000.0;
        String levelsCompacted = filesToMerge.stream().map(v -> v.getMetadata().getCompactionLevel()).distinct().map(v -> Integer.toString(v)).sorted().collect(Collectors.joining(","));
        Object[] fileToMergeIndexes = filesToMerge.stream().map(reader -> reader.getMetadata().getIndex()).toArray();
        Object[] allFileIndexes = fileCollection.getAllCompletedFiles().stream().map(reader -> reader.getMetadata().getIndex()).toArray();
        logger.info(LogMarker.MERKLE_DB.getMarker(), "[{}] Compacted {} file(s) / {} at level {} into {} file(s) of level {} / {} in {} second(s)\n        effectively read at {} effectively written at {},\n        compactedFiles[{}] = {},\n        filesToMerge[{}] = {}\n        allFilesAfter[{}] = {}", new Object[]{storeName, filesToMerge.size(), DataFileCommon.formatSizeBytes(filesToMergeSize), levelsCompacted, mergedFilesCount, targetCompactionLevel, DataFileCommon.formatSizeBytes(mergedFilesSize), tookSeconds, DataFileCommon.formatSizeBytes((long)((double)filesToMergeSize / tookSeconds)) + "/sec", DataFileCommon.formatSizeBytes((long)((double)mergedFilesSize / tookSeconds)) + "/sec", mergedFilesCount, Arrays.toString(mergedFiles.stream().map(Path::getFileName).toArray()), fileToMergeIndexes.length, Arrays.toString(fileToMergeIndexes), allFileIndexes.length, Arrays.toString(allFileIndexes)});
    }

    public static void deleteDirectoryAndContents(Path dir) {
        if (Files.exists(dir, new LinkOption[0]) && Files.isDirectory(dir, new LinkOption[0])) {
            try (Stream<Path> filesStream = Files.walk(dir, new FileVisitOption[0]);){
                filesStream.sorted(Comparator.reverseOrder()).map(Path::toFile).forEach(File::delete);
                Files.deleteIfExists(dir);
                logger.info(LogMarker.MERKLE_DB.getMarker(), "Deleted data directory [{}]", (Object)dir.toFile().getAbsolutePath());
            }
            catch (Exception e) {
                logger.warn(LogMarker.EXCEPTION.getMarker(), "Failed to delete test directory [{}]", (Object)dir.toFile().getAbsolutePath(), (Object)e);
            }
        }
    }
}

