/*
 * Decompiled with CFR 0.152.
 */
package org.hyperledger.besu.evm.processor;

import com.google.common.annotations.VisibleForTesting;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.hyperledger.besu.datatypes.Address;
import org.hyperledger.besu.datatypes.Wei;
import org.hyperledger.besu.evm.EVM;
import org.hyperledger.besu.evm.ModificationNotAllowedException;
import org.hyperledger.besu.evm.account.MutableAccount;
import org.hyperledger.besu.evm.frame.ExceptionalHaltReason;
import org.hyperledger.besu.evm.frame.MessageFrame;
import org.hyperledger.besu.evm.precompile.PrecompileContractRegistry;
import org.hyperledger.besu.evm.precompile.PrecompiledContract;
import org.hyperledger.besu.evm.processor.AbstractMessageProcessor;
import org.hyperledger.besu.evm.tracing.OperationTracer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageCallProcessor
extends AbstractMessageProcessor {
    private static final Logger LOG = LoggerFactory.getLogger(MessageCallProcessor.class);
    protected final PrecompileContractRegistry precompiles;

    public MessageCallProcessor(EVM evm, PrecompileContractRegistry precompiles, Collection<Address> forceCommitAddresses) {
        super(evm, forceCommitAddresses);
        this.precompiles = precompiles;
    }

    public MessageCallProcessor(EVM evm, PrecompileContractRegistry precompiles) {
        super(evm, Set.of());
        this.precompiles = precompiles;
    }

    @Override
    public void start(MessageFrame frame, OperationTracer operationTracer) {
        LOG.trace("Executing message-call");
        try {
            this.transferValue(frame);
            PrecompiledContract precompile = this.precompiles.get(frame.getContractAddress());
            if (precompile != null) {
                this.executePrecompile(precompile, frame, operationTracer);
            } else {
                frame.setState(MessageFrame.State.CODE_EXECUTING);
            }
        }
        catch (ModificationNotAllowedException ex) {
            LOG.trace("Message call error: attempt to mutate an immutable account");
            frame.setExceptionalHaltReason(Optional.of(ExceptionalHaltReason.ILLEGAL_STATE_CHANGE));
            frame.setState(MessageFrame.State.EXCEPTIONAL_HALT);
        }
    }

    @Override
    protected void codeSuccess(MessageFrame frame, OperationTracer operationTracer) {
        LOG.trace("Successful message call of {} to {} (Gas remaining: {})", new Object[]{frame.getSenderAddress(), frame.getRecipientAddress(), frame.getRemainingGas()});
        frame.setState(MessageFrame.State.COMPLETED_SUCCESS);
    }

    private void transferValue(MessageFrame frame) {
        MutableAccount senderAccount = frame.getWorldUpdater().getSenderAccount(frame);
        MutableAccount recipientAccount = frame.getWorldUpdater().getOrCreate(frame.getRecipientAddress());
        if (Objects.equals(frame.getValue(), Wei.ZERO)) {
            LOG.trace("Message call from {} to {} has zero value: no fund transferred", (Object)frame.getSenderAddress(), (Object)frame.getRecipientAddress());
            return;
        }
        if (frame.getRecipientAddress().equals((Object)frame.getSenderAddress())) {
            LOG.trace("Message call of {} to itself: no fund transferred", (Object)frame.getSenderAddress());
        } else {
            Wei prevSenderBalance = senderAccount.decrementBalance(frame.getValue());
            Wei prevRecipientBalance = recipientAccount.incrementBalance(frame.getValue());
            LOG.trace("Transferred value {} for message call from {} ({} -> {}) to {} ({} -> {})", new Object[]{frame.getValue(), frame.getSenderAddress(), prevSenderBalance, senderAccount.getBalance(), frame.getRecipientAddress(), prevRecipientBalance, recipientAccount.getBalance()});
        }
    }

    private void executePrecompile(PrecompiledContract contract, MessageFrame frame, OperationTracer operationTracer) {
        long gasRequirement = contract.gasRequirement(frame.getInputData());
        if (frame.getRemainingGas() < gasRequirement) {
            frame.setExceptionalHaltReason(Optional.of(ExceptionalHaltReason.INSUFFICIENT_GAS));
            frame.setState(MessageFrame.State.EXCEPTIONAL_HALT);
            operationTracer.tracePrecompileCall(frame, gasRequirement, null);
        } else {
            frame.decrementRemainingGas(gasRequirement);
            PrecompiledContract.PrecompileContractResult result = contract.computePrecompile(frame.getInputData(), frame);
            operationTracer.tracePrecompileCall(frame, gasRequirement, result.getOutput());
            if (result.isRefundGas()) {
                frame.incrementRemainingGas(gasRequirement);
            }
            if (frame.getState() == MessageFrame.State.REVERT) {
                frame.setRevertReason(result.getOutput());
            } else {
                frame.setOutputData(result.getOutput());
            }
            frame.setState(result.getState());
            frame.setExceptionalHaltReason(result.getHaltReason());
        }
    }

    @VisibleForTesting
    public Set<Address> getPrecompileAddresses() {
        return this.precompiles.getPrecompileAddresses();
    }
}

