/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.hints.impl;

import com.hedera.hapi.node.state.hints.HintsConstruction;
import com.hedera.hapi.node.state.hints.NodePartyId;
import com.hedera.hapi.node.state.hints.PreprocessedKeys;
import com.hedera.hapi.services.auxiliary.hints.HintsPartialSignatureTransactionBody;
import com.hedera.node.app.hints.HintsLibrary;
import com.hedera.node.config.data.TssConfig;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import com.swirlds.config.api.Configuration;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

@Singleton
public class HintsContext {
    private static final Logger log = LogManager.getLogger(HintsContext.class);
    private static final Duration SIGNING_ATTEMPT_TIMEOUT = Duration.ofDays(1L);
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final HintsLibrary library;
    private final Supplier<Configuration> configProvider;
    @Nullable
    private HintsConstruction construction;
    @Nullable
    private Map<Long, Integer> nodePartyIds;
    private long schemeId;

    @Inject
    public HintsContext(@NonNull HintsLibrary library, @NonNull Supplier<Configuration> configProvider) {
        this.library = Objects.requireNonNull(library);
        this.configProvider = Objects.requireNonNull(configProvider);
    }

    public void setConstruction(@NonNull HintsConstruction construction) {
        Objects.requireNonNull(construction);
        if (!construction.hasHintsScheme()) {
            throw new IllegalArgumentException("Given construction #" + construction.constructionId() + " has no hinTS scheme");
        }
        this.construction = Objects.requireNonNull(construction);
        this.nodePartyIds = HintsContext.asNodePartyIds(construction.hintsSchemeOrThrow().nodePartyIds());
        this.schemeId = this.construction.constructionId();
    }

    public boolean isReady() {
        return this.construction != null && this.construction.hasHintsScheme();
    }

    public long activeSchemeIdOrThrow() {
        if (this.schemeId == 0L) {
            throw new IllegalStateException("No scheme id set");
        }
        return this.schemeId;
    }

    public Bytes verificationKeyOrThrow() {
        this.throwIfNotReady();
        return Objects.requireNonNull(this.construction).hintsSchemeOrThrow().preprocessedKeysOrThrow().verificationKey();
    }

    public long constructionIdOrThrow() {
        this.throwIfNotReady();
        return Objects.requireNonNull(this.construction).constructionId();
    }

    @Nullable
    public HintsConstruction activeConstruction() {
        return this.construction;
    }

    public boolean validate(long nodeId, @Nullable Bytes crs, @NonNull HintsPartialSignatureTransactionBody body) {
        Objects.requireNonNull(crs);
        if (this.construction == null || this.nodePartyIds == null) {
            return false;
        }
        if (this.construction.constructionId() == body.constructionId() && this.nodePartyIds.containsKey(nodeId)) {
            PreprocessedKeys preprocessedKeys = this.construction.hintsSchemeOrThrow().preprocessedKeysOrThrow();
            Bytes aggregationKey = preprocessedKeys.aggregationKey();
            Integer partyId = this.nodePartyIds.get(nodeId);
            return this.library.verifyBls(crs, body.partialSignature(), body.message(), aggregationKey, partyId);
        }
        return false;
    }

    @NonNull
    public Signing newSigning(@NonNull Bytes blockHash, @NonNull Runnable onCompletion) {
        Objects.requireNonNull(blockHash);
        Objects.requireNonNull(onCompletion);
        this.throwIfNotReady();
        Objects.requireNonNull(this.construction);
        PreprocessedKeys preprocessedKeys = this.construction.hintsSchemeOrThrow().preprocessedKeysOrThrow();
        Bytes verificationKey = preprocessedKeys.verificationKey();
        HashMap<Long, Long> nodeWeights = new HashMap<Long, Long>();
        long totalWeight = 0L;
        for (NodePartyId nodePartyId : this.construction.hintsSchemeOrThrow().nodePartyIds()) {
            totalWeight += nodePartyId.partyWeight();
            nodeWeights.put(nodePartyId.nodeId(), nodePartyId.partyWeight());
        }
        TssConfig tssConfig = (TssConfig)this.configProvider.get().getConfigData(TssConfig.class);
        int divisor = Math.max(1, tssConfig.signingThresholdDivisor());
        long threshold = totalWeight / (long)divisor;
        return new Signing(blockHash, threshold, preprocessedKeys.aggregationKey(), Objects.requireNonNull(this.nodePartyIds), nodeWeights, verificationKey, onCompletion);
    }

    private static Map<Long, Integer> asNodePartyIds(@NonNull List<NodePartyId> nodePartyIds) {
        return nodePartyIds.stream().collect(Collectors.toMap(NodePartyId::nodeId, NodePartyId::partyId));
    }

    private void throwIfNotReady() {
        if (!this.isReady()) {
            throw new IllegalStateException("Signing context not ready");
        }
    }

    public class Signing {
        private final long thresholdWeight;
        private final Bytes aggregationKey;
        private final Bytes verificationKey;
        private final Map<Long, Long> nodeWeights;
        private final Map<Long, Integer> partyIds;
        private final CompletableFuture<Bytes> future = new CompletableFuture();
        private final ConcurrentMap<Integer, Bytes> signatures = new ConcurrentHashMap<Integer, Bytes>();
        private final AtomicLong weightOfSignatures = new AtomicLong();
        private final AtomicBoolean completed = new AtomicBoolean();

        public Signing(Bytes blockHash, @NonNull long thresholdWeight, @NonNull Bytes aggregationKey, @NonNull Map<Long, Integer> partyIds, @NonNull Map<Long, Long> nodeWeights, @NonNull Bytes verificationKey, Runnable onCompletion) {
            this.thresholdWeight = thresholdWeight;
            Objects.requireNonNull(onCompletion);
            this.aggregationKey = Objects.requireNonNull(aggregationKey);
            this.partyIds = Objects.requireNonNull(partyIds);
            this.nodeWeights = Objects.requireNonNull(nodeWeights);
            this.verificationKey = Objects.requireNonNull(verificationKey);
            HintsContext.this.executor.schedule(() -> {
                if (!this.future.isDone()) {
                    log.warn("Completing signing attempt on '{}' without obtaining a signature (had {} from parties {} for total weight {}/{} required)", (Object)blockHash, (Object)this.signatures.size(), this.signatures.keySet(), (Object)this.weightOfSignatures.get(), (Object)thresholdWeight);
                }
                onCompletion.run();
            }, SIGNING_ATTEMPT_TIMEOUT.getSeconds(), TimeUnit.SECONDS);
        }

        public CompletableFuture<Bytes> future() {
            return this.future;
        }

        public Bytes verificationKey() {
            return this.verificationKey;
        }

        public void incorporateValid(@NonNull Bytes crs, long nodeId, @NonNull Bytes signature) {
            boolean reachedThreshold;
            Objects.requireNonNull(crs);
            Objects.requireNonNull(signature);
            if (this.completed.get()) {
                return;
            }
            Integer partyId = this.partyIds.get(nodeId);
            if (this.signatures.put(partyId, signature) != null) {
                return;
            }
            Long weight = this.nodeWeights.getOrDefault(nodeId, 0L);
            long totalWeight = this.weightOfSignatures.addAndGet(weight);
            boolean bl = reachedThreshold = totalWeight > this.thresholdWeight;
            if (reachedThreshold && this.completed.compareAndSet(false, true)) {
                Bytes aggregatedSignature = HintsContext.this.library.aggregateSignatures(crs, this.aggregationKey, this.verificationKey, this.signatures);
                this.future.complete(aggregatedSignature);
            }
        }
    }
}

