/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.service.networkadmin.impl.handlers;

import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Objects;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class UnzipUtility {
    private static final Logger log = LogManager.getLogger(UnzipUtility.class);
    private static final int BUFFER_SIZE = 4096;
    private static final int MAX_ENTRIES = 10000;
    private static final long MAX_TOTAL_EXTRACTED_BYTES = 0x280000000L;
    private static final int MAX_DEPTH = 20;
    private static final int MAX_NAME_LENGTH = 4096;
    private static final long MAX_ENTRY_BYTES = 0x80000000L;

    private UnzipUtility() {
    }

    public static void unzip(@NonNull byte[] bytes, @NonNull Path dstDir) throws IOException {
        Objects.requireNonNull(bytes);
        Objects.requireNonNull(dstDir);
        try (ZipInputStream zipIn = new ZipInputStream(new ByteArrayInputStream(bytes));){
            Path resolvedDstDir = dstDir.toAbsolutePath().normalize();
            Path canonicalDstDir = resolvedDstDir.toFile().getCanonicalFile().toPath();
            ZipEntry entry = zipIn.getNextEntry();
            if (entry == null) {
                throw new IOException("No zip entry found in bytes");
            }
            int entryCount = 0;
            long totalExtractedBytes = 0L;
            while (entry != null) {
                if (++entryCount > 10000) {
                    throw new IOException("Zip archive contains too many entries (>10000)");
                }
                String entryName = entry.getName();
                UnzipUtility.validateEntryName(entryName);
                int depth = UnzipUtility.normalizedRelativePathDepth(entryName);
                if (depth > 20) {
                    throw new IOException("Zip entry is nested too deeply (depth=" + depth + ", max=20): " + entryName);
                }
                Path filePath = resolvedDstDir.resolve(entryName).normalize();
                if (!filePath.startsWith(resolvedDstDir)) {
                    throw new IOException("Zip file entry is outside of the destination directory: " + String.valueOf(filePath));
                }
                File fileOrDir = filePath.toFile();
                Path canonicalFilePath = fileOrDir.getCanonicalFile().toPath();
                if (!canonicalFilePath.startsWith(canonicalDstDir)) {
                    throw new IOException("Zip file entry is outside of the destination directory: " + String.valueOf(filePath));
                }
                File directory = fileOrDir.getParentFile();
                if (!directory.exists() && !directory.mkdirs()) {
                    throw new IOException("Unable to create the parent directories for the file: " + String.valueOf(fileOrDir));
                }
                if (!entry.isDirectory()) {
                    long remainingTotal = 0x280000000L - totalExtractedBytes;
                    long written = UnzipUtility.extractSingleFileWithLimits(zipIn, filePath, 0x80000000L, remainingTotal);
                    totalExtractedBytes += written;
                    log.info(" - Extracted update file {}", (Object)filePath);
                } else {
                    if (!fileOrDir.exists() && !fileOrDir.mkdirs()) {
                        throw new IOException("Unable to create assets sub-directory: " + String.valueOf(fileOrDir));
                    }
                    log.info(" - Created assets sub-directory {}", (Object)fileOrDir);
                }
                zipIn.closeEntry();
                entry = zipIn.getNextEntry();
            }
        }
    }

    public static void extractSingleFile(@NonNull ZipInputStream inputStream, @NonNull Path filePath) throws IOException {
        Objects.requireNonNull(inputStream);
        Objects.requireNonNull(filePath);
        UnzipUtility.extractSingleFileWithLimits(inputStream, filePath, 0x80000000L, Long.MAX_VALUE);
    }

    private static long extractSingleFileWithLimits(@NonNull ZipInputStream inputStream, @NonNull Path filePath, long maxEntryBytes, long remainingTotalBytes) throws IOException {
        long written = 0L;
        try (BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(filePath.toFile()));){
            int read;
            byte[] bytesIn = new byte[4096];
            while ((read = inputStream.read(bytesIn)) != -1) {
                if (written + (long)read > maxEntryBytes) {
                    throw new IOException("Zip entry exceeds max allowed size (" + maxEntryBytes + " bytes): " + String.valueOf(filePath));
                }
                if (written + (long)read > remainingTotalBytes) {
                    throw new IOException("Zip archive exceeds max allowed extracted size (10737418240 bytes): " + String.valueOf(filePath));
                }
                bos.write(bytesIn, 0, read);
                written += (long)read;
            }
        }
        catch (IOException e) {
            try {
                File f = filePath.toFile();
                if (f.exists() && !f.delete()) {
                    log.warn("Unable to delete partially extracted file {}", (Object)filePath);
                }
            }
            catch (Exception exception) {
                // empty catch block
            }
            throw e;
        }
        return written;
    }

    private static void validateEntryName(String name) throws IOException {
        if (name == null || name.isEmpty()) {
            throw new IOException("Zip entry has an empty name");
        }
        if (name.length() > 4096) {
            throw new IOException("Zip entry name is too long (>4096 chars)");
        }
        for (int i = 0; i < name.length(); ++i) {
            char c = name.charAt(i);
            if (c != '\u0000' && !Character.isISOControl(c)) continue;
            throw new IOException("Zip entry name contains a control character");
        }
        if (name.startsWith("/") || name.startsWith("\\") || name.matches("^[A-Za-z]:[\\\\/].*")) {
            throw new IOException("Zip entry name is an absolute path");
        }
        if (name.indexOf(92) != -1) {
            throw new IOException("Zip entry name contains invalid path separator");
        }
        if (name.equals("..") || name.startsWith("../") || name.contains("/../") || name.endsWith("/..")) {
            throw new IOException("Zip entry name contains path traversal");
        }
    }

    private static int normalizedRelativePathDepth(String entryName) throws IOException {
        Path normalized = Path.of(entryName, new String[0]).normalize();
        if (normalized.isAbsolute()) {
            throw new IOException("Zip entry name is an absolute path");
        }
        return normalized.getNameCount();
    }
}

