/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.concurrency.limits;

import io.helidon.common.concurrency.limits.AimdLimitConfig;
import io.helidon.common.concurrency.limits.IgnoreTaskException;
import io.helidon.common.concurrency.limits.LimitAlgorithm;
import io.helidon.common.concurrency.limits.LimitException;
import io.helidon.common.concurrency.limits.LimitHandlers;
import io.helidon.common.config.ConfigException;
import io.helidon.metrics.api.Gauge;
import io.helidon.metrics.api.Meter;
import io.helidon.metrics.api.MeterRegistry;
import io.helidon.metrics.api.Metrics;
import io.helidon.metrics.api.MetricsFactory;
import io.helidon.metrics.api.Tag;
import io.helidon.metrics.api.Timer;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

class AimdLimitImpl {
    private final double backoffRatio;
    private final long timeoutInNanos;
    private final int minLimit;
    private final int maxLimit;
    private final Supplier<Long> clock;
    private final AtomicInteger concurrentRequests;
    private final AtomicInteger rejectedRequests;
    private final AdjustableSemaphore semaphore;
    private final LimitHandlers.LimiterHandler handler;
    private final AtomicInteger limit;
    private final Lock limitLock = new ReentrantLock();
    private final int queueLength;
    private Timer rttTimer;
    private Timer queueWaitTimer;

    AimdLimitImpl(AimdLimitConfig config) {
        int initialLimit = config.initialLimit();
        this.backoffRatio = config.backoffRatio();
        this.timeoutInNanos = config.timeout().toNanos();
        this.minLimit = config.minLimit();
        this.maxLimit = config.maxLimit();
        this.clock = config.clock().orElseGet(() -> System::nanoTime);
        this.concurrentRequests = new AtomicInteger();
        this.rejectedRequests = new AtomicInteger();
        this.limit = new AtomicInteger(initialLimit);
        this.queueLength = config.queueLength();
        this.semaphore = new AdjustableSemaphore(initialLimit, config.fair());
        this.handler = new LimitHandlers.QueuedSemaphoreHandler(this.semaphore, this.queueLength, config.queueTimeout(), () -> new AimdToken(this.clock, this.concurrentRequests));
        if (!(this.backoffRatio < 1.0) || !(this.backoffRatio >= 0.5)) {
            throw new ConfigException("Backoff ratio must be within [0.5, 1.0)");
        }
        if (this.maxLimit < this.minLimit) {
            throw new ConfigException("Max limit must be higher than min limit, or equal to it");
        }
        if (initialLimit > this.maxLimit) {
            throw new ConfigException("Initial limit must be lower than max limit, or equal to it");
        }
        if (initialLimit < this.minLimit) {
            throw new ConfigException("Initial limit must be higher than minimum limit, or equal to it");
        }
    }

    Semaphore semaphore() {
        return this.semaphore;
    }

    int currentLimit() {
        return this.limit.get();
    }

    Optional<LimitAlgorithm.Token> tryAcquire(boolean wait) {
        Optional<LimitAlgorithm.Token> token = this.handler.tryAcquire(false);
        if (token.isPresent()) {
            return token;
        }
        if (wait && this.queueLength > 0) {
            long startWait = this.clock.get();
            token = this.handler.tryAcquire(true);
            if (token.isPresent()) {
                if (this.queueWaitTimer != null) {
                    this.queueWaitTimer.record(this.clock.get() - startWait, TimeUnit.NANOSECONDS);
                }
                return token;
            }
        }
        this.rejectedRequests.getAndIncrement();
        return token;
    }

    void invoke(Runnable runnable) throws Exception {
        this.invoke(() -> {
            runnable.run();
            return null;
        });
    }

    <T> T invoke(Callable<T> callable) throws Exception {
        Optional<LimitAlgorithm.Token> optionalToken = this.tryAcquire(true);
        if (optionalToken.isPresent()) {
            LimitAlgorithm.Token token = optionalToken.get();
            try {
                T response = callable.call();
                token.success();
                return response;
            }
            catch (IgnoreTaskException e) {
                token.ignore();
                return e.handle();
            }
            catch (Throwable e) {
                token.dropped();
                throw e;
            }
        }
        throw new LimitException("No more permits available for the semaphore");
    }

    void updateWithSample(long startTime, long endTime, int currentRequests, boolean success) {
        long rtt = endTime - startTime;
        if (this.rttTimer != null) {
            this.rttTimer.record(rtt, TimeUnit.NANOSECONDS);
        }
        int currentLimit = this.limit.get();
        if (rtt > this.timeoutInNanos || !success) {
            currentLimit = (int)((double)currentLimit * this.backoffRatio);
        } else if (currentRequests * 2 >= currentLimit) {
            ++currentLimit;
        }
        this.setLimit(Math.min(this.maxLimit, Math.max(this.minLimit, currentLimit)));
    }

    private void setLimit(int newLimit) {
        if (newLimit == this.limit.get()) {
            return;
        }
        this.limitLock.lock();
        try {
            int oldLimit = this.limit.get();
            if (oldLimit == newLimit) {
                return;
            }
            this.limit.set(newLimit);
            if (newLimit > oldLimit) {
                this.semaphore.release(newLimit - oldLimit);
            } else {
                this.semaphore.reducePermits(oldLimit - newLimit);
            }
        }
        finally {
            this.limitLock.unlock();
        }
    }

    void initMetrics(String socketName, AimdLimitConfig config) {
        if (config.enableMetrics()) {
            MetricsFactory metricsFactory = MetricsFactory.getInstance();
            MeterRegistry meterRegistry = Metrics.globalRegistry();
            Tag socketNameTag = null;
            if (!socketName.equals("@default")) {
                socketNameTag = Tag.create((String)"socketName", (String)socketName);
            }
            Gauge.Builder limitBuilder = (Gauge.Builder)metricsFactory.gaugeBuilder(config.name() + "_limit", this.limit::get).scope("vendor");
            if (socketNameTag != null) {
                limitBuilder.tags(List.of(socketNameTag));
            }
            meterRegistry.getOrCreate((Meter.Builder)limitBuilder);
            Gauge.Builder concurrentRequestsBuilder = (Gauge.Builder)metricsFactory.gaugeBuilder(config.name() + "_concurrent_requests", this.concurrentRequests::get).scope("vendor");
            if (socketNameTag != null) {
                concurrentRequestsBuilder.tags(List.of(socketNameTag));
            }
            meterRegistry.getOrCreate((Meter.Builder)concurrentRequestsBuilder);
            Gauge.Builder rejectedRequestsBuilder = (Gauge.Builder)metricsFactory.gaugeBuilder(config.name() + "_rejected_requests", this.rejectedRequests::get).scope("vendor");
            if (socketNameTag != null) {
                rejectedRequestsBuilder.tags(List.of(socketNameTag));
            }
            meterRegistry.getOrCreate((Meter.Builder)rejectedRequestsBuilder);
            Gauge.Builder queueLengthBuilder = (Gauge.Builder)metricsFactory.gaugeBuilder(config.name() + "_queue_length", this.semaphore::getQueueLength).scope("vendor");
            if (socketNameTag != null) {
                queueLengthBuilder.tags(List.of(socketNameTag));
            }
            meterRegistry.getOrCreate((Meter.Builder)queueLengthBuilder);
            Timer.Builder rttTimerBuilder = (Timer.Builder)((Timer.Builder)metricsFactory.timerBuilder(config.name() + "_rtt").scope("vendor")).baseUnit("milliseconds");
            if (socketNameTag != null) {
                rttTimerBuilder.tags(List.of(socketNameTag));
            }
            this.rttTimer = (Timer)meterRegistry.getOrCreate((Meter.Builder)rttTimerBuilder);
            Timer.Builder waitTimerBuilder = (Timer.Builder)((Timer.Builder)metricsFactory.timerBuilder(config.name() + "_queue_wait_time").scope("vendor")).baseUnit("milliseconds");
            if (socketNameTag != null) {
                waitTimerBuilder.tags(List.of(socketNameTag));
            }
            this.queueWaitTimer = (Timer)meterRegistry.getOrCreate((Meter.Builder)waitTimerBuilder);
        }
    }

    private static final class AdjustableSemaphore
    extends Semaphore {
        private static final long serialVersionUID = 114L;

        private AdjustableSemaphore(int permits, boolean fair) {
            super(permits, fair);
        }

        @Override
        protected void reducePermits(int reduction) {
            super.reducePermits(reduction);
        }
    }

    private class AimdToken
    implements LimitAlgorithm.Token {
        private final long startTime;
        private final int currentRequests;

        private AimdToken(Supplier<Long> clock, AtomicInteger concurrentRequests) {
            this.startTime = clock.get();
            this.currentRequests = concurrentRequests.incrementAndGet();
        }

        @Override
        public void dropped() {
            try {
                AimdLimitImpl.this.updateWithSample(this.startTime, AimdLimitImpl.this.clock.get(), this.currentRequests, false);
            }
            finally {
                AimdLimitImpl.this.semaphore.release();
            }
        }

        @Override
        public void ignore() {
            AimdLimitImpl.this.concurrentRequests.decrementAndGet();
            AimdLimitImpl.this.semaphore.release();
        }

        @Override
        public void success() {
            try {
                AimdLimitImpl.this.updateWithSample(this.startTime, AimdLimitImpl.this.clock.get(), this.currentRequests, true);
                AimdLimitImpl.this.concurrentRequests.decrementAndGet();
            }
            finally {
                AimdLimitImpl.this.semaphore.release();
            }
        }
    }
}

