/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.collections;

import com.swirlds.base.utility.ToStringBuilder;
import com.swirlds.config.api.Configuration;
import com.swirlds.merkledb.collections.HashList;
import com.swirlds.merkledb.collections.OffHeapUser;
import com.swirlds.merkledb.config.MerkleDbConfig;
import com.swirlds.merkledb.utilities.HashTools;
import com.swirlds.merkledb.utilities.MerkleDbFileUtils;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicLong;
import org.hiero.base.crypto.Hash;
import org.hiero.base.utility.MemoryUtils;

public final class HashListByteBuffer
implements HashList,
OffHeapUser {
    static final int FILE_FORMAT_VERSION_V1 = 1;
    static final int FILE_FORMAT_VERSION_V2 = 2;
    static final int FILE_HEADER_SIZE_V1 = 33;
    static final int FILE_HEADER_SIZE_V2 = 16;
    private final List<ByteBuffer> data = new CopyOnWriteArrayList<ByteBuffer>();
    private final long capacity;
    private final AtomicLong size = new AtomicLong(0L);
    private final int hashesPerBuffer;
    private final int memoryBufferSize;
    private final boolean offHeap;

    public HashListByteBuffer(@NonNull Path file, long capacity, @NonNull Configuration configuration) throws IOException {
        Objects.requireNonNull(file);
        Objects.requireNonNull(configuration);
        MerkleDbConfig merkleDbConfig = (MerkleDbConfig)configuration.getConfigData(MerkleDbConfig.class);
        if (!Files.exists(file, new LinkOption[0])) {
            throw new IOException("Cannot load hash list, file doesn't exist: " + String.valueOf(file.toAbsolutePath()));
        }
        try (FileChannel fc = FileChannel.open(file, StandardOpenOption.READ);){
            ByteBuffer versionBuffer = ByteBuffer.allocate(4);
            if (MerkleDbFileUtils.completelyRead(fc, versionBuffer) != 4) {
                throw new IOException("Failed to read hash list file version");
            }
            int formatVersion = versionBuffer.getInt(0);
            this.capacity = capacity;
            this.hashesPerBuffer = merkleDbConfig.hashStoreRamBufferSize();
            this.memoryBufferSize = this.hashesPerBuffer * HashTools.HASH_SIZE_BYTES;
            this.offHeap = merkleDbConfig.hashStoreRamOffHeapBuffers();
            if (formatVersion == 1) {
                headerBuffer = ByteBuffer.allocate(33);
                MerkleDbFileUtils.completelyRead(fc, headerBuffer);
                headerBuffer.flip();
                headerBuffer.getInt();
                headerBuffer.getLong();
                headerBuffer.get();
                headerBuffer.getLong();
                this.size.set(headerBuffer.getLong());
                if (this.size.get() > capacity) {
                    throw new IllegalArgumentException("Hash list in the file is too large, size=" + this.size.get() + ", capacity=" + capacity);
                }
                headerBuffer.getInt();
            } else if (formatVersion == 2) {
                headerBuffer = ByteBuffer.allocate(16);
                if (MerkleDbFileUtils.completelyRead(fc, headerBuffer) != 16) {
                    throw new IOException("Failed to read hash list file header");
                }
                headerBuffer.clear();
                this.size.set(headerBuffer.getLong());
                long capacityFromFile = headerBuffer.getLong();
                if (capacityFromFile != capacity) {
                    throw new IllegalArgumentException("Hash list capacity mismatch, expected=" + capacity + ", loaded=" + capacityFromFile);
                }
            } else {
                throw new UnsupportedOperationException("Hash list file version " + formatVersion + " is not supported");
            }
            int numOfBuffers = Math.toIntExact(this.size.get() / (long)this.hashesPerBuffer);
            if (this.size.get() % (long)this.hashesPerBuffer != 0L) {
                ++numOfBuffers;
            }
            for (int i = 0; i < numOfBuffers; ++i) {
                ByteBuffer buffer = this.offHeap ? ByteBuffer.allocateDirect(this.memoryBufferSize) : ByteBuffer.allocate(this.memoryBufferSize);
                buffer.position(0);
                int toRead = this.memoryBufferSize;
                if (i == numOfBuffers - 1 && this.size.get() % (long)this.hashesPerBuffer != 0L) {
                    toRead = Math.toIntExact(this.size.get() % (long)this.hashesPerBuffer * (long)HashTools.HASH_SIZE_BYTES);
                }
                buffer.limit(toRead);
                int read = MerkleDbFileUtils.completelyRead(fc, buffer);
                if (read != toRead) {
                    throw new IOException("Failed to read hashes, buffer=" + i + " toRead=" + toRead + " read=" + read);
                }
                buffer.clear();
                this.data.add(buffer);
            }
        }
    }

    public HashListByteBuffer(long capacity, @NonNull Configuration configuration) {
        Objects.requireNonNull(configuration);
        MerkleDbConfig merkleDbConfig = (MerkleDbConfig)configuration.getConfigData(MerkleDbConfig.class);
        if (capacity < 0L) {
            throw new IllegalArgumentException("The maximum number of hashes must be non-negative");
        }
        this.capacity = capacity;
        this.hashesPerBuffer = merkleDbConfig.hashStoreRamBufferSize();
        this.memoryBufferSize = this.hashesPerBuffer * HashTools.HASH_SIZE_BYTES;
        this.offHeap = merkleDbConfig.hashStoreRamOffHeapBuffers();
    }

    @Override
    public void close() {
        this.size.set(0L);
        if (this.offHeap) {
            for (ByteBuffer directBuffer : this.data) {
                MemoryUtils.closeDirectByteBuffer((ByteBuffer)directBuffer);
            }
        }
        this.data.clear();
    }

    @Override
    public Hash get(long index) throws IOException {
        if (index < 0L || index >= this.capacity) {
            throw new IndexOutOfBoundsException();
        }
        if (index < this.size.get()) {
            return HashTools.byteBufferToHash(this.getBuffer(index), HashTools.getSerializationVersion());
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(long index, Hash hash) {
        if (index < 0L || index >= this.capacity) {
            throw new IndexOutOfBoundsException("Cannot put a hash at index " + index + " given " + this.capacity + " capacity");
        }
        long currentMaxIndex = (long)this.data.size() * (long)this.hashesPerBuffer - 1L;
        if (currentMaxIndex < index) {
            HashListByteBuffer hashListByteBuffer = this;
            synchronized (hashListByteBuffer) {
                for (currentMaxIndex = (long)this.data.size() * (long)this.hashesPerBuffer - 1L; currentMaxIndex < index; currentMaxIndex += (long)this.hashesPerBuffer) {
                    this.data.add(this.offHeap ? ByteBuffer.allocateDirect(this.memoryBufferSize) : ByteBuffer.allocate(this.memoryBufferSize));
                }
            }
        }
        this.size.updateAndGet(currentValue -> Math.max(currentValue, index + 1L));
        HashTools.hashToByteBuffer(hash, this.getBuffer(index));
    }

    @Override
    public long capacity() {
        return this.capacity;
    }

    @Override
    public long size() {
        return this.size.get();
    }

    @Override
    public void writeToFile(Path file) throws IOException {
        int numOfBuffers = this.data.size();
        try (FileChannel fc = FileChannel.open(file, StandardOpenOption.CREATE, StandardOpenOption.WRITE);){
            ByteBuffer headerBuffer = ByteBuffer.allocate(20);
            headerBuffer.putInt(2);
            headerBuffer.putLong(this.size.get());
            headerBuffer.putLong(this.capacity);
            headerBuffer.flip();
            assert (headerBuffer.remaining() == 20);
            if (MerkleDbFileUtils.completelyWrite(fc, headerBuffer) != headerBuffer.limit()) {
                throw new IOException("Failed to write hash list header to file");
            }
            for (int i = 0; i < numOfBuffers; ++i) {
                ByteBuffer dataBuffer = this.data.get(i).slice();
                dataBuffer.position(0);
                if (i == numOfBuffers - 1) {
                    long bytesWrittenSoFar = (long)this.memoryBufferSize * (long)i;
                    int remainingBytes = Math.toIntExact(this.size() * (long)HashTools.HASH_SIZE_BYTES - bytesWrittenSoFar);
                    dataBuffer.limit(remainingBytes);
                } else {
                    dataBuffer.limit(this.memoryBufferSize);
                }
                int toWrite = dataBuffer.limit();
                int written = MerkleDbFileUtils.completelyWrite(fc, dataBuffer);
                if (written == toWrite) continue;
                throw new IOException("Failed to write hash list data buffer to file");
            }
        }
    }

    @Override
    public long getOffHeapConsumption() {
        return this.offHeap ? (long)this.data.size() * (long)this.hashesPerBuffer * (long)HashTools.HASH_SIZE_BYTES : 0L;
    }

    private ByteBuffer getBuffer(long index) {
        assert (index >= 0L && index < (long)this.hashesPerBuffer * (long)this.data.size()) : "The index " + index + " was out of range";
        int bufferIndex = Math.toIntExact(index / (long)this.hashesPerBuffer);
        ByteBuffer buffer = this.data.get(bufferIndex).slice();
        int subIndex = Math.toIntExact(index % (long)this.hashesPerBuffer);
        int offset = HashTools.HASH_SIZE_BYTES * subIndex;
        buffer.position(offset);
        buffer.limit(offset + HashTools.HASH_SIZE_BYTES);
        return buffer;
    }

    int getCurrentBufferCount() {
        return this.data.size();
    }

    public String toString() {
        return new ToStringBuilder((Object)this).append("size", (Object)this.size.get()).append("capacity", (Object)this.capacity).append("hashesPerBuffer", (Object)this.hashesPerBuffer).append("num of buffers", (Object)this.data.size()).toString();
    }
}

