/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.workflows;

import com.hedera.hapi.node.base.AccountAmount;
import com.hedera.hapi.node.base.AccountID;
import com.hedera.hapi.node.base.FeeData;
import com.hedera.hapi.node.base.HederaFunctionality;
import com.hedera.hapi.node.base.ResponseCodeEnum;
import com.hedera.hapi.node.base.SubType;
import com.hedera.hapi.node.base.Timestamp;
import com.hedera.hapi.node.contract.ContractCallTransactionBody;
import com.hedera.hapi.node.contract.ContractCreateTransactionBody;
import com.hedera.hapi.node.contract.EthereumTransactionBody;
import com.hedera.hapi.node.state.token.Account;
import com.hedera.hapi.node.transaction.TransactionBody;
import com.hedera.hapi.util.HapiUtils;
import com.hedera.node.app.fees.ExchangeRateManager;
import com.hedera.node.app.fees.FeeManager;
import com.hedera.node.app.service.token.ReadableAccountStore;
import com.hedera.node.app.spi.authorization.Authorizer;
import com.hedera.node.app.spi.fees.Fees;
import com.hedera.node.app.spi.workflows.InsufficientBalanceException;
import com.hedera.node.app.spi.workflows.InsufficientNetworkFeeException;
import com.hedera.node.app.spi.workflows.InsufficientNonFeeDebitsException;
import com.hedera.node.app.spi.workflows.InsufficientServiceFeeException;
import com.hedera.node.app.spi.workflows.PreCheckException;
import com.hedera.node.app.store.ReadableStoreFactory;
import com.hedera.node.app.validation.ExpiryValidation;
import com.hedera.node.app.workflows.TransactionInfo;
import com.hedera.node.app.workflows.handle.dispatch.DispatchValidator;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.time.Instant;
import java.util.Objects;
import javax.inject.Inject;
import javax.inject.Singleton;

@Singleton
public class SolvencyPreCheck {
    private final ExchangeRateManager exchangeRateManager;
    private final FeeManager feeManager;
    private final ExpiryValidation expiryValidation;
    private final Authorizer authorizer;

    @Inject
    public SolvencyPreCheck(@NonNull ExchangeRateManager exchangeRateManager, @NonNull FeeManager feeManager, @NonNull ExpiryValidation expiryValidation, @NonNull Authorizer authorizer) {
        this.exchangeRateManager = Objects.requireNonNull(exchangeRateManager, "exchangeRateManager must not be null");
        this.feeManager = Objects.requireNonNull(feeManager, "feeManager must not be null");
        this.expiryValidation = Objects.requireNonNull(expiryValidation, "expiryValidation must not be null");
        this.authorizer = Objects.requireNonNull(authorizer, "authorizer must not be null");
    }

    @NonNull
    public Account getPayerAccount(@NonNull ReadableStoreFactory storeFactory, @NonNull AccountID accountID) throws PreCheckException {
        ReadableAccountStore accountStore = storeFactory.getStore(ReadableAccountStore.class);
        Account account = accountStore.getAccountById(accountID);
        if (account == null) {
            throw new PreCheckException(ResponseCodeEnum.PAYER_ACCOUNT_NOT_FOUND);
        }
        if (account.deleted()) {
            throw new PreCheckException(ResponseCodeEnum.PAYER_ACCOUNT_DELETED);
        }
        if (account.smartContract()) {
            throw new PreCheckException(ResponseCodeEnum.PAYER_ACCOUNT_NOT_FOUND);
        }
        return account;
    }

    public void checkSolvency(@NonNull TransactionInfo txInfo, @NonNull Account account, @NonNull Fees fees, @NonNull DispatchValidator.WorkflowCheck workflowCheck) throws PreCheckException {
        this.checkSolvency(txInfo.txBody(), txInfo.payerID(), txInfo.functionality(), account, fees, workflowCheck, DispatchValidator.OfferedFeeCheck.CHECK_OFFERED_FEE);
    }

    public void checkSolvency(@NonNull TransactionBody txBody, @NonNull AccountID payerID, @NonNull HederaFunctionality functionality, @NonNull Account account, @NonNull Fees fees, @NonNull DispatchValidator.WorkflowCheck workflowCheck, @NonNull DispatchValidator.OfferedFeeCheck offeredFeeCheck) throws PreCheckException {
        boolean isIngest = workflowCheck.equals((Object)DispatchValidator.WorkflowCheck.INGEST);
        boolean checkOfferedFee = offeredFeeCheck.equals((Object)DispatchValidator.OfferedFeeCheck.CHECK_OFFERED_FEE);
        if (this.authorizer.hasWaivedFees(payerID, functionality, txBody)) {
            return;
        }
        long totalFee = isIngest ? fees.totalWithoutServiceFee() : fees.totalFee();
        long availableBalance = account.tinybarBalance();
        long offeredFee = txBody.transactionFee();
        if (checkOfferedFee && offeredFee < fees.networkFee()) {
            throw new InsufficientNetworkFeeException(ResponseCodeEnum.INSUFFICIENT_TX_FEE, totalFee);
        }
        if (availableBalance < fees.networkFee()) {
            throw new InsufficientNetworkFeeException(ResponseCodeEnum.INSUFFICIENT_PAYER_BALANCE, totalFee);
        }
        if (checkOfferedFee && offeredFee < totalFee) {
            throw new InsufficientServiceFeeException(ResponseCodeEnum.INSUFFICIENT_TX_FEE, totalFee);
        }
        if (availableBalance < totalFee) {
            throw new InsufficientServiceFeeException(ResponseCodeEnum.INSUFFICIENT_PAYER_BALANCE, totalFee);
        }
        long additionalCosts = 0L;
        try {
            if (isIngest) {
                Timestamp now = txBody.transactionIDOrThrow().transactionValidStartOrThrow();
                additionalCosts = Math.max(0L, this.estimateAdditionalCosts(txBody, functionality, HapiUtils.asInstant((Timestamp)now)));
            }
        }
        catch (NullPointerException ex) {
            throw new InsufficientBalanceException(ResponseCodeEnum.INVALID_TRANSACTION_BODY, totalFee);
        }
        if (availableBalance < totalFee + additionalCosts) {
            this.expiryValidation.checkAccountExpiry(account);
            throw new InsufficientNonFeeDebitsException(ResponseCodeEnum.INSUFFICIENT_PAYER_BALANCE, totalFee);
        }
    }

    public long estimateAdditionalCosts(@NonNull TransactionBody txBody, @NonNull HederaFunctionality functionality, @NonNull Instant consensusTime) {
        return switch (functionality) {
            case HederaFunctionality.CRYPTO_CREATE -> txBody.cryptoCreateAccountOrThrow().initialBalance();
            case HederaFunctionality.CRYPTO_TRANSFER -> {
                if (!txBody.cryptoTransferOrThrow().hasTransfers()) {
                    yield 0L;
                }
                AccountID payerID = txBody.transactionIDOrThrow().accountIDOrThrow();
                yield -txBody.cryptoTransferOrThrow().transfersOrThrow().accountAmounts().stream().filter(aa -> Objects.equals(aa.accountID(), payerID)).mapToLong(AccountAmount::amount).sum();
            }
            case HederaFunctionality.CONTRACT_CREATE -> {
                ContractCreateTransactionBody contractCreate = txBody.contractCreateInstanceOrThrow();
                yield contractCreate.initialBalance() + contractCreate.gas() * this.estimatedGasPriceInTinybars(HederaFunctionality.CONTRACT_CREATE, consensusTime);
            }
            case HederaFunctionality.CONTRACT_CALL -> {
                ContractCallTransactionBody contractCall = txBody.contractCallOrThrow();
                yield contractCall.amount() + contractCall.gas() * this.estimatedGasPriceInTinybars(HederaFunctionality.CONTRACT_CALL, consensusTime);
            }
            case HederaFunctionality.ETHEREUM_TRANSACTION -> {
                EthereumTransactionBody ethTxn = txBody.ethereumTransactionOrThrow();
                yield ethTxn.maxGasAllowance();
            }
            default -> 0L;
        };
    }

    private long estimatedGasPriceInTinybars(@NonNull HederaFunctionality functionality, @NonNull Instant consensusTime) {
        FeeData feeData = this.feeManager.getFeeData(functionality, consensusTime, SubType.DEFAULT);
        long priceInTinyCents = feeData.servicedataOrThrow().gas() / 1000L;
        long priceInTinyBars = this.exchangeRateManager.getTinybarsFromTinycents(priceInTinyCents, consensusTime);
        return Math.max(priceInTinyBars, 1L);
    }
}

