/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.metadata;

import io.helidon.metadata.MetadataDiscovery;
import io.helidon.metadata.MetadataDiscoveryContext;
import io.helidon.metadata.MetadataFile;
import io.helidon.metadata.MetadataFileImpl;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.jar.Manifest;
import java.util.stream.Stream;

class MetadataDiscoveryImpl
implements MetadataDiscovery {
    static final String SYSTEM_PROPERTY_MODE = "io.helidon.metadata.mode";
    private static final System.Logger LOGGER = System.getLogger(MetadataDiscoveryImpl.class.getName());
    private final Map<String, List<MetadataFile>> metadata;

    private MetadataDiscoveryImpl(Map<String, List<MetadataFile>> metadata) {
        this.metadata = metadata;
    }

    static MetadataDiscovery create(MetadataDiscovery.Mode mode) {
        return MetadataDiscoveryImpl.create(mode, MetadataDiscoveryContext.create(MetadataDiscoveryImpl.classLoader()));
    }

    static MetadataDiscovery create(MetadataDiscovery.Mode mode, MetadataDiscoveryContext context) {
        if (mode == MetadataDiscovery.Mode.AUTO) {
            mode = MetadataDiscoveryImpl.guessMode(context);
        }
        if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
            LOGGER.log(System.Logger.Level.DEBUG, "Metadata discovery mode: " + String.valueOf((Object)mode));
        }
        return switch (mode) {
            default -> throw new MatchException(null, null);
            case MetadataDiscovery.Mode.AUTO -> throw new IllegalStateException("We have already guessed mode, this should never be reached");
            case MetadataDiscovery.Mode.RESOURCES -> MetadataDiscoveryImpl.createFromResources(context);
            case MetadataDiscovery.Mode.SCANNING -> MetadataDiscoveryImpl.createFromClasspathScanning(context);
            case MetadataDiscovery.Mode.NONE -> new MetadataDiscoveryImpl(Map.of());
        };
    }

    static MetadataDiscovery create(ClassLoader cl) {
        String modeString = System.getProperty(SYSTEM_PROPERTY_MODE, MetadataDiscovery.Mode.AUTO.name()).toUpperCase(Locale.ROOT);
        MetadataDiscovery.Mode mode = MetadataDiscovery.Mode.valueOf(modeString);
        MetadataDiscoveryContext context = MetadataDiscoveryContext.create(cl);
        return MetadataDiscoveryImpl.create(mode, context);
    }

    static MetadataDiscovery createFromClasspathScanning(MetadataDiscoveryContext ctx) {
        Set<Path> configuredClasspath = MetadataDiscoveryImpl.modules();
        HashSet<MetadataFile> metadatums = new HashSet<MetadataFile>();
        ArrayDeque<Path> pathStack = new ArrayDeque<Path>(configuredClasspath);
        String location = ctx.location();
        Set<String> metadataFiles = ctx.metadataFiles();
        if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
            LOGGER.log(System.Logger.Level.DEBUG, "Classpath configured: " + String.valueOf(configuredClasspath) + ", location: " + location + ", metadataFiles: " + String.valueOf(metadataFiles));
        }
        while (!pathStack.isEmpty()) {
            Path path = (Path)pathStack.pop();
            if (!Files.exists(path, new LinkOption[0])) continue;
            if (Files.isDirectory(path, new LinkOption[0])) {
                if (LOGGER.isLoggable(System.Logger.Level.TRACE)) {
                    LOGGER.log(System.Logger.Level.TRACE, "Analyzing directory: " + String.valueOf(path));
                }
                try {
                    Files.walkFileTree(path, new MetadataFileVisitor(location, metadataFiles, path, metadatums));
                    continue;
                }
                catch (IOException e) {
                    throw new UncheckedIOException("Failed to do classpath scanning on path: " + String.valueOf(path), e);
                }
            }
            if (MetadataDiscoveryImpl.isZipFile(path)) {
                if (LOGGER.isLoggable(System.Logger.Level.TRACE)) {
                    LOGGER.log(System.Logger.Level.TRACE, "Analyzing zip file: " + String.valueOf(path));
                }
                try (FileSystem fs = FileSystems.newFileSystem(path, Map.of());){
                    for (Path jarRoot : fs.getRootDirectories()) {
                        Manifest manifest;
                        String cp;
                        if (LOGGER.isLoggable(System.Logger.Level.TRACE)) {
                            LOGGER.log(System.Logger.Level.TRACE, "Analyzing zip root directory: " + String.valueOf(jarRoot));
                        }
                        Files.walkFileTree(jarRoot, new ZipMetadataFileVisitor(path, location, metadataFiles, jarRoot, metadatums));
                        Path mf = path.resolve("META-INF/MANIFEST.MF");
                        if (!Files.exists(mf, new LinkOption[0]) || (cp = (manifest = new Manifest(Files.newInputStream(mf, new OpenOption[0]))).getMainAttributes().getValue("Class-Path")) == null) continue;
                        Path parent = path.getParent();
                        for (String e : cp.split("\\s")) {
                            Path path0;
                            if (e.isEmpty() || !configuredClasspath.add(path0 = parent.resolve(e).toAbsolutePath().normalize())) continue;
                            pathStack.push(path0);
                        }
                    }
                }
                catch (IOException e) {
                    LOGGER.log(System.Logger.Level.WARNING, "Failed to close zip file " + String.valueOf(path), (Throwable)e);
                }
                continue;
            }
            throw new IllegalArgumentException("Invalid classpath element, neither jar nor directory: " + String.valueOf(path.toAbsolutePath()));
        }
        HashMap<String, List<MetadataFile>> metadataMap = new HashMap<String, List<MetadataFile>>();
        metadatums.forEach(it -> metadataMap.computeIfAbsent(it.fileName(), k -> new ArrayList()).add(it));
        return new MetadataDiscoveryImpl(metadataMap);
    }

    @Override
    public List<MetadataFile> list(String fileName) {
        Objects.requireNonNull(fileName, "fileName is null");
        List<MetadataFile> found = this.metadata.get(fileName);
        if (found == null) {
            return List.of();
        }
        return found;
    }

    public String toString() {
        return "MetadataImpl{metadata=" + String.valueOf(this.metadata) + "}";
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static MetadataDiscovery.Mode guessMode(MetadataDiscoveryContext ctx) {
        if (MetadataDiscoveryImpl.isMultipleJars(ctx.classLoader())) {
            LOGGER.log(System.Logger.Level.TRACE, "Multiple jars on classpath, using resource discovery");
            return MetadataDiscovery.Mode.RESOURCES;
        }
        List<URL> manifests = ctx.classLoader().resources(ctx.location() + "/" + ctx.manifestFile()).distinct().toList();
        if (manifests.isEmpty()) {
            LOGGER.log(System.Logger.Level.TRACE, "Single jar file, no manifests - using scanning");
            return MetadataDiscovery.Mode.SCANNING;
        }
        if (manifests.size() > 1) {
            LOGGER.log(System.Logger.Level.TRACE, "Multiple manifests, using resource discovery");
            return MetadataDiscovery.Mode.RESOURCES;
        }
        try (BufferedReader br = new BufferedReader(new InputStreamReader(manifests.getFirst().openStream(), StandardCharsets.UTF_8));){
            String line;
            boolean found = false;
            while ((line = br.readLine()) != null) {
                if (!"#HELIDON MANIFEST#".equals(line)) continue;
                if (found) {
                    LOGGER.log(System.Logger.Level.TRACE, "Manifest is merged, using resource discovery");
                    MetadataDiscovery.Mode mode = MetadataDiscovery.Mode.RESOURCES;
                    return mode;
                }
                found = true;
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to read manifest " + String.valueOf(manifests.getFirst()), e);
        }
        LOGGER.log(System.Logger.Level.TRACE, "Single jar file, unmerged manifest - using scanning");
        return MetadataDiscovery.Mode.SCANNING;
    }

    private static boolean isZipFile(Path path) {
        block9: {
            boolean bl;
            block10: {
                String fileName;
                Path fileNamePath = path.getFileName();
                String string = fileName = fileNamePath == null ? "" : fileNamePath.toString();
                if (fileName.endsWith(".jar") || fileName.endsWith(".zip")) {
                    return true;
                }
                if (Files.size(path) <= 2L) break block9;
                InputStream is = Files.newInputStream(path, new OpenOption[0]);
                try {
                    boolean bl2 = bl = is.read() == 80 && is.read() == 75;
                    if (is == null) break block10;
                }
                catch (Throwable throwable) {
                    try {
                        if (is != null) {
                            try {
                                is.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        LOGGER.log(System.Logger.Level.WARNING, "Failed to read magic number of " + String.valueOf(path), (Throwable)e);
                    }
                }
                is.close();
            }
            return bl;
        }
        return false;
    }

    private static Set<Path> modules() {
        HashSet<Path> modules = new HashSet<Path>();
        for (String e : System.getProperty("java.class.path", "").split(File.pathSeparator)) {
            if (e.isEmpty()) continue;
            modules.add(Path.of(e, new String[0]).toAbsolutePath().normalize());
        }
        for (String e : System.getProperty("jdk.module.path", "").split(File.pathSeparator)) {
            if (e.isEmpty()) continue;
            modules.add(Path.of(e, new String[0]).toAbsolutePath().normalize());
        }
        return modules;
    }

    private static MetadataDiscovery createFromResources(MetadataDiscoveryContext config) {
        HashMap<String, List<MetadataFile>> metadataMap = new HashMap<String, List<MetadataFile>>();
        ClassLoader cl = config.classLoader();
        String location = config.location();
        String manifestFile = config.manifestFile();
        Set<String> metadataFiles = config.metadataFiles();
        Stream manfiestStream = cl.resources(location + "/" + manifestFile).distinct().flatMap(it -> MetadataDiscoveryImpl.parseHelidonManifest(cl, it));
        Stream<MetadataFile> nonManfiestStream = MetadataDiscoveryImpl.defaultMetadata(cl, location, metadataFiles);
        Stream.concat(manfiestStream, nonManfiestStream).distinct().forEach(it -> metadataMap.computeIfAbsent(it.fileName(), k -> new ArrayList()).add(it));
        return new MetadataDiscoveryImpl(metadataMap);
    }

    private static Stream<MetadataFile> defaultMetadata(ClassLoader cl, String defaultLocation, Set<String> metadataFiles) {
        ArrayList foundFiles = new ArrayList();
        for (String metadataFile : metadataFiles) {
            String location = defaultLocation + "/" + metadataFile;
            cl.resources(location).distinct().map(url -> MetadataFileImpl.create(location, metadataFile, url)).forEach(foundFiles::add);
        }
        if (LOGGER.isLoggable(System.Logger.Level.TRACE)) {
            LOGGER.log(System.Logger.Level.TRACE, "Found " + foundFiles.size() + " metadata files directly in " + defaultLocation + ". Details in Debug (if more than 0).");
        }
        if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
            foundFiles.forEach(it -> LOGGER.log(System.Logger.Level.DEBUG, "Found metadata file: " + String.valueOf(it)));
        }
        return foundFiles.stream();
    }

    private static Stream<MetadataFile> parseHelidonManifest(ClassLoader cl, URL url) {
        Stream<MetadataFile> stream;
        if (LOGGER.isLoggable(System.Logger.Level.TRACE)) {
            LOGGER.log(System.Logger.Level.TRACE, "Parsing manifest: " + String.valueOf(url));
        }
        BufferedReader br = new BufferedReader(new InputStreamReader(url.openStream(), StandardCharsets.UTF_8));
        try {
            String line;
            ArrayList<MetadataFile> foundFiles = new ArrayList<MetadataFile>();
            int lineNumber = 0;
            while ((line = br.readLine()) != null) {
                ++lineNumber;
                if ((line = line.trim()).isEmpty() || line.startsWith("#")) continue;
                String resourceLocation = line;
                FoundFile fileName = MetadataDiscoveryImpl.fileName(line);
                URL resourceUrl = cl.getResource(resourceLocation);
                if (resourceUrl == null) {
                    throw new IllegalArgumentException("Metadata file " + resourceLocation + " not found, it is defined in manifest " + String.valueOf(url) + " at line " + lineNumber);
                }
                if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                    LOGGER.log(System.Logger.Level.DEBUG, "Adding file from manifest: " + resourceLocation + " at line " + lineNumber);
                }
                foundFiles.add(MetadataFileImpl.create(resourceLocation, fileName.fileName(), resourceUrl));
            }
            stream = foundFiles.stream();
        }
        catch (Throwable throwable) {
            try {
                try {
                    br.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                LOGGER.log(System.Logger.Level.WARNING, "Failed to read services from " + String.valueOf(url), (Throwable)e);
                return Stream.of(new MetadataFile[0]);
            }
        }
        br.close();
        return stream;
    }

    private static FoundFile fileName(String resourceLocation) {
        int lastSlash = resourceLocation.lastIndexOf(47);
        if (lastSlash == -1) {
            return new FoundFile("", resourceLocation);
        }
        String directory = resourceLocation.substring(0, lastSlash);
        String fileName = resourceLocation.substring(lastSlash + 1);
        return new FoundFile(directory, fileName);
    }

    private static boolean isMultipleJars(ClassLoader cl) {
        try {
            Enumeration<URL> enumeration = cl.getResources("META-INF/MANIFEST.MF");
            boolean found = false;
            while (enumeration.hasMoreElements()) {
                if (found) {
                    return true;
                }
                enumeration.nextElement();
                found = true;
            }
            return false;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static ClassLoader classLoader() {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        if (classLoader == null) {
            classLoader = MetadataDiscoveryImpl.class.getClassLoader();
        }
        return classLoader;
    }

    private static class MetadataFileVisitor
    extends SimpleFileVisitor<Path> {
        private final Path metaDir;
        private final Set<MetadataFile> metadata;
        private final Set<String> metadataFiles;
        private final String rootLocation;

        MetadataFileVisitor(String rootLocation, Set<String> metadataFiles, Path root, Set<MetadataFile> metadata) {
            this.rootLocation = rootLocation;
            this.metaDir = root.resolve(rootLocation);
            this.metadata = metadata;
            this.metadataFiles = metadataFiles;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
            if (this.metaDir.startsWith(dir) || dir.startsWith(this.metaDir)) {
                if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                    LOGGER.log(System.Logger.Level.DEBUG, "Visitor: continue with directory: " + String.valueOf(dir));
                }
                return FileVisitResult.CONTINUE;
            }
            if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                LOGGER.log(System.Logger.Level.DEBUG, "Visitor: ignore directory: " + String.valueOf(dir));
            }
            return FileVisitResult.SKIP_SUBTREE;
        }

        @Override
        public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
            if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                LOGGER.log(System.Logger.Level.DEBUG, "Visitor: visit file: " + String.valueOf(file));
            }
            if (file.startsWith(this.metaDir)) {
                Path fileNamePath;
                String fileName;
                if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                    LOGGER.log(System.Logger.Level.DEBUG, "Visitor: found valid location: " + String.valueOf(file));
                }
                String string = fileName = (fileNamePath = file.getFileName()) == null ? "" : fileNamePath.toString();
                if (this.metadataFiles.contains(fileName)) {
                    if (LOGGER.isLoggable(System.Logger.Level.DEBUG)) {
                        LOGGER.log(System.Logger.Level.DEBUG, "Visitor: found valid file: " + String.valueOf(file));
                    }
                    String relativePath = this.metaDir.relativize(file).toString().replace('\\', '/');
                    this.addMetadataFile(this.metadata, this.rootLocation + "/" + relativePath, fileName, file);
                }
            }
            return FileVisitResult.CONTINUE;
        }

        void addMetadataFile(Set<MetadataFile> metadata, String location, String fileName, Path file) {
            metadata.add(MetadataFileImpl.create(location, fileName, file));
        }
    }

    private static class ZipMetadataFileVisitor
    extends MetadataFileVisitor {
        private final Path zipFile;

        ZipMetadataFileVisitor(Path zipFile, String rootLocation, Set<String> metadataFiles, Path root, Set<MetadataFile> metadata) {
            super(rootLocation, metadataFiles, root, metadata);
            this.zipFile = zipFile;
        }

        @Override
        void addMetadataFile(Set<MetadataFile> metadata, String location, String fileName, Path file) {
            metadata.add(MetadataFileImpl.create(this.zipFile, location, fileName, file));
        }
    }

    private record FoundFile(String resourceDirectory, String fileName) {
    }

    static class InstanceHolder {
        private static final Lock CACHE_LOCK = new ReentrantLock();
        private static final LinkedHashMap<ClassLoader, MetadataDiscovery> CACHE = new LinkedHashMap();

        InstanceHolder() {
        }

        static MetadataDiscovery getInstance() {
            MetadataDiscovery instance;
            CACHE_LOCK.lock();
            try {
                instance = CACHE.computeIfAbsent(InstanceHolder.classLoader(), MetadataDiscoveryImpl::create);
                if (CACHE.size() > 10) {
                    Iterator<Map.Entry<ClassLoader, MetadataDiscovery>> iterator = CACHE.entrySet().iterator();
                    while (CACHE.size() > 10 && iterator.hasNext()) {
                        iterator.remove();
                    }
                }
            }
            finally {
                CACHE_LOCK.unlock();
            }
            return instance;
        }

        private static ClassLoader classLoader() {
            ClassLoader cl = Thread.currentThread().getContextClassLoader();
            if (cl == null) {
                cl = MetadataDiscoveryImpl.class.getClassLoader();
            }
            return cl;
        }
    }
}

