/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.service.token.impl.api;

import com.hedera.hapi.node.base.AccountID;
import com.hedera.hapi.node.base.ContractID;
import com.hedera.hapi.node.base.Key;
import com.hedera.hapi.node.base.ResponseCodeEnum;
import com.hedera.hapi.node.state.token.Account;
import com.hedera.hapi.node.token.CryptoTransferTransactionBody;
import com.hedera.node.app.hapi.utils.EntityType;
import com.hedera.node.app.hapi.utils.keys.KeyUtils;
import com.hedera.node.app.service.addressbook.ReadableAccountNodeRelStore;
import com.hedera.node.app.service.entityid.WritableEntityCounters;
import com.hedera.node.app.service.token.ReadableAccountStore;
import com.hedera.node.app.service.token.api.ContractChangeSummary;
import com.hedera.node.app.service.token.api.FeeStreamBuilder;
import com.hedera.node.app.service.token.api.TokenServiceApi;
import com.hedera.node.app.service.token.impl.WritableAccountStore;
import com.hedera.node.app.service.token.impl.validators.StakingValidator;
import com.hedera.node.app.spi.fees.Fees;
import com.hedera.node.app.spi.info.NetworkInfo;
import com.hedera.node.app.spi.validation.ExpiryValidator;
import com.hedera.node.app.spi.workflows.HandleException;
import com.hedera.node.app.spi.workflows.record.DeleteCapableTransactionStreamBuilder;
import com.hedera.node.app.spi.workflows.record.StreamBuilder;
import com.hedera.node.config.data.AccountsConfig;
import com.hedera.node.config.data.HederaConfig;
import com.hedera.node.config.data.LedgerConfig;
import com.hedera.node.config.data.NodesConfig;
import com.hedera.node.config.data.StakingConfig;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import com.swirlds.config.api.Configuration;
import com.swirlds.state.spi.WritableStates;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.util.Objects;
import java.util.function.LongConsumer;
import java.util.function.ObjLongConsumer;
import java.util.function.Predicate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class TokenServiceApiImpl
implements TokenServiceApi {
    private static final Logger logger = LogManager.getLogger(TokenServiceApiImpl.class);
    private final WritableAccountStore accountStore;
    private final AccountID fundingAccountID;
    private final AccountID stakingRewardAccountID;
    private final AccountID nodeRewardAccountID;
    private final NodesConfig nodesConfig;
    private final StakingConfig stakingConfig;
    private final Predicate<CryptoTransferTransactionBody> customFeeTest;

    public TokenServiceApiImpl(@NonNull Configuration config, @NonNull WritableStates writableStates, @NonNull Predicate<CryptoTransferTransactionBody> customFeeTest, @NonNull WritableEntityCounters entityCounters) {
        this.customFeeTest = customFeeTest;
        Objects.requireNonNull(config);
        this.accountStore = new WritableAccountStore(writableStates, entityCounters);
        this.nodesConfig = (NodesConfig)config.getConfigData(NodesConfig.class);
        this.stakingConfig = (StakingConfig)config.getConfigData(StakingConfig.class);
        HederaConfig hederaConfig = (HederaConfig)config.getConfigData(HederaConfig.class);
        this.fundingAccountID = AccountID.newBuilder().shardNum(hederaConfig.shard()).realmNum(hederaConfig.realm()).accountNum(((LedgerConfig)config.getConfigData(LedgerConfig.class)).fundingAccount()).build();
        this.stakingRewardAccountID = AccountID.newBuilder().shardNum(hederaConfig.shard()).realmNum(hederaConfig.realm()).accountNum(((AccountsConfig)config.getConfigData(AccountsConfig.class)).stakingRewardAccount()).build();
        this.nodeRewardAccountID = AccountID.newBuilder().shardNum(hederaConfig.shard()).realmNum(hederaConfig.realm()).accountNum(((AccountsConfig)config.getConfigData(AccountsConfig.class)).nodeRewardAccount()).build();
    }

    public void assertValidStakingElectionForCreation(boolean hasDeclineRewardChange, @NonNull String stakedIdKind, @Nullable AccountID stakedAccountIdInOp, @Nullable Long stakedNodeIdInOp, @NonNull ReadableAccountStore accountStore, @NonNull NetworkInfo networkInfo) {
        StakingValidator.validateStakedIdForCreation(hasDeclineRewardChange, stakedIdKind, stakedAccountIdInOp, stakedNodeIdInOp, accountStore, networkInfo);
    }

    public void assertValidStakingElectionForUpdate(boolean hasDeclineRewardChange, @NonNull String stakedIdKind, @Nullable AccountID stakedAccountIdInOp, @Nullable Long stakedNodeIdInOp, @NonNull ReadableAccountStore accountStore, @NonNull NetworkInfo networkInfo) {
        StakingValidator.validateStakedIdForUpdate(hasDeclineRewardChange, stakedIdKind, stakedAccountIdInOp, stakedNodeIdInOp, accountStore, networkInfo);
    }

    public void markAsContract(@NonNull AccountID accountId, @Nullable AccountID autoRenewAccountId) {
        Objects.requireNonNull(accountId);
        Account accountAsContract = Objects.requireNonNull(this.accountStore.get(accountId)).copyBuilder().smartContract(true).autoRenewAccountId(autoRenewAccountId).build();
        this.accountStore.put(accountAsContract);
    }

    public void finalizeHollowAccountAsContract(@NonNull AccountID hollowAccountId) {
        Objects.requireNonNull(hollowAccountId);
        Account hollowAccount = Objects.requireNonNull(this.accountStore.get(hollowAccountId));
        if (!KeyUtils.IMMUTABILITY_SENTINEL_KEY.equals((Object)hollowAccount.keyOrThrow())) {
            throw new IllegalArgumentException("Cannot finalize non-hollow account " + String.valueOf(hollowAccountId) + " as contract");
        }
        Account accountAsContract = hollowAccount.copyBuilder().key(Key.newBuilder().contractID(ContractID.newBuilder().shardNum(hollowAccountId.shardNum()).realmNum(hollowAccountId.realmNum()).contractNum(hollowAccountId.accountNumOrThrow().longValue())).build()).smartContract(true).maxAutoAssociations(hollowAccount.numberAssociations()).build();
        this.accountStore.put(accountAsContract);
    }

    public void deleteContract(@NonNull ContractID contractId) {
        Objects.requireNonNull(contractId);
        Account contract = this.accountStore.getContractById(contractId);
        if (contract == null) {
            logger.warn("Contract {} does not exist, so cannot be deleted", (Object)contractId);
            return;
        }
        if (contract.deleted()) {
            logger.warn("Trying to delete Contract {}, which is already deleted", (Object)contractId);
        }
        Bytes evmAddress = contract.alias();
        this.accountStore.removeAlias(evmAddress);
        Account.Builder builder = contract.copyBuilder().deleted(true);
        Account originalContract = this.accountStore.getOriginalValue(contract.accountIdOrThrow());
        if (originalContract != null && originalContract.smartContract()) {
            builder.alias(Bytes.EMPTY);
        }
        this.accountStore.put(builder.build());
        Bytes usedEvmAddress = contractId.evmAddressOrElse(Bytes.EMPTY);
        if (!usedEvmAddress.equals((Object)evmAddress)) {
            logger.error("Contract {} has an alias {} different than its referencing EVM address {}", (Object)contractId, (Object)evmAddress, (Object)usedEvmAddress);
            this.accountStore.removeAlias(usedEvmAddress);
        }
    }

    public void incrementParentNonce(@NonNull ContractID parentId) {
        Objects.requireNonNull(parentId);
        Account contract = Objects.requireNonNull(this.accountStore.getContractById(parentId));
        this.accountStore.put(contract.copyBuilder().ethereumNonce(contract.ethereumNonce() + 1L).build());
    }

    public void incrementSenderNonce(@NonNull AccountID senderId) {
        Objects.requireNonNull(senderId);
        Account sender = Objects.requireNonNull(this.accountStore.get(senderId));
        this.accountStore.put(sender.copyBuilder().ethereumNonce(sender.ethereumNonce() + 1L).build());
    }

    public void setNonce(@NonNull AccountID accountId, long nonce) {
        Objects.requireNonNull(accountId);
        Account target = Objects.requireNonNull(this.accountStore.get(accountId));
        this.accountStore.put(target.copyBuilder().ethereumNonce(nonce).build());
    }

    public void transferFromTo(@NonNull AccountID fromId, @NonNull AccountID toId, long amount) {
        if (amount < 0L) {
            throw new IllegalArgumentException("Cannot transfer negative value (" + amount + " tinybars) from " + String.valueOf(fromId) + " to " + String.valueOf(toId));
        }
        Account from = Objects.requireNonNull(this.accountStore.get(fromId));
        Account to = Objects.requireNonNull(this.accountStore.get(toId));
        if (from.tinybarBalance() < amount) {
            throw new IllegalArgumentException("Insufficient balance to transfer " + amount + " tinybars from " + String.valueOf(fromId) + " to " + String.valueOf(toId));
        }
        if (to.tinybarBalance() + amount < 0L) {
            throw new IllegalArgumentException("Overflow on transfer of " + amount + " tinybars from " + String.valueOf(fromId) + " to " + String.valueOf(toId));
        }
        if (!from.accountIdOrThrow().equals((Object)to.accountIdOrThrow())) {
            this.accountStore.put(from.copyBuilder().tinybarBalance(from.tinybarBalance() - amount).build());
            this.accountStore.put(to.copyBuilder().tinybarBalance(to.tinybarBalance() + amount).build());
        }
    }

    public ContractChangeSummary summarizeContractChanges() {
        return this.accountStore.summarizeContractChanges();
    }

    public void updateStorageMetadata(@NonNull ContractID contractID, @NonNull Bytes firstKey, int netChangeInSlotsUsed) {
        Objects.requireNonNull(firstKey);
        Objects.requireNonNull(contractID);
        Account target = this.accountStore.getContractById(contractID);
        if (target == null) {
            throw new IllegalArgumentException("No contract found for ID " + String.valueOf(contractID));
        }
        int newNumKvPairs = target.contractKvPairsNumber() + netChangeInSlotsUsed;
        if (newNumKvPairs < 0) {
            throw new IllegalArgumentException("Cannot change # of storage slots (currently " + target.contractKvPairsNumber() + ") by " + netChangeInSlotsUsed + " for contract " + String.valueOf(contractID));
        }
        this.accountStore.put(target.copyBuilder().firstContractStorageKey(firstKey).contractKvPairsNumber(newNumKvPairs).build());
    }

    public void updateLambdaStorageSlots(@NonNull AccountID accountId, int netChangeInSlotsUsed, boolean requireContract) {
        Objects.requireNonNull(accountId);
        Account account = this.accountStore.get(accountId);
        if (account == null) {
            throw new IllegalArgumentException("No account found for ID " + String.valueOf(accountId));
        }
        long newSlotsUsed = account.numberLambdaStorageSlots() + (long)netChangeInSlotsUsed;
        if (newSlotsUsed < 0L) {
            throw new IllegalArgumentException("Cannot change # of lambda storage slots (currently " + account.numberLambdaStorageSlots() + ") by " + netChangeInSlotsUsed + " for account " + String.valueOf(accountId));
        }
        if (requireContract && !account.smartContract()) {
            throw new HandleException(ResponseCodeEnum.WRONG_HOOK_ENTITY_TYPE);
        }
        if (netChangeInSlotsUsed == 0) {
            return;
        }
        this.accountStore.put(account.copyBuilder().numberLambdaStorageSlots(newSlotsUsed).build());
    }

    public Fees chargeFee(@NonNull AccountID payerId, long amount, @NonNull StreamBuilder streamBuilder, @Nullable ObjLongConsumer<AccountID> cb) {
        Objects.requireNonNull(payerId);
        Objects.requireNonNull(streamBuilder);
        if (!(streamBuilder instanceof FeeStreamBuilder)) {
            throw new IllegalArgumentException("StreamBuilder must be a FeeStreamBuilder");
        }
        FeeStreamBuilder feeBuilder = (FeeStreamBuilder)streamBuilder;
        Account payerAccount = this.lookupAccount("Payer", payerId);
        long amountToCharge = Math.min(amount, payerAccount.tinybarBalance());
        this.chargePayer(payerAccount, amountToCharge, cb);
        if (cb == null) {
            feeBuilder.transactionFee(feeBuilder.transactionFee() + amountToCharge);
        }
        this.distributeToNetworkFundingAccounts(amountToCharge, cb);
        return new Fees(0L, amountToCharge, 0L);
    }

    public void refundFee(@NonNull AccountID payerId, long amount, @NonNull FeeStreamBuilder rb) {
        Objects.requireNonNull(payerId);
        Objects.requireNonNull(rb);
        long retractedAmount = this.retractFromNetworkFundingAccounts(amount);
        Account payerAccount = this.lookupAccount("Payer", payerId);
        this.refundPayer(payerAccount, retractedAmount);
        rb.transactionFee(Math.max(0L, rb.transactionFee() - retractedAmount));
    }

    public Fees chargeFees(@NonNull AccountID payerId, @NonNull AccountID nodeAccountId, @NonNull Fees fees, @NonNull FeeStreamBuilder rb, @Nullable ObjLongConsumer<AccountID> cb, @NonNull LongConsumer onNodeFee) {
        Objects.requireNonNull(rb);
        Objects.requireNonNull(fees);
        Objects.requireNonNull(payerId);
        Objects.requireNonNull(nodeAccountId);
        Objects.requireNonNull(onNodeFee);
        Account payerAccount = this.lookupAccount("Payer", payerId);
        if (payerAccount.tinybarBalance() < fees.networkFee()) {
            throw new IllegalArgumentException("Payer %s (balance=%d) cannot afford network fee of %d, which should have been a due diligence failure".formatted(payerId, payerAccount.tinybarBalance(), fees.networkFee()));
        }
        if (fees.serviceFee() > 0L && payerAccount.tinybarBalance() < fees.totalFee()) {
            throw new IllegalArgumentException("Payer %s (balance=%d) cannot afford total fee of %d, which means service component should have been zeroed out".formatted(payerId, payerAccount.tinybarBalance(), fees.totalFee()));
        }
        long chargeableNodeFee = Math.min(fees.nodeFee(), payerAccount.tinybarBalance() - fees.networkFee());
        long amountToCharge = fees.totalWithoutNodeFee() + chargeableNodeFee;
        long amountToDistributeToFundingAccounts = amountToCharge - chargeableNodeFee;
        this.chargePayer(payerAccount, amountToCharge, cb);
        rb.transactionFee(amountToCharge);
        this.distributeToNetworkFundingAccounts(amountToDistributeToFundingAccounts, cb);
        if (chargeableNodeFee > 0L) {
            Account nodeAccount = this.lookupAccount("Node account", nodeAccountId);
            this.accountStore.put(nodeAccount.copyBuilder().tinybarBalance(nodeAccount.tinybarBalance() + chargeableNodeFee).build());
            if (cb != null) {
                cb.accept(nodeAccountId, chargeableNodeFee);
            }
            onNodeFee.accept(chargeableNodeFee);
        }
        if (amountToCharge == fees.totalFee()) {
            return fees;
        }
        return fees.withChargedNodeComponent(chargeableNodeFee);
    }

    public void refundFees(@NonNull AccountID payerId, @NonNull AccountID nodeAccountId, @NonNull Fees fees, @NonNull FeeStreamBuilder rb, @NonNull LongConsumer onNodeRefund) {
        Objects.requireNonNull(payerId);
        Objects.requireNonNull(nodeAccountId);
        Objects.requireNonNull(fees);
        Objects.requireNonNull(rb);
        Objects.requireNonNull(onNodeRefund);
        long amountRetracted = 0L;
        if (fees.nodeFee() > 0L) {
            Account nodeAccount = this.lookupAccount("Node account", nodeAccountId);
            long nodeBalance = nodeAccount.tinybarBalance();
            long amountToRetract = Math.min(fees.nodeFee(), nodeBalance);
            this.accountStore.put(nodeAccount.copyBuilder().tinybarBalance(nodeBalance - amountToRetract).build());
            onNodeRefund.accept(amountToRetract);
            amountRetracted += amountToRetract;
        }
        Account payerAccount = this.lookupAccount("Payer", payerId);
        this.refundPayer(payerAccount, amountRetracted += this.retractFromNetworkFundingAccounts(fees.totalWithoutNodeFee()));
        rb.transactionFee(Math.max(0L, rb.transactionFee() - amountRetracted));
    }

    public long originalKvUsageFor(@NonNull ContractID id) {
        Account account = this.accountStore.getContractById(id);
        Account oldAccount = account == null ? null : this.accountStore.getOriginalValue(account.accountIdOrThrow());
        return oldAccount == null ? 0L : (long)oldAccount.contractKvPairsNumber();
    }

    public void updateContract(Account contract) {
        this.accountStore.put(contract);
    }

    private void chargePayer(@NonNull Account payerAccount, long amount, @Nullable ObjLongConsumer<AccountID> cb) {
        if (amount > payerAccount.tinybarBalance()) {
            throw new IllegalArgumentException("Payer %s (balance=%d) cannot afford fee of %d".formatted(payerAccount, payerAccount.tinybarBalance(), amount));
        }
        long currentBalance = payerAccount.tinybarBalance();
        this.accountStore.put(payerAccount.copyBuilder().tinybarBalance(currentBalance - amount).build());
        if (cb != null) {
            cb.accept(payerAccount.accountId(), -amount);
        }
    }

    private void refundPayer(@NonNull Account payerAccount, long amount) {
        long currentBalance = payerAccount.tinybarBalance();
        this.accountStore.put(payerAccount.copyBuilder().tinybarBalance(currentBalance + amount).build());
    }

    private void payNodeRewardAccount(long amount) {
        if (amount == 0L) {
            return;
        }
        Account nodeAccount = this.lookupAccount("Node reward", this.nodeRewardAccountID);
        this.accountStore.put(nodeAccount.copyBuilder().tinybarBalance(nodeAccount.tinybarBalance() + amount).build());
    }

    private long retractNodeRewardAccount(long amount) {
        if (amount == 0L) {
            return 0L;
        }
        Account nodeAccount = this.lookupAccount("Node reward", this.nodeRewardAccountID);
        long balance = nodeAccount.tinybarBalance();
        long amountToRetract = Math.min(amount, balance);
        this.accountStore.put(nodeAccount.copyBuilder().tinybarBalance(balance - amountToRetract).build());
        return amountToRetract;
    }

    private void payStakingRewardAccount(long amount) {
        if (amount == 0L) {
            return;
        }
        Account stakingAccount = this.lookupAccount("Staking reward", this.stakingRewardAccountID);
        this.accountStore.put(stakingAccount.copyBuilder().tinybarBalance(stakingAccount.tinybarBalance() + amount).build());
    }

    private long retractStakingRewardAccount(long amount) {
        if (amount == 0L) {
            return 0L;
        }
        Account stakingAccount = this.lookupAccount("Staking reward", this.stakingRewardAccountID);
        long balance = stakingAccount.tinybarBalance();
        long amountToRetract = Math.min(amount, balance);
        this.accountStore.put(stakingAccount.copyBuilder().tinybarBalance(balance - amountToRetract).build());
        return amountToRetract;
    }

    @NonNull
    private Account lookupAccount(String logName, AccountID id) {
        Account account = this.accountStore.get(id);
        if (account == null) {
            throw new IllegalStateException(logName + " account " + String.valueOf(id) + " does not exist");
        }
        return account;
    }

    public boolean checkForCustomFees(@NonNull CryptoTransferTransactionBody op) {
        return this.customFeeTest.test(op);
    }

    public void deleteAndTransfer(@NonNull AccountID deletedId, @NonNull AccountID obtainerId, @NonNull ExpiryValidator expiryValidator, @NonNull DeleteCapableTransactionStreamBuilder recordBuilder, @NonNull ReadableAccountNodeRelStore accountNodeRelStore) {
        InvolvedAccounts deleteAndTransferAccounts = this.validateSemantics(deletedId, obtainerId, expiryValidator, accountNodeRelStore);
        this.transferRemainingBalance(expiryValidator, deleteAndTransferAccounts);
        Account updatedDeleteAccount = Objects.requireNonNull(this.accountStore.get(deletedId));
        Account.Builder builder = updatedDeleteAccount.copyBuilder().deleted(true);
        this.accountStore.removeAlias(updatedDeleteAccount.alias());
        builder.alias(Bytes.EMPTY);
        this.accountStore.put(builder.build());
        recordBuilder.addBeneficiaryForDeletedAccount(deletedId, obtainerId);
    }

    private InvolvedAccounts validateSemantics(@NonNull AccountID deletedId, @NonNull AccountID obtainerId, @NonNull ExpiryValidator expiryValidator, @NonNull ReadableAccountNodeRelStore accountNodeRelStore) {
        Account deletedAccount = this.accountStore.get(deletedId);
        HandleException.validateTrue((deletedAccount != null ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.INVALID_ACCOUNT_ID);
        HandleException.validateFalse((boolean)deletedAccount.hasHeadPendingAirdropId(), (ResponseCodeEnum)ResponseCodeEnum.ACCOUNT_HAS_PENDING_AIRDROPS);
        Account transferAccount = this.accountStore.get(obtainerId);
        HandleException.validateTrue((transferAccount != null ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.INVALID_TRANSFER_ACCOUNT_ID);
        HandleException.validateFalse((deletedAccount.numberTreasuryTitles() > 0 ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.ACCOUNT_IS_TREASURY);
        boolean isExpired = this.areAccountsDetached(deletedAccount, transferAccount, expiryValidator);
        HandleException.validateFalse((boolean)isExpired, (ResponseCodeEnum)ResponseCodeEnum.ACCOUNT_EXPIRED_AND_PENDING_REMOVAL);
        HandleException.validateTrue((deletedAccount.numberPositiveBalances() == 0 ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.TRANSACTION_REQUIRES_ZERO_TOKEN_BALANCES);
        HandleException.validateTrue((deletedAccount.numberHooksInUse() == 0L ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.TRANSACTION_REQUIRES_ZERO_HOOKS);
        HandleException.validateTrue((accountNodeRelStore.get(deletedAccount.accountId()) == null ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.ACCOUNT_IS_LINKED_TO_A_NODE);
        return new InvolvedAccounts(deletedAccount, transferAccount);
    }

    private void transferRemainingBalance(@NonNull ExpiryValidator expiryValidator, @NonNull InvolvedAccounts involvedAccounts) {
        Account fromAccount = involvedAccounts.deletedAccount();
        Account toAccount = involvedAccounts.obtainerAccount();
        long newFromBalance = this.computeNewBalance(expiryValidator, fromAccount, -1L * fromAccount.tinybarBalance());
        long newToBalance = this.computeNewBalance(expiryValidator, toAccount, fromAccount.tinybarBalance());
        this.accountStore.put(fromAccount.copyBuilder().tinybarBalance(newFromBalance).build());
        this.accountStore.put(toAccount.copyBuilder().tinybarBalance(newToBalance).build());
    }

    private long computeNewBalance(ExpiryValidator expiryValidator, Account account, long adjustment) {
        HandleException.validateTrue((!account.deleted() ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.ACCOUNT_DELETED);
        HandleException.validateTrue((!expiryValidator.isDetached(EntityType.ACCOUNT, account.expiredAndPendingRemoval(), account.tinybarBalance()) ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.ACCOUNT_EXPIRED_AND_PENDING_REMOVAL);
        long balance = account.tinybarBalance();
        HandleException.validateTrue((balance + adjustment >= 0L ? 1 : 0) != 0, (ResponseCodeEnum)ResponseCodeEnum.INSUFFICIENT_ACCOUNT_BALANCE);
        return balance + adjustment;
    }

    private boolean areAccountsDetached(@NonNull Account deleteAccount, @NonNull Account transferAccount, @NonNull ExpiryValidator expiryValidator) {
        return expiryValidator.isDetached(this.getEntityType(deleteAccount), deleteAccount.expiredAndPendingRemoval(), deleteAccount.tinybarBalance()) || expiryValidator.isDetached(this.getEntityType(transferAccount), transferAccount.expiredAndPendingRemoval(), transferAccount.tinybarBalance());
    }

    private EntityType getEntityType(@NonNull Account account) {
        return account.smartContract() ? EntityType.CONTRACT_BYTECODE : EntityType.ACCOUNT;
    }

    private void distributeToNetworkFundingAccounts(long amount, @Nullable ObjLongConsumer<AccountID> cb) {
        boolean preservingRewardBalance;
        long balance = amount;
        Account nodeRewardAccount = this.lookupAccount("Node rewards", this.nodeRewardAccountID);
        boolean bl = preservingRewardBalance = this.nodesConfig.nodeRewardsEnabled() && this.nodesConfig.preserveMinNodeRewardBalance();
        if (!preservingRewardBalance || nodeRewardAccount.tinybarBalance() > this.nodesConfig.minNodeRewardBalance()) {
            long nodeReward = (long)this.stakingConfig.feesNodeRewardPercentage() * amount / 100L;
            balance -= nodeReward;
            this.payNodeRewardAccount(nodeReward);
            if (cb != null) {
                cb.accept(this.nodeRewardAccountID, nodeReward);
            }
            long stakingReward = (long)this.stakingConfig.feesStakingRewardPercentage() * amount / 100L;
            balance -= stakingReward;
            this.payStakingRewardAccount(stakingReward);
            if (cb != null) {
                cb.accept(this.stakingRewardAccountID, stakingReward);
            }
            Account fundingAccount = this.lookupAccount("Funding", this.fundingAccountID);
            this.accountStore.put(fundingAccount.copyBuilder().tinybarBalance(fundingAccount.tinybarBalance() + balance).build());
            if (cb != null) {
                cb.accept(this.fundingAccountID, balance);
            }
        } else {
            this.payNodeRewardAccount(balance);
            if (cb != null) {
                cb.accept(this.nodeRewardAccountID, balance);
            }
        }
    }

    private long retractFromNetworkFundingAccounts(long amount) {
        boolean preservingRewardBalance;
        long balance = amount;
        Account nodeRewardAccount = this.lookupAccount("Node rewards", this.nodeRewardAccountID);
        boolean bl = preservingRewardBalance = this.nodesConfig.nodeRewardsEnabled() && this.nodesConfig.preserveMinNodeRewardBalance();
        if (!preservingRewardBalance || nodeRewardAccount.tinybarBalance() > this.nodesConfig.minNodeRewardBalance()) {
            long amountRetracted = 0L;
            long nodeReward = (long)this.stakingConfig.feesNodeRewardPercentage() * amount / 100L;
            balance -= nodeReward;
            amountRetracted += this.retractNodeRewardAccount(nodeReward);
            long stakingReward = (long)this.stakingConfig.feesStakingRewardPercentage() * amount / 100L;
            Account fundingAccount = this.lookupAccount("Funding", this.fundingAccountID);
            long fundingBalance = fundingAccount.tinybarBalance();
            long amountToRetract = Math.min(balance -= stakingReward, fundingBalance);
            this.accountStore.put(fundingAccount.copyBuilder().tinybarBalance(fundingBalance - amountToRetract).build());
            return (amountRetracted += this.retractStakingRewardAccount(stakingReward)) + amountToRetract;
        }
        return this.retractNodeRewardAccount(balance);
    }

    private record InvolvedAccounts(@NonNull Account deletedAccount, @NonNull Account obtainerAccount) {
        private InvolvedAccounts {
            Objects.requireNonNull(deletedAccount);
            Objects.requireNonNull(obtainerAccount);
        }
    }
}

