/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.hints.impl;

import com.google.common.annotations.VisibleForTesting;
import com.hedera.cryptography.hints.AggregationAndVerificationKeys;
import com.hedera.hapi.node.base.Timestamp;
import com.hedera.hapi.node.state.hints.CRSStage;
import com.hedera.hapi.node.state.hints.CRSState;
import com.hedera.hapi.node.state.hints.HintsConstruction;
import com.hedera.hapi.node.state.hints.PreprocessedKeys;
import com.hedera.hapi.node.state.hints.PreprocessingVote;
import com.hedera.hapi.services.auxiliary.hints.CrsPublicationTransactionBody;
import com.hedera.hapi.util.HapiUtils;
import com.hedera.node.app.hints.HintsLibrary;
import com.hedera.node.app.hints.HintsService;
import com.hedera.node.app.hints.ReadableHintsStore;
import com.hedera.node.app.hints.WritableHintsStore;
import com.hedera.node.app.hints.impl.HintsContext;
import com.hedera.node.app.hints.impl.HintsController;
import com.hedera.node.app.hints.impl.HintsSubmissions;
import com.hedera.node.app.hints.impl.OnHintsFinished;
import com.hedera.node.app.service.roster.impl.RosterTransitionWeights;
import com.hedera.node.config.data.TssConfig;
import com.hedera.pbj.runtime.io.buffer.Bytes;
import com.swirlds.config.api.Configuration;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.security.SecureRandom;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class HintsControllerImpl
implements HintsController {
    private static final SecureRandom SECURE_RANDOM = new SecureRandom();
    private static final Logger log = LogManager.getLogger(HintsControllerImpl.class);
    private final int numParties;
    private final long selfId;
    private final Executor executor;
    private final Bytes blsPrivateKey;
    private final HintsLibrary library;
    private final HintsSubmissions submissions;
    private final HintsContext context;
    private final Map<Long, Integer> nodePartyIds = new HashMap<Long, Integer>();
    private final Map<Integer, Long> partyNodeIds = new HashMap<Integer, Long>();
    private final RosterTransitionWeights weights;
    private final Map<Long, PreprocessingVote> votes = new ConcurrentHashMap<Long, PreprocessingVote>();
    private final NavigableMap<Instant, CompletableFuture<Validation>> validationFutures = new TreeMap<Instant, CompletableFuture<Validation>>();
    private final Supplier<Configuration> configurationSupplier;
    private final OnHintsFinished onHintsFinished;
    @Nullable
    private CompletableFuture<CRSValidation> finalCrsFuture;
    private HintsConstruction construction;
    @Nullable
    private CompletableFuture<Void> preprocessingVoteFuture;
    @Nullable
    private CompletableFuture<Void> publicationFuture;
    @Nullable
    private CompletableFuture<Void> crsPublicationFuture;

    public HintsControllerImpl(long selfId, @NonNull Bytes blsPrivateKey, @NonNull HintsConstruction construction, @NonNull RosterTransitionWeights weights, @NonNull Executor executor, @NonNull HintsLibrary library, @NonNull Map<Long, PreprocessingVote> votes, @NonNull List<ReadableHintsStore.HintsKeyPublication> publications, @NonNull HintsSubmissions submissions, @NonNull HintsContext context, @NonNull Supplier<Configuration> configuration, @NonNull WritableHintsStore hintsStore, @NonNull OnHintsFinished onHintsFinished) {
        this.selfId = selfId;
        this.blsPrivateKey = Objects.requireNonNull(blsPrivateKey);
        this.weights = Objects.requireNonNull(weights);
        this.numParties = HintsService.partySizeForRosterNodeCount(weights.targetRosterSize());
        this.executor = Objects.requireNonNull(executor);
        this.context = Objects.requireNonNull(context);
        this.submissions = Objects.requireNonNull(submissions);
        this.library = Objects.requireNonNull(library);
        this.construction = Objects.requireNonNull(construction);
        this.onHintsFinished = Objects.requireNonNull(onHintsFinished);
        this.votes.putAll(votes);
        this.configurationSupplier = Objects.requireNonNull(configuration);
        CRSState crsState = hintsStore.getCrsState();
        if (crsState.stage() == CRSStage.GATHERING_CONTRIBUTIONS) {
            Map<Long, CrsPublicationTransactionBody> crsPublications = hintsStore.getOrderedCrsPublications(weights.sourceNodeIds());
            crsPublications.forEach((nodeId, publication) -> {
                if (publication != null) {
                    this.verifyCrsUpdate((CrsPublicationTransactionBody)publication, hintsStore, (long)nodeId);
                }
            });
        }
        if (crsState.stage() == CRSStage.COMPLETED && !construction.hasHintsScheme()) {
            Instant cutoffTime = construction.hasPreprocessingStartTime() ? HapiUtils.asInstant((Timestamp)construction.preprocessingStartTimeOrThrow()) : Instant.MAX;
            publications.forEach(publication -> {
                if (!publication.adoptionTime().isAfter(cutoffTime)) {
                    this.updateHintsKey(crsState.crs(), (ReadableHintsStore.HintsKeyPublication)publication);
                }
            });
        }
    }

    @Override
    public long constructionId() {
        return this.construction.constructionId();
    }

    @Override
    public boolean isStillInProgress() {
        return !this.construction.hasHintsScheme();
    }

    @Override
    public boolean hasNumParties(int numParties) {
        return this.numParties == numParties;
    }

    @Override
    public void advanceConstruction(@NonNull Instant now, @NonNull WritableHintsStore hintsStore, boolean isActive) {
        Objects.requireNonNull(now);
        Objects.requireNonNull(hintsStore);
        if (hintsStore.getCrsState().stage() != CRSStage.COMPLETED || this.construction.hasHintsScheme()) {
            return;
        }
        if (this.construction.hasPreprocessingStartTime()) {
            if (isActive && !this.votes.containsKey(this.selfId) && this.preprocessingVoteFuture == null) {
                this.preprocessingVoteFuture = this.startPreprocessingVoteFuture(HapiUtils.asInstant((Timestamp)this.construction.preprocessingStartTimeOrThrow()), hintsStore.getCrsState().crs());
            }
        } else {
            Bytes crs = hintsStore.getCrsState().crs();
            if (this.shouldStartPreprocessing(now)) {
                this.construction = hintsStore.setPreprocessingStartTime(this.construction.constructionId(), now);
                if (isActive) {
                    this.preprocessingVoteFuture = this.startPreprocessingVoteFuture(now, crs);
                }
            } else if (isActive) {
                this.ensureHintsKeyPublished(crs);
            }
        }
    }

    @Override
    public void advanceCrsWork(@NonNull Instant now, @NonNull WritableHintsStore hintsStore, boolean isActive) {
        CRSState crsState = hintsStore.getCrsState();
        TssConfig tssConfig = (TssConfig)this.configurationSupplier.get().getConfigData(TssConfig.class);
        try {
            if (!crsState.hasNextContributingNodeId()) {
                this.tryToFinalizeCrs(now, hintsStore, crsState, tssConfig);
            } else if (crsState.hasContributionEndTime() && now.isAfter(HapiUtils.asInstant((Timestamp)crsState.contributionEndTimeOrThrow()))) {
                this.moveToNextNode(now, hintsStore);
            } else if (crsState.nextContributingNodeIdOrThrow() == this.selfId && this.crsPublicationFuture == null && isActive) {
                this.submitUpdatedCRS(hintsStore);
            }
        }
        catch (CancellationException cancellationException) {
        }
        catch (Exception e) {
            log.error("Failed to advance CRS work", (Throwable)e);
        }
    }

    private void tryToFinalizeCrs(@NonNull Instant now, @NonNull WritableHintsStore hintsStore, @NonNull CRSState crsState, @NonNull TssConfig tssConfig) {
        if (crsState.stage() == CRSStage.GATHERING_CONTRIBUTIONS) {
            Duration delay = tssConfig.crsFinalizationDelay();
            CRSState updatedState = crsState.copyBuilder().stage(CRSStage.WAITING_FOR_ADOPTING_FINAL_CRS).contributionEndTime(HapiUtils.asTimestamp((Instant)now.plus(delay))).build();
            hintsStore.setCrsState(updatedState);
            log.info("All nodes have contributed to the CRS, waiting for final adoption");
        } else if (now.isAfter(HapiUtils.asInstant((Timestamp)crsState.contributionEndTimeOrThrow()))) {
            boolean thresholdMet = this.validateWeightOfContributions();
            if (!thresholdMet) {
                this.restartFromFirstNode(now, hintsStore, tssConfig);
            } else {
                Bytes crs = Objects.requireNonNull(this.finalCrsFuture).join().crs();
                CRSState updatedState = crsState.copyBuilder().crs(crs).stage(CRSStage.COMPLETED).contributionEndTime((Timestamp)null).build();
                hintsStore.setCrsState(updatedState);
                log.info("Finished constructing CRS");
            }
        }
    }

    private void restartFromFirstNode(@NonNull Instant now, @NonNull WritableHintsStore hintsStore, @NonNull TssConfig tssConfig) {
        log.warn("Restarting CRS ceremony from the first node because threshold not met for CRS contributions");
        if (this.crsPublicationFuture != null && !this.crsPublicationFuture.isDone()) {
            this.crsPublicationFuture.cancel(true);
        }
        this.crsPublicationFuture = null;
        CRSState crsState = hintsStore.getCrsState();
        Long firstNodeId = this.weights.sourceNodeIds().stream().min(Long::compareTo).orElse(0L);
        Duration contributionTime = tssConfig.crsUpdateContributionTime();
        CRSState updatedState = crsState.copyBuilder().stage(CRSStage.GATHERING_CONTRIBUTIONS).contributionEndTime(HapiUtils.asTimestamp((Instant)now.plus(contributionTime))).nextContributingNodeId(firstNodeId).build();
        hintsStore.setCrsState(updatedState);
    }

    private boolean validateWeightOfContributions() {
        long contributedWeight = this.finalCrsFuture == null ? 0L : this.finalCrsFuture.join().weightContributedSoFar();
        long totalWeight = this.weights.sourceNodeWeights().values().stream().mapToLong(Long::longValue).sum();
        log.info("Total weight of CRS contributions is {} (of {} total)", (Object)contributedWeight, (Object)totalWeight);
        return contributedWeight >= RosterTransitionWeights.moreThanTwoThirdsOfTotal((long)totalWeight);
    }

    private void moveToNextNode(@NonNull Instant now, @NonNull WritableHintsStore hintsStore) {
        CRSState crsState = hintsStore.getCrsState();
        TssConfig tssConfig = (TssConfig)this.configurationSupplier.get().getConfigData(TssConfig.class);
        OptionalLong optionalNextNodeId = this.nextNodeId(this.weights.sourceNodeIds(), crsState);
        log.info("{} for CRS contribution", (Object)optionalNextNodeId.stream().mapToObj(l -> "Moving on to node" + l).findFirst().orElse("No remaining nodes to consider"));
        hintsStore.moveToNextNode(optionalNextNodeId.isEmpty() ? null : Long.valueOf(optionalNextNodeId.getAsLong()), now.plusSeconds(tssConfig.crsUpdateContributionTime().toSeconds()));
    }

    private void submitUpdatedCRS(@NonNull WritableHintsStore hintsStore) {
        this.crsPublicationFuture = CompletableFuture.runAsync(() -> {
            try {
                Bytes previousCrs = this.finalCrsFuture != null ? this.finalCrsFuture.join().crs() : hintsStore.getCrsState().crs();
                Bytes updatedCrs = this.library.updateCrs(previousCrs, this.generateEntropy());
                CrsUpdateOutput newCrs = HintsControllerImpl.decodeCrsUpdate(previousCrs.length(), updatedCrs);
                this.submissions.submitCrsUpdate(newCrs.crs(), newCrs.proof()).join();
            }
            catch (CancellationException previousCrs) {
            }
            catch (Exception e) {
                log.error("Failed to submit updated CRS", (Throwable)e);
            }
        }, this.executor);
    }

    private OptionalLong nextNodeId(Set<Long> nodeIds, CRSState crsState) {
        if (!crsState.hasNextContributingNodeId()) {
            return OptionalLong.empty();
        }
        return nodeIds.stream().mapToLong(Long::longValue).filter(nodeId -> nodeId > crsState.nextContributingNodeIdOrThrow()).findFirst();
    }

    private Bytes generateEntropy() {
        byte[] entropyBytes = new byte[32];
        SECURE_RANDOM.nextBytes(entropyBytes);
        return Bytes.wrap((byte[])entropyBytes);
    }

    @Override
    @NonNull
    public OptionalInt partyIdOf(long nodeId) {
        if (!this.weights.targetIncludes(nodeId)) {
            return OptionalInt.empty();
        }
        return this.nodePartyIds.containsKey(nodeId) ? OptionalInt.of(this.nodePartyIds.get(nodeId)) : OptionalInt.of(this.expectedPartyId(nodeId));
    }

    @Override
    public void addHintsKeyPublication(@NonNull ReadableHintsStore.HintsKeyPublication publication, Bytes crs) {
        Objects.requireNonNull(publication);
        if (!this.construction.hasGracePeriodEndTime()) {
            log.info("Ignoring tardy hinTS key from node{}", (Object)publication.nodeId());
            return;
        }
        this.maybeUpdateForHintsKey(crs, publication);
    }

    @Override
    public boolean addPreprocessingVote(long nodeId, @NonNull PreprocessingVote vote, @NonNull WritableHintsStore hintsStore) {
        Objects.requireNonNull(vote);
        Objects.requireNonNull(hintsStore);
        if (!this.construction.hasHintsScheme() && !this.votes.containsKey(nodeId)) {
            PreprocessingVote congruentVote;
            hintsStore.addPreprocessingVote(nodeId, this.constructionId(), vote);
            if (vote.hasPreprocessedKeys()) {
                this.votes.put(nodeId, vote);
            } else if (vote.hasCongruentNodeId() && (congruentVote = this.votes.get(vote.congruentNodeIdOrThrow())) != null && congruentVote.hasPreprocessedKeys()) {
                this.votes.put(nodeId, congruentVote);
            }
            Map<PreprocessedKeys, Long> outputWeights = this.votes.entrySet().stream().collect(Collectors.groupingBy(entry -> ((PreprocessingVote)entry.getValue()).preprocessedKeysOrThrow(), Collectors.summingLong(entry -> this.weights.sourceWeightOf(((Long)entry.getKey()).longValue()))));
            Optional<PreprocessedKeys> maybeWinningOutputs = outputWeights.entrySet().stream().filter(entry -> (Long)entry.getValue() >= this.weights.sourceWeightThreshold()).map(Map.Entry::getKey).findFirst();
            maybeWinningOutputs.ifPresent(keys -> {
                this.construction = hintsStore.setHintsScheme(this.construction.constructionId(), (PreprocessedKeys)keys, this.nodePartyIds, this.weights.targetNodeWeights());
                log.info("Completed hinTS Scheme for construction #{}", (Object)this.construction.constructionId());
                this.onHintsFinished.accept(hintsStore, this.construction, this.context);
            });
            return true;
        }
        return false;
    }

    @Override
    public void cancelPendingWork() {
        if (this.publicationFuture != null) {
            this.publicationFuture.cancel(true);
        }
        if (this.preprocessingVoteFuture != null) {
            this.preprocessingVoteFuture.cancel(true);
        }
        if (this.crsPublicationFuture != null) {
            this.crsPublicationFuture.cancel(true);
        }
        if (this.finalCrsFuture != null) {
            this.finalCrsFuture.cancel(true);
        }
        this.validationFutures.values().forEach(future -> future.cancel(true));
    }

    @Override
    public void addCrsPublication(@NonNull CrsPublicationTransactionBody publication, @NonNull Instant consensusTime, @NonNull WritableHintsStore hintsStore, long creatorId) {
        Objects.requireNonNull(publication);
        Objects.requireNonNull(consensusTime);
        Objects.requireNonNull(hintsStore);
        this.verifyCrsUpdate(publication, hintsStore, creatorId);
        this.moveToNextNode(consensusTime, hintsStore);
    }

    @Override
    public void verifyCrsUpdate(@NonNull CrsPublicationTransactionBody publication, @NonNull ReadableHintsStore hintsStore, long creatorId) {
        Objects.requireNonNull(publication);
        Objects.requireNonNull(hintsStore);
        long creatorWeight = this.weights.sourceWeightOf(creatorId);
        if (this.finalCrsFuture == null) {
            Bytes initialCrs = hintsStore.getCrsState().crs();
            this.finalCrsFuture = CompletableFuture.supplyAsync(() -> {
                boolean isValid = this.library.verifyCrsUpdate(initialCrs, publication.newCrs(), publication.proof());
                if (isValid) {
                    return new CRSValidation(publication.newCrs(), creatorWeight);
                }
                return new CRSValidation(initialCrs, 0L);
            }, this.executor);
        } else {
            this.finalCrsFuture = this.finalCrsFuture.thenApplyAsync(previousValidation -> {
                boolean isValid = this.library.verifyCrsUpdate(previousValidation.crs(), publication.newCrs(), publication.proof());
                if (isValid) {
                    return new CRSValidation(publication.newCrs(), previousValidation.weightContributedSoFar() + creatorWeight);
                }
                return new CRSValidation(previousValidation.crs(), previousValidation.weightContributedSoFar());
            }, this.executor);
        }
    }

    private boolean shouldStartPreprocessing(@NonNull Instant now) {
        if (this.validationFutures.size() == this.weights.numTargetNodesInSource()) {
            log.info("All nodes have published hinTS keys. Starting preprocessing.");
            return true;
        }
        if (now.isBefore(HapiUtils.asInstant((Timestamp)this.construction.gracePeriodEndTimeOrThrow()))) {
            return false;
        }
        return this.weightOfValidHintsKeysAt(now) >= this.weights.targetWeightThreshold();
    }

    private void maybeUpdateForHintsKey(@NonNull Bytes crs, @NonNull ReadableHintsStore.HintsKeyPublication publication) {
        Objects.requireNonNull(publication);
        Objects.requireNonNull(crs);
        if (publication.partyId() == this.expectedPartyId(publication.nodeId())) {
            this.updateHintsKey(crs, publication);
        }
    }

    private void updateHintsKey(@NonNull Bytes crs, @NonNull ReadableHintsStore.HintsKeyPublication publication) {
        int partyId = publication.partyId();
        long nodeId = publication.nodeId();
        this.nodePartyIds.put(nodeId, partyId);
        this.partyNodeIds.put(partyId, nodeId);
        this.validationFutures.put(publication.adoptionTime(), this.validationFuture(crs, partyId, publication.hintsKey()));
        log.info("Updated hinTS key for node #{} (weight={} of target threshold={})", (Object)nodeId, (Object)this.weights.targetWeightOf(nodeId), (Object)this.weights.targetWeightThreshold());
    }

    private int expectedPartyId(long nodeId) {
        List<Long> unassignedNodeIds = this.weights.targetNodeWeights().keySet().stream().filter(id -> !this.nodePartyIds.containsKey(id)).sorted().toList();
        List<Integer> unusedPartyIds = IntStream.range(1, this.numParties + 1).filter(id -> !this.partyNodeIds.containsKey(id)).boxed().toList();
        return unusedPartyIds.get(unassignedNodeIds.indexOf(nodeId));
    }

    private CompletableFuture<Validation> validationFuture(Bytes crs, int partyId, @NonNull Bytes hintsKey) {
        return CompletableFuture.supplyAsync(() -> {
            boolean isValid = false;
            try {
                isValid = this.library.validateHintsKey(crs, hintsKey, partyId, this.numParties);
            }
            catch (Exception e) {
                log.warn("Failed to validate hints key {} for party{} of {}", (Object)hintsKey, (Object)partyId, (Object)this.numParties, (Object)e);
            }
            return new Validation(partyId, hintsKey, isValid);
        }, this.executor);
    }

    private long weightOfValidHintsKeysAt(@NonNull Instant now) {
        return this.validationFutures.headMap(now, true).values().stream().map(CompletableFuture::join).filter(Validation::isValid).mapToLong(validation -> this.weights.targetWeightOf(this.partyNodeIds.get(validation.partyId()).longValue())).sum();
    }

    private void ensureHintsKeyPublished(Bytes crs) {
        if (this.publicationFuture == null && this.weights.targetIncludes(this.selfId) && !this.nodePartyIds.containsKey(this.selfId)) {
            int selfPartyId = this.expectedPartyId(this.selfId);
            this.publicationFuture = CompletableFuture.runAsync(() -> {
                try {
                    Bytes hints = this.library.computeHints(crs, this.blsPrivateKey, selfPartyId, this.numParties);
                    this.submissions.submitHintsKey(selfPartyId, this.numParties, hints).join();
                }
                catch (CancellationException hints) {
                }
                catch (Exception e) {
                    log.error("Failed to publish hinTS key", (Throwable)e);
                }
            }, this.executor);
        }
    }

    private CompletableFuture<Void> startPreprocessingVoteFuture(@NonNull Instant cutoff, Bytes crs) {
        return CompletableFuture.runAsync(() -> {
            try {
                TreeMap hintKeys = this.validationFutures.headMap(cutoff, true).values().stream().map(CompletableFuture::join).filter(Validation::isValid).collect(Collectors.toMap(Validation::partyId, Validation::hintsKey, (a, b) -> a, TreeMap::new));
                TreeMap aggregatedWeights = this.nodePartyIds.entrySet().stream().filter(entry -> hintKeys.containsKey(entry.getValue())).collect(Collectors.toMap(Map.Entry::getValue, entry -> this.weights.targetWeightOf(((Long)entry.getKey()).longValue()), (a, b) -> a, TreeMap::new));
                AggregationAndVerificationKeys output = this.library.preprocess(crs, hintKeys, aggregatedWeights, this.numParties);
                PreprocessedKeys preprocessedKeys = PreprocessedKeys.newBuilder().verificationKey(Bytes.wrap((byte[])output.verificationKey())).aggregationKey(Bytes.wrap((byte[])output.aggregationKey())).build();
                long congruentNodeId = -1L;
                for (Map.Entry<Long, PreprocessingVote> entry2 : this.votes.entrySet()) {
                    if (!entry2.getValue().preprocessedKeysOrThrow().equals((Object)preprocessedKeys)) continue;
                    congruentNodeId = entry2.getKey();
                    break;
                }
                if (congruentNodeId != -1L) {
                    log.info("Voting for congruent node's preprocessed keys: {}", (Object)congruentNodeId);
                    this.submissions.submitHintsVote(this.construction.constructionId(), congruentNodeId).join();
                } else {
                    log.info("Voting for own preprocessed keys");
                    this.submissions.submitHintsVote(this.construction.constructionId(), preprocessedKeys).join();
                }
            }
            catch (CancellationException hintKeys) {
            }
            catch (Exception e) {
                log.error("Failed to submit preprocessing vote", (Throwable)e);
            }
        }, this.executor);
    }

    @VisibleForTesting
    public void setFinalCrsFuture(@Nullable CompletableFuture<CRSValidation> finalCrsFuture) {
        this.finalCrsFuture = finalCrsFuture;
    }

    public static CrsUpdateOutput decodeCrsUpdate(long oldCrsLength, @NonNull Bytes output) {
        Objects.requireNonNull(output);
        Bytes crs = output.slice(0L, oldCrsLength);
        Bytes proof = output.slice(oldCrsLength, output.length() - oldCrsLength);
        return new CrsUpdateOutput(crs, proof);
    }

    public record CRSValidation(@NonNull Bytes crs, long weightContributedSoFar) {
    }

    public record CrsUpdateOutput(@NonNull Bytes crs, @NonNull Bytes proof) {
    }

    private record Validation(int partyId, @NonNull Bytes hintsKey, boolean isValid) {
    }
}

