/*
 * Decompiled with CFR 0.152.
 */
package com.hedera.node.app.blocks.impl.streaming;

import com.hedera.hapi.block.stream.BlockItem;
import com.hedera.hapi.block.stream.BlockProof;
import com.hedera.node.app.blocks.impl.streaming.BlockBufferService;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeClientFactory;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeConnectionManager;
import com.hedera.node.app.blocks.impl.streaming.BlockNodeStats;
import com.hedera.node.app.blocks.impl.streaming.BlockState;
import com.hedera.node.app.metrics.BlockStreamMetrics;
import com.hedera.node.config.ConfigProvider;
import com.hedera.node.config.data.BlockNodeConnectionConfig;
import com.hedera.node.internal.network.BlockNodeConfig;
import com.hedera.pbj.grpc.client.helidon.PbjGrpcClientConfig;
import com.hedera.pbj.runtime.grpc.GrpcException;
import com.hedera.pbj.runtime.grpc.Pipeline;
import com.hedera.pbj.runtime.grpc.ServiceInterface;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import io.helidon.common.tls.Tls;
import io.helidon.common.tls.TlsConfig;
import io.helidon.webclient.api.WebClient;
import io.helidon.webclient.api.WebClientConfig;
import io.helidon.webclient.grpc.GrpcClientProtocolConfig;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Flow;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.block.api.BlockEnd;
import org.hiero.block.api.BlockItemSet;
import org.hiero.block.api.BlockStreamPublishServiceInterface;
import org.hiero.block.api.PublishStreamRequest;
import org.hiero.block.api.PublishStreamResponse;

public class BlockNodeConnection
implements Pipeline<PublishStreamResponse> {
    private static final Logger logger = LogManager.getLogger(BlockNodeConnection.class);
    private static final int MAX_BYTES_PER_REQUEST = 0x200000;
    private static final Options OPTIONS = new Options(Optional.empty(), "application/grpc");
    private static final AtomicLong connectionIdCounter = new AtomicLong(0L);
    public static final Duration THIRTY_SECONDS = Duration.ofSeconds(30L);
    private final BlockNodeConfig blockNodeConfig;
    private final BlockNodeConnectionManager blockNodeConnectionManager;
    private final BlockBufferService blockBufferService;
    private final BlockStreamMetrics blockStreamMetrics;
    private final Duration streamResetPeriod;
    private final AtomicBoolean streamShutdownInProgress = new AtomicBoolean(false);
    private BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient blockStreamPublishServiceClient;
    private final AtomicReference<Pipeline<? super PublishStreamRequest>> requestPipelineRef = new AtomicReference();
    private final AtomicReference<ConnectionState> connectionState;
    private final ScheduledExecutorService executorService;
    private ScheduledFuture<?> streamResetTask;
    private final String connectionId;
    private final AtomicLong streamingBlockNumber = new AtomicLong(-1L);
    private final AtomicReference<Thread> workerThreadRef = new AtomicReference();
    private final ConfigProvider configProvider;
    private final BlockNodeClientFactory clientFactory;

    public BlockNodeConnection(@NonNull ConfigProvider configProvider, @NonNull BlockNodeConfig nodeConfig, @NonNull BlockNodeConnectionManager blockNodeConnectionManager, @NonNull BlockBufferService blockBufferService, @NonNull BlockStreamMetrics blockStreamMetrics, @NonNull ScheduledExecutorService executorService, @Nullable Long initialBlockToStream, @NonNull BlockNodeClientFactory clientFactory) {
        this.configProvider = Objects.requireNonNull(configProvider, "configProvider must not be null");
        this.blockNodeConfig = Objects.requireNonNull(nodeConfig, "nodeConfig must not be null");
        this.blockNodeConnectionManager = Objects.requireNonNull(blockNodeConnectionManager, "blockNodeConnectionManager must not be null");
        this.blockBufferService = Objects.requireNonNull(blockBufferService, "blockBufferService must not be null");
        this.blockStreamMetrics = Objects.requireNonNull(blockStreamMetrics, "blockStreamMetrics must not be null");
        this.connectionState = new AtomicReference<ConnectionState>(ConnectionState.UNINITIALIZED);
        this.executorService = Objects.requireNonNull(executorService, "executorService must not be null");
        BlockNodeConnectionConfig blockNodeConnectionConfig = (BlockNodeConnectionConfig)configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class);
        this.streamResetPeriod = blockNodeConnectionConfig.streamResetPeriod();
        this.clientFactory = Objects.requireNonNull(clientFactory, "clientFactory must not be null");
        this.connectionId = String.format("%04d", connectionIdCounter.incrementAndGet());
        if (initialBlockToStream != null) {
            this.streamingBlockNumber.set(initialBlockToStream);
            logger.info("Block node connection will initially stream with block {}", (Object)initialBlockToStream);
        }
    }

    public synchronized void createRequestPipeline() {
        if (this.requestPipelineRef.get() == null) {
            this.blockStreamPublishServiceClient = this.createNewGrpcClient();
            Pipeline pipeline = this.blockStreamPublishServiceClient.publishBlockStream((Pipeline)this);
            this.requestPipelineRef.set((Pipeline<? super PublishStreamRequest>)pipeline);
            logger.debug("{} Request pipeline initialized.", (Object)this);
            this.updateConnectionState(ConnectionState.PENDING);
            this.blockStreamMetrics.recordConnectionOpened();
        } else {
            logger.debug("{} Request pipeline already available.", (Object)this);
        }
    }

    @NonNull
    private BlockStreamPublishServiceInterface.BlockStreamPublishServiceClient createNewGrpcClient() {
        Duration timeoutDuration = ((BlockNodeConnectionConfig)this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).grpcOverallTimeout();
        Tls tls = ((TlsConfig.Builder)Tls.builder().enabled(false)).build();
        PbjGrpcClientConfig grpcConfig = new PbjGrpcClientConfig(timeoutDuration, tls, Optional.of(""), "application/grpc");
        WebClient webClient = ((WebClientConfig.Builder)((WebClientConfig.Builder)((WebClientConfig.Builder)((WebClientConfig.Builder)WebClient.builder().baseUri("http://" + this.blockNodeConfig.address() + ":" + this.blockNodeConfig.port())).tls(tls)).protocolConfigs(List.of(((GrpcClientProtocolConfig.Builder)((GrpcClientProtocolConfig.Builder)GrpcClientProtocolConfig.builder().abortPollTimeExpired(false)).pollWaitTime(timeoutDuration)).build()))).connectTimeout(timeoutDuration)).build();
        if (logger.isDebugEnabled()) {
            logger.debug("Created BlockStreamPublishServiceClient for {}:{}.", (Object)this.blockNodeConfig.address(), (Object)this.blockNodeConfig.port());
        }
        return this.clientFactory.createClient(webClient, grpcConfig, OPTIONS);
    }

    public void updateConnectionState(@NonNull ConnectionState newState) {
        this.updateConnectionState(null, newState);
    }

    /*
     * Enabled aggressive block sorting
     */
    private boolean updateConnectionState(@Nullable ConnectionState expectedCurrentState, @NonNull ConnectionState newState) {
        Objects.requireNonNull(newState, "newState must not be null");
        if (expectedCurrentState != null) {
            if (!this.connectionState.compareAndSet(expectedCurrentState, newState)) {
                logger.debug("{} Failed to transition state from {} to {} because current state does not match expected state.", (Object)this, (Object)expectedCurrentState, (Object)newState);
                return false;
            }
            logger.info("{} Connection state transitioned from {} to {}.", (Object)this, (Object)expectedCurrentState, (Object)newState);
        } else {
            ConnectionState oldState = this.connectionState.getAndSet(newState);
            logger.info("{} Connection state transitioned from {} to {}.", (Object)this, (Object)oldState, (Object)newState);
        }
        if (newState == ConnectionState.ACTIVE) {
            this.scheduleStreamReset();
            Thread workerThread = new Thread((Runnable)new ConnectionWorkerLoopTask(), "bn-conn-worker-" + this.connectionId);
            if (!this.workerThreadRef.compareAndSet(null, workerThread)) return true;
            workerThread.start();
            return true;
        }
        this.cancelStreamReset();
        return true;
    }

    private void scheduleStreamReset() {
        if (this.streamResetTask != null && !this.streamResetTask.isDone()) {
            this.streamResetTask.cancel(false);
        }
        this.streamResetTask = this.executorService.scheduleAtFixedRate(this::performStreamReset, this.streamResetPeriod.toMillis(), this.streamResetPeriod.toMillis(), TimeUnit.MILLISECONDS);
        logger.debug("{} Scheduled periodic stream reset every {}.", (Object)this, (Object)this.streamResetPeriod);
    }

    private void performStreamReset() {
        if (this.getConnectionState() == ConnectionState.ACTIVE) {
            logger.info("{} Performing scheduled stream reset.", (Object)this);
            this.endTheStreamWith(PublishStreamRequest.EndStream.Code.RESET);
            this.blockNodeConnectionManager.selectNewBlockNodeForStreaming(false);
        }
    }

    private void cancelStreamReset() {
        if (this.streamResetTask != null) {
            this.streamResetTask.cancel(false);
            this.streamResetTask = null;
            logger.debug("{} Cancelled periodic stream reset.", (Object)this);
        }
    }

    private void closeAndReschedule(@Nullable Duration delay, boolean callOnComplete) {
        this.close(callOnComplete);
        this.blockNodeConnectionManager.rescheduleConnection(this, delay, null, true);
    }

    private void endStreamAndReschedule(@NonNull PublishStreamRequest.EndStream.Code code) {
        Objects.requireNonNull(code, "code must not be null");
        this.endTheStreamWith(code);
        this.blockNodeConnectionManager.rescheduleConnection(this, THIRTY_SECONDS, null, true);
    }

    private void closeAndRestart(long blockNumber) {
        this.close(true);
        this.blockNodeConnectionManager.rescheduleConnection(this, null, blockNumber, false);
    }

    public void handleStreamFailure() {
        logger.debug("{} Handling failed stream.", (Object)this);
        this.closeAndReschedule(THIRTY_SECONDS, true);
    }

    public void handleStreamFailureWithoutOnComplete() {
        logger.debug("{} Handling failed stream without onComplete.", (Object)this);
        this.closeAndReschedule(THIRTY_SECONDS, false);
    }

    private void handleAcknowledgement(@NonNull PublishStreamResponse.BlockAcknowledgement acknowledgement) {
        long acknowledgedBlockNumber = acknowledgement.blockNumber();
        logger.debug("{} BlockAcknowledgement received for block {}.", (Object)this, (Object)acknowledgedBlockNumber);
        this.acknowledgeBlocks(acknowledgedBlockNumber, true);
        BlockNodeStats.HighLatencyResult result = this.blockNodeConnectionManager.recordBlockAckAndCheckLatency(this.blockNodeConfig, acknowledgedBlockNumber, Instant.now());
        if (result.shouldSwitch() && !this.blockNodeConnectionManager.isOnlyOneBlockNodeConfigured()) {
            if (logger.isInfoEnabled()) {
                logger.info("{} Block node has exceeded high latency threshold {} times consecutively.", (Object)this, (Object)result.consecutiveHighLatencyEvents());
            }
            this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TIMEOUT);
        }
    }

    private void acknowledgeBlocks(long acknowledgedBlockNumber, boolean maybeJumpToBlock) {
        logger.debug("{} Acknowledging blocks <= {}.", (Object)this, (Object)acknowledgedBlockNumber);
        long currentBlockStreaming = this.streamingBlockNumber.get();
        long currentBlockProducing = this.blockBufferService.getLastBlockNumberProduced();
        this.blockBufferService.setLatestAcknowledgedBlock(acknowledgedBlockNumber);
        if (maybeJumpToBlock && (acknowledgedBlockNumber > currentBlockProducing || acknowledgedBlockNumber > currentBlockStreaming)) {
            long blockToJumpTo = acknowledgedBlockNumber + 1L;
            logger.debug("{} Received acknowledgement for block {}, later than current streamed ({}) or produced ({}).", (Object)this, (Object)acknowledgedBlockNumber, (Object)currentBlockStreaming, (Object)currentBlockProducing);
            this.streamingBlockNumber.updateAndGet(current -> Math.max(current, blockToJumpTo));
        }
    }

    private void handleEndOfStream(@NonNull PublishStreamResponse.EndOfStream endOfStream) {
        Objects.requireNonNull(endOfStream, "endOfStream must not be null");
        long blockNumber = endOfStream.blockNumber();
        PublishStreamResponse.EndOfStream.Code responseCode = endOfStream.status();
        logger.info("{} Received EndOfStream response (block={}, responseCode={}).", (Object)this, (Object)blockNumber, (Object)responseCode);
        this.acknowledgeBlocks(blockNumber, false);
        if (this.blockNodeConnectionManager.recordEndOfStreamAndCheckLimit(this.blockNodeConfig, Instant.now())) {
            if (logger.isInfoEnabled()) {
                logger.info("{} Block node has exceeded the allowed number of EndOfStream responses (received={}, permitted={}, timeWindow={}). Reconnection scheduled for {}.", (Object)this, (Object)this.blockNodeConnectionManager.getEndOfStreamCount(this.blockNodeConfig), (Object)this.blockNodeConnectionManager.getMaxEndOfStreamsAllowed(), (Object)this.blockNodeConnectionManager.getEndOfStreamTimeframe(), (Object)this.blockNodeConnectionManager.getEndOfStreamScheduleDelay());
            }
            this.blockStreamMetrics.recordEndOfStreamLimitExceeded();
            this.closeAndReschedule(this.blockNodeConnectionManager.getEndOfStreamScheduleDelay(), true);
            return;
        }
        switch (responseCode) {
            case ERROR: 
            case PERSISTENCE_FAILED: {
                logger.debug("{} Block node reported an error at block {}. Will attempt to reestablish the stream later.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case TIMEOUT: 
            case DUPLICATE_BLOCK: 
            case BAD_BLOCK_PROOF: 
            case INVALID_REQUEST: {
                long restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                logger.debug("{} Block node reported status indicating immediate restart should be attempted. Will restart stream at block {}.", (Object)this, (Object)restartBlockNumber);
                this.closeAndRestart(restartBlockNumber);
                break;
            }
            case SUCCESS: {
                logger.info("{} Block node orderly ended the stream at block {}.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
                break;
            }
            case BEHIND: {
                long restartBlockNumber;
                long l = restartBlockNumber = blockNumber == Long.MAX_VALUE ? 0L : blockNumber + 1L;
                if (this.blockBufferService.getBlockState(restartBlockNumber) != null) {
                    logger.debug("{} Block node reported it is behind. Will restart stream at block {}.", (Object)this, (Object)restartBlockNumber);
                    this.closeAndRestart(restartBlockNumber);
                    break;
                }
                logger.debug("{} Block node is behind and block state is not available. Ending the stream.", (Object)this);
                this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
                break;
            }
            case UNKNOWN: {
                logger.debug("{} Block node reported an unknown error at block {}.", (Object)this, (Object)blockNumber);
                this.closeAndReschedule(THIRTY_SECONDS, true);
            }
        }
    }

    private void handleSkipBlock(@NonNull PublishStreamResponse.SkipBlock skipBlock) {
        long nextBlock;
        Objects.requireNonNull(skipBlock, "skipBlock must not be null");
        long skipBlockNumber = skipBlock.blockNumber();
        long activeBlockNumber = this.streamingBlockNumber.get();
        if (skipBlockNumber == activeBlockNumber && this.streamingBlockNumber.compareAndSet(activeBlockNumber, nextBlock = skipBlockNumber + 1L)) {
            logger.debug("{} Received SkipBlock response; skipping to block {}", (Object)this, (Object)nextBlock);
            return;
        }
        logger.debug("{} Received SkipBlock response (blockToSkip={}), but we've moved on to another block. Ignoring skip request", (Object)this, (Object)skipBlockNumber);
    }

    private void handleResendBlock(@NonNull PublishStreamResponse.ResendBlock resendBlock) {
        Objects.requireNonNull(resendBlock, "resendBlock must not be null");
        long resendBlockNumber = resendBlock.blockNumber();
        logger.debug("{} Received ResendBlock response for block {}.", (Object)this, (Object)resendBlockNumber);
        if (this.blockBufferService.getBlockState(resendBlockNumber) != null) {
            this.streamingBlockNumber.set(resendBlockNumber);
        } else {
            logger.debug("{} Block node requested a ResendBlock for block {} but that block does not exist on this consensus node. Closing connection and will retry later.", (Object)this, (Object)resendBlockNumber);
            this.closeAndReschedule(THIRTY_SECONDS, true);
        }
    }

    public void endTheStreamWith(PublishStreamRequest.EndStream.Code code) {
        long earliestBlockNumber = this.blockBufferService.getEarliestAvailableBlockNumber();
        long highestAckedBlockNumber = this.blockBufferService.getHighestAckedBlockNumber();
        PublishStreamRequest endStream = PublishStreamRequest.newBuilder().endStream(PublishStreamRequest.EndStream.newBuilder().endCode(code).earliestBlockNumber(earliestBlockNumber).latestBlockNumber(highestAckedBlockNumber)).build();
        logger.info("{} Sending EndStream (code={}, earliestBlock={}, latestAcked={}).", (Object)this, (Object)code, (Object)earliestBlockNumber, (Object)highestAckedBlockNumber);
        try {
            this.sendRequest(endStream);
        }
        catch (RuntimeException e) {
            logger.warn("{} Error sending EndStream request", (Object)this, (Object)e);
        }
        this.close(true);
    }

    public boolean sendRequest(@NonNull PublishStreamRequest request) {
        return this.sendRequest(-1L, -1, request);
    }

    private boolean sendRequest(long blockNumber, int requestNumber, @NonNull PublishStreamRequest request) {
        block13: {
            Objects.requireNonNull(request, "request must not be null");
            Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
            if (this.getConnectionState() == ConnectionState.ACTIVE && pipeline != null) {
                try {
                    if (blockNumber == -1L && requestNumber == -1) {
                        logger.debug("{} Sending ad hoc request to block node (type={})", (Object)this, (Object)request.request().kind());
                    } else {
                        logger.debug("{} [block={}, request={}] Sending request to block node (type={})", (Object)this, (Object)blockNumber, (Object)requestNumber, (Object)request.request().kind());
                    }
                    long startMs = System.currentTimeMillis();
                    pipeline.onNext((Object)request);
                    long durationMs = System.currentTimeMillis() - startMs;
                    this.blockStreamMetrics.recordRequestLatency(durationMs);
                    if (blockNumber == -1L && requestNumber == -1) {
                        logger.trace("{} Ad hoc request took {}ms to send", (Object)this, (Object)durationMs);
                    } else {
                        logger.trace("{} [block={}, request={}] Request took {}ms to send", (Object)this, (Object)blockNumber, (Object)requestNumber, (Object)durationMs);
                    }
                    if (request.hasEndStream()) {
                        this.blockStreamMetrics.recordRequestEndStreamSent(request.endStream().endCode());
                    } else if (request.hasBlockItems()) {
                        this.blockStreamMetrics.recordRequestSent((PublishStreamRequest.RequestOneOfType)request.request().kind());
                        BlockItemSet itemSet = request.blockItems();
                        if (itemSet != null) {
                            List items = itemSet.blockItems();
                            this.blockStreamMetrics.recordBlockItemsSent(items.size());
                            for (BlockItem item : items) {
                                BlockProof blockProof = item.blockProof();
                                if (blockProof == null) continue;
                                this.blockNodeConnectionManager.recordBlockProofSent(this.blockNodeConfig, blockProof.block(), Instant.now());
                            }
                        }
                    } else {
                        this.blockStreamMetrics.recordRequestSent((PublishStreamRequest.RequestOneOfType)request.request().kind());
                    }
                    return true;
                }
                catch (RuntimeException e) {
                    if (this.getConnectionState() != ConnectionState.ACTIVE) break block13;
                    this.blockStreamMetrics.recordRequestSendFailure();
                    throw e;
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void close(boolean callOnComplete) {
        ConnectionState connState = this.getConnectionState();
        if (connState.isTerminal()) {
            logger.debug("{} Connection already in terminal state ({}).", (Object)this, (Object)connState);
            return;
        }
        if (!this.updateConnectionState(connState, ConnectionState.CLOSING)) {
            logger.debug("{} State changed while trying to close connection. Aborting close attempt.", (Object)this);
            return;
        }
        logger.debug("{} Closing connection.", (Object)this);
        try {
            this.closePipeline(callOnComplete);
            logger.debug("{} Connection successfully closed.", (Object)this);
        }
        catch (RuntimeException e) {
            logger.warn("{} Error occurred while attempting to close connection.", (Object)this, (Object)e);
        }
        finally {
            try {
                if (this.blockStreamPublishServiceClient != null) {
                    this.blockStreamPublishServiceClient.close();
                }
            }
            catch (Exception e) {
                logger.error("{} Error occurred while closing gRPC client.", (Object)this, (Object)e);
            }
            this.blockStreamMetrics.recordConnectionClosed();
            this.blockStreamMetrics.recordActiveConnectionIp(-1L);
            this.blockNodeConnectionManager.notifyConnectionClosed(this);
            this.updateConnectionState(ConnectionState.CLOSED);
        }
    }

    private void closePipeline(boolean callOnComplete) {
        Pipeline<? super PublishStreamRequest> pipeline = this.requestPipelineRef.get();
        if (pipeline != null) {
            logger.debug("{} Closing request pipeline for block node.", (Object)this);
            this.streamShutdownInProgress.set(true);
            try {
                ConnectionState state = this.getConnectionState();
                if (state == ConnectionState.CLOSING && callOnComplete) {
                    pipeline.onComplete();
                    logger.debug("{} Request pipeline successfully closed.", (Object)this);
                }
            }
            catch (Exception e) {
                logger.warn("{} Error while completing request pipeline.", (Object)this, (Object)e);
            }
            logger.debug("{} Request pipeline removed.", (Object)this);
            this.requestPipelineRef.compareAndSet(pipeline, null);
        }
    }

    public BlockNodeConfig getNodeConfig() {
        return this.blockNodeConfig;
    }

    public void onSubscribe(Flow.Subscription subscription) {
        logger.debug("{} OnSubscribe invoked.", (Object)this);
        subscription.request(Long.MAX_VALUE);
    }

    public void clientEndStreamReceived() {
        logger.debug("{} Client End Stream received.", (Object)this);
        super.clientEndStreamReceived();
    }

    public void onNext(@NonNull PublishStreamResponse response) {
        Objects.requireNonNull(response, "response must not be null");
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            logger.debug("{} onNext invoked but connection is already closed ({}).", (Object)this, (Object)response);
            return;
        }
        if (response.hasAcknowledgement()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.handleAcknowledgement(response.acknowledgement());
        } else if (response.hasEndStream()) {
            this.blockStreamMetrics.recordResponseEndOfStreamReceived(response.endStream().status());
            this.blockStreamMetrics.recordLatestBlockEndOfStream(response.endStream().blockNumber());
            this.handleEndOfStream(response.endStream());
        } else if (response.hasSkipBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.blockStreamMetrics.recordLatestBlockSkipBlock(response.skipBlock().blockNumber());
            this.handleSkipBlock(response.skipBlock());
        } else if (response.hasResendBlock()) {
            this.blockStreamMetrics.recordResponseReceived((PublishStreamResponse.ResponseOneOfType)response.response().kind());
            this.blockStreamMetrics.recordLatestBlockResendBlock(response.resendBlock().blockNumber());
            this.handleResendBlock(response.resendBlock());
        } else {
            this.blockStreamMetrics.recordUnknownResponseReceived();
            logger.debug("{} Unexpected response received: {}.", (Object)this, (Object)response);
        }
    }

    public void onError(Throwable error) {
        if (!this.getConnectionState().isTerminal()) {
            this.blockStreamMetrics.recordConnectionOnError();
            if (error instanceof GrpcException) {
                GrpcException grpcException = (GrpcException)error;
                if (logger.isWarnEnabled()) {
                    logger.warn("{} Error received (grpcStatus={}).", (Object)this, (Object)grpcException.status(), (Object)grpcException);
                }
            } else if (logger.isWarnEnabled()) {
                logger.warn("{} Error received.", (Object)this, (Object)error);
            }
            this.handleStreamFailure();
        }
    }

    public void onComplete() {
        this.blockStreamMetrics.recordConnectionOnComplete();
        if (this.getConnectionState() == ConnectionState.CLOSED) {
            logger.debug("{} onComplete invoked but connection is already closed.", (Object)this);
            return;
        }
        if (this.streamShutdownInProgress.getAndSet(false)) {
            logger.debug("{} Stream completed (stream close was in progress).", (Object)this);
        } else {
            logger.debug("{} Stream completed unexpectedly.", (Object)this);
            this.handleStreamFailure();
        }
    }

    @NonNull
    public ConnectionState getConnectionState() {
        return this.connectionState.get();
    }

    public String toString() {
        return "[" + this.connectionId + "/" + this.blockNodeConfig.address() + ":" + this.blockNodeConfig.port() + "/" + String.valueOf((Object)this.getConnectionState()) + "]";
    }

    public boolean equals(Object o) {
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BlockNodeConnection that = (BlockNodeConnection)o;
        return Objects.equals(this.connectionId, that.connectionId) && Objects.equals(this.blockNodeConfig, that.blockNodeConfig);
    }

    public int hashCode() {
        return Objects.hash(this.blockNodeConfig, this.connectionId);
    }

    public static enum ConnectionState {
        UNINITIALIZED(false),
        PENDING(false),
        ACTIVE(false),
        CLOSING(true),
        CLOSED(true);

        private final boolean isTerminal;

        private ConnectionState(boolean isTerminal) {
            this.isTerminal = isTerminal;
        }

        boolean isTerminal() {
            return this.isTerminal;
        }
    }

    private record Options(Optional<String> authority, String contentType) implements ServiceInterface.RequestOptions
    {
    }

    private class ConnectionWorkerLoopTask
    implements Runnable {
        private static final int BYTES_PADDING = 100;
        private final List<BlockItem> pendingRequestItems = new ArrayList<BlockItem>();
        private long pendingRequestBytes = 100L;
        private int itemIndex = 0;
        private BlockState block;
        private long lastSendTimeMillis = -1L;
        private final AtomicInteger requestCtr = new AtomicInteger(1);

        private ConnectionWorkerLoopTask() {
        }

        @Override
        public void run() {
            logger.info("{} Worker thread started", (Object)BlockNodeConnection.this);
            block3: while (true) {
                try {
                    while (!BlockNodeConnection.this.connectionState.get().isTerminal()) {
                        this.doWork();
                        if (BlockNodeConnection.this.connectionState.get().isTerminal()) break block3;
                        Thread.sleep(this.connectionWorkerSleepMillis());
                    }
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.warn("{} Worker loop was interrupted", (Object)BlockNodeConnection.this);
                    continue;
                }
                catch (Exception e) {
                    logger.warn("{} Error caught in connection worker loop", (Object)BlockNodeConnection.this, (Object)e);
                    continue;
                }
                break;
            }
            logger.info("Worker thread exiting");
            BlockNodeConnection.this.workerThreadRef.compareAndSet(Thread.currentThread(), null);
        }

        private void doWork() {
            long diffMillis;
            BlockItem item;
            this.switchBlockIfNeeded();
            if (this.block == null) {
                return;
            }
            while ((item = this.block.blockItem(this.itemIndex)) != null) {
                int itemSize;
                long newRequestBytes;
                if (this.itemIndex == 0) {
                    logger.trace("{} Starting to process items for block {}", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber());
                }
                if ((newRequestBytes = this.pendingRequestBytes + (long)(itemSize = item.protobufSize() + 5)) > 0x200000L) {
                    if (!this.pendingRequestItems.isEmpty()) {
                        if (this.sendPendingRequest()) continue;
                        break;
                    }
                    logger.error("{} !!! FATAL: Request would contain a block item that is too big to send (block={}, itemIndex={}, expectedRequestSize={}, maxAllowed={}). Closing connection.", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.itemIndex, (Object)newRequestBytes, (Object)0x200000);
                    BlockNodeConnection.this.endTheStreamWith(PublishStreamRequest.EndStream.Code.ERROR);
                    break;
                }
                this.pendingRequestItems.add(item);
                this.pendingRequestBytes = newRequestBytes;
                ++this.itemIndex;
            }
            if (!this.pendingRequestItems.isEmpty() && (diffMillis = System.currentTimeMillis() - this.lastSendTimeMillis) >= this.maxRequestDelayMillis()) {
                this.sendPendingRequest();
            }
            if (this.pendingRequestItems.isEmpty() && this.block.isClosed() && this.block.itemCount() == this.itemIndex) {
                PublishStreamRequest endOfBlock = PublishStreamRequest.newBuilder().endOfBlock(BlockEnd.newBuilder().blockNumber(this.block.blockNumber())).build();
                try {
                    BlockNodeConnection.this.sendRequest(endOfBlock);
                }
                catch (RuntimeException e) {
                    logger.warn("{} Error sending EndOfBlock request", (Object)BlockNodeConnection.this, (Object)e);
                    BlockNodeConnection.this.handleStreamFailureWithoutOnComplete();
                }
                long nextBlockNumber = this.block.blockNumber() + 1L;
                if (BlockNodeConnection.this.streamingBlockNumber.compareAndSet(this.block.blockNumber(), nextBlockNumber)) {
                    logger.trace("{} Advancing to block {}", (Object)BlockNodeConnection.this, (Object)nextBlockNumber);
                } else {
                    logger.trace("{} Tried to advance to block {} but the block to stream was updated externally", (Object)BlockNodeConnection.this, (Object)nextBlockNumber);
                }
            }
        }

        private boolean sendPendingRequest() {
            BlockItemSet itemSet = BlockItemSet.newBuilder().blockItems(List.copyOf(this.pendingRequestItems)).build();
            PublishStreamRequest req = PublishStreamRequest.newBuilder().blockItems(itemSet).build();
            if (logger.isTraceEnabled()) {
                logger.trace("{} Attempting to send request (block={}, request={}, itemCount={}, estimatedBytes={} actualBytes={})", (Object)BlockNodeConnection.this, (Object)this.block.blockNumber(), (Object)this.requestCtr.get(), (Object)this.pendingRequestItems.size(), (Object)this.pendingRequestBytes, (Object)req.protobufSize());
            }
            try {
                if (BlockNodeConnection.this.sendRequest(this.block.blockNumber(), this.requestCtr.get(), req)) {
                    this.lastSendTimeMillis = System.currentTimeMillis();
                    this.pendingRequestBytes = 100L;
                    this.pendingRequestItems.clear();
                    this.requestCtr.incrementAndGet();
                    return true;
                }
            }
            catch (UncheckedIOException e) {
                logger.debug("{} UncheckedIOException caught in connection worker thread", (Object)BlockNodeConnection.this, (Object)e);
                BlockNodeConnection.this.handleStreamFailureWithoutOnComplete();
            }
            catch (Exception e) {
                logger.debug("{} Exception caught in connection worker thread", (Object)BlockNodeConnection.this, (Object)e);
                BlockNodeConnection.this.handleStreamFailure();
            }
            return false;
        }

        private void switchBlockIfNeeded() {
            long latestActiveBlockNumber;
            long latestBlock;
            long activeBlockNum = BlockNodeConnection.this.streamingBlockNumber.get();
            if (activeBlockNum == -1L && BlockNodeConnection.this.streamingBlockNumber.compareAndSet(-1L, latestBlock = BlockNodeConnection.this.blockBufferService.getLastBlockNumberProduced())) {
                logger.info("{} Connection was not initialized with a starting block; defaulting to latest block produced ({})", (Object)BlockNodeConnection.this, (Object)latestBlock);
            }
            if ((latestActiveBlockNumber = BlockNodeConnection.this.streamingBlockNumber.get()) == -1L) {
                return;
            }
            if (this.block != null && this.block.blockNumber() == latestActiveBlockNumber) {
                return;
            }
            if (logger.isTraceEnabled()) {
                long oldBlock = this.block == null ? -1L : this.block.blockNumber();
                logger.trace("{} Worker switching from block {} to block {}", (Object)BlockNodeConnection.this, (Object)oldBlock, (Object)latestActiveBlockNumber);
            }
            this.block = BlockNodeConnection.this.blockBufferService.getBlockState(latestActiveBlockNumber);
            if (this.block == null && latestActiveBlockNumber < BlockNodeConnection.this.blockBufferService.getEarliestAvailableBlockNumber()) {
                BlockNodeConnection.this.endStreamAndReschedule(PublishStreamRequest.EndStream.Code.TOO_FAR_BEHIND);
            }
            this.pendingRequestBytes = 100L;
            this.itemIndex = 0;
            this.pendingRequestItems.clear();
            this.requestCtr.set(1);
        }

        private long maxRequestDelayMillis() {
            return ((BlockNodeConnectionConfig)BlockNodeConnection.this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).maxRequestDelay().toMillis();
        }

        private long connectionWorkerSleepMillis() {
            return ((BlockNodeConnectionConfig)BlockNodeConnection.this.configProvider.getConfiguration().getConfigData(BlockNodeConnectionConfig.class)).connectionWorkerSleepDuration().toMillis();
        }
    }
}

