/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.platform.reconnect;

import com.hedera.hapi.node.base.SemanticVersion;
import com.hedera.hapi.node.state.roster.Roster;
import com.swirlds.base.state.Startable;
import com.swirlds.base.time.Time;
import com.swirlds.common.context.PlatformContext;
import com.swirlds.common.merkle.crypto.MerkleCryptography;
import com.swirlds.common.merkle.synchronization.config.ReconnectConfig;
import com.swirlds.common.threading.framework.config.ThreadConfiguration;
import com.swirlds.common.threading.manager.ThreadManager;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.logging.legacy.payload.ReconnectFailurePayload;
import com.swirlds.logging.legacy.payload.UnableToReconnectPayload;
import com.swirlds.platform.components.SavedStateController;
import com.swirlds.platform.network.protocol.ReservedSignedStatePromise;
import com.swirlds.platform.reconnect.DefaultSignedStateValidator;
import com.swirlds.platform.reconnect.StateSyncException;
import com.swirlds.platform.state.ConsensusStateEventHandler;
import com.swirlds.platform.state.SwirldStateManager;
import com.swirlds.platform.state.service.PlatformStateFacade;
import com.swirlds.platform.state.signed.ReservedSignedState;
import com.swirlds.platform.state.signed.SignedState;
import com.swirlds.platform.state.signed.SignedStateValidationData;
import com.swirlds.platform.state.signed.SignedStateValidator;
import com.swirlds.platform.state.snapshot.SignedStateFileReader;
import com.swirlds.platform.system.InitTrigger;
import com.swirlds.platform.system.Platform;
import com.swirlds.platform.system.SystemExitCode;
import com.swirlds.platform.system.SystemExitUtils;
import com.swirlds.platform.wiring.PlatformCoordinator;
import com.swirlds.state.MerkleNodeState;
import com.swirlds.state.State;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Semaphore;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.base.crypto.Hash;
import org.hiero.consensus.model.node.NodeId;
import org.hiero.consensus.roster.RosterRetriever;

public class PlatformReconnecter
implements Startable {
    private static final Logger logger = LogManager.getLogger(PlatformReconnecter.class);
    private final PlatformStateFacade platformStateFacade;
    private final ThreadManager threadManager;
    private final Roster roster;
    private final SignedStateValidator validator;
    private final MerkleCryptography merkleCryptography;
    private final Semaphore threadRunning = new Semaphore(1);
    private final Platform platform;
    private final PlatformContext platformContext;
    private final PlatformCoordinator platformCoordinator;
    private final SwirldStateManager swirldStateManager;
    private final SavedStateController savedStateController;
    private final ConsensusStateEventHandler<MerkleNodeState> consensusStateEventHandler;
    private final ReservedSignedStatePromise reservedSignedStatePromise;
    private final NodeId selfId;
    private final ReconnectConfig reconnectConfig;
    private final Time time;
    private final Instant startupTime;
    private int failedReconnectsInARow;

    public PlatformReconnecter(@NonNull PlatformStateFacade platformStateFacade, @NonNull ThreadManager threadManager, @NonNull Roster roster, @NonNull MerkleCryptography merkleCryptography, @NonNull Platform platform, @NonNull PlatformContext platformContext, @NonNull PlatformCoordinator platformCoordinator, @NonNull SwirldStateManager swirldStateManager, @NonNull SavedStateController savedStateController, @NonNull ConsensusStateEventHandler<MerkleNodeState> consensusStateEventHandler, @NonNull ReservedSignedStatePromise reservedSignedStatePromise, @NonNull NodeId selfId) {
        this.platformStateFacade = Objects.requireNonNull(platformStateFacade);
        this.threadManager = Objects.requireNonNull(threadManager);
        this.roster = Objects.requireNonNull(roster);
        this.platformCoordinator = Objects.requireNonNull(platformCoordinator);
        this.reservedSignedStatePromise = Objects.requireNonNull(reservedSignedStatePromise);
        this.validator = new DefaultSignedStateValidator(platformContext, platformStateFacade);
        this.merkleCryptography = Objects.requireNonNull(merkleCryptography);
        this.reconnectConfig = (ReconnectConfig)platformContext.getConfiguration().getConfigData(ReconnectConfig.class);
        this.platform = Objects.requireNonNull(platform);
        this.platformContext = Objects.requireNonNull(platformContext);
        this.swirldStateManager = Objects.requireNonNull(swirldStateManager);
        this.savedStateController = Objects.requireNonNull(savedStateController);
        this.consensusStateEventHandler = Objects.requireNonNull(consensusStateEventHandler);
        this.time = platformContext.getTime();
        this.selfId = selfId;
        this.startupTime = this.time.now();
    }

    static void hashStateForReconnect(MerkleCryptography merkleCryptography, MerkleNodeState workingState) {
        try {
            merkleCryptography.digestTreeAsync(workingState.getRoot()).get();
        }
        catch (ExecutionException e) {
            logger.error(LogMarker.EXCEPTION.getMarker(), () -> new ReconnectFailurePayload("Error encountered while hashing state for reconnect", ReconnectFailurePayload.CauseOfFailure.ERROR).toString(), (Throwable)e);
            throw new StateSyncException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.error(LogMarker.EXCEPTION.getMarker(), () -> new ReconnectFailurePayload("Interrupted while attempting to hash state", ReconnectFailurePayload.CauseOfFailure.ERROR).toString(), (Throwable)e);
        }
    }

    public void start() {
        if (!this.threadRunning.tryAcquire()) {
            logger.error(LogMarker.EXCEPTION.getMarker(), "Attempting to start reconnect controller while it's already running");
            return;
        }
        logger.info(LogMarker.RECONNECT.getMarker(), "Starting ReconnectController");
        ((ThreadConfiguration)((ThreadConfiguration)new ThreadConfiguration(this.threadManager).setComponent("reconnect")).setThreadName("reconnect-controller")).setRunnable(() -> {
            try {
                while (!this.executeReconnect()) {
                    logger.error(LogMarker.EXCEPTION.getMarker(), "Reconnect failed, retrying");
                    Thread.sleep(((ReconnectConfig)this.platformContext.getConfiguration().getConfigData(ReconnectConfig.class)).minimumTimeBetweenReconnects().toMillis());
                }
            }
            catch (InterruptedException | RuntimeException e) {
                logger.error(LogMarker.EXCEPTION.getMarker(), "Unexpected error occurred while reconnecting", (Throwable)e);
                SystemExitUtils.exitSystem(SystemExitCode.RECONNECT_FAILURE);
            }
            finally {
                this.threadRunning.release();
            }
        }).build(true);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean executeReconnect() throws InterruptedException {
        this.exitIfReconnectIsDisabled();
        MerkleNodeState currentState = this.swirldStateManager.getConsensusState();
        this.prepareForReconnect(currentState);
        logger.info(LogMarker.RECONNECT.getMarker(), "waiting for reconnect connection");
        try {
            logger.info(LogMarker.RECONNECT.getMarker(), "acquired reconnect connection");
            try (ReservedSignedState reservedState = this.reservedSignedStatePromise.get();){
                this.validator.validate(reservedState.get(), this.roster, new SignedStateValidationData((State)currentState, this.roster, this.platformStateFacade));
                SignedStateFileReader.registerServiceStates(reservedState.get());
                this.successfulReconnect();
                if (!this.loadReconnectState(reservedState.get())) {
                    this.handleFailedReconnect();
                    boolean bl = false;
                    return bl;
                }
            }
        }
        catch (RuntimeException e) {
            this.handleFailedReconnect();
            logger.info(LogMarker.RECONNECT.getMarker(), "receiving signed state failed", (Throwable)e);
            return false;
        }
        this.platformCoordinator.resumeGossip();
        return true;
    }

    public void prepareForReconnect(MerkleNodeState currentState) {
        logger.info(LogMarker.RECONNECT.getMarker(), "Preparing for reconnect, stopping gossip");
        this.platformCoordinator.pauseGossip();
        logger.info(LogMarker.RECONNECT.getMarker(), "Preparing for reconnect, start clearing queues");
        this.platformCoordinator.clear();
        logger.info(LogMarker.RECONNECT.getMarker(), "Queues have been cleared");
        PlatformReconnecter.hashStateForReconnect(this.merkleCryptography, currentState);
    }

    public boolean loadReconnectState(@NonNull SignedState signedState) {
        try {
            Hash reconnectHash = signedState.getState().getHash();
            MerkleNodeState state = signedState.getState();
            SemanticVersion creationSoftwareVersion = this.platformStateFacade.creationSoftwareVersionOf((State)state);
            signedState.init(this.platformContext);
            this.consensusStateEventHandler.onStateInitialized(state, this.platform, InitTrigger.RECONNECT, creationSoftwareVersion);
            if (!Objects.equals(signedState.getState().getHash(), reconnectHash)) {
                throw new IllegalStateException("State hash is not permitted to change during a reconnect init() call. Previous hash was " + String.valueOf(reconnectHash) + ", new hash is " + String.valueOf(signedState.getState().getHash()));
            }
            long round = this.platformStateFacade.roundOf((State)state);
            Roster stateRoster = RosterRetriever.retrieveActive((State)state, (long)round);
            if (!this.roster.equals((Object)stateRoster)) {
                throw new IllegalStateException("Current roster and state-based roster do not contain the same nodes  (currentRoster=" + Roster.JSON.toJSON((Object)this.roster) + ") (stateRoster=" + Roster.JSON.toJSON((Object)stateRoster) + ")");
            }
            this.swirldStateManager.loadFromSignedState(signedState);
            this.platformCoordinator.loadReconnectState(this.platformContext.getConfiguration(), signedState);
            this.savedStateController.reconnectStateReceived(signedState.reserve("savedStateController.reconnectStateReceived"));
            return true;
        }
        catch (RuntimeException e) {
            logger.debug(LogMarker.RECONNECT.getMarker(), "`loadReconnectState` : FAILED, reason: {}", (Object)e.getMessage());
            this.platformCoordinator.clear();
            return false;
        }
    }

    @NonNull
    private Duration getTimeSinceStartup() {
        return Duration.between(this.startupTime, this.time.now());
    }

    public void successfulReconnect() {
        this.failedReconnectsInARow = 0;
    }

    public void handleFailedReconnect() {
        ++this.failedReconnectsInARow;
        this.killNodeIfThresholdMet();
    }

    private void killNodeIfThresholdMet() {
        if (this.failedReconnectsInARow >= this.reconnectConfig.maximumReconnectFailuresBeforeShutdown()) {
            logger.error(LogMarker.EXCEPTION.getMarker(), "Too many reconnect failures in a row, killing node");
            SystemExitUtils.exitSystem(SystemExitCode.RECONNECT_FAILURE);
        }
    }

    public void exitIfReconnectIsDisabled() {
        if (!this.reconnectConfig.active()) {
            logger.warn(LogMarker.STARTUP.getMarker(), () -> new UnableToReconnectPayload("Node has fallen behind, reconnect is disabled, will die", this.selfId.id()).toString());
            SystemExitUtils.exitSystem(SystemExitCode.BEHIND_RECONNECT_DISABLED);
        }
        if (this.reconnectConfig.reconnectWindowSeconds() >= 0 && (long)this.reconnectConfig.reconnectWindowSeconds() < this.getTimeSinceStartup().toSeconds()) {
            logger.warn(LogMarker.STARTUP.getMarker(), () -> new UnableToReconnectPayload("Node has fallen behind, reconnect is disabled outside of time window, will die", this.selfId.id()).toString());
            SystemExitUtils.exitSystem(SystemExitCode.BEHIND_RECONNECT_DISABLED);
        }
    }
}

