/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.platform.reconnect;

import com.hedera.hapi.node.base.SemanticVersion;
import com.hedera.hapi.node.state.roster.Roster;
import com.swirlds.base.time.Time;
import com.swirlds.common.context.PlatformContext;
import com.swirlds.common.merkle.crypto.MerkleCryptography;
import com.swirlds.common.merkle.synchronization.config.ReconnectConfig;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.logging.legacy.payload.ReconnectFailurePayload;
import com.swirlds.logging.legacy.payload.UnableToReconnectPayload;
import com.swirlds.platform.components.SavedStateController;
import com.swirlds.platform.network.protocol.ReservedSignedStatePromise;
import com.swirlds.platform.reconnect.DefaultSignedStateValidator;
import com.swirlds.platform.reconnect.FallenBehindMonitor;
import com.swirlds.platform.reconnect.StateSyncException;
import com.swirlds.platform.state.ConsensusStateEventHandler;
import com.swirlds.platform.state.SwirldStateManager;
import com.swirlds.platform.state.service.PlatformStateFacade;
import com.swirlds.platform.state.signed.ReservedSignedState;
import com.swirlds.platform.state.signed.SignedState;
import com.swirlds.platform.state.signed.SignedStateValidationData;
import com.swirlds.platform.state.signed.SignedStateValidator;
import com.swirlds.platform.state.snapshot.SignedStateFileReader;
import com.swirlds.platform.system.InitTrigger;
import com.swirlds.platform.system.Platform;
import com.swirlds.platform.system.SystemExitCode;
import com.swirlds.platform.system.SystemExitUtils;
import com.swirlds.platform.system.status.actions.FallenBehindAction;
import com.swirlds.platform.system.status.actions.ReconnectCompleteAction;
import com.swirlds.platform.wiring.PlatformCoordinator;
import com.swirlds.state.MerkleNodeState;
import com.swirlds.state.State;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.time.Duration;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.hiero.base.concurrent.locks.locked.LockedResource;
import org.hiero.base.crypto.Hash;
import org.hiero.consensus.model.node.NodeId;
import org.hiero.consensus.roster.RosterRetriever;

public class ReconnectController {
    private static final Logger logger = LogManager.getLogger(ReconnectController.class);
    private final PlatformStateFacade platformStateFacade;
    private final Roster roster;
    private final SignedStateValidator validator;
    private final MerkleCryptography merkleCryptography;
    private final Platform platform;
    private final PlatformContext platformContext;
    private final PlatformCoordinator platformCoordinator;
    private final SwirldStateManager swirldStateManager;
    private final SavedStateController savedStateController;
    private final ConsensusStateEventHandler<MerkleNodeState> consensusStateEventHandler;
    private final ReservedSignedStatePromise peerReservedSignedStatePromise;
    private final NodeId selfId;
    private final ReconnectConfig reconnectConfig;
    private final Time time;
    private final Instant startupTime;
    private final FallenBehindMonitor fallenBehindMonitor;
    private final AtomicBoolean run = new AtomicBoolean(true);

    public ReconnectController(@NonNull PlatformStateFacade platformStateFacade, @NonNull Roster roster, @NonNull MerkleCryptography merkleCryptography, @NonNull Platform platform, @NonNull PlatformContext platformContext, @NonNull PlatformCoordinator platformCoordinator, @NonNull SwirldStateManager swirldStateManager, @NonNull SavedStateController savedStateController, @NonNull ConsensusStateEventHandler<MerkleNodeState> consensusStateEventHandler, @NonNull ReservedSignedStatePromise peerReservedSignedStatePromise, @NonNull NodeId selfId, @NonNull FallenBehindMonitor fallenBehindMonitor) {
        this.platformStateFacade = Objects.requireNonNull(platformStateFacade);
        this.roster = Objects.requireNonNull(roster);
        this.platformCoordinator = Objects.requireNonNull(platformCoordinator);
        this.peerReservedSignedStatePromise = Objects.requireNonNull(peerReservedSignedStatePromise);
        this.fallenBehindMonitor = Objects.requireNonNull(fallenBehindMonitor);
        this.validator = new DefaultSignedStateValidator(platformContext, platformStateFacade);
        this.merkleCryptography = Objects.requireNonNull(merkleCryptography);
        this.reconnectConfig = (ReconnectConfig)platformContext.getConfiguration().getConfigData(ReconnectConfig.class);
        this.platform = Objects.requireNonNull(platform);
        this.platformContext = Objects.requireNonNull(platformContext);
        this.swirldStateManager = Objects.requireNonNull(swirldStateManager);
        this.savedStateController = Objects.requireNonNull(savedStateController);
        this.consensusStateEventHandler = Objects.requireNonNull(consensusStateEventHandler);
        this.time = platformContext.getTime();
        this.selfId = selfId;
        this.startupTime = this.time.now();
    }

    static void hashStateForReconnect(MerkleCryptography merkleCryptography, MerkleNodeState workingState) {
        try {
            merkleCryptography.digestTreeAsync(workingState.getRoot()).get();
        }
        catch (ExecutionException e) {
            logger.error(LogMarker.EXCEPTION.getMarker(), () -> new ReconnectFailurePayload("Error encountered while hashing state for reconnect", ReconnectFailurePayload.CauseOfFailure.ERROR).toString(), (Throwable)e);
            throw new StateSyncException(e);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            logger.error(LogMarker.EXCEPTION.getMarker(), () -> new ReconnectFailurePayload("Interrupted while attempting to hash state", ReconnectFailurePayload.CauseOfFailure.ERROR).toString(), (Throwable)e);
        }
    }

    public void stop() {
        this.run.set(false);
    }

    public void start() {
        logger.info(LogMarker.RECONNECT.getMarker(), "Starting the ReconnectController");
        this.exitIfReconnectIsDisabled();
        try {
            while (this.run.get()) {
                int failedReconnectsInARow = 0;
                this.fallenBehindMonitor.awaitFallenBehind();
                this.exitIfReconnectTimeTimeElapsed();
                this.platformCoordinator.submitStatusAction(new FallenBehindAction());
                while (!this.doReconnect()) {
                    this.exitIfThresholdMet(++failedReconnectsInARow);
                    logger.info(LogMarker.RECONNECT.getMarker(), "Reconnect failed, retrying");
                    Thread.sleep(this.reconnectConfig.minimumTimeBetweenReconnects().toMillis());
                }
                this.fallenBehindMonitor.reset();
            }
        }
        catch (InterruptedException | RuntimeException e) {
            logger.error(LogMarker.EXCEPTION.getMarker(), "Unexpected error occurred while reconnecting", (Throwable)e);
            SystemExitUtils.exitSystem(SystemExitCode.RECONNECT_FAILURE);
        }
    }

    private boolean doReconnect() throws InterruptedException {
        MerkleNodeState currentState = this.swirldStateManager.getConsensusState();
        logger.info(LogMarker.RECONNECT.getMarker(), "Preparing for reconnect, stopping gossip");
        this.platformCoordinator.pauseGossip();
        logger.info(LogMarker.RECONNECT.getMarker(), "Preparing for reconnect, start clearing queues");
        this.platformCoordinator.clear();
        logger.info(LogMarker.RECONNECT.getMarker(), "Queues have been cleared");
        ReconnectController.hashStateForReconnect(this.merkleCryptography, currentState);
        logger.info(LogMarker.RECONNECT.getMarker(), "Waiting for a state to be obtained from a peer");
        try (LockedResource<ReservedSignedState> reservedStateResource = Objects.requireNonNull(this.peerReservedSignedStatePromise.await());
             ReservedSignedState reservedState = Objects.requireNonNull((ReservedSignedState)reservedStateResource.getResource());){
            logger.info(LogMarker.RECONNECT.getMarker(), "A state was obtained from a peer");
            this.validator.validate(reservedState.get(), this.roster, new SignedStateValidationData((State)currentState, this.roster, this.platformStateFacade));
            logger.info(LogMarker.RECONNECT.getMarker(), "The state obtained from a peer was validated");
            SignedStateFileReader.registerServiceStates(reservedState.get());
            this.loadState(reservedState.get());
            this.platformCoordinator.sendReconnectCompleteNotification(reservedState.get());
        }
        catch (RuntimeException e) {
            logger.info(LogMarker.RECONNECT.getMarker(), "Reconnect failed with the following exception", (Throwable)e);
            this.platformCoordinator.clear();
            return false;
        }
        this.platformCoordinator.resumeGossip();
        return true;
    }

    private void loadState(@NonNull SignedState signedState) {
        logger.info(LogMarker.STATE_HASH.getMarker(), "RECONNECT: loadReconnectState: reloading state");
        logger.debug(LogMarker.RECONNECT.getMarker(), "`loadReconnectState` : reloading state");
        Hash reconnectHash = signedState.getState().getHash();
        MerkleNodeState state = signedState.getState();
        SemanticVersion creationSoftwareVersion = this.platformStateFacade.creationSoftwareVersionOf((State)state);
        signedState.init(this.platformContext);
        this.consensusStateEventHandler.onStateInitialized(state, this.platform, InitTrigger.RECONNECT, creationSoftwareVersion);
        if (!Objects.equals(signedState.getState().getHash(), reconnectHash)) {
            throw new IllegalStateException("State hash is not permitted to change during a reconnect init() call. Previous hash was " + String.valueOf(reconnectHash) + ", new hash is " + String.valueOf(signedState.getState().getHash()));
        }
        long round = this.platformStateFacade.roundOf((State)state);
        Roster stateRoster = RosterRetriever.retrieveActive((State)state, (long)round);
        if (!this.roster.equals((Object)stateRoster)) {
            throw new IllegalStateException("Current roster and state-based roster do not contain the same nodes  (currentRoster=" + Roster.JSON.toJSON((Object)this.roster) + ") (stateRoster=" + Roster.JSON.toJSON((Object)stateRoster) + ")");
        }
        this.swirldStateManager.loadFromSignedState(signedState);
        this.platformCoordinator.submitStatusAction(new ReconnectCompleteAction(signedState.getRound()));
        this.savedStateController.reconnectStateReceived(signedState.reserve("savedStateController.reconnectStateReceived"));
        this.platformCoordinator.loadReconnectState(this.platformContext.getConfiguration(), signedState);
    }

    private void exitIfThresholdMet(int failedReconnectsInARow) {
        if (failedReconnectsInARow >= this.reconnectConfig.maximumReconnectFailuresBeforeShutdown()) {
            logger.error(LogMarker.EXCEPTION.getMarker(), "Too many reconnect failures in a row, killing node");
            SystemExitUtils.exitSystem(SystemExitCode.RECONNECT_FAILURE);
        }
    }

    private void exitIfReconnectTimeTimeElapsed() {
        if (this.reconnectConfig.reconnectWindowSeconds() >= 0 && (long)this.reconnectConfig.reconnectWindowSeconds() < Duration.between(this.startupTime, this.time.now()).toSeconds()) {
            logger.warn(LogMarker.STARTUP.getMarker(), () -> new UnableToReconnectPayload("Node has fallen behind, reconnect is disabled outside of time window, will die", this.selfId.id()).toString());
            SystemExitUtils.exitSystem(SystemExitCode.BEHIND_RECONNECT_DISABLED);
        }
    }

    private void exitIfReconnectIsDisabled() {
        if (!this.reconnectConfig.active()) {
            logger.warn(LogMarker.STARTUP.getMarker(), () -> new UnableToReconnectPayload("Node has fallen behind, reconnect is disabled, will die", this.selfId.id()).toString());
            SystemExitUtils.exitSystem(SystemExitCode.BEHIND_RECONNECT_DISABLED);
        }
    }
}

