/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.platform.test.fixtures.resource;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.PosixFilePermissions;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

public class ResourceLoader<T> {
    private final Class<T> requester;
    private final List<ClassLoader> resolvers;

    public ResourceLoader(Class<T> requester) {
        Objects.requireNonNull(requester, "requester cannot be null");
        this.requester = requester;
        this.resolvers = new ArrayList<ClassLoader>();
        this.resolvers.add(this.requester.getClassLoader());
        this.resolvers.add(ClassLoader.getSystemClassLoader());
        this.resolvers.add(ClassLoader.getPlatformClassLoader());
    }

    public Path load(String path) throws IOException {
        Objects.requireNonNull(path, "path cannot be null");
        InputStream resourceStream = this.acquireResourceStream(path);
        String fileName = Path.of(path, new String[0]).getFileName().toString();
        Path tempDirectory = this.createTempDirectory();
        Path tempFile = tempDirectory.resolve(fileName);
        Files.copy(resourceStream, tempFile, new CopyOption[0]);
        if (this.isPosixCompliant()) {
            Files.setPosixFilePermissions(tempFile, PosixFilePermissions.fromString("rwxrwxrwx"));
        } else {
            File f = tempFile.toFile();
            f.setExecutable(true, false);
            f.setReadable(true, false);
            f.setWritable(true, false);
        }
        return tempFile;
    }

    public Path loadDirectory(String path) throws IOException {
        Objects.requireNonNull(path, "path cannot be null");
        URI resourceUri = this.acquireResourceUri(path);
        Path tempDirectory = this.createTempDirectory();
        try (FileSystem fs = this.acquireFileSystem(resourceUri);){
            Path root = fs.getPath(resourceUri.getPath(), new String[0]);
            try (Stream<Path> stream = Files.walk(root, new FileVisitOption[0]);){
                stream.forEach(source -> {
                    try {
                        Path destination = tempDirectory.resolve(root.relativize((Path)source).toString());
                        Files.copy(source, destination, StandardCopyOption.REPLACE_EXISTING);
                        if (this.isPosixCompliant()) {
                            Files.setPosixFilePermissions(destination, PosixFilePermissions.fromString("rwxrwxrwx"));
                        } else {
                            File f = destination.toFile();
                            f.setExecutable(true, false);
                            f.setReadable(true, false);
                            f.setWritable(true, false);
                        }
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(e);
                    }
                });
            }
            catch (IOException | UncheckedIOException e) {
                throw new IOException("Unable to load directory: " + path, e);
            }
        }
        catch (UnsupportedOperationException unsupportedOperationException) {
            // empty catch block
        }
        return tempDirectory;
    }

    private InputStream acquireResourceStream(String path) throws IOException {
        Objects.requireNonNull(path, "path cannot be null");
        for (ClassLoader resolver : this.resolvers) {
            InputStream stream = resolver.getResourceAsStream(path);
            if (stream == null) continue;
            return stream;
        }
        throw new IOException("Unable to acquire resource stream for path: " + path);
    }

    private URI acquireResourceUri(String path) throws IOException {
        Objects.requireNonNull(path, "path cannot be null");
        for (ClassLoader resolver : this.resolvers) {
            URL url = resolver.getResource(path);
            if (url == null) continue;
            try {
                return url.toURI();
            }
            catch (URISyntaxException e) {
                throw new IOException("Unable to acquire resource URI for path: " + path, e);
            }
        }
        throw new IOException("Unable to acquire resource URI for path: " + path);
    }

    private FileSystem acquireFileSystem(URI uri) throws IOException {
        try {
            if (uri.getScheme().equals("file")) {
                return FileSystems.getFileSystem(uri.resolve("/"));
            }
            return FileSystems.getFileSystem(uri);
        }
        catch (FileSystemNotFoundException e) {
            return FileSystems.newFileSystem(uri, Collections.emptyMap());
        }
    }

    private Path createTempDirectory() throws IOException {
        Path tempDirectory = Files.createTempDirectory(this.requester.getSimpleName(), new FileAttribute[0]);
        tempDirectory.toFile().mkdir();
        tempDirectory.toFile().deleteOnExit();
        return tempDirectory;
    }

    protected boolean isPosixCompliant() {
        try {
            return FileSystems.getDefault().supportedFileAttributeViews().contains("posix");
        }
        catch (SecurityException | FileSystemNotFoundException | ProviderNotFoundException e) {
            return false;
        }
    }
}

