/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.platform.crypto;

import com.swirlds.config.api.Configuration;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.platform.config.PathsConfig;
import com.swirlds.platform.crypto.CryptoStatic;
import com.swirlds.platform.crypto.KeyCertPurpose;
import com.swirlds.platform.crypto.KeyGeneratingException;
import com.swirlds.platform.crypto.KeyLoadingException;
import com.swirlds.platform.crypto.KeysAndCertsGenerator;
import com.swirlds.platform.crypto.PublicStores;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.lang.runtime.SwitchBootstraps;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.security.Key;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.Security;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openssl.PEMDecryptorProvider;
import org.bouncycastle.openssl.PEMEncryptedKeyPair;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcePEMDecryptorProviderBuilder;
import org.bouncycastle.operator.InputDecryptorProvider;
import org.bouncycastle.operator.jcajce.JceInputDecryptorProviderBuilder;
import org.bouncycastle.pkcs.PKCS8EncryptedPrivateKeyInfo;
import org.bouncycastle.pkcs.PKCSException;
import org.bouncycastle.util.encoders.DecoderException;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;
import org.bouncycastle.util.io.pem.PemWriter;
import org.hiero.base.crypto.config.CryptoConfig;
import org.hiero.consensus.model.node.KeysAndCerts;
import org.hiero.consensus.model.node.NodeId;
import org.hiero.consensus.model.roster.Address;
import org.hiero.consensus.model.roster.AddressBook;
import org.hiero.consensus.roster.RosterUtils;

public class EnhancedKeyStoreLoader {
    private static final String MSG_NODE_ID_NON_NULL = "nodeId must not be null";
    private static final String MSG_NODE_ALIAS_NON_NULL = "nodeAlias must not be null";
    private static final String MSG_PURPOSE_NON_NULL = "purpose must not be null";
    private static final String MSG_LEGACY_PUBLIC_STORE_NON_NULL = "legacyPublicStore must not be null";
    private static final String MSG_LOCATION_NON_NULL = "location must not be null";
    private static final String MSG_ENTRY_TYPE_NON_NULL = "entryType must not be null";
    private static final String MSG_ENTRY_NON_NULL = "entry must not be null";
    private static final String MSG_ADDRESS_BOOK_NON_NULL = "addressBook must not be null";
    private static final String MSG_FUNCTION_NON_NULL = "function must not be null";
    private static final String MSG_KEY_STORE_DIRECTORY_NON_NULL = "keyStoreDirectory must not be null";
    private static final String MSG_KEY_STORE_PASSPHRASE_NON_NULL = "keyStorePassphrase must not be null";
    private static final String MSG_NODES_TO_START_NON_NULL = "the local nodes must not be null";
    private static final Logger logger = LogManager.getLogger(EnhancedKeyStoreLoader.class);
    private final AddressBook addressBook;
    private final Path keyStoreDirectory;
    private final char[] keyStorePassphrase;
    private final Map<NodeId, PrivateKey> sigPrivateKeys;
    private final Map<NodeId, Certificate> sigCertificates;
    private final Map<NodeId, PrivateKey> agrPrivateKeys;
    private final Map<NodeId, Certificate> agrCertificates;
    private final Set<NodeId> localNodes;

    private EnhancedKeyStoreLoader(@NonNull AddressBook addressBook, @NonNull Path keyStoreDirectory, @NonNull char[] keyStorePassphrase, @NonNull Set<NodeId> localNodes) {
        this.addressBook = Objects.requireNonNull(addressBook, MSG_ADDRESS_BOOK_NON_NULL);
        this.keyStoreDirectory = Objects.requireNonNull(keyStoreDirectory, MSG_KEY_STORE_DIRECTORY_NON_NULL);
        this.keyStorePassphrase = Objects.requireNonNull(keyStorePassphrase, MSG_KEY_STORE_PASSPHRASE_NON_NULL);
        this.sigPrivateKeys = HashMap.newHashMap(addressBook.getSize());
        this.sigCertificates = HashMap.newHashMap(addressBook.getSize());
        this.agrPrivateKeys = HashMap.newHashMap(addressBook.getSize());
        this.agrCertificates = HashMap.newHashMap(addressBook.getSize());
        this.localNodes = Collections.unmodifiableSet(Objects.requireNonNull(localNodes, MSG_NODES_TO_START_NON_NULL));
    }

    @NonNull
    public static EnhancedKeyStoreLoader using(@NonNull AddressBook addressBook, @NonNull Configuration configuration, @NonNull Set<NodeId> localNodes) {
        Objects.requireNonNull(addressBook, MSG_ADDRESS_BOOK_NON_NULL);
        Objects.requireNonNull(configuration, "configuration must not be null");
        Objects.requireNonNull(localNodes, MSG_NODES_TO_START_NON_NULL);
        String keyStorePassphrase = ((CryptoConfig)configuration.getConfigData(CryptoConfig.class)).keystorePassword();
        Path keyStoreDirectory = ((PathsConfig)configuration.getConfigData(PathsConfig.class)).getKeysDirPath();
        if (keyStorePassphrase == null || keyStorePassphrase.isBlank()) {
            throw new IllegalArgumentException("keyStorePassphrase must not be null or blank");
        }
        return new EnhancedKeyStoreLoader(addressBook, keyStoreDirectory, keyStorePassphrase.toCharArray(), localNodes);
    }

    @NonNull
    public EnhancedKeyStoreLoader scan() throws KeyLoadingException, KeyStoreException {
        logger.debug(LogMarker.STARTUP.getMarker(), "Starting key store enumeration");
        KeyStore legacyPublicStore = this.resolveLegacyPublicStore();
        EnhancedKeyStoreLoader.iterateAddressBook(this.addressBook, (nodeId, address) -> {
            logger.debug(LogMarker.STARTUP.getMarker(), "Attempting to locate key stores for nodeId {}", (Object)nodeId);
            if (this.localNodes.contains(address.getNodeId())) {
                this.sigPrivateKeys.compute(nodeId, (k, v) -> this.resolveNodePrivateKey(nodeId, KeyCertPurpose.SIGNING));
            }
            this.sigCertificates.compute(nodeId, (k, v) -> this.resolveNodeCertificate(nodeId, KeyCertPurpose.SIGNING, legacyPublicStore));
        });
        logger.trace(LogMarker.STARTUP.getMarker(), "Completed key store enumeration");
        return this;
    }

    public EnhancedKeyStoreLoader generate() throws NoSuchAlgorithmException, NoSuchProviderException, KeyGeneratingException {
        for (NodeId nodeId : this.localNodes) {
            if (this.agrPrivateKeys.containsKey(nodeId)) continue;
            logger.info(LogMarker.STARTUP.getMarker(), "Generating agreement key pair for local nodeId {}", (Object)nodeId);
            KeyPair agrKeyPair = KeysAndCertsGenerator.generateAgreementKeyPair();
            this.agrPrivateKeys.put(nodeId, agrKeyPair.getPrivate());
            PrivateKey privateSigningKey = this.sigPrivateKeys.get(nodeId);
            X509Certificate signingCert = (X509Certificate)this.sigCertificates.get(nodeId);
            if (privateSigningKey == null || signingCert == null) continue;
            PublicKey publicSigningKey = signingCert.getPublicKey();
            KeyPair signingKeyPair = new KeyPair(publicSigningKey, privateSigningKey);
            String dnA = CryptoStatic.distinguishedName(KeyCertPurpose.AGREEMENT.storeName(nodeId));
            X509Certificate agrCert = CryptoStatic.generateCertificate(dnA, agrKeyPair, signingCert.getSubjectX500Principal().getName(), signingKeyPair, SecureRandom.getInstanceStrong(), "SHA384withRSA");
            this.agrCertificates.put(nodeId, agrCert);
        }
        return this;
    }

    @NonNull
    public EnhancedKeyStoreLoader verify() throws KeyLoadingException, KeyStoreException {
        return this.verify(this.addressBook);
    }

    @NonNull
    public EnhancedKeyStoreLoader verify(@NonNull AddressBook validatingBook) throws KeyLoadingException, KeyStoreException {
        Objects.requireNonNull(validatingBook, "validatingBook must not be null");
        if (this.addressBook.getSize() != validatingBook.getSize()) {
            throw new KeyLoadingException("The validating address book size differs from the address book used during initialization [ validatingSize = %d, initializedSize = %d ]".formatted(validatingBook.getSize(), this.addressBook.getSize()));
        }
        EnhancedKeyStoreLoader.iterateAddressBook(validatingBook, (nodeId, address) -> {
            try {
                if (this.localNodes.contains(address.getNodeId())) {
                    if (!this.sigPrivateKeys.containsKey(nodeId)) {
                        throw new KeyLoadingException("No private key found for nodeId %s [ purpose = %s ]".formatted(new Object[]{nodeId, KeyCertPurpose.SIGNING}));
                    }
                    if (!this.agrPrivateKeys.containsKey(nodeId)) {
                        throw new KeyLoadingException("No private key found for nodeId %s [purpose = %s ]".formatted(new Object[]{nodeId, KeyCertPurpose.AGREEMENT}));
                    }
                    if (!this.agrCertificates.containsKey(nodeId)) {
                        throw new KeyLoadingException("No certificate found for nodeId %s [purpose = %s ]".formatted(new Object[]{nodeId, KeyCertPurpose.AGREEMENT}));
                    }
                }
                if (!this.sigCertificates.containsKey(nodeId)) {
                    throw new KeyLoadingException("No certificate found for nodeId %s [purpose = %s ]".formatted(new Object[]{nodeId, KeyCertPurpose.SIGNING}));
                }
            }
            catch (KeyLoadingException e) {
                logger.warn(LogMarker.STARTUP.getMarker(), e.getMessage());
                throw e;
            }
        });
        return this;
    }

    @NonNull
    public Map<NodeId, KeysAndCerts> keysAndCerts() throws KeyStoreException, KeyLoadingException {
        return this.keysAndCerts(this.addressBook);
    }

    @NonNull
    public Map<NodeId, KeysAndCerts> keysAndCerts(@NonNull AddressBook validatingBook) throws KeyStoreException, KeyLoadingException {
        Objects.requireNonNull(validatingBook, "validatingBook must not be null");
        HashMap<NodeId, KeysAndCerts> keysAndCerts = HashMap.newHashMap(validatingBook.getSize());
        PublicStores publicStores = this.publicStores(validatingBook);
        EnhancedKeyStoreLoader.iterateAddressBook(validatingBook, (nodeId, address) -> {
            X509Certificate sigCert = publicStores.getCertificate(KeyCertPurpose.SIGNING, nodeId);
            if (sigCert == null) {
                throw new KeyLoadingException("No signing certificate found for nodeId: %s".formatted(nodeId));
            }
            if (this.localNodes.contains(nodeId)) {
                X509Certificate agrCert = publicStores.getCertificate(KeyCertPurpose.AGREEMENT, nodeId);
                PrivateKey sigPrivateKey = this.sigPrivateKeys.get(nodeId);
                PrivateKey agrPrivateKey = this.agrPrivateKeys.get(nodeId);
                if (sigPrivateKey == null) {
                    throw new KeyLoadingException("No signing private key found for nodeId: %s".formatted(nodeId));
                }
                if (agrPrivateKey == null) {
                    throw new KeyLoadingException("No agreement private key found for nodeId: %s".formatted(nodeId));
                }
                if (agrCert == null) {
                    throw new KeyLoadingException("No agreement certificate found for nodeId: %s".formatted(nodeId));
                }
                KeyPair sigKeyPair = new KeyPair(sigCert.getPublicKey(), sigPrivateKey);
                KeyPair agrKeyPair = new KeyPair(agrCert.getPublicKey(), agrPrivateKey);
                KeysAndCerts kc = new KeysAndCerts(sigKeyPair, agrKeyPair, sigCert, agrCert);
                keysAndCerts.put(nodeId, kc);
            }
        });
        return keysAndCerts;
    }

    @NonNull
    public EnhancedKeyStoreLoader injectInAddressBook() throws KeyLoadingException, KeyStoreException {
        return this.injectInAddressBook(this.addressBook);
    }

    @NonNull
    public EnhancedKeyStoreLoader injectInAddressBook(@NonNull AddressBook validatingBook) throws KeyStoreException, KeyLoadingException {
        PublicStores publicStores = this.publicStores(validatingBook);
        CryptoStatic.copyPublicKeys(publicStores, validatingBook);
        return this;
    }

    @NonNull
    public PublicStores publicStores(@NonNull AddressBook validatingBook) throws KeyStoreException, KeyLoadingException {
        PublicStores publicStores = new PublicStores();
        EnhancedKeyStoreLoader.iterateAddressBook(validatingBook, (nodeId, address) -> {
            Certificate sigCert = this.sigCertificates.get(nodeId);
            Certificate agrCert = this.agrCertificates.get(nodeId);
            if (!(sigCert instanceof X509Certificate)) {
                throw new KeyLoadingException("Illegal signing certificate type for nodeId: %s [ purpose = %s ]".formatted(new Object[]{nodeId, KeyCertPurpose.SIGNING}));
            }
            if (this.localNodes.contains(nodeId)) {
                logger.trace(LogMarker.STARTUP.getMarker(), "Injecting agreement certificate for local nodeId {} into public stores", (Object)nodeId);
                if (!(agrCert instanceof X509Certificate)) {
                    throw new KeyLoadingException("Illegal agreement certificate type for nodeId: %s [ purpose = %s ]".formatted(new Object[]{nodeId, KeyCertPurpose.AGREEMENT}));
                }
                publicStores.setCertificate(KeyCertPurpose.AGREEMENT, (X509Certificate)agrCert, nodeId);
            }
            publicStores.setCertificate(KeyCertPurpose.SIGNING, (X509Certificate)sigCert, nodeId);
        });
        return publicStores;
    }

    @NonNull
    private KeyStore resolveLegacyPublicStore() throws KeyLoadingException, KeyStoreException {
        Path legacyStorePath = this.legacyCertificateStore();
        logger.trace(LogMarker.STARTUP.getMarker(), "Searching for the legacy public key store [ path = {} ]", (Object)legacyStorePath);
        if (Files.exists(legacyStorePath, new LinkOption[0])) {
            logger.debug(LogMarker.STARTUP.getMarker(), "Loading the legacy public key store [ path = {} ]", (Object)legacyStorePath);
            return CryptoStatic.loadKeys(legacyStorePath, this.keyStorePassphrase);
        }
        logger.debug(LogMarker.STARTUP.getMarker(), "No Legacy public key store found");
        return CryptoStatic.createEmptyTrustStore();
    }

    @Nullable
    private PrivateKey resolveNodePrivateKey(@NonNull NodeId nodeId, @NonNull KeyCertPurpose purpose) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(purpose, MSG_PURPOSE_NON_NULL);
        Path ksLocation = this.privateKeyStore(nodeId, purpose);
        if (Files.exists(ksLocation, new LinkOption[0])) {
            logger.trace(LogMarker.STARTUP.getMarker(), "Found enhanced private key store for nodeId: {} [ purpose = {}, fileName = {} ]", (Object)nodeId, (Object)purpose, (Object)ksLocation.getFileName());
            return this.readPrivateKey(nodeId, ksLocation);
        }
        ksLocation = this.legacyPrivateKeyStore(nodeId);
        if (Files.exists(ksLocation, new LinkOption[0])) {
            logger.trace(LogMarker.STARTUP.getMarker(), "Found legacy private key store for nodeId: {} [ purpose = {}, fileName = {} ]", (Object)nodeId, (Object)purpose, (Object)ksLocation.getFileName());
            return this.readLegacyPrivateKey(nodeId, ksLocation, purpose.storeName(nodeId));
        }
        logger.warn(LogMarker.STARTUP.getMarker(), "No private key store found for nodeId: {} [ purpose = {} ]", (Object)nodeId, (Object)purpose);
        return null;
    }

    @Nullable
    private Certificate resolveNodeCertificate(@NonNull NodeId nodeId, @NonNull KeyCertPurpose purpose, @NonNull KeyStore legacyPublicStore) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(purpose, MSG_PURPOSE_NON_NULL);
        Objects.requireNonNull(legacyPublicStore, MSG_LEGACY_PUBLIC_STORE_NON_NULL);
        Path ksLocation = this.certificateStore(nodeId, purpose);
        if (Files.exists(ksLocation, new LinkOption[0])) {
            logger.trace(LogMarker.STARTUP.getMarker(), "Found enhanced certificate store for nodeId: {} [ purpose = {}, fileName = {} ]", (Object)nodeId, (Object)purpose, (Object)ksLocation.getFileName());
            return this.readCertificate(nodeId, ksLocation);
        }
        ksLocation = this.legacyCertificateStore();
        if (Files.exists(ksLocation, new LinkOption[0])) {
            logger.trace(LogMarker.STARTUP.getMarker(), "Found legacy certificate store for nodeId: {} [ purpose = {}, fileName = {} ]", (Object)nodeId, (Object)purpose, (Object)ksLocation.getFileName());
            return this.readLegacyCertificate(nodeId, purpose, legacyPublicStore);
        }
        logger.warn(LogMarker.STARTUP.getMarker(), "No certificate store found for nodeId: {} [ purpose = {} ]", (Object)nodeId, (Object)purpose);
        return null;
    }

    @Nullable
    private Certificate readCertificate(@NonNull NodeId nodeId, @NonNull Path location) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        try {
            return this.readEnhancedStore(location, Certificate.class);
        }
        catch (KeyLoadingException e) {
            logger.warn(LogMarker.STARTUP.getMarker(), "Unable to load the enhanced certificate store for nodeId: {} [ fileName = {} ]", (Object)nodeId, (Object)location.getFileName(), (Object)e);
            return null;
        }
    }

    @Nullable
    private Certificate readLegacyCertificate(@NonNull NodeId nodeId, @NonNull KeyCertPurpose purpose, @NonNull KeyStore legacyPublicStore) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(purpose, MSG_PURPOSE_NON_NULL);
        Objects.requireNonNull(legacyPublicStore, MSG_LEGACY_PUBLIC_STORE_NON_NULL);
        try {
            Certificate cert = legacyPublicStore.getCertificate(purpose.storeName(nodeId));
            if (cert == null) {
                logger.warn(LogMarker.STARTUP.getMarker(), "No certificate found for nodeId: {} [ entryName = {} ]", (Object)nodeId, (Object)purpose.storeName(nodeId));
            }
            return cert;
        }
        catch (KeyStoreException e) {
            logger.warn(LogMarker.STARTUP.getMarker(), "Unable to load the legacy certificate store [ fileName = {} ]", (Object)"public.pfx", (Object)e);
            return null;
        }
    }

    @Nullable
    PrivateKey readPrivateKey(@NonNull NodeId nodeId, @NonNull Path location) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        try {
            return this.readEnhancedStore(location, PrivateKey.class);
        }
        catch (KeyLoadingException e) {
            logger.warn(LogMarker.STARTUP.getMarker(), "Unable to load the enhanced private key store for nodeId: {} [ fileName = {} ]", (Object)nodeId, (Object)location.getFileName(), (Object)e);
            return null;
        }
    }

    @Nullable
    private PrivateKey readLegacyPrivateKey(@NonNull NodeId nodeId, @NonNull Path location, @NonNull String entryName) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        try {
            PrivateKey pk;
            KeyStore ks = CryptoStatic.loadKeys(location, this.keyStorePassphrase);
            Key k = ks.getKey(entryName, this.keyStorePassphrase);
            if (!(k instanceof PrivateKey)) {
                logger.warn(LogMarker.STARTUP.getMarker(), "No private key found for nodeId: {} [ entryName = {} ]", (Object)nodeId, (Object)entryName);
            }
            return k instanceof PrivateKey ? (pk = (PrivateKey)k) : null;
        }
        catch (KeyLoadingException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException e) {
            logger.warn(LogMarker.STARTUP.getMarker(), "Unable to load the legacy private key store [ fileName = {} ]", (Object)location.getFileName(), (Object)e);
            return null;
        }
    }

    @NonNull
    private Path privateKeyStore(@NonNull NodeId nodeId, @NonNull KeyCertPurpose purpose) {
        Objects.requireNonNull(purpose, MSG_PURPOSE_NON_NULL);
        return this.keyStoreDirectory.resolve(String.format("%s-private-%s.pem", purpose.prefix(), RosterUtils.formatNodeName((NodeId)nodeId)));
    }

    @NonNull
    private Path legacyPrivateKeyStore(@NonNull NodeId nodeId) {
        Objects.requireNonNull(nodeId, MSG_NODE_ALIAS_NON_NULL);
        return this.keyStoreDirectory.resolve(String.format("private-%s.pfx", RosterUtils.formatNodeName((NodeId)nodeId)));
    }

    @NonNull
    private Path certificateStore(@NonNull NodeId nodeId, @NonNull KeyCertPurpose purpose) {
        Objects.requireNonNull(nodeId, MSG_NODE_ID_NON_NULL);
        Objects.requireNonNull(purpose, MSG_PURPOSE_NON_NULL);
        return this.keyStoreDirectory.resolve(String.format("%s-public-%s.pem", purpose.prefix(), RosterUtils.formatNodeName((NodeId)nodeId)));
    }

    @NonNull
    private Path legacyCertificateStore() {
        return this.keyStoreDirectory.resolve("public.pfx");
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @NonNull
    private <T> T readEnhancedStore(@NonNull Path location, @NonNull Class<T> entryType) throws KeyLoadingException {
        Objects.requireNonNull(location, MSG_LOCATION_NON_NULL);
        Objects.requireNonNull(entryType, MSG_ENTRY_TYPE_NON_NULL);
        try (PEMParser parser = new PEMParser((Reader)new InputStreamReader(Files.newInputStream(location, new OpenOption[0]), StandardCharsets.UTF_8));){
            Object entry;
            while ((entry = parser.readObject()) != null && !EnhancedKeyStoreLoader.isCompatibleStoreEntry(entry, entryType)) {
            }
            if (entry == null) {
                throw new KeyLoadingException("No entry of the requested type found [ entryType = %s, fileName = %s ]".formatted(entryType.getName(), location.getFileName()));
            }
            T t = this.extractEntityOfType(entry, entryType);
            return t;
        }
        catch (IOException | DecoderException e) {
            throw new KeyLoadingException("Unable to read enhanced store [ fileName = %s ]".formatted(location.getFileName()), e);
        }
    }

    @NonNull
    private <T> T extractEntityOfType(@NonNull Object entry, @NonNull Class<T> entryType) throws KeyLoadingException {
        Objects.requireNonNull(entry, MSG_ENTRY_NON_NULL);
        Objects.requireNonNull(entryType, MSG_ENTRY_TYPE_NON_NULL);
        if (entryType.isAssignableFrom(PublicKey.class)) {
            return (T)this.extractPublicKeyEntity(entry);
        }
        if (entryType.isAssignableFrom(PrivateKey.class)) {
            return (T)this.extractPrivateKeyEntity(entry);
        }
        if (entryType.isAssignableFrom(Certificate.class)) {
            return (T)this.extractCertificateEntity(entry);
        }
        throw new KeyLoadingException("Unsupported entry type [ entryType = %s ]".formatted(entryType.getName()));
    }

    @NonNull
    private PublicKey extractPublicKeyEntity(@NonNull Object entry) throws KeyLoadingException {
        Objects.requireNonNull(entry, MSG_ENTRY_NON_NULL);
        try {
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter();
            PEMDecryptorProvider decrypter = new JcePEMDecryptorProviderBuilder().build(this.keyStorePassphrase);
            Object object = entry;
            Objects.requireNonNull(object);
            Object object2 = object;
            int n = 0;
            return switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{SubjectPublicKeyInfo.class, PEMKeyPair.class, PEMEncryptedKeyPair.class}, (Object)object2, n)) {
                case 0 -> {
                    SubjectPublicKeyInfo spki = (SubjectPublicKeyInfo)object2;
                    yield converter.getPublicKey(spki);
                }
                case 1 -> {
                    PEMKeyPair kp = (PEMKeyPair)object2;
                    yield converter.getPublicKey(kp.getPublicKeyInfo());
                }
                case 2 -> {
                    PEMEncryptedKeyPair ekp = (PEMEncryptedKeyPair)object2;
                    yield converter.getPublicKey(ekp.decryptKeyPair(decrypter).getPublicKeyInfo());
                }
                default -> throw new KeyLoadingException("Unsupported entry type [ entryType = %s ]".formatted(entry.getClass().getName()));
            };
        }
        catch (IOException e) {
            throw new KeyLoadingException("Unable to extract a public key from the specified entry [ entryType = %s ]".formatted(entry.getClass().getName()), e);
        }
    }

    @NonNull
    private PrivateKey extractPrivateKeyEntity(@NonNull Object entry) throws KeyLoadingException {
        Objects.requireNonNull(entry, MSG_ENTRY_NON_NULL);
        try {
            JcaPEMKeyConverter converter = new JcaPEMKeyConverter();
            PEMDecryptorProvider decrypter = new JcePEMDecryptorProviderBuilder().build(this.keyStorePassphrase);
            InputDecryptorProvider inputDecrypter = new JceInputDecryptorProviderBuilder().build(new String(this.keyStorePassphrase).getBytes());
            Object object = entry;
            Objects.requireNonNull(object);
            Object object2 = object;
            int n = 0;
            return switch (SwitchBootstraps.typeSwitch("typeSwitch", new Object[]{PrivateKeyInfo.class, PKCS8EncryptedPrivateKeyInfo.class, PEMKeyPair.class, PEMEncryptedKeyPair.class}, (Object)object2, n)) {
                case 0 -> {
                    PrivateKeyInfo pki = (PrivateKeyInfo)object2;
                    yield converter.getPrivateKey(pki);
                }
                case 1 -> {
                    PKCS8EncryptedPrivateKeyInfo epki = (PKCS8EncryptedPrivateKeyInfo)object2;
                    yield converter.getPrivateKey(epki.decryptPrivateKeyInfo(inputDecrypter));
                }
                case 2 -> {
                    PEMKeyPair kp = (PEMKeyPair)object2;
                    yield converter.getPrivateKey(kp.getPrivateKeyInfo());
                }
                case 3 -> {
                    PEMEncryptedKeyPair ekp = (PEMEncryptedKeyPair)object2;
                    yield converter.getPrivateKey(ekp.decryptKeyPair(decrypter).getPrivateKeyInfo());
                }
                default -> throw new KeyLoadingException("Unsupported entry type [ entryType = %s ]".formatted(entry.getClass().getName()));
            };
        }
        catch (IOException | PKCSException e) {
            throw new KeyLoadingException("Unable to extract a private key from the specified entry [ entryType = %s ]".formatted(entry.getClass().getName()), e);
        }
    }

    @NonNull
    private Certificate extractCertificateEntity(@NonNull Object entry) throws KeyLoadingException {
        Objects.requireNonNull(entry, MSG_ENTRY_NON_NULL);
        try {
            if (entry instanceof X509CertificateHolder) {
                X509CertificateHolder ch = (X509CertificateHolder)entry;
                return new JcaX509CertificateConverter().getCertificate(ch);
            }
            throw new KeyLoadingException("Unsupported entry type [ entryType = %s ]".formatted(entry.getClass().getName()));
        }
        catch (CertificateException e) {
            throw new KeyLoadingException("Unable to extract a certificate from the specified entry [ entryType = %s ]".formatted(entry.getClass().getName()), e);
        }
    }

    private static <T> boolean isCompatibleStoreEntry(@NonNull Object entry, @NonNull Class<T> entryType) {
        Objects.requireNonNull(entry, MSG_ENTRY_NON_NULL);
        Objects.requireNonNull(entryType, MSG_ENTRY_TYPE_NON_NULL);
        if (entryType.isAssignableFrom(PublicKey.class) && (entry instanceof SubjectPublicKeyInfo || entry instanceof PEMKeyPair || entry instanceof PEMEncryptedKeyPair)) {
            return true;
        }
        if (entryType.isAssignableFrom(PrivateKey.class) && (entry instanceof PEMKeyPair || entry instanceof PrivateKeyInfo || entry instanceof PKCS8EncryptedPrivateKeyInfo || entry instanceof PEMEncryptedKeyPair)) {
            return true;
        }
        if (entryType.isAssignableFrom(KeyPair.class) && (entry instanceof PEMKeyPair || entry instanceof PEMEncryptedKeyPair)) {
            return true;
        }
        return entryType.isAssignableFrom(Certificate.class) && entry instanceof X509CertificateHolder;
    }

    private static void iterateAddressBook(@NonNull AddressBook addressBook, @NonNull AddressBookCallback function) throws KeyStoreException, KeyLoadingException {
        Objects.requireNonNull(addressBook, MSG_ADDRESS_BOOK_NON_NULL);
        Objects.requireNonNull(function, MSG_FUNCTION_NON_NULL);
        for (int i = 0; i < addressBook.getSize(); ++i) {
            NodeId nodeId = addressBook.getNodeId(i);
            Address address = addressBook.getAddress(nodeId);
            function.apply(nodeId, address);
        }
    }

    @NonNull
    public EnhancedKeyStoreLoader migrate() throws KeyLoadingException, KeyStoreException {
        logger.info(LogMarker.STARTUP.getMarker(), "Starting key store migration");
        HashMap<NodeId, PrivateKey> pfxPrivateKeys = new HashMap<NodeId, PrivateKey>();
        HashMap<NodeId, Certificate> pfxCertificates = new HashMap<NodeId, Certificate>();
        this.deleteAgreementKeys();
        long errorCount = this.extractPrivateKeysAndCertsFromPfxFiles(pfxPrivateKeys, pfxCertificates);
        if (errorCount == 0L) {
            errorCount = this.validateKeysAndCertsAreLoadableFromPemFiles(pfxPrivateKeys, pfxCertificates);
        }
        if (errorCount > 0L) {
            logger.error(LogMarker.STARTUP.getMarker(), "Due to {} errors, reverting pem file creation.", (Object)errorCount);
            this.rollBackSigningKeysAndCertsChanges(pfxPrivateKeys, pfxCertificates);
        } else {
            this.cleanupByMovingPfxFilesToSubDirectory();
            logger.info(LogMarker.STARTUP.getMarker(), "Finished key store migration.");
        }
        return this;
    }

    private void deleteAgreementKeys() {
        File[] agreementKeyFiles = this.keyStoreDirectory.toFile().listFiles((dir, name) -> name.startsWith("a-"));
        if (agreementKeyFiles != null) {
            for (File agreementKeyFile : agreementKeyFiles) {
                if (!agreementKeyFile.isFile()) continue;
                try {
                    Files.delete(agreementKeyFile.toPath());
                    logger.debug(LogMarker.STARTUP.getMarker(), "Deleted agreement key file {}", (Object)agreementKeyFile.getName());
                }
                catch (IOException e) {
                    logger.error(LogMarker.ERROR.getMarker(), "Failed to delete agreement key file {}", (Object)agreementKeyFile.getName());
                }
            }
        }
    }

    private long extractPrivateKeysAndCertsFromPfxFiles(Map<NodeId, PrivateKey> pfxPrivateKeys, Map<NodeId, Certificate> pfxCertificates) throws KeyStoreException, KeyLoadingException {
        KeyStore legacyPublicStore = this.resolveLegacyPublicStore();
        AtomicLong errorCount = new AtomicLong(0L);
        EnhancedKeyStoreLoader.iterateAddressBook(this.addressBook, (nodeId, address) -> {
            Path ksLocation;
            if (this.localNodes.contains(nodeId)) {
                Path sPrivateKeyLocation = this.keyStoreDirectory.resolve(String.format("s-private-%s.pem", RosterUtils.formatNodeName((NodeId)nodeId)));
                ksLocation = this.legacyPrivateKeyStore(nodeId);
                if (!Files.exists(sPrivateKeyLocation, new LinkOption[0]) && Files.exists(ksLocation, new LinkOption[0])) {
                    logger.info(LogMarker.STARTUP.getMarker(), "Extracting private signing key for nodeId: {} from file {}", (Object)nodeId, (Object)ksLocation.getFileName());
                    PrivateKey privateKey = this.readLegacyPrivateKey(nodeId, ksLocation, KeyCertPurpose.SIGNING.storeName(nodeId));
                    pfxPrivateKeys.put(nodeId, privateKey);
                    if (privateKey == null) {
                        logger.error(LogMarker.ERROR.getMarker(), "Failed to extract private signing key for nodeId: {} from file {}", (Object)nodeId, (Object)ksLocation.getFileName());
                        errorCount.incrementAndGet();
                    } else {
                        logger.info(LogMarker.STARTUP.getMarker(), "Writing private signing key for nodeId: {} to PEM file {}", (Object)nodeId, (Object)sPrivateKeyLocation.getFileName());
                        try {
                            EnhancedKeyStoreLoader.writePemFile(true, sPrivateKeyLocation, privateKey.getEncoded());
                        }
                        catch (IOException e) {
                            logger.error(LogMarker.ERROR.getMarker(), "Failed to write private key for nodeId: {} to PEM file {}", (Object)nodeId, (Object)sPrivateKeyLocation.getFileName());
                            errorCount.incrementAndGet();
                        }
                    }
                }
            }
            Path sCertificateLocation = this.keyStoreDirectory.resolve(String.format("s-public-%s.pem", RosterUtils.formatNodeName((NodeId)nodeId)));
            ksLocation = this.legacyCertificateStore();
            if (!Files.exists(sCertificateLocation, new LinkOption[0]) && Files.exists(ksLocation, new LinkOption[0])) {
                logger.info(LogMarker.STARTUP.getMarker(), "Extracting signing certificate for nodeId: {} from file {} ", (Object)nodeId, (Object)ksLocation.getFileName());
                Certificate certificate = this.readLegacyCertificate(nodeId, KeyCertPurpose.SIGNING, legacyPublicStore);
                pfxCertificates.put(nodeId, certificate);
                if (certificate == null) {
                    logger.error(LogMarker.ERROR.getMarker(), "Failed to extract signing certificate for nodeId: {} from file {}", (Object)nodeId, (Object)ksLocation.getFileName());
                    errorCount.incrementAndGet();
                } else {
                    logger.info(LogMarker.STARTUP.getMarker(), "Writing signing certificate for nodeId: {} to PEM file {}", (Object)nodeId, (Object)sCertificateLocation.getFileName());
                    try {
                        EnhancedKeyStoreLoader.writePemFile(false, sCertificateLocation, certificate.getEncoded());
                    }
                    catch (IOException | CertificateEncodingException e) {
                        logger.error(LogMarker.ERROR.getMarker(), "Failed to write signing certificate for nodeId: {} to PEM file {}", (Object)nodeId, (Object)sCertificateLocation.getFileName());
                        errorCount.incrementAndGet();
                    }
                }
            }
        });
        return errorCount.get();
    }

    private long validateKeysAndCertsAreLoadableFromPemFiles(Map<NodeId, PrivateKey> pfxPrivateKeys, Map<NodeId, Certificate> pfxCertificates) throws KeyStoreException, KeyLoadingException {
        AtomicLong errorCount = new AtomicLong(0L);
        EnhancedKeyStoreLoader.iterateAddressBook(this.addressBook, (nodeId, address) -> {
            Path ksLocation;
            PrivateKey pemPrivateKey;
            if (this.localNodes.contains(nodeId) && pfxCertificates.containsKey(nodeId) && ((pemPrivateKey = this.readPrivateKey(nodeId, ksLocation = this.privateKeyStore(nodeId, KeyCertPurpose.SIGNING))) == null || !Arrays.equals(pemPrivateKey.getEncoded(), ((PrivateKey)pfxPrivateKeys.get(nodeId)).getEncoded()))) {
                logger.error(LogMarker.ERROR.getMarker(), "Private key for nodeId: {} does not match the migrated key", (Object)nodeId);
                errorCount.incrementAndGet();
            }
            if (pfxCertificates.containsKey(nodeId)) {
                ksLocation = this.certificateStore(nodeId, KeyCertPurpose.SIGNING);
                Certificate pemCertificate = this.readCertificate(nodeId, ksLocation);
                try {
                    if (pemCertificate == null || !Arrays.equals(pemCertificate.getEncoded(), ((Certificate)pfxCertificates.get(nodeId)).getEncoded())) {
                        logger.error(LogMarker.ERROR.getMarker(), "Certificate for nodeId: {} does not match the migrated certificate", (Object)nodeId);
                        errorCount.incrementAndGet();
                    }
                }
                catch (CertificateEncodingException e) {
                    logger.error(LogMarker.ERROR.getMarker(), "Encoding error while validating certificate for nodeId: {}.", (Object)nodeId);
                    errorCount.incrementAndGet();
                }
            }
        });
        return errorCount.get();
    }

    private void rollBackSigningKeysAndCertsChanges(Map<NodeId, PrivateKey> pfxPrivateKeys, Map<NodeId, Certificate> pfxCertificates) throws KeyStoreException, KeyLoadingException {
        AtomicLong cleanupErrorCount = new AtomicLong(0L);
        EnhancedKeyStoreLoader.iterateAddressBook(this.addressBook, (nodeId, address) -> {
            if (this.localNodes.contains(nodeId) && pfxPrivateKeys.containsKey(address.getNodeId())) {
                try {
                    Files.deleteIfExists(this.privateKeyStore(nodeId, KeyCertPurpose.SIGNING));
                }
                catch (IOException e) {
                    cleanupErrorCount.incrementAndGet();
                }
            }
            if (pfxCertificates.containsKey(address.getNodeId())) {
                try {
                    Files.deleteIfExists(this.certificateStore(nodeId, KeyCertPurpose.SIGNING));
                }
                catch (IOException e) {
                    cleanupErrorCount.incrementAndGet();
                }
            }
        });
        if (cleanupErrorCount.get() > 0L) {
            logger.error(LogMarker.ERROR.getMarker(), "Failed to rollback {} pem files created. Manual cleanup required.", (Object)cleanupErrorCount.get());
            throw new IllegalStateException("Cryptography Migration failed to generate or validate PEM files.");
        }
    }

    private void cleanupByMovingPfxFilesToSubDirectory() throws KeyStoreException, KeyLoadingException {
        AtomicLong cleanupErrorCount = new AtomicLong(0L);
        AtomicBoolean doCleanup = new AtomicBoolean(false);
        EnhancedKeyStoreLoader.iterateAddressBook(this.addressBook, (nodeId, address) -> {
            File sPrivatePfx;
            if (this.localNodes.contains(nodeId) && (sPrivatePfx = this.legacyPrivateKeyStore(nodeId).toFile()).exists() && sPrivatePfx.isFile()) {
                doCleanup.set(true);
            }
        });
        if (!doCleanup.get()) {
            return;
        }
        String archiveDirectory = ".archive";
        String now = DateTimeFormatter.ofPattern("yyyy-MM-dd_HH-mm-ss").format(LocalDateTime.now());
        String newDirectory = ".archive" + File.pathSeparator + now;
        Path pfxArchiveDirectory = this.keyStoreDirectory.resolve(".archive");
        Path pfxDateDirectory = pfxArchiveDirectory.resolve(now);
        logger.info(LogMarker.STARTUP.getMarker(), "Cryptography Migration Cleanup: Moving PFX files to {}", (Object)pfxDateDirectory);
        if (!Files.exists(pfxDateDirectory, new LinkOption[0])) {
            try {
                if (!Files.exists(pfxArchiveDirectory, new LinkOption[0])) {
                    Files.createDirectory(pfxArchiveDirectory, new FileAttribute[0]);
                }
                Files.createDirectory(pfxDateDirectory, new FileAttribute[0]);
            }
            catch (IOException e) {
                logger.error(LogMarker.ERROR.getMarker(), "Failed to create [{}] subdirectory. Manual cleanup required.", (Object)newDirectory);
                return;
            }
        }
        EnhancedKeyStoreLoader.iterateAddressBook(this.addressBook, (nodeId, address) -> {
            File sPrivatePfx;
            if (this.localNodes.contains(nodeId) && (sPrivatePfx = this.legacyPrivateKeyStore(nodeId).toFile()).exists() && sPrivatePfx.isFile() && !sPrivatePfx.renameTo(pfxDateDirectory.resolve(sPrivatePfx.getName()).toFile())) {
                cleanupErrorCount.incrementAndGet();
            }
        });
        File sPublicPfx = this.legacyCertificateStore().toFile();
        if (sPublicPfx.exists() && sPublicPfx.isFile() && !sPublicPfx.renameTo(pfxDateDirectory.resolve(sPublicPfx.getName()).toFile())) {
            cleanupErrorCount.incrementAndGet();
        }
        if (cleanupErrorCount.get() > 0L) {
            logger.error(LogMarker.ERROR.getMarker(), "Failed to move {} PFX files to [{}] subdirectory. Manual cleanup required.", (Object)cleanupErrorCount.get(), (Object)newDirectory);
            throw new IllegalStateException("Cryptography Migration failed to move PFX files to [" + newDirectory + "] subdirectory.");
        }
    }

    public static void writePemFile(boolean isPrivateKey, @NonNull Path location, @NonNull byte[] encoded) throws IOException {
        PemObject pemObj = new PemObject(isPrivateKey ? "PRIVATE KEY" : "CERTIFICATE", encoded);
        try (FileOutputStream file = new FileOutputStream(location.toFile(), false);
             OutputStreamWriter out = new OutputStreamWriter(file);
             PemWriter writer = new PemWriter((Writer)out);){
            writer.writeObject((PemObjectGenerator)pemObj);
            file.getFD().sync();
        }
    }

    static {
        if (Arrays.stream(Security.getProviders()).noneMatch(p -> p instanceof BouncyCastleProvider)) {
            Security.addProvider((Provider)new BouncyCastleProvider());
        }
    }

    @FunctionalInterface
    private static interface AddressBookCallback {
        public void apply(NodeId var1, Address var2) throws KeyStoreException, KeyLoadingException;
    }
}

