#!/bin/bash


#
# Copyright 2016-2022 Hedera Hashgraph, LLC
#
# This software is the confidential and proprietary information of
# Hedera Hashgraph, LLC. ("Confidential Information"). You shall not
# disclose such Confidential Information and shall use it only in
# accordance with the terms of the license agreement you entered into
# with Hedera Hashgraph.
#
# HEDERA HASHGRAPH MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
# THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
# TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
# PARTICULAR PURPOSE, OR NON-INFRINGEMENT. HEDERA HASHGRAPH SHALL NOT BE LIABLE FOR
# ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING OR
# DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
#

#
# Read Previous-file-hash field from record stream file
# for event stream file v5, read startRunningHash
#
# Event file V5 header
#   FILE_VERSION    : 4-byte (value: 5)
#   OBJECT_STREAM_VERSION: 4-byte (value: 1)
#   CLASS_ID        : 8-byte
#   CLASS_VERSION   : 4-byte
#   DIGEST_TYPE_ID  : 4-byte
#   byte array length: 4-byte
#   Hash            : 48-byte
#
# record file V2 header
#   Version : 4-byte
#   API Ver : 4-byte
#   Type    : 1-byte
#   Hash    : 48-byte
#
# record file V5 header
#   FILE_VERSION              : 4-byte (value: 5)
#   Hapi Proto Major Version  : 4-byte (value: 0)
#   Hapi Proto Minor Version  : 4-byte (value: 9)
#   Hapi Proto patch Version  : 4-byte (value: 0)
#   OBJECT_STREAM_VERSION     : 4-byte (value: 1)
#   CLASS_ID                  : 8-byte
#   CLASS_VERSION             : 4-byte (value: 1)
#   DIGEST_TYPE_ID            : 4-byte
#   byte array length         : 4-byte
#   Hash                      : 48-byte
#
#
# Input arguement is a directory, the script
# iterate each file sorted by alphabetical order
# and append prev file hash string to output file
#

# Read bytes in hex format at given offset from the file.
read_data() {
  file="$1"
  offset="$2"   # how many bytes to skip
  count="$3:-1"

  # skip some bytes, dump 48 bytes in hex format to putput file
  #  "-s" start offset
  #  "-c" output column width
  #  "-l" how many byte to dump
  #  "-p" hexdump
  xxd -c $3 -p -l $count -s $offset "$file"
}

function read_hash {
    file_name=$1
    out_file=$2
    offset=$3  # how many bytes to skip from the beginining of the file
    basename="$(basename -- $file_name)"
    echo -n $basename, >> $out_file
    read_data $file_name $(($offset)) 48 >> $out_file
}

function readFileVersion {
  file_name=$1
  # read the first 4 bytes in the given file
  file_version=$(xxd -c 4 -p -l 4 -s 0 $file_name)
}

dir_name=$1
out_file_name=$2
record_file_version=$3

if [[ $dir_name == *event* ]]; then
  for i in `ls -v $dir_name/*.evts`; do
      echo $i
      readFileVersion $i
      if [ $file_version -eq 3 ]; then
        read_hash $i $out_file_name 5; # skip 5 bytes for event v3 files
      elif [ $file_version -eq 5 ]; then
        read_hash $i $out_file_name 28; # skip 28 byte for event v5 files
      else
        echo "unsupported event stream file version ${file_version}"
      fi
  done;
fi

if [[ $dir_name == *record* ]]; then
  for i in `ls -v $dir_name/*.rcd`; do
      echo $i
      readFileVersion $i
      if [ $file_version -eq 2 ]; then
        # skipping reading hash from v2 files as we are validating hashes of v5 files only.
        echo "skipping v2 files"
        #      read_hash $i $out_file_name 9; # skip 9 byte for rcd files
      elif [ $file_version -eq $((record_file_version)) ]; then
        read_hash $i $out_file_name 40; # skip 40 byte for rcd files
      else
        echo "unsupported record stream file version ${file_version}"
      fi
  done;
fi


