/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.files;

import com.hedera.pbj.runtime.FieldDefinition;
import com.hedera.pbj.runtime.FieldType;
import com.hedera.pbj.runtime.ProtoWriterTools;
import com.hedera.pbj.runtime.io.WritableSequentialData;
import com.hedera.pbj.runtime.io.buffer.BufferedData;
import com.hedera.pbj.runtime.io.stream.ReadableStreamingData;
import com.hedera.pbj.runtime.io.stream.WritableStreamingData;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.merkledb.FileStatisticAware;
import com.swirlds.merkledb.KeyRange;
import com.swirlds.merkledb.Snapshotable;
import com.swirlds.merkledb.collections.ImmutableIndexedObjectList;
import com.swirlds.merkledb.collections.ImmutableIndexedObjectListUsingArray;
import com.swirlds.merkledb.collections.LongList;
import com.swirlds.merkledb.config.MerkleDbConfig;
import com.swirlds.merkledb.files.DataFileCommon;
import com.swirlds.merkledb.files.DataFileIterator;
import com.swirlds.merkledb.files.DataFileMetadata;
import com.swirlds.merkledb.files.DataFileReader;
import com.swirlds.merkledb.files.DataFileWriter;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class DataFileCollection
implements FileStatisticAware,
Snapshotable {
    private static final Logger logger = LogManager.getLogger(DataFileCollection.class);
    private static final int MOVE_LIST_CHUNK_SIZE = 500000;
    private static final int METADATA_FILE_FORMAT_VERSION = 1;
    private static final String METADATA_FILENAME_SUFFIX = "_metadata.pbj";
    private static final int NUM_OF_READ_RETRIES = 5;
    private static final FieldDefinition FIELD_FILECOLLECTION_MINVALIDKEY = new FieldDefinition("minValidKey", FieldType.UINT64, false, true, false, 1);
    private static final FieldDefinition FIELD_FILECOLLECTION_MAXVALIDKEY = new FieldDefinition("maxValidKey", FieldType.UINT64, false, true, false, 2);
    private final MerkleDbConfig dbConfig;
    private final Path storeDir;
    private final String storeName;
    private final String legacyStoreName;
    private final boolean loadedFromExistingFiles;
    private final AtomicInteger nextFileIndex = new AtomicInteger();
    private volatile KeyRange validKeyRange = KeyRange.INVALID_KEY_RANGE;
    private final AtomicReference<ImmutableIndexedObjectList<DataFileReader>> dataFiles = new AtomicReference();
    private final AtomicReference<DataFileWriter> currentDataFileWriter = new AtomicReference();
    private final AtomicReference<DataFileReader> currentDataFileReader = new AtomicReference();
    private final Function<List<DataFileReader>, ImmutableIndexedObjectList<DataFileReader>> indexedObjectListConstructor;
    private final ConcurrentSkipListSet<Integer> setOfNewFileIndexes = logger.isTraceEnabled() ? new ConcurrentSkipListSet() : null;

    public DataFileCollection(MerkleDbConfig config, Path storeDir, String storeName, LoadedDataCallback loadedDataCallback) throws IOException {
        this(config, storeDir, storeName, null, loadedDataCallback, l -> new ImmutableIndexedObjectListUsingArray((Function<Integer, T[]>)(DataFileReader[]::new), l));
    }

    public DataFileCollection(MerkleDbConfig dbConfig, Path storeDir, String storeName, String legacyStoreName, LoadedDataCallback loadedDataCallback) throws IOException {
        this(dbConfig, storeDir, storeName, legacyStoreName, loadedDataCallback, l -> new ImmutableIndexedObjectListUsingArray((Function<Integer, T[]>)(DataFileReader[]::new), l));
    }

    protected DataFileCollection(MerkleDbConfig dbConfig, Path storeDir, String storeName, String legacyStoreName, LoadedDataCallback loadedDataCallback, Function<List<DataFileReader>, ImmutableIndexedObjectList<DataFileReader>> indexedObjectListConstructor) throws IOException {
        this.dbConfig = dbConfig;
        this.storeDir = storeDir;
        this.storeName = storeName;
        this.legacyStoreName = legacyStoreName;
        this.indexedObjectListConstructor = indexedObjectListConstructor;
        if (Files.exists(storeDir, new LinkOption[0])) {
            this.loadedFromExistingFiles = this.tryLoadFromExistingStore(loadedDataCallback);
        } else {
            this.loadedFromExistingFiles = false;
            Files.createDirectories(storeDir, new FileAttribute[0]);
            this.nextFileIndex.set(0);
        }
    }

    public KeyRange getValidKeyRange() {
        return this.validKeyRange;
    }

    public boolean isLoadedFromExistingFiles() {
        return this.loadedFromExistingFiles;
    }

    public int getNumOfFiles() {
        return this.dataFiles.get().size();
    }

    public List<DataFileReader> getAllCompletedFiles() {
        ImmutableIndexedObjectList<DataFileReader> activeIndexedFiles = this.dataFiles.get();
        if (activeIndexedFiles == null) {
            return Collections.emptyList();
        }
        return activeIndexedFiles.stream().filter(DataFileReader::isFileCompleted).toList();
    }

    @Override
    public LongSummaryStatistics getFilesSizeStatistics() {
        ImmutableIndexedObjectList<DataFileReader> activeIndexedFiles = this.dataFiles.get();
        return activeIndexedFiles == null ? new LongSummaryStatistics() : activeIndexedFiles.stream().filter(DataFileReader::isFileCompleted).mapToLong(DataFileReader::getSize).summaryStatistics();
    }

    public void close() throws IOException {
        DataFileWriter currentDataFileForWriting = this.currentDataFileWriter.getAndSet(null);
        if (currentDataFileForWriting != null) {
            currentDataFileForWriting.close();
        }
        this.saveMetadata(this.storeDir);
        ImmutableIndexedObjectList fileList = this.dataFiles.getAndSet(null);
        if (fileList != null) {
            for (DataFileReader file : fileList.stream()::iterator) {
                file.close();
            }
        }
    }

    public void startWriting() throws IOException {
        DataFileWriter activeDataFileWriter = this.currentDataFileWriter.get();
        if (activeDataFileWriter != null) {
            throw new IOException("Tried to start writing when we were already writing.");
        }
        DataFileWriter writer = this.newDataFile(Instant.now(), 0);
        this.currentDataFileWriter.set(writer);
        DataFileMetadata metadata = writer.getMetadata();
        DataFileReader reader = this.addNewDataFileReader(writer.getPath(), metadata);
        this.currentDataFileReader.set(reader);
    }

    public long storeDataItem(BufferedData dataItem) throws IOException {
        DataFileWriter currentDataFileForWriting = this.currentDataFileWriter.get();
        if (currentDataFileForWriting == null) {
            throw new IOException("Tried to put data " + String.valueOf(dataItem) + " when we never started writing.");
        }
        return currentDataFileForWriting.storeDataItem(dataItem);
    }

    public long storeDataItem(Consumer<BufferedData> dataItemWriter, int dataItemSize) throws IOException {
        DataFileWriter currentDataFileForWriting = this.currentDataFileWriter.get();
        if (currentDataFileForWriting == null) {
            throw new IOException("Tried to put data " + String.valueOf(dataItemWriter) + " when we never started writing.");
        }
        return currentDataFileForWriting.storeDataItem(dataItemWriter, dataItemSize);
    }

    public DataFileReader endWriting() throws IOException {
        DataFileWriter dataWriter = this.currentDataFileWriter.getAndSet(null);
        if (dataWriter == null) {
            throw new IOException("Tried to end writing when we never started writing.");
        }
        dataWriter.close();
        DataFileReader dataReader = this.currentDataFileReader.getAndSet(null);
        if (logger.isTraceEnabled()) {
            DataFileMetadata metadata = dataReader.getMetadata();
            this.setOfNewFileIndexes.remove(metadata.getIndex());
        }
        dataReader.setFileCompleted();
        return dataReader;
    }

    public void updateValidKeyRange(long minimumValidKey, long maximumValidKey) {
        this.validKeyRange = new KeyRange(minimumValidKey, maximumValidKey);
    }

    private DataFileReader readerForDataLocation(long dataLocation) throws IOException {
        DataFileReader file;
        if (dataLocation == 0L) {
            return null;
        }
        int fileIndex = DataFileCommon.fileIndexFromDataLocation(dataLocation);
        ImmutableIndexedObjectList<DataFileReader> currentIndexedFileList = this.dataFiles.get();
        if (fileIndex < 0 || currentIndexedFileList == null || (file = currentIndexedFileList.get(fileIndex)) == null) {
            throw new IOException("Got a data location from index for a file that doesn't exist. dataLocation=" + DataFileCommon.dataLocationToString(dataLocation) + " fileIndex=" + fileIndex + " validKeyRange=" + String.valueOf(this.validKeyRange) + "\ncurrentIndexedFileList=" + String.valueOf(currentIndexedFileList));
        }
        if (file.isOpen()) {
            return file;
        }
        logger.warn(LogMarker.EXCEPTION.getMarker(), "Store [{}] DataFile was closed while trying to read from file", (Object)this.storeName);
        return null;
    }

    public BufferedData readDataItem(long dataLocation) throws IOException {
        DataFileReader file = this.readerForDataLocation(dataLocation);
        return file != null ? file.readDataItem(dataLocation) : null;
    }

    public BufferedData readDataItemUsingIndex(LongList index, long keyIntoIndex) throws IOException {
        for (int retries = 0; retries < 5; ++retries) {
            long dataLocation = index.get(keyIntoIndex, 0L);
            if (dataLocation == 0L) {
                return null;
            }
            try {
                BufferedData readData = this.readDataItem(dataLocation);
                if (readData == null) continue;
                return readData;
            }
            catch (IOException e) {
                int currentRetry = retries;
                logger.warn(LogMarker.EXCEPTION.getMarker(), () -> {
                    String currentFiles = this.dataFiles.get() == null ? "UNKNOWN" : this.dataFiles.get().prettyPrintedIndices();
                    return "Store [" + this.storeName + "] had IOException while trying to read key [" + keyIntoIndex + "] at offset [" + DataFileCommon.byteOffsetFromDataLocation(dataLocation) + "] from file [" + DataFileCommon.fileIndexFromDataLocation(dataLocation) + "] on retry [" + currentRetry + "]. Current files are [" + currentFiles + "], validKeyRange=" + String.valueOf(this.validKeyRange) + ", storeDir=[" + String.valueOf(this.storeDir.toAbsolutePath()) + "]";
                }, (Throwable)e);
            }
        }
        throw new IOException("Read failed after 5 retries");
    }

    @Override
    public void snapshot(Path snapshotDirectory) throws IOException {
        this.saveMetadata(snapshotDirectory);
        List<DataFileReader> snapshotIndexedFiles = this.getAllCompletedFiles();
        for (DataFileReader fileReader : snapshotIndexedFiles) {
            Path existingFile = fileReader.getPath();
            Files.createLink(snapshotDirectory.resolve(existingFile.getFileName()), existingFile);
        }
    }

    Set<Integer> getSetOfNewFileIndexes() {
        if (logger.isTraceEnabled()) {
            return this.setOfNewFileIndexes;
        }
        throw new IllegalStateException("getSetOfNewFileIndexes can only be called if trace logging is enabled");
    }

    DataFileReader getDataFile(int index) {
        ImmutableIndexedObjectList<DataFileReader> fileList = this.dataFiles.get();
        return fileList == null ? null : fileList.get(index);
    }

    DataFileReader addNewDataFileReader(Path filePath, DataFileMetadata metadata) throws IOException {
        DataFileReader newDataFileReader = new DataFileReader(this.dbConfig, filePath, metadata);
        this.dataFiles.getAndUpdate(currentFileList -> {
            try {
                return currentFileList == null ? this.indexedObjectListConstructor.apply(Collections.singletonList(newDataFileReader)) : currentFileList.withAddedObject(newDataFileReader);
            }
            catch (IllegalArgumentException e) {
                logger.error(LogMarker.EXCEPTION.getMarker(), "Failed when updated the indexed files list", (Throwable)e);
                throw e;
            }
        });
        return newDataFileReader;
    }

    void deleteFiles(@NonNull Collection<?> filesToDelete) throws IOException {
        HashSet files = new HashSet(filesToDelete);
        this.dataFiles.getAndUpdate(currentFileList -> currentFileList == null ? null : currentFileList.withDeletedObjects(files));
        for (DataFileReader fileReader : files) {
            fileReader.close();
            Files.delete(fileReader.getPath());
        }
    }

    DataFileWriter newDataFile(Instant creationTime, int compactionLevel) throws IOException {
        int newFileIndex = this.nextFileIndex.getAndIncrement();
        if (logger.isTraceEnabled()) {
            this.setOfNewFileIndexes.add(newFileIndex);
        }
        return new DataFileWriter(this.storeName, this.storeDir, newFileIndex, creationTime, compactionLevel);
    }

    private void saveMetadata(Path directory) throws IOException {
        Files.createDirectories(directory, new FileAttribute[0]);
        KeyRange keyRange = this.validKeyRange;
        Path metadataFile = directory.resolve(this.storeName + METADATA_FILENAME_SUFFIX);
        try (OutputStream fileOut = Files.newOutputStream(metadataFile, new OpenOption[0]);){
            WritableStreamingData out = new WritableStreamingData(fileOut);
            if (keyRange.getMinValidKey() != 0L) {
                ProtoWriterTools.writeTag((WritableSequentialData)out, (FieldDefinition)FIELD_FILECOLLECTION_MINVALIDKEY);
                out.writeVarLong(keyRange.getMinValidKey(), false);
            }
            if (keyRange.getMaxValidKey() != 0L) {
                ProtoWriterTools.writeTag((WritableSequentialData)out, (FieldDefinition)FIELD_FILECOLLECTION_MAXVALIDKEY);
                out.writeVarLong(keyRange.getMaxValidKey(), false);
            }
            fileOut.flush();
        }
    }

    private boolean tryLoadFromExistingStore(LoadedDataCallback loadedDataCallback) throws IOException {
        if (!Files.isDirectory(this.storeDir, new LinkOption[0])) {
            throw new IOException("Tried to initialize DataFileCollection with a storage directory that is not a directory. [" + String.valueOf(this.storeDir.toAbsolutePath()) + "]");
        }
        try (Stream<Path> storePaths = Files.list(this.storeDir);){
            boolean bl;
            Path[] fullWrittenFilePaths = (Path[])storePaths.filter(path -> DataFileCommon.isFullyWrittenDataFile(this.storeName, path) || DataFileCommon.isFullyWrittenDataFile(this.legacyStoreName, path)).toArray(Path[]::new);
            Object[] dataFileReaders = new DataFileReader[fullWrittenFilePaths.length];
            try {
                for (int i = 0; i < fullWrittenFilePaths.length; ++i) {
                    assert (fullWrittenFilePaths[i].toString().endsWith(".pbj"));
                    dataFileReaders[i] = new DataFileReader(this.dbConfig, fullWrittenFilePaths[i]);
                }
                Arrays.sort(dataFileReaders);
            }
            catch (IOException e) {
                for (Object dataFileReader : dataFileReaders) {
                    if (dataFileReader == null) continue;
                    ((DataFileReader)dataFileReader).close();
                }
                throw e;
            }
            if (dataFileReaders.length > 0) {
                this.loadFromExistingFiles((DataFileReader[])dataFileReaders, loadedDataCallback);
                bl = true;
                return bl;
            }
            this.nextFileIndex.set(0);
            bl = false;
            return bl;
        }
    }

    private boolean loadMetadata() throws IOException {
        boolean loadedLegacyMetadata = false;
        Path metadataFile = this.storeDir.resolve(this.storeName + METADATA_FILENAME_SUFFIX);
        if (!Files.exists(metadataFile, new LinkOption[0])) {
            metadataFile = this.storeDir.resolve(this.legacyStoreName + METADATA_FILENAME_SUFFIX);
            loadedLegacyMetadata = true;
        }
        if (!Files.exists(metadataFile, new LinkOption[0])) {
            return false;
        }
        try (ReadableStreamingData in = new ReadableStreamingData(metadataFile);){
            long minValidKey = 0L;
            long maxValidKey = 0L;
            while (in.hasRemaining()) {
                int tag = in.readVarInt(false);
                int fieldNum = tag >> 3;
                if (fieldNum == FIELD_FILECOLLECTION_MINVALIDKEY.number()) {
                    minValidKey = in.readVarLong(false);
                    continue;
                }
                if (fieldNum == FIELD_FILECOLLECTION_MAXVALIDKEY.number()) {
                    maxValidKey = in.readVarLong(false);
                    continue;
                }
                throw new IllegalArgumentException("Unknown file collection metadata field: " + fieldNum);
            }
            this.validKeyRange = new KeyRange(minValidKey, maxValidKey);
        }
        if (loadedLegacyMetadata) {
            Files.delete(metadataFile);
        }
        return true;
    }

    private void loadFromExistingFiles(DataFileReader[] dataFileReaders, LoadedDataCallback loadedDataCallback) throws IOException {
        logger.info(LogMarker.MERKLE_DB.getMarker(), "Loading existing set of [{}] data files for DataFileCollection [{}]", (Object)dataFileReaders.length, (Object)this.storeName);
        if (!this.loadMetadata()) {
            logger.warn(LogMarker.EXCEPTION.getMarker(), "Loading existing set of data files but no metadata file was found in [{}]", (Object)this.storeDir.toAbsolutePath());
        }
        this.dataFiles.set(this.indexedObjectListConstructor.apply(List.of(dataFileReaders)));
        this.nextFileIndex.set(this.getMaxFileReaderIndex(dataFileReaders) + 1);
        if (loadedDataCallback != null) {
            for (DataFileReader reader : dataFileReaders) {
                try (DataFileIterator iterator = reader.createIterator();){
                    while (iterator.next()) {
                        loadedDataCallback.newIndexEntry(iterator.getDataItemDataLocation(), iterator.getDataItemData());
                    }
                }
            }
        }
        for (DataFileReader dataFileReader : dataFileReaders) {
            dataFileReader.setFileCompleted();
        }
        logger.info(LogMarker.MERKLE_DB.getMarker(), "Finished loading existing data files for DataFileCollection [{}]", (Object)this.storeName);
    }

    private int getMaxFileReaderIndex(DataFileReader[] dataFileReaders) {
        return Stream.of(dataFileReaders).mapToInt(DataFileReader::getIndex).max().orElse(-1);
    }

    @FunctionalInterface
    public static interface LoadedDataCallback {
        public void newIndexEntry(long var1, @NonNull BufferedData var3);
    }
}

