/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb;

import com.hedera.pbj.runtime.FieldDefinition;
import com.hedera.pbj.runtime.FieldType;
import com.hedera.pbj.runtime.ProtoConstants;
import com.hedera.pbj.runtime.ProtoWriterTools;
import com.hedera.pbj.runtime.io.ReadableSequentialData;
import com.hedera.pbj.runtime.io.WritableSequentialData;
import com.hedera.pbj.runtime.io.stream.ReadableStreamingData;
import com.hedera.pbj.runtime.io.stream.WritableStreamingData;
import com.swirlds.common.io.utility.FileUtils;
import com.swirlds.common.io.utility.LegacyTemporaryFileBuilder;
import com.swirlds.config.api.Configuration;
import com.swirlds.logging.legacy.LogMarker;
import com.swirlds.merkledb.MerkleDbDataSource;
import com.swirlds.merkledb.MerkleDbTableConfig;
import com.swirlds.merkledb.files.DataFileCommon;
import edu.umd.cs.findbugs.annotations.NonNull;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.AtomicReferenceArray;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class MerkleDb {
    private static final Logger logger = LogManager.getLogger(MerkleDb.class);
    private static final int MAX_TABLES = 1024;
    private static final String SHARED_DIRNAME = "shared";
    private static final String TABLES_DIRNAME = "tables";
    private static final String METADATA_FILENAME = "database_metadata.pbj";
    public static final String MERKLEDB_COMPONENT = "merkledb";
    @NonNull
    private final Configuration configuration;
    private static final ConcurrentHashMap<Path, MerkleDb> instances = new ConcurrentHashMap();
    private static final AtomicReference<Path> defaultInstancePath = new AtomicReference();
    private final Path storageDir;
    private final AtomicInteger nextTableId = new AtomicInteger(0);
    private final AtomicReferenceArray<TableMetadata> tableConfigs;
    private final AtomicReferenceArray<MerkleDbDataSource> dataSources = new AtomicReferenceArray(1024);
    private final Set<Integer> primaryTables = ConcurrentHashMap.newKeySet();
    private static final FieldDefinition FIELD_DBMETADATA_TABLEMETADATA = new FieldDefinition("tableMetadata", FieldType.MESSAGE, true, true, false, 11);

    public static MerkleDb getInstance(Path path, @NonNull Configuration configuration) {
        Objects.requireNonNull(configuration);
        return instances.computeIfAbsent(path != null ? path : MerkleDb.getDefaultPath(configuration), p -> new MerkleDb((Path)p, configuration));
    }

    private static Path getDefaultPath(@NonNull Configuration configuration) {
        Objects.requireNonNull(configuration);
        return defaultInstancePath.updateAndGet(p -> {
            if (p == null) {
                try {
                    p = LegacyTemporaryFileBuilder.buildTemporaryFile((String)MERKLEDB_COMPONENT, (Configuration)configuration);
                }
                catch (IOException z) {
                    throw new UncheckedIOException(z);
                }
            }
            return p;
        });
    }

    public static void setDefaultPath(@NonNull Path value) {
        Objects.requireNonNull(value);
        defaultInstancePath.set(value);
    }

    public static void resetDefaultInstancePath() {
        defaultInstancePath.set(null);
    }

    public static MerkleDb getDefaultInstance(@NonNull Configuration configuration) {
        Objects.requireNonNull(configuration);
        return MerkleDb.getInstance(MerkleDb.getDefaultPath(configuration), configuration);
    }

    private MerkleDb(Path storageDir, @NonNull Configuration configuration) {
        Objects.requireNonNull(configuration);
        this.configuration = configuration;
        if (storageDir == null) {
            throw new IllegalArgumentException("Cannot create a MerkleDatabase instance with null storageDir");
        }
        this.storageDir = storageDir;
        this.tableConfigs = this.loadMetadata();
        try {
            Path tablesDir;
            Path sharedDir = this.getSharedDir();
            if (!Files.exists(sharedDir, new LinkOption[0])) {
                Files.createDirectories(sharedDir, new FileAttribute[0]);
            }
            if (!Files.exists(tablesDir = this.getTablesDir(), new LinkOption[0])) {
                Files.createDirectories(tablesDir, new FileAttribute[0]);
            }
        }
        catch (IOException z) {
            throw new UncheckedIOException(z);
        }
        if (!Files.exists(storageDir.resolve(METADATA_FILENAME), new LinkOption[0])) {
            this.storeMetadata();
        }
        logger.info(LogMarker.MERKLE_DB.getMarker(), "New MerkleDb instance is created, storageDir={}", (Object)storageDir);
    }

    int getNextTableId() {
        for (int tablesCount = 0; tablesCount < 1024; ++tablesCount) {
            int id = Math.abs(this.nextTableId.getAndIncrement() % 1024);
            if (this.tableConfigs.get(id) != null) continue;
            return id;
        }
        throw new IllegalStateException("Tables limit is reached");
    }

    public Path getStorageDir() {
        return this.storageDir;
    }

    public Path getSharedDir() {
        return MerkleDb.getSharedDir(this.storageDir);
    }

    private static Path getSharedDir(Path baseDir) {
        return baseDir.resolve(SHARED_DIRNAME);
    }

    public Path getTablesDir() {
        return MerkleDb.getTablesDir(this.storageDir);
    }

    private static Path getTablesDir(Path baseDir) {
        return baseDir.resolve(TABLES_DIRNAME);
    }

    public Path getTableDir(String tableName, int tableId) {
        return MerkleDb.getTableDir(this.storageDir, tableName, tableId);
    }

    private static Path getTableDir(Path baseDir, String tableName, int tableId) {
        return MerkleDb.getTablesDir(baseDir).resolve(tableName + "-" + tableId);
    }

    public Configuration getConfiguration() {
        return this.configuration;
    }

    public MerkleDbDataSource createDataSource(String label, MerkleDbTableConfig tableConfig, boolean dbCompactionEnabled) throws IOException {
        if (this.tableExists(label)) {
            throw new IllegalStateException("Table already exists: " + label);
        }
        int tableId = this.getNextTableId();
        this.tableConfigs.set(tableId, new TableMetadata(tableId, label, tableConfig));
        MerkleDbDataSource dataSource = new MerkleDbDataSource(this, this.configuration, label, tableId, tableConfig, dbCompactionEnabled, false);
        this.dataSources.set(tableId, dataSource);
        this.primaryTables.add(tableId);
        this.storeMetadata();
        return dataSource;
    }

    public MerkleDbDataSource copyDataSource(MerkleDbDataSource dataSource, boolean makeCopyPrimary, boolean offlineUse) throws IOException {
        String label = dataSource.getTableName();
        int tableId = this.getNextTableId();
        this.importDataSource(dataSource, tableId, !makeCopyPrimary, makeCopyPrimary);
        return this.getDataSource(this.configuration, tableId, label, makeCopyPrimary, offlineUse);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void importDataSource(MerkleDbDataSource dataSource, int tableId, boolean leaveSourcePrimary, boolean makeCopyPrimary) throws IOException {
        String label = dataSource.getTableName();
        MerkleDbTableConfig tableConfig = dataSource.getTableConfig().copy();
        if (this.tableConfigs.get(tableId) != null) {
            throw new IllegalStateException("Table with ID " + tableId + " already exists");
        }
        this.tableConfigs.set(tableId, new TableMetadata(tableId, label, tableConfig));
        try {
            dataSource.pauseCompaction();
            dataSource.snapshot(this.getTableDir(label, tableId));
        }
        finally {
            dataSource.resumeCompaction();
        }
        if (!leaveSourcePrimary) {
            dataSource.stopAndDisableBackgroundCompaction();
            this.primaryTables.remove(dataSource.getTableId());
        }
        if (makeCopyPrimary) {
            this.primaryTables.add(tableId);
        }
        this.storeMetadata();
    }

    public MerkleDbDataSource getDataSource(String name, boolean dbCompactionEnabled) throws IOException {
        return this.getDataSource(this.configuration, name, dbCompactionEnabled);
    }

    public MerkleDbDataSource getDataSource(Configuration config, String name, boolean dbCompactionEnabled) throws IOException {
        TableMetadata metadata = this.getTableMetadata(name);
        if (metadata == null) {
            throw new IllegalStateException("Unknown table: " + name);
        }
        int tableId = metadata.getTableId();
        return this.getDataSource(config, tableId, name, dbCompactionEnabled, false);
    }

    private MerkleDbDataSource getDataSource(Configuration config, int tableId, String tableName, boolean dbCompactionEnabled, boolean offlineUse) throws IOException {
        MerkleDbTableConfig tableConfig = this.getTableConfig(tableId);
        if (tableConfig == null) {
            throw new IllegalStateException("Unknown table: " + tableId);
        }
        AtomicReference<Object> rethrowIO = new AtomicReference<Object>(null);
        MerkleDbDataSource dataSource = this.dataSources.updateAndGet(tableId, ds -> {
            if (ds != null) {
                return ds;
            }
            try {
                return new MerkleDbDataSource(this, config, tableName, tableId, tableConfig, dbCompactionEnabled, offlineUse);
            }
            catch (IOException z) {
                rethrowIO.set(z);
                return null;
            }
        });
        if (rethrowIO.get() != null) {
            throw (IOException)rethrowIO.get();
        }
        return dataSource;
    }

    void closeDataSource(MerkleDbDataSource dataSource, boolean deleteData) {
        if (this != dataSource.getDatabase()) {
            throw new IllegalStateException("Can't close table in a different database");
        }
        int tableId = dataSource.getTableId();
        assert (this.dataSources.get(tableId) != null);
        this.dataSources.set(tableId, null);
        if (deleteData) {
            TableMetadata metadata = this.tableConfigs.get(tableId);
            if (metadata == null) {
                throw new IllegalArgumentException("Unknown table ID: " + tableId);
            }
            String label = metadata.getTableName();
            this.tableConfigs.set(tableId, null);
            DataFileCommon.deleteDirectoryAndContents(this.getTableDir(label, tableId));
        }
        this.storeMetadata();
    }

    public MerkleDbTableConfig getTableConfig(int tableId) {
        if (tableId < 0 || tableId >= 1024) {
            return null;
        }
        TableMetadata metadata = this.tableConfigs.get(tableId);
        MerkleDbTableConfig tableConfig = metadata != null ? metadata.getTableConfig() : null;
        return tableConfig;
    }

    public Map<String, MerkleDbTableConfig> getTableConfigs() {
        HashMap<String, MerkleDbTableConfig> result = new HashMap<String, MerkleDbTableConfig>();
        for (int i = 0; i < this.tableConfigs.length(); ++i) {
            TableMetadata tableMetadata = this.tableConfigs.get(i);
            if (tableMetadata == null || !this.primaryTables.contains(tableMetadata.getTableId())) continue;
            result.put(tableMetadata.getTableName(), tableMetadata.getTableConfig());
        }
        return result;
    }

    public void snapshot(Path destination, MerkleDbDataSource dataSource) throws IOException {
        MerkleDb targetDb;
        if (this != dataSource.getDatabase()) {
            logger.error(LogMarker.EXCEPTION.getMarker(), "Trying to snapshot a data source from a different database. This storageDir={}, other storageDir={}", (Object)this.getStorageDir(), (Object)dataSource.getDatabase().getStorageDir());
            throw new IllegalArgumentException("Cannot snapshot a data source from a different database");
        }
        String tableName = dataSource.getTableName();
        boolean isPrimary = this.primaryTables.contains(dataSource.getTableId());
        if (!isPrimary) {
            logger.info(LogMarker.MERKLE_DB.getMarker(), "A snapshot is taken for a secondary table, it may happen during reconnect or ISS reporting. Table name={}", (Object)tableName);
        }
        if ((targetDb = MerkleDb.getInstance(destination, this.configuration)).tableExists(tableName)) {
            throw new IllegalStateException("Table already exists in the target database, " + tableName);
        }
        targetDb.importDataSource(dataSource, targetDb.getNextTableId(), true, true);
        targetDb.storeMetadata();
    }

    public static MerkleDb restore(Path source, Path target, @NonNull Configuration configuration) throws IOException {
        Path defaultInstancePath;
        Objects.requireNonNull(configuration);
        Path path = defaultInstancePath = target != null ? target : MerkleDb.getDefaultPath(configuration);
        if (!Files.exists(defaultInstancePath.resolve(METADATA_FILENAME), new LinkOption[0])) {
            Files.createDirectories(defaultInstancePath, new FileAttribute[0]);
            assert (Files.exists(source.resolve(METADATA_FILENAME), new LinkOption[0]));
            Files.copy(source.resolve(METADATA_FILENAME), defaultInstancePath.resolve(METADATA_FILENAME), new CopyOption[0]);
            Path sharedDirPath = source.resolve(SHARED_DIRNAME);
            if (Files.exists(sharedDirPath, new LinkOption[0])) {
                FileUtils.hardLinkTree((Path)sharedDirPath, (Path)defaultInstancePath.resolve(SHARED_DIRNAME));
            }
            FileUtils.hardLinkTree((Path)source.resolve(TABLES_DIRNAME), (Path)defaultInstancePath.resolve(TABLES_DIRNAME));
        }
        return MerkleDb.getInstance(defaultInstancePath, configuration);
    }

    private synchronized void storeMetadata() {
        HashSet<TableMetadata> tables = new HashSet<TableMetadata>();
        for (int i = 0; i < this.tableConfigs.length(); ++i) {
            TableMetadata tableMetadata = this.tableConfigs.get(i);
            if (tableMetadata == null || !this.primaryTables.contains(i)) continue;
            tables.add(tableMetadata);
        }
        Path dbMetadataFile = this.storageDir.resolve(METADATA_FILENAME);
        try (OutputStream fileOut = Files.newOutputStream(dbMetadataFile, StandardOpenOption.WRITE, StandardOpenOption.CREATE);){
            WritableStreamingData out = new WritableStreamingData(fileOut);
            for (TableMetadata metadata : tables) {
                int len = metadata.pbjSizeInBytes();
                ProtoWriterTools.writeDelimited((WritableSequentialData)out, (FieldDefinition)FIELD_DBMETADATA_TABLEMETADATA, (int)len, metadata::writeTo);
            }
        }
        catch (IOException z) {
            throw new UncheckedIOException(z);
        }
    }

    private AtomicReferenceArray<TableMetadata> loadMetadata() {
        AtomicReferenceArray<TableMetadata> metadata = MerkleDb.loadMetadata(this.storageDir);
        for (int i = 0; i < 1024; ++i) {
            if (metadata.get(i) == null) continue;
            this.primaryTables.add(i);
        }
        return metadata;
    }

    private static AtomicReferenceArray<TableMetadata> loadMetadata(Path dir) {
        AtomicReferenceArray<TableMetadata> tableConfigs = new AtomicReferenceArray<TableMetadata>(1024);
        Path tableConfigFile = dir.resolve(METADATA_FILENAME);
        if (Files.exists(tableConfigFile, new LinkOption[0])) {
            try (ReadableStreamingData in = new ReadableStreamingData(tableConfigFile);){
                while (in.hasRemaining()) {
                    int tag = in.readVarInt(false);
                    int fieldNum = tag >> 3;
                    if (fieldNum == FIELD_DBMETADATA_TABLEMETADATA.number()) {
                        int size = in.readVarInt(false);
                        long oldLimit = in.limit();
                        in.limit(in.position() + (long)size);
                        TableMetadata tableMetadata = new TableMetadata((ReadableSequentialData)in);
                        in.limit(oldLimit);
                        int tableId = tableMetadata.getTableId();
                        if (tableId < 0 || tableId >= 1024) {
                            throw new IllegalStateException("Corrupted MerkleDb metadata: wrong table ID");
                        }
                        tableConfigs.set(tableId, tableMetadata);
                        continue;
                    }
                    throw new IllegalArgumentException("Unknown database metadata field: " + fieldNum);
                }
            }
            catch (IOException z) {
                throw new UncheckedIOException(z);
            }
        }
        return tableConfigs;
    }

    private boolean tableExists(String tableName) {
        for (int i = 0; i < this.tableConfigs.length(); ++i) {
            TableMetadata tableMetadata = this.tableConfigs.get(i);
            if (tableMetadata == null || !tableName.equals(tableMetadata.getTableName())) continue;
            return true;
        }
        return false;
    }

    private TableMetadata getTableMetadata(String tableName) {
        for (int i = 0; i < this.tableConfigs.length(); ++i) {
            TableMetadata metadata = this.tableConfigs.get(i);
            if (metadata == null || !tableName.equals(metadata.getTableName()) || !this.primaryTables.contains(i)) continue;
            return metadata;
        }
        return null;
    }

    public int hashCode() {
        return this.storageDir.hashCode();
    }

    public boolean equals(Object o) {
        if (!(o instanceof MerkleDb)) {
            return false;
        }
        MerkleDb other = (MerkleDb)o;
        return Objects.equals(this.storageDir, other.storageDir);
    }

    private static class TableMetadata {
        private final int tableId;
        private final String tableName;
        private final MerkleDbTableConfig tableConfig;
        private static final FieldDefinition FIELD_TABLEMETADATA_TABLEID = new FieldDefinition("tableId", FieldType.UINT32, false, true, false, 1);
        private static final FieldDefinition FIELD_TABLEMETADATA_TABLENAME = new FieldDefinition("tableName", FieldType.BYTES, false, false, false, 2);
        private static final FieldDefinition FIELD_TABLEMETADATA_TABLECONFIG = new FieldDefinition("tableConfig", FieldType.MESSAGE, false, false, false, 3);

        public TableMetadata(int tableId, String tableName, MerkleDbTableConfig tableConfig) {
            if (tableId < 0) {
                throw new IllegalArgumentException("Table ID < 0");
            }
            if (tableId >= 1024) {
                throw new IllegalArgumentException("Table ID >= MAX_TABLES");
            }
            this.tableId = tableId;
            if (tableName == null) {
                throw new IllegalArgumentException("Table name is null");
            }
            this.tableName = tableName;
            if (tableConfig == null) {
                throw new IllegalArgumentException("Table config is null");
            }
            this.tableConfig = tableConfig;
        }

        public TableMetadata(ReadableSequentialData in) {
            int tableId = 0;
            String tableName = null;
            MerkleDbTableConfig tableConfig = null;
            while (in.hasRemaining()) {
                int len;
                int tag = in.readVarInt(false);
                int fieldNum = tag >> 3;
                if (fieldNum == FIELD_TABLEMETADATA_TABLEID.number()) {
                    tableId = in.readVarInt(false);
                    continue;
                }
                if (fieldNum == FIELD_TABLEMETADATA_TABLENAME.number()) {
                    len = in.readVarInt(false);
                    byte[] bb = new byte[len];
                    in.readBytes(bb);
                    tableName = new String(bb, StandardCharsets.UTF_8);
                    continue;
                }
                if (fieldNum == FIELD_TABLEMETADATA_TABLECONFIG.number()) {
                    len = in.readVarInt(false);
                    long oldLimit = in.limit();
                    in.limit(in.position() + (long)len);
                    tableConfig = new MerkleDbTableConfig(in);
                    in.limit(oldLimit);
                    continue;
                }
                throw new IllegalArgumentException("Unknown table metadata field: " + fieldNum);
            }
            Objects.requireNonNull(tableName, "Null table name");
            Objects.requireNonNull(tableConfig, "Null table config");
            this.tableId = tableId;
            this.tableName = tableName;
            this.tableConfig = tableConfig;
        }

        public int getTableId() {
            return this.tableId;
        }

        public String getTableName() {
            return this.tableName;
        }

        public MerkleDbTableConfig getTableConfig() {
            return this.tableConfig;
        }

        public int pbjSizeInBytes() {
            int size = 0;
            if (this.tableId != 0) {
                size += ProtoWriterTools.sizeOfTag((FieldDefinition)FIELD_TABLEMETADATA_TABLEID, (ProtoConstants)ProtoConstants.WIRE_TYPE_VARINT_OR_ZIGZAG);
                size += ProtoWriterTools.sizeOfVarInt32((int)this.tableId);
            }
            size += ProtoWriterTools.sizeOfDelimited((FieldDefinition)FIELD_TABLEMETADATA_TABLENAME, (int)this.tableName.getBytes(StandardCharsets.UTF_8).length);
            return size += ProtoWriterTools.sizeOfDelimited((FieldDefinition)FIELD_TABLEMETADATA_TABLECONFIG, (int)this.tableConfig.pbjSizeInBytes());
        }

        public void writeTo(WritableSequentialData out) {
            if (this.tableId != 0) {
                ProtoWriterTools.writeTag((WritableSequentialData)out, (FieldDefinition)FIELD_TABLEMETADATA_TABLEID);
                out.writeVarInt(this.tableId, false);
            }
            byte[] bb = this.tableName.getBytes(StandardCharsets.UTF_8);
            ProtoWriterTools.writeDelimited((WritableSequentialData)out, (FieldDefinition)FIELD_TABLEMETADATA_TABLENAME, (int)bb.length, t -> t.writeBytes(bb));
            ProtoWriterTools.writeDelimited((WritableSequentialData)out, (FieldDefinition)FIELD_TABLEMETADATA_TABLECONFIG, (int)this.tableConfig.pbjSizeInBytes(), this.tableConfig::writeTo);
        }
    }
}

