/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.merkledb.collections;

import com.swirlds.config.api.Configuration;
import com.swirlds.merkledb.collections.CASableLongIndex;
import com.swirlds.merkledb.collections.LongList;
import com.swirlds.merkledb.collections.LongListSpliterator;
import com.swirlds.merkledb.config.MerkleDbConfig;
import com.swirlds.merkledb.utilities.MerkleDbFileUtils;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.function.BooleanSupplier;
import java.util.stream.LongStream;
import java.util.stream.StreamSupport;

public abstract class AbstractLongList<C>
implements LongList {
    public static final String MAX_CHUNKS_EXCEEDED_MSG = "The maximum number of memory chunks should not exceed %s. Either increase longsPerChunk or decrease capacity";
    public static final String CHUNK_SIZE_ZERO_OR_NEGATIVE_MSG = "Cannot store %d per chunk (min is 1)";
    public static final String CHUNK_SIZE_EXCEEDED_MSG = "Cannot store %d per chunk (max is %d)";
    public static final String INVALID_RANGE_MSG = "Invalid range %d - %d";
    public static final String MAX_VALID_INDEX_LIMIT = "Max valid index %d must be less than max capacity %d";
    protected static final int MAX_NUM_LONGS_PER_CHUNK = Math.toIntExact(0xFA00000L);
    public static final int MAX_NUM_CHUNKS = 0x200000;
    private static final int MIN_VALID_INDEX_SUPPORT_VERSION = 2;
    private static final int NO_CAPACITY_VERSION = 3;
    private static final int CURRENT_FILE_FORMAT_VERSION = 3;
    protected static final int VERSION_METADATA_SIZE = 4;
    protected static final int FORMAT_METADATA_SIZE_V2 = 20;
    protected static final int FORMAT_METADATA_SIZE_V3 = 8;
    protected static final int FILE_HEADER_SIZE_V2 = 24;
    protected static final int FILE_HEADER_SIZE_V3 = 12;
    protected final int longsPerChunk;
    protected final int memoryChunkSize;
    protected final AtomicLong size = new AtomicLong(0L);
    protected final long capacity;
    protected final AtomicLong minValidIndex = new AtomicLong(-1L);
    protected final AtomicLong maxValidIndex = new AtomicLong(-1L);
    protected final AtomicReferenceArray<C> chunkList;
    protected final long reservedBufferSize;

    protected AbstractLongList(long capacity, Configuration configuration) {
        MerkleDbConfig merkleDbConfig = (MerkleDbConfig)configuration.getConfigData(MerkleDbConfig.class);
        this.checkCapacity(capacity);
        this.capacity = capacity;
        this.longsPerChunk = merkleDbConfig.longListChunkSize();
        this.reservedBufferSize = merkleDbConfig.longListReservedBufferSize();
        this.chunkList = new AtomicReferenceArray(this.calculateNumberOfChunks(capacity));
        this.memoryChunkSize = Math.multiplyExact(this.longsPerChunk, 8);
    }

    protected AbstractLongList(int longsPerChunk, long capacity, long reservedBufferSize) {
        this.checkCapacity(capacity);
        this.capacity = capacity;
        this.checkLongsPerChunk(longsPerChunk);
        this.longsPerChunk = longsPerChunk;
        this.reservedBufferSize = reservedBufferSize;
        this.chunkList = new AtomicReferenceArray(this.calculateNumberOfChunks(capacity));
        this.memoryChunkSize = Math.multiplyExact(this.longsPerChunk, 8);
    }

    public AbstractLongList(@NonNull Path file, long capacity, @NonNull Configuration configuration) throws IOException {
        MerkleDbConfig merkleDbConfig = (MerkleDbConfig)configuration.getConfigData(MerkleDbConfig.class);
        this.checkCapacity(capacity);
        this.capacity = capacity;
        this.longsPerChunk = merkleDbConfig.longListChunkSize();
        this.memoryChunkSize = this.longsPerChunk * 8;
        this.reservedBufferSize = merkleDbConfig.longListReservedBufferSize();
        this.chunkList = new AtomicReferenceArray(this.calculateNumberOfChunks(this.capacity));
        this.loadFromFile(file, configuration);
    }

    protected AbstractLongList(@NonNull Path path, int longsPerChunk, long capacity, long reservedBufferSize, @NonNull Configuration configuration) throws IOException {
        this.longsPerChunk = longsPerChunk;
        this.memoryChunkSize = longsPerChunk * 8;
        this.capacity = capacity;
        this.reservedBufferSize = reservedBufferSize;
        this.chunkList = new AtomicReferenceArray(this.calculateNumberOfChunks(this.capacity));
        this.loadFromFile(path, configuration);
    }

    private void loadFromFile(@NonNull Path file, @NonNull Configuration configuration) throws IOException {
        Objects.requireNonNull(file);
        Objects.requireNonNull(configuration);
        if (!Files.exists(file, new LinkOption[0])) {
            throw new IOException("Cannot load index, file doesn't exist: " + String.valueOf(file.toAbsolutePath()));
        }
        try (FileChannel fileChannel = FileChannel.open(file, StandardOpenOption.READ);){
            int currentFileHeaderSize;
            int formatMetadataSize;
            ByteBuffer versionBuffer = MerkleDbFileUtils.readFromFileChannel(fileChannel, 4);
            int formatVersion = versionBuffer.getInt();
            if (formatVersion == 2) {
                formatMetadataSize = 20;
                currentFileHeaderSize = 24;
            } else if (formatVersion == 3) {
                formatMetadataSize = 8;
                currentFileHeaderSize = 12;
            } else {
                throw new IOException("File format version is not supported. File format version [" + formatVersion + "], the latest supported version is [3].");
            }
            ByteBuffer headerBuffer = MerkleDbFileUtils.readFromFileChannel(fileChannel, formatMetadataSize);
            if (formatVersion == 2) {
                headerBuffer.getInt();
            }
            if (formatVersion == 2) {
                headerBuffer.getLong();
            }
            long longsInFile = (fileChannel.size() - (long)currentFileHeaderSize) / 8L;
            long readMinValidIndex = headerBuffer.getLong();
            if (longsInFile <= 0L || readMinValidIndex < 0L) {
                this.size.set(0L);
                this.minValidIndex.set(-1L);
                this.maxValidIndex.set(-1L);
            } else {
                this.minValidIndex.set(readMinValidIndex);
                this.size.set(readMinValidIndex + longsInFile);
                this.maxValidIndex.set(this.size.get() - 1L);
            }
            if (this.size.get() > this.capacity) {
                throw new IllegalArgumentException("Failed to read index from file, size=" + this.size.get() + ", capacity=" + this.capacity);
            }
            this.readBodyFromFileChannelOnInit(file.getFileName().toString(), fileChannel, configuration);
        }
    }

    protected void readBodyFromFileChannelOnInit(String sourceFileName, FileChannel fileChannel, Configuration configuration) throws IOException {
        if (this.minValidIndex.get() < 0L) {
            return;
        }
        int firstChunkIndex = Math.toIntExact(this.minValidIndex.get() / (long)this.longsPerChunk);
        int lastChunkIndex = Math.toIntExact(this.maxValidIndex.get() / (long)this.longsPerChunk);
        int minValidIndexInChunk = Math.toIntExact(this.minValidIndex.get() % (long)this.longsPerChunk);
        int maxValidIndexInChunk = Math.toIntExact(this.maxValidIndex.get() % (long)this.longsPerChunk);
        for (int chunkIndex = firstChunkIndex; chunkIndex <= lastChunkIndex; ++chunkIndex) {
            int startIndexInChunk = chunkIndex == firstChunkIndex ? minValidIndexInChunk : 0;
            int endIndexInChunk = chunkIndex == lastChunkIndex ? maxValidIndexInChunk + 1 : this.longsPerChunk;
            C chunk = this.readChunkData(fileChannel, chunkIndex, startIndexInChunk, endIndexInChunk);
            this.setChunk(chunkIndex, chunk);
        }
    }

    protected abstract C readChunkData(FileChannel var1, int var2, int var3, int var4) throws IOException;

    protected void setChunk(int chunkIndex, C chunk) {
        this.chunkList.set(chunkIndex, chunk);
    }

    protected static void readDataIntoBuffer(FileChannel fileChannel, int chunkIndex, int startIndex, int endIndex, ByteBuffer buffer) throws IOException {
        int startOffset = startIndex * 8;
        int endOffset = endIndex * 8;
        buffer.position(startOffset);
        buffer.limit(endOffset);
        int bytesToRead = endOffset - startOffset;
        long bytesRead = MerkleDbFileUtils.completelyRead(fileChannel, buffer);
        if (bytesRead != (long)bytesToRead) {
            throw new IOException("Failed to read chunks, chunkIndex=" + chunkIndex + " expected=" + bytesToRead + " actual=" + bytesRead);
        }
    }

    @Override
    public long get(long index, long defaultValue) {
        if (index < 0L || index >= this.capacity) {
            throw new IndexOutOfBoundsException(index);
        }
        if (index >= this.size.get()) {
            return defaultValue;
        }
        int chunkIndex = Math.toIntExact(index / (long)this.longsPerChunk);
        long subIndex = index % (long)this.longsPerChunk;
        C chunk = this.chunkList.get(chunkIndex);
        if (chunk == null) {
            return defaultValue;
        }
        long presentValue = this.lookupInChunk(chunk, subIndex);
        return presentValue == 0L ? defaultValue : presentValue;
    }

    @Override
    public final void put(long index, long value) {
        this.checkIndex(index);
        this.checkValue(value);
        this.putImpl(index, value);
    }

    @Override
    public final void remove(long index) {
        this.checkIndex(index);
        this.putImpl(index, 0L);
    }

    private void putImpl(long index, long value) {
        assert (index >= this.minValidIndex.get()) : String.format("Index %d is less than min valid index %d", index, this.minValidIndex.get());
        assert (index <= this.maxValidIndex.get()) : String.format("Index %d is greater than max valid index %d", index, this.maxValidIndex.get());
        C chunk = this.createOrGetChunk(index);
        int subIndex = Math.toIntExact(index % (long)this.longsPerChunk);
        this.putToChunk(chunk, subIndex, value);
    }

    protected abstract void putToChunk(C var1, int var2, long var3);

    @Override
    public final boolean putIfEqual(long index, long oldValue, long newValue) {
        this.checkIndex(index);
        this.checkValue(newValue);
        int chunkIndex = Math.toIntExact(index / (long)this.longsPerChunk);
        C chunk = this.chunkList.get(chunkIndex);
        if (chunk == null) {
            return false;
        }
        int subIndex = Math.toIntExact(index % (long)this.longsPerChunk);
        boolean result = this.putIfEqual(chunk, subIndex, oldValue, newValue);
        if (result) {
            this.size.getAndUpdate(oldSize -> index >= oldSize ? index + 1L : oldSize);
        }
        return result;
    }

    protected abstract boolean putIfEqual(C var1, int var2, long var3, long var5);

    @Override
    public long get(long index) {
        return this.get(index, 0L);
    }

    @Override
    public final long capacity() {
        return this.capacity;
    }

    @Override
    public final long size() {
        return this.size.get();
    }

    public final int getLongsPerChunk() {
        return this.longsPerChunk;
    }

    final long getReservedBufferSize() {
        return this.reservedBufferSize;
    }

    @Override
    public LongStream stream() {
        return StreamSupport.longStream(new LongListSpliterator(this), false);
    }

    @Override
    public void writeToFile(Path file) throws IOException {
        try (FileChannel fc = FileChannel.open(file, StandardOpenOption.CREATE_NEW, StandardOpenOption.WRITE);){
            this.writeHeader(fc);
            if (this.size() > 0L) {
                this.writeLongsData(fc);
            }
            fc.force(true);
        }
    }

    protected final void writeHeader(FileChannel fc) throws IOException {
        int currentFileHeaderSize = 12;
        ByteBuffer headerBuffer = ByteBuffer.allocate(12);
        headerBuffer.rewind();
        headerBuffer.putInt(3);
        headerBuffer.putLong(this.minValidIndex.get());
        headerBuffer.flip();
        if (MerkleDbFileUtils.completelyWrite(fc, headerBuffer, 0L) != 12) {
            throw new IOException("Failed to write long list header to the file channel " + String.valueOf(fc));
        }
        fc.position(12L);
    }

    protected abstract void writeLongsData(FileChannel var1) throws IOException;

    protected abstract long lookupInChunk(@NonNull C var1, long var2);

    @Override
    public final void updateValidRange(long newMinValidIndex, long newMaxValidIndex) {
        if (newMinValidIndex < -1L || newMinValidIndex > newMaxValidIndex) {
            throw new IndexOutOfBoundsException(INVALID_RANGE_MSG.formatted(newMinValidIndex, newMaxValidIndex));
        }
        if (newMaxValidIndex >= this.capacity) {
            throw new IndexOutOfBoundsException(MAX_VALID_INDEX_LIMIT.formatted(newMaxValidIndex, this.capacity));
        }
        this.minValidIndex.set(newMinValidIndex);
        long oldMaxValidIndex = this.maxValidIndex.getAndSet(newMaxValidIndex);
        this.size.updateAndGet(v -> Math.min(v, newMaxValidIndex + 1L));
        this.shrinkLeftSideIfNeeded(newMinValidIndex);
        this.shrinkRightSideIfNeeded(oldMaxValidIndex, newMaxValidIndex);
    }

    @Override
    public long getMinValidIndex() {
        return this.minValidIndex.get();
    }

    @Override
    public long getMaxValidIndex() {
        return this.maxValidIndex.get();
    }

    protected C createOrGetChunk(long newIndex) {
        this.size.getAndUpdate(oldSize -> newIndex >= oldSize ? newIndex + 1L : oldSize);
        int chunkIndex = Math.toIntExact(newIndex / (long)this.longsPerChunk);
        C result = this.chunkList.get(chunkIndex);
        if (result == null) {
            C newChunk = this.createChunk();
            C oldChunk = this.chunkList.compareAndExchange(chunkIndex, null, newChunk);
            if (oldChunk == null) {
                return newChunk;
            }
            this.closeChunk(newChunk);
            return oldChunk;
        }
        return result;
    }

    private void shrinkLeftSideIfNeeded(long newMinValidIndex) {
        int firstChunkIndexToDelete;
        int firstValidChunkWithBuffer = Math.toIntExact(Math.max((newMinValidIndex - this.reservedBufferSize) / (long)this.longsPerChunk, 0L));
        for (int i = firstChunkIndexToDelete = firstValidChunkWithBuffer - 1; i >= 0; --i) {
            C chunk = this.chunkList.get(i);
            if (chunk == null || !this.chunkList.compareAndSet(i, chunk, null)) continue;
            this.closeChunk(chunk);
        }
        int firstChunkWithDataIndex = Math.toIntExact(newMinValidIndex / (long)this.longsPerChunk);
        long numberOfElementsToCleanUp = newMinValidIndex % (long)this.longsPerChunk;
        C chunk = this.chunkList.get(firstChunkWithDataIndex);
        if (chunk != null && numberOfElementsToCleanUp > 0L) {
            this.partialChunkCleanup(chunk, true, numberOfElementsToCleanUp);
        }
        for (int i = firstValidChunkWithBuffer; i < firstChunkWithDataIndex; ++i) {
            chunk = this.chunkList.get(i);
            if (chunk == null) continue;
            this.partialChunkCleanup(chunk, true, this.longsPerChunk);
        }
    }

    private void shrinkRightSideIfNeeded(long oldMaxValidIndex, long newMaxValidIndex) {
        int listLength = this.chunkList.length();
        int lastValidChunkWithBufferIndex = Math.toIntExact(Math.min((newMaxValidIndex + this.reservedBufferSize) / (long)this.longsPerChunk, (long)(listLength - 1)));
        int firstChunkIndexToDelete = lastValidChunkWithBufferIndex + 1;
        int numberOfChunks = this.calculateNumberOfChunks(oldMaxValidIndex);
        for (int i = firstChunkIndexToDelete; i < numberOfChunks; ++i) {
            C chunk = this.chunkList.get(i);
            if (chunk == null || !this.chunkList.compareAndSet(i, chunk, null)) continue;
            this.closeChunk(chunk);
        }
        int firstChunkWithDataIndex = Math.toIntExact(newMaxValidIndex / (long)this.longsPerChunk);
        long numberOfEntriesToCleanUp = (long)this.longsPerChunk - newMaxValidIndex % (long)this.longsPerChunk - 1L;
        C chunk = this.chunkList.get(firstChunkWithDataIndex);
        if (chunk != null && numberOfEntriesToCleanUp > 0L) {
            this.partialChunkCleanup(chunk, false, numberOfEntriesToCleanUp);
        }
        for (int i = firstChunkWithDataIndex + 1; i <= lastValidChunkWithBufferIndex; ++i) {
            chunk = this.chunkList.get(i);
            if (chunk == null) continue;
            this.partialChunkCleanup(chunk, false, this.longsPerChunk);
        }
    }

    protected abstract void partialChunkCleanup(@NonNull C var1, boolean var2, long var3);

    protected abstract C createChunk();

    protected void closeChunk(@NonNull C chunk) {
    }

    protected int calculateNumberOfChunks(long totalNumberOfElements) {
        int chunkCount = Math.toIntExact((totalNumberOfElements - 1L) / (long)this.longsPerChunk + 1L);
        this.checkChunkCount(chunkCount);
        return chunkCount;
    }

    private void checkIndex(long index) {
        if (index < 0L || index >= this.capacity) {
            throw new IndexOutOfBoundsException("Index " + index + " is out-of-bounds given capacity " + this.capacity);
        }
    }

    private void checkValue(long value) {
        if (value == 0L) {
            throw new IllegalArgumentException("Cannot put 0 into a LongList");
        }
    }

    @Override
    public <T extends Throwable> boolean forEach(@NonNull CASableLongIndex.LongAction<T> action, @Nullable BooleanSupplier cond) throws InterruptedException, T {
        long i;
        long max = this.maxValidIndex.get();
        if (max < 0L) {
            return true;
        }
        Objects.requireNonNull(action);
        BooleanSupplier condition = cond != null ? cond : () -> true;
        for (i = this.minValidIndex.get(); i <= max && condition.getAsBoolean(); ++i) {
            long value = this.get(i);
            if (value == 0L) continue;
            action.handle(i, value);
        }
        return i == max + 1L;
    }

    List<C> dataCopy() {
        ArrayList<C> result = new ArrayList<C>();
        for (int i = 0; i < this.chunkList.length(); ++i) {
            result.add(this.chunkList.get(i));
        }
        return result;
    }

    @Override
    public void close() {
        this.size.set(0L);
        for (int i = 0; i < this.chunkList.length(); ++i) {
            C chunk = this.chunkList.getAndSet(i, null);
            if (chunk == null) continue;
            this.closeChunk(chunk);
        }
    }

    private void checkCapacity(long capacity) {
        if (capacity < 0L) {
            throw new IllegalArgumentException("The maximum number of longs must be non-negative, not " + capacity);
        }
    }

    private void checkLongsPerChunk(long longsPerChunk) {
        if (longsPerChunk <= 0L) {
            throw new IllegalArgumentException(CHUNK_SIZE_ZERO_OR_NEGATIVE_MSG.formatted(longsPerChunk));
        }
        if (longsPerChunk > (long)MAX_NUM_LONGS_PER_CHUNK) {
            throw new IllegalArgumentException(CHUNK_SIZE_EXCEEDED_MSG.formatted(longsPerChunk, MAX_NUM_LONGS_PER_CHUNK));
        }
    }

    private void checkChunkCount(int chunkCount) {
        if (chunkCount > 0x200000) {
            throw new IllegalArgumentException(MAX_CHUNKS_EXCEEDED_MSG.formatted(0x200000));
        }
    }
}

