/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.platform.testreader;

import com.swirlds.common.formatting.HorizontalAlignment;
import com.swirlds.platform.testreader.JrsReportData;
import com.swirlds.platform.testreader.JrsTestIdentifier;
import com.swirlds.platform.testreader.JrsTestMetadata;
import com.swirlds.platform.testreader.JrsTestResult;
import com.swirlds.platform.testreader.TestStatus;
import com.swirlds.platform.util.CommandResult;
import com.swirlds.platform.util.VirtualTerminal;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import org.hiero.base.CompareTo;
import org.hiero.base.concurrent.interrupt.Uninterruptable;

public final class JrsTestReader {
    private JrsTestReader() {
    }

    @Nullable
    public static Instant parseTimestampFromDirectory(@NonNull String timestampString) {
        String[] parts = timestampString.split("-");
        if (parts.length < 2 || parts[0].length() != 8 || parts[1].length() != 6) {
            return null;
        }
        try {
            int year = Integer.parseInt(parts[0].substring(0, 4));
            int month = Integer.parseInt(parts[0].substring(4, 6));
            int day = Integer.parseInt(parts[0].substring(6, 8));
            int hour = Integer.parseInt(parts[1].substring(0, 2));
            int minute = Integer.parseInt(parts[1].substring(2, 4));
            int second = Integer.parseInt(parts[1].substring(4, 6));
            String instantString = year + "-" + HorizontalAlignment.ALIGNED_RIGHT.pad(Integer.toString(month), '0', 2, false) + "-" + HorizontalAlignment.ALIGNED_RIGHT.pad(Integer.toString(day), '0', 2, false) + "T" + HorizontalAlignment.ALIGNED_RIGHT.pad(Integer.toString(hour), '0', 2, false) + ":" + HorizontalAlignment.ALIGNED_RIGHT.pad(Integer.toString(minute), '0', 2, false) + ":" + HorizontalAlignment.ALIGNED_RIGHT.pad(Integer.toString(second), '0', 2, false) + "Z";
            return Instant.parse(instantString);
        }
        catch (NumberFormatException | DateTimeParseException e) {
            return null;
        }
    }

    @Nullable
    public static Instant parseTimestampFromPath(@NonNull String pathString) {
        String[] directories;
        for (String directory : directories = pathString.split("/")) {
            Instant instant = JrsTestReader.parseTimestampFromDirectory(directory);
            if (instant == null) continue;
            return instant;
        }
        return null;
    }

    @NonNull
    public static List<String> lsRemoteDir(@NonNull VirtualTerminal terminal, @NonNull String remoteDir) {
        CommandResult result = terminal.run("gsutil", "ls", remoteDir);
        if (!result.isSuccessful()) {
            throw new RuntimeException("Failed to list remote directory: " + remoteDir);
        }
        String[] directories = result.out().split("\n");
        return List.of(directories);
    }

    public static List<String> findTestPanelDirectories(@NonNull VirtualTerminal terminal, @NonNull ExecutorService executorService, @NonNull String rootDirectory, @NonNull Instant now, @NonNull Duration maximumAge) {
        terminal.getProgressIndicator().writeMessage("Searching for panel directories.");
        LinkedBlockingQueue<String> directoriesToExplore = new LinkedBlockingQueue<String>();
        directoriesToExplore.add(rootDirectory);
        LinkedBlockingQueue directoriesWithTimestamps = new LinkedBlockingQueue();
        while (!directoriesToExplore.isEmpty()) {
            CountDownLatch latch = new CountDownLatch(directoriesToExplore.size());
            while (!directoriesToExplore.isEmpty()) {
                String next = (String)directoriesToExplore.remove();
                executorService.submit(() -> {
                    Instant timestamp = JrsTestReader.parseTimestampFromPath(next);
                    if (timestamp == null) {
                        List<String> subDirectories = JrsTestReader.lsRemoteDir(terminal, next);
                        directoriesToExplore.addAll(subDirectories);
                    } else {
                        Duration age = Duration.between(timestamp, now);
                        if (CompareTo.isGreaterThan((Comparable)age, (Object)maximumAge)) {
                            latch.countDown();
                            return;
                        }
                        directoriesWithTimestamps.add(next);
                    }
                    latch.countDown();
                });
            }
            Uninterruptable.abortAndThrowIfInterrupted(latch::await, (String)"interrupted while waiting for directory search to complete");
        }
        terminal.getProgressIndicator().writeMessage("Found " + directoriesWithTimestamps.size() + " panel directories.");
        ArrayList<String> dirList = new ArrayList<String>(directoriesWithTimestamps.size());
        dirList.addAll(directoriesWithTimestamps);
        return dirList;
    }

    public static List<String> findTestDirectories(@NonNull VirtualTerminal terminal, @NonNull ExecutorService executorService, @NonNull String rootDirectory, @NonNull Instant now, @NonNull Duration maximumAge) {
        List<String> panelDirectories = JrsTestReader.findTestPanelDirectories(terminal, executorService, rootDirectory, now, maximumAge);
        terminal.getProgressIndicator().writeMessage("Searching for test directories.");
        LinkedBlockingDeque testDirectories = new LinkedBlockingDeque();
        CountDownLatch latch = new CountDownLatch(panelDirectories.size());
        for (String panelDirectory : panelDirectories) {
            executorService.submit(() -> {
                List<String> panelContents = JrsTestReader.lsRemoteDir(terminal, panelDirectory);
                for (String potentialTestDirectory : panelContents) {
                    if (potentialTestDirectory.endsWith(".log")) continue;
                    testDirectories.add(potentialTestDirectory);
                }
                latch.countDown();
            });
        }
        Uninterruptable.abortAndThrowIfInterrupted(latch::await, (String)"interrupted while waiting for directory search to complete");
        terminal.getProgressIndicator().writeMessage("Found " + testDirectories.size() + " test directories.");
        ArrayList<String> dirList = new ArrayList<String>(testDirectories.size());
        dirList.addAll(testDirectories);
        return dirList;
    }

    @NonNull
    public static JrsReportData findTestResults(@NonNull VirtualTerminal terminal, @NonNull ExecutorService executorService, @NonNull String rootDirectory, @NonNull Instant now, @NonNull Duration maximumAge) {
        List<String> testDirectories = JrsTestReader.findTestDirectories(terminal, executorService, rootDirectory, now, maximumAge);
        terminal.getProgressIndicator().writeMessage("Scanning tests for data.");
        LinkedBlockingQueue testResults = new LinkedBlockingQueue();
        CountDownLatch latch = new CountDownLatch(testDirectories.size());
        for (String testDirectory : testDirectories) {
            Runnable task = () -> {
                String[] parts;
                List<String> testFiles = JrsTestReader.lsRemoteDir(terminal, testDirectory);
                TestStatus status = TestStatus.UNKNOWN;
                for (String testFile : testFiles) {
                    if (testFile.endsWith("test-passed")) {
                        status = TestStatus.PASS;
                        break;
                    }
                    if (!testFile.endsWith("test-failed")) continue;
                    status = TestStatus.FAIL;
                    break;
                }
                if ((parts = testDirectory.split("/")).length < 3) {
                    System.out.println("Invalid test directory structure");
                    latch.countDown();
                    return;
                }
                String timestampString = parts[parts.length - 2];
                Instant timestamp = JrsTestReader.parseTimestampFromDirectory(timestampString);
                if (timestamp == null) {
                    System.out.println("Unable to parse timestamp from string: " + testDirectory);
                    latch.countDown();
                    return;
                }
                String testName = parts[parts.length - 1];
                String panelName = parts[parts.length - 3];
                JrsTestIdentifier id = new JrsTestIdentifier(panelName, testName);
                JrsTestResult result = new JrsTestResult(id, status, timestamp, testDirectory);
                testResults.add(result);
                latch.countDown();
            };
            executorService.submit(task);
        }
        Uninterruptable.abortAndThrowIfInterrupted(latch::await, (String)"interrupted while waiting for test search to complete");
        terminal.getProgressIndicator().writeMessage("Found results for " + testResults.size() + " tests.");
        return new JrsReportData(rootDirectory, now, (int)maximumAge.toDays(), new ArrayList<JrsTestResult>(testResults));
    }

    @NonNull
    public static Map<JrsTestIdentifier, JrsTestMetadata> parseMetadataFile(@Nullable Path notesFile) {
        HashMap<JrsTestIdentifier, JrsTestMetadata> metadata = new HashMap<JrsTestIdentifier, JrsTestMetadata>();
        if (notesFile == null) {
            return metadata;
        }
        try (BufferedReader reader = Files.newBufferedReader(notesFile);){
            String line = reader.readLine();
            while (line != null) {
                if (!line.isEmpty() && !line.strip().startsWith("#")) {
                    String[] parts = line.split(",");
                    if (parts.length != 4) {
                        System.out.println("Invalid line in notes file: " + line);
                    } else {
                        String url;
                        String testOwner;
                        String testName;
                        String panel = parts[0].strip();
                        JrsTestIdentifier id = new JrsTestIdentifier(panel, testName = parts[1].strip());
                        JrsTestMetadata previous = metadata.put(id, new JrsTestMetadata(testOwner = parts[2].strip(), url = parts[3]));
                        if (previous != null) {
                            System.out.println("Duplicate note URL found for " + String.valueOf(id));
                        }
                    }
                }
                line = reader.readLine();
            }
        }
        catch (IOException e) {
            System.out.println("Unable to parse notes file " + String.valueOf(notesFile));
            e.printStackTrace();
        }
        return metadata;
    }

    public static void saveTestResults(@NonNull JrsReportData reportData, @NonNull Path resultsFile) {
        StringBuilder sb = new StringBuilder();
        sb.append("# report directory, report timestamp, time span in days\n");
        sb.append(reportData.directory()).append(",").append(reportData.reportTime()).append(",").append(reportData.reportSpan()).append("\n");
        sb.append("# panel, test name, status, timestamp, test directory\n");
        for (JrsTestResult result : reportData.testResults()) {
            sb.append(result.toCsvLine()).append("\n");
        }
        String data = sb.toString();
        try {
            Files.write(resultsFile, data.getBytes(), new OpenOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException("unable to generate results csv", e);
        }
    }

    @NonNull
    public static JrsReportData loadTestResults(@NonNull Path resultsFile) {
        String data;
        try {
            data = Files.readString(resultsFile);
        }
        catch (IOException e) {
            throw new UncheckedIOException("unable to read results csv", e);
        }
        boolean firstLine = true;
        String testDirectory = "?";
        Instant testTimestamp = null;
        int testSpan = -1;
        String[] lines = data.split("\n");
        ArrayList<JrsTestResult> results = new ArrayList<JrsTestResult>(lines.length);
        for (String line : lines) {
            if (line.isEmpty() || line.strip().startsWith("#")) continue;
            if (firstLine) {
                firstLine = false;
                String[] parts = line.split(",");
                if (parts.length != 3) {
                    System.out.println("Invalid first line in data file: " + line);
                    continue;
                }
                testDirectory = parts[0].strip();
                try {
                    testTimestamp = Instant.parse(parts[1].strip());
                    testSpan = Integer.parseInt(parts[2].strip());
                }
                catch (NumberFormatException | DateTimeParseException e) {
                    System.out.println("Invalid first line in data file: " + line);
                }
                continue;
            }
            JrsTestResult result = JrsTestResult.parseFromCsvLine(line);
            if (result == null) {
                System.out.println("Unable to parse line: " + line);
                continue;
            }
            results.add(result);
        }
        testTimestamp = testTimestamp == null ? Instant.now() : testTimestamp;
        return new JrsReportData(testDirectory, testTimestamp, testSpan, results);
    }
}

